/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <stdlib.h>

#include <gtk/gtkgl.h>
#include <GL/gl.h>
#include <GL/glext.h>

#include "config.h"

#include "glarea.h"
#include "disk.h"
#include "ww_texture.h"
#include "util.h"


typedef struct _WwTexturePrivate WwTexturePrivate;
struct _WwTexturePrivate {
	gboolean dispose_has_run;

	guint t_num;
};

#define WW_TEXTURE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), WW_TYPE_TEXTURE, WwTexturePrivate))

static GObjectClass *parent_class = NULL;

static void ww_texture_class_init (WwTextureClass * klass);
static void ww_texture_init (WwTexture * object);
static void ww_texture_finalize (GObject * object);
static void ww_texture_dispose (GObject * object);
static GObject *ww_texture_constructor (GType type, guint n_props, GObjectConstructParam * props);

/*****************************************************************************/


/*****************************************************************************/

GType
ww_texture_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (WwTextureClass),
			NULL,				/* base_init */
			NULL,				/* base_finalize */
			(GClassInitFunc) ww_texture_class_init,
			NULL,				/* class_finalize */
			NULL,				/* class_data */
			sizeof (WwTexture),
			0,					/* n_preallocs */
			(GInstanceInitFunc) ww_texture_init
		};

		type = g_type_register_static (G_TYPE_OBJECT, "WwTexture", &info, 0);
	}

	return type;
}

static void
ww_texture_class_init (WwTextureClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	g_print ("ww_texture_class_init:c:%p:\n", klass);

	g_type_class_add_private (klass, sizeof (WwTexturePrivate));

	parent_class = g_type_class_peek_parent (klass);
	object_class->constructor = ww_texture_constructor;
	object_class->finalize = ww_texture_finalize;
	object_class->dispose = ww_texture_dispose;

}

static void
ww_texture_init (WwTexture * self)
{
	WwTexturePrivate *priv = WW_TEXTURE_GET_PRIVATE (self);

	/*
	   g_print ("ww_texture_init:o:%p:\n", self);
	 */

	priv->dispose_has_run = FALSE;
	priv->t_num = 0;
}

static void
ww_texture_dispose (GObject * obj)
{
	WwTexture *self = WW_TEXTURE (obj);
	WwTexturePrivate *priv = WW_TEXTURE_GET_PRIVATE (self);
	/*
	   g_print ("ww_texture_dispose\n");
	 */
	if (priv->dispose_has_run) {
		return;
	}
	priv->dispose_has_run = TRUE;
	/*
	   g_object_unref(self->***);
	 */
	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void
ww_texture_finalize (GObject * obj)
{
	WwTexture *self = WW_TEXTURE (obj);
	WwTexturePrivate *priv = WW_TEXTURE_GET_PRIVATE (self);
	/*
	   g_print ("ww_texture_finalize\n");
	 */

	glarea_util_texture_free (priv->t_num);

	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static GObject *
ww_texture_constructor (GType type, guint n_props, GObjectConstructParam * props)
{
	GObject *object;
	GObjectClass *object_class = G_OBJECT_CLASS (parent_class);
	/*
	   g_print ("ww_texture_constructor\n");
	 */
	object = object_class->constructor (type, n_props, props);

	return object;
}

WwTexture *
ww_texture_new (void)
{
	GObject *object;

	object = g_object_new (WW_TYPE_TEXTURE, NULL);
	/*
	   g_print ("ww_texture_new:o:%p\n", object);
	 */
	return WW_TEXTURE (object);
}

/********************************************************************/

static guint
texture_compress_to_t_num (const gchar * data, gint w, gint h, gint size)
{
	guint num;
	gint texture_format;

	/*
	   g_print ("texture_compress_to_t_num\n");
	 */

	glarea_dummy_gl_begin ();

	glGenTextures (1, &num);
	glBindTexture (GL_TEXTURE_2D, num);
	/*
	   glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	   glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
	 */
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);

	texture_format = glarea_get_texture_format ();

	glCompressedTexImage2D (GL_TEXTURE_2D, 0, texture_format, w, h, 0, size, data);

	glarea_dummy_gl_end ();

	return num;
}


static guint
texture_pixels_to_t_num (const guchar * pixels, gint w, gint h, const gchar * savepath)
{
	guint num;

	glarea_dummy_gl_begin ();

	glGenTextures (1, &num);
	glBindTexture (GL_TEXTURE_2D, num);
	/*
	   glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	   glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
	 */
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);

	if (glarea_is_support_compress () == TRUE && glarea_is_direct () == TRUE) {
		gint bytes;
		gint format;

		/*
		   g_print ("before_compress\n");
		 */
		glTexImage2D (GL_TEXTURE_2D, 0, GL_COMPRESSED_RGB, w, h, 0, GL_RGB, GL_UNSIGNED_BYTE, pixels);
		/*
		   glTexImage2D (GL_TEXTURE_2D, 0, GL_RGB, w, h, 0, GL_RGB, GL_UNSIGNED_BYTE, pixels);
		   g_print ("after_compress\n");
		 */

		if (savepath != NULL) {
			gchar *tex = NULL;

			glGetTexLevelParameteriv (GL_TEXTURE_2D, 0, GL_TEXTURE_COMPRESSED_IMAGE_SIZE, &bytes);
			glGetTexLevelParameteriv (GL_TEXTURE_2D, 0, GL_TEXTURE_INTERNAL_FORMAT, &format);
			/*
			   g_print ("internal_format:%x compressed_texture_size:%d\n", format, bytes);
			 */
			tex = g_new (gchar, bytes);
			glGetCompressedTexImage (GL_TEXTURE_2D, 0, tex);

			disk_save_bin (savepath, tex, bytes);

			g_free (tex);
		}
	} else {
		g_print ("before glteximage2d\n");
		glTexImage2D (GL_TEXTURE_2D, 0, GL_RGB, w, h, 0, GL_RGB, GL_UNSIGNED_BYTE, pixels);
	}

	glarea_dummy_gl_end ();

	return num;
}

static guint
texture_pixbuf_to_t_num (GdkPixbuf * pixbuf, const gchar * savepath)
{
	guint num;
	guchar *pixels;
	gint w, h;
	/*
	   gint rowstride;
	   gint size;
	 */

	if (pixbuf == NULL) {
		return 0;
	}

	w = gdk_pixbuf_get_width (pixbuf);
	h = gdk_pixbuf_get_height (pixbuf);
	g_print ("texture_pixbuf_to_t_num:w:%d h:%d\n", w, h);
	/*
	   rowstride = gdk_pixbuf_get_rowstride (pixbuf);
	   size = rowstride * h;
	 */
	pixels = gdk_pixbuf_get_pixels (pixbuf);

	num = texture_pixels_to_t_num (pixels, w, h, savepath);

	return num;
}

WwTexture *
ww_texture_new_dds (const gchar * path_dds, gint width, gint height)
{
	WwTexture *texture;
	WwTexturePrivate *priv;
	gchar *data;
	gint size;
	/*
	   g_print ("ww_texture_new_dds:dds:%s\n", path_dds);
	 */
	texture = ww_texture_new ();
	priv = WW_TEXTURE_GET_PRIVATE (texture);

	data = disk_load_bin (path_dds, &size);
	priv->t_num = texture_compress_to_t_num (data, width, height, size);
	g_free (data);

	return texture;
}

WwTexture *
ww_texture_new_jpg (const gchar * path_jpg, const gchar * path_dds_save)
{
	WwTexture *texture;
	WwTexturePrivate *priv;
	GdkPixbuf *pixbuf;
	/*
	   g_print ("ww_texture_new_jpg:jpg:%s dds:%s\n", path_jpg, path_dds_save);
	 */
	texture = ww_texture_new ();
	priv = WW_TEXTURE_GET_PRIVATE (texture);

	pixbuf = disk_load_jpg (path_jpg);
	/* テクスチャーを作って ddspath で圧縮済テクスチャーを保存 */
	priv->t_num = texture_pixbuf_to_t_num (pixbuf, path_dds_save);
	g_object_unref (pixbuf);

	return texture;
}


WwTexture *
ww_texture_new_pixbuf (GdkPixbuf * pixbuf, const gchar * path_dds_save)
{
	WwTexture *texture;
	WwTexturePrivate *priv;

	texture = ww_texture_new ();
	priv = WW_TEXTURE_GET_PRIVATE (texture);

	priv->t_num = texture_pixbuf_to_t_num (pixbuf, path_dds_save);

	return texture;
}

WwTexture *
ww_texture_new_data (const gchar * data, gint size, const gchar * path_jpg_save, const gchar * path_dds_save)
{
	WwTexture *texture;
	WwTexturePrivate *priv;

	texture = ww_texture_new ();
	priv = WW_TEXTURE_GET_PRIVATE (texture);

	if (data != NULL && size > 10) {	/* サイズが 1 byte のデータを受ける時がある。たぶんネットワークからのデータの取得
										   部分でプログラムを間違っている・・。要修正。 */
		GdkPixbuf *pixbuf;

		disk_save_bin (path_jpg_save, (gchar *) data, size);
		pixbuf = util_bin_to_pixbuf (data, size);

		priv->t_num = texture_pixbuf_to_t_num (pixbuf, path_dds_save);
		g_object_unref (pixbuf);
	} else {
		priv->t_num = 0;
	}

	return texture;
}

WwTexture *
ww_texture_copy (WwTexture * texture)
{
	g_object_ref (texture);

	return texture;
}

gboolean
ww_texture_bind (WwTexture * texture)
{
	WwTexturePrivate *priv;

	if (texture == NULL) {
		glBindTexture (GL_TEXTURE_2D, 0);

		return FALSE;
	}


	priv = WW_TEXTURE_GET_PRIVATE (texture);

	if (priv->t_num == 0) {
		return FALSE;

	}

	glBindTexture (GL_TEXTURE_2D, priv->t_num);

	return TRUE;
}
