/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <stdlib.h>
#include <stdio.h>
#include <math.h>


#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <gtk/gtkgl.h>
#include <gdk/x11/gdkglx.h>

/*
#include <GL/glext.h>
*/

#include "glarea.h"


gboolean
saveimage_check_glx (GtkWidget * glarea)
{
	GdkGLConfig *glconfig;
	const gchar ext_pbuffer[] = "GLX_SGIX_pbuffer";
	const gchar ext_fbconfig[] = "GLX_SGIX_fbconfig";
	gboolean bool_pbuffer;
	gboolean bool_fbconfig;

	glconfig = gtk_widget_get_gl_config (glarea);

	bool_pbuffer = gdk_x11_gl_query_glx_extension (glconfig, ext_pbuffer);
	bool_fbconfig = gdk_x11_gl_query_glx_extension (glconfig, ext_fbconfig);

	if (bool_pbuffer && bool_fbconfig) {
		g_print ("pbuffer is supported\n");
	}

	return (bool_pbuffer && bool_fbconfig);
}

static guchar *
reverse_pixels (const guchar * pixels, gint w, gint h)
{
	guchar *ret;
	gint x, y;

	DB (g_print ("reverse_pixels:w:%d h:%d\n", w, h));

	ret = g_new (guchar, w * h * 3);
	for (y = 0; y < h; ++y) {
		for (x = 0; x < w; ++x) {
			guchar *src;
			guchar *dest;

			dest = ret + (w * y + x) * 3;
			src = (guchar *) pixels + (w * (h - y - 1) + x) * 3;

			dest[0] = src[0];
			dest[1] = src[1];
			dest[2] = src[2];
		}
	}
	return ret;
}

static void
write_png (gint w, gint h, const gchar * filename)
{
	guchar *image;
	guchar *image_reverse;
	GdkPixbuf *pixbuf = NULL;
	GError *err = NULL;
	gchar *filename_local;

	image = g_malloc0 (w * h * 3 * sizeof (guchar));
	glPixelStorei (GL_PACK_ALIGNMENT, 1);
	glReadPixels (0, 0, w, h, GL_RGB, GL_UNSIGNED_BYTE, image);
	image_reverse = reverse_pixels (image, w, h);
	pixbuf = gdk_pixbuf_new_from_data (image_reverse, GDK_COLORSPACE_RGB, FALSE, 8, w, h, w * 3, NULL, NULL);
	/*
	   {
	   gint bit_per_sample;
	   gint width;
	   gint height;
	   gint rowstride;

	   if (pixbuf == NULL) {
	   g_print ("pixbuf is null\n");
	   }
	   bit_per_sample = gdk_pixbuf_get_bits_per_sample (pixbuf);
	   width = gdk_pixbuf_get_width (pixbuf);
	   height = gdk_pixbuf_get_height (pixbuf);
	   rowstride = gdk_pixbuf_get_rowstride (pixbuf);
	   g_print ("bit_per_sample:%d\n", bit_per_sample);
	   g_print ("w:%d\n", width);
	   g_print ("h:%d\n", height);
	   g_print ("rowstride:%d\n", rowstride);

	   if (gdk_pixbuf_get_has_alpha (pixbuf) == FALSE) {
	   g_print ("pixbuf dont has alpha\n");
	   }
	   }
	 */
	filename_local = g_locale_from_utf8 (filename, -1, NULL, NULL, NULL);
	if (filename_local == NULL) {
		/* gtkfilechooser の出力が環境変数によってかわるので、とりあえず UTF-8 の変換
		 * をして成功するかしないかで UTF-8 かどうかを見分ける。
		 * g_utf8_validate() は、一部の EUC-JP の文字("姫"とか)に対して TRUE を返す(?)ので
		 * 使えない。
		 */
		filename_local = g_strdup (filename);
	}
	printf ("printf:%s:\n", filename_local);
	gdk_pixbuf_save (pixbuf, filename_local, "png", &err, NULL);
	g_free (filename_local);
	if (err != NULL) {
		g_print ("pixbuf_save error:%s:\n", err->message);
	}

	g_object_unref (pixbuf);
	g_free (image);
	g_free (image_reverse);
}


static void
output_png_real (GtkWidget * draw, const gchar * filename)
{
	GdkGLConfig *glconfig;

	GdkGL_GLX_SGIX_fbconfig *fbc = NULL;
	GdkGL_GLX_SGIX_pbuffer *pb = NULL;

	Display *xdisplay;
	XVisualInfo *xvinfo;

	GLXFBConfigSGIX fbconfig;
	GLXPbufferSGIX pbuffer;

	int pb_attrib_list[] = { GLX_LARGEST_PBUFFER_SGIX, True,
		GLX_PRESERVED_CONTENTS_SGIX, False,
		None
	};

	GdkGLContext *glcontext;
	GLXContext glxcontext;
	gint w, h;
	gint w3, h3;

	g_print ("output_png_real:%s:\n", filename);

	w = draw->allocation.width;
	h = draw->allocation.height;
	w3 = w * 3;
	h3 = h * 3;

	glconfig = gtk_widget_get_gl_config (draw);

	fbc = gdk_gl_get_GLX_SGIX_fbconfig (glconfig);
	pb = gdk_gl_get_GLX_SGIX_pbuffer (glconfig);
	if (!fbc || !pb) {
		g_print ("GLX extension is not supported.\n");
		exit (1);
	}

	xdisplay = gdk_x11_gl_config_get_xdisplay (glconfig);
	xvinfo = gdk_x11_gl_config_get_xvinfo (glconfig);
	fbconfig = fbc->glXGetFBConfigFromVisualSGIX (xdisplay, xvinfo);
	if (!fbconfig) {
		g_print ("cannot get FBConfig\n");
		return;
	}
	pbuffer = pb->glXCreateGLXPbufferSGIX (xdisplay, fbconfig, w3, h3, pb_attrib_list);
	if (!pbuffer) {
		g_print ("cannot create GLXPbuffer\n");
		return;
	}

	glcontext = gtk_widget_get_gl_context (draw);
	glxcontext = gdk_x11_gl_context_get_glxcontext (glcontext);

	glXMakeCurrent (xdisplay, pbuffer, glxcontext);

	glViewport (0, 0, w3, h3);
	glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glarea_render_file ();
	glFlush ();

	write_png (w3, h3, filename);

	glXMakeCurrent (xdisplay, None, NULL);


	pb->glXDestroyGLXPbufferSGIX (xdisplay, pbuffer);
}

/***********************************************************************/

static GtkWidget *label_w;
static GtkWidget *label_h;

static void
set_label_width_height (gint w, gint h)
{
	gchar *w_char, *h_char;

	w_char = g_strdup_printf ("Width  : %4d pixels", w);
	h_char = g_strdup_printf ("Height : %4d pixels", h);
	gtk_label_set_text (GTK_LABEL (label_w), w_char);
	gtk_label_set_text (GTK_LABEL (label_h), h_char);
	g_free (w_char);
	g_free (h_char);
}

static gboolean
glarea_configure_event_cb (GtkWidget * widget, GdkEventConfigure * event, gpointer user_data)
{
	gint w, h;
	/*
	   g_print("draw_configure_event_cb\n");
	 */
	w = 3 * widget->allocation.width;
	h = 3 * widget->allocation.height;

	set_label_width_height (w, h);

	return FALSE;
}

void
saveimage_dialog (GtkWidget * glarea, const gchar * filename)
{
	GtkWidget *toplevel;
	GtkWidget *dialog;
	GtkWidget *vbox;
	GtkWidget *frame;
	gint w, h;

	g_signal_connect (glarea, "configure_event", G_CALLBACK (glarea_configure_event_cb), NULL);

	toplevel = gtk_widget_get_toplevel (glarea);
	dialog = gtk_file_chooser_dialog_new ("画像の出力", GTK_WINDOW (toplevel),
										  GTK_FILE_CHOOSER_ACTION_SAVE,
										  GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
										  GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, NULL);

	gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (dialog), filename);

	w = 3 * glarea->allocation.width;
	h = 3 * glarea->allocation.height;

	label_w = gtk_label_new ("");
	label_h = gtk_label_new ("");
	set_label_width_height (w, h);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), label_w, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), label_h, FALSE, FALSE, 0);

	frame = gtk_frame_new (_("Size"));
	gtk_container_add (GTK_CONTAINER (frame), vbox);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), frame, FALSE, FALSE, 0);

	gtk_widget_show_all (frame);

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
		gchar *filename;

		/*
		 * EUC-JP の環境で G_FILENAME_ENCODIN が設定されてない時、
		 * gtk_file_chooser_get_filename() の出力は、UTF-8。
		 * G_FILENAME_ENCODIN=@locale の時、同出力は、EUC-JP。
		 */
		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));

		if (g_str_has_suffix (filename, "png") || g_str_has_suffix (filename, "PNG")) {
			;
		} else {
			gchar *tmp;

			tmp = g_strconcat (filename, ".png", NULL);
			g_free (filename);

			filename = tmp;
		}

		/*
		   printf("p filename:%s\n",filename);
		   g_print("g filename:%s\n",filename);
		 */

		output_png_real (glarea, filename);
		g_free (filename);
	}

	g_signal_handlers_disconnect_by_func (glarea, G_CALLBACK (glarea_configure_event_cb), NULL);

	gtk_widget_destroy (dialog);
}
