/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*
#define DB(x) (x)
*/
#define DB(x)

#include <gtk/gtk.h>
#include <gtk/gtkgl.h>

#include <math.h>
#include <stdlib.h>
#include <GL/gl.h>

#include "config.h"
#include "mmap.h"
#include "ww_quadtile.h"
#include "ww_quadchild.h"
#include "ww_texture.h"
#include "ww_mesh.h"
#include "camera.h"
#include "util.h"
#include "glarea.h"
#include "net.h"
#include "disk.h"
#include "mmapthread.h"
#include "ww_terrainaccessor.h"


typedef struct _WwQuadchildPrivate WwQuadchildPrivate;

struct _WwQuadchildPrivate {
	gboolean dispose_has_run;

	WwQuadchildState state;

	gint x;
	gint y;
	gint level;
	gdouble tile_deg;

	WwQuadchild *north_west;
	WwQuadchild *north_east;
	WwQuadchild *south_west;
	WwQuadchild *south_east;

	gboolean need_flag;

	gchar *path_dds;
	gchar *path_jpg;
	gchar *uri;
	gboolean texture_initialized;	/* テクスチャーがあろうがなかろうがテクスチャー処理が終了してたら TRUE */
	gboolean mesh_initialized;
	gchar *data;
	gint size;
	WwTexture *texture;
	WwMesh *mesh;
	GMutex *mutex_texture_and_mesh;

	WwQuadtile *quadtile;

	WwTerrainaccessor *ta;

	gboolean is_detached;		/* ww_quadtile から外れると（描画対象から外れると）TRUE */
};

#define WW_QUADCHILD_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), WW_TYPE_QUADCHILD, WwQuadchildPrivate))

static GObjectClass *parent_class = NULL;

static void ww_quadchild_class_init (WwQuadchildClass * klass);
static void ww_quadchild_init (WwQuadchild * object);
static void ww_quadchild_finalize (GObject * object);
static void ww_quadchild_dispose (GObject * object);
static GObject *ww_quadchild_constructor (GType type, guint n_props, GObjectConstructParam * props);

static gint SPLIT = 40;

/*****************************************************************************/


/*****************************************************************************/

GType
ww_quadchild_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (WwQuadchildClass),
			NULL,				/* base_init */
			NULL,				/* base_finalize */
			(GClassInitFunc) ww_quadchild_class_init,
			NULL,				/* class_finalize */
			NULL,				/* class_data */
			sizeof (WwQuadchild),
			0,					/* n_preallocs */
			(GInstanceInitFunc) ww_quadchild_init
		};

		type = g_type_register_static (G_TYPE_OBJECT, "WwQuadchild", &info, 0);
        /*
		g_print ("ww_quadchild_get_type\n");
        */
	}

	return type;
}

static void
ww_quadchild_class_init (WwQuadchildClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	g_print ("ww_quadchild_class_init:c:%p:\n", klass);

	g_type_class_add_private (klass, sizeof (WwQuadchildPrivate));

	parent_class = g_type_class_peek_parent (klass);

	object_class->constructor = ww_quadchild_constructor;
	object_class->finalize = ww_quadchild_finalize;
	object_class->dispose = ww_quadchild_dispose;
}

static void
ww_quadchild_init (WwQuadchild * self)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (self);

	/*
	   g_print ("ww_quadchild_init:o:%p:\n", self);
	 */

	priv->dispose_has_run = FALSE;

	priv->state = WW_QUADCHILD_STATE_ORIG;
	priv->x = 0;
	priv->y = 0;
	priv->level = 0;
	priv->tile_deg = 0.0;
	priv->north_west = NULL;
	priv->north_east = NULL;
	priv->south_west = NULL;
	priv->south_east = NULL;
	priv->need_flag = TRUE;
	priv->quadtile = NULL;

	priv->path_dds = NULL;
	priv->path_jpg = NULL;
	priv->uri = NULL;
	priv->texture_initialized = FALSE;
	priv->mesh_initialized = FALSE;
	priv->data = NULL;
	priv->size = 0;
	priv->texture = NULL;
	priv->mesh = NULL;
	priv->mutex_texture_and_mesh = g_mutex_new ();

	priv->ta = NULL;

	priv->is_detached = FALSE;
}

static void
ww_quadchild_dispose (GObject * obj)
{
	WwQuadchild *self = WW_QUADCHILD (obj);
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (self);

	/*
	   g_print ("ww_quadchild_dispose:x:%d y:%d level:%d\n", priv->x, priv->y, priv->level);
	 */

	if (priv->dispose_has_run) {
		return;
	}
	priv->dispose_has_run = TRUE;

	if (priv->north_west != NULL) {
		ww_quadchild_is_detached_from_render_list (priv->north_west);
		g_object_unref (priv->north_west);
	}
	if (priv->north_east != NULL) {
		ww_quadchild_is_detached_from_render_list (priv->north_east);
		g_object_unref (priv->north_east);
	}
	if (priv->south_west != NULL) {
		ww_quadchild_is_detached_from_render_list (priv->south_west);
		g_object_unref (priv->south_west);
	}
	if (priv->south_east != NULL) {
		ww_quadchild_is_detached_from_render_list (priv->south_east);
		g_object_unref (priv->south_east);
	}

	g_object_unref (priv->quadtile);

	if (priv->texture != NULL) {
		g_object_unref (priv->texture);
	}

	if (priv->mesh != NULL) {
		g_object_unref (priv->mesh);
	}

	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void
ww_quadchild_finalize (GObject * obj)
{
	WwQuadchild *self = WW_QUADCHILD (obj);
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (self);

	/*
	   g_print ("ww_quadchild_finalize\n");
	 */

	g_free (priv->path_dds);
	g_free (priv->path_jpg);
	g_free (priv->uri);
	if (priv->data != NULL) {
		g_free (priv->data);
	}

	g_mutex_free (priv->mutex_texture_and_mesh);

	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static GObject *
ww_quadchild_constructor (GType type, guint n_props, GObjectConstructParam * props)
{
	GObject *object;
	GObjectClass *object_class = G_OBJECT_CLASS (parent_class);
	/*
	   g_print ("ww_quadchild_constructor\n");
	 */
	object = object_class->constructor (type, n_props, props);

	return object;
}

WwQuadchild *
ww_quadchild_new (void)
{
	GObject *object;

	object = g_object_new (WW_TYPE_QUADCHILD, NULL);
	/*
	   g_print ("ww_quadchild_new:o:%p\n", object);
	 */
	return WW_QUADCHILD (object);
}

/********************************************************************/

static void
quadchild_change_texture_cb (WwTexture * texture, gpointer data)
{
	WwQuadchildPrivate *priv;
	WwQuadchild *child;

	g_return_if_fail (WW_IS_TEXTURE (texture));
	g_return_if_fail (WW_IS_QUADCHILD (data));

	child = WW_QUADCHILD (data);

	priv = WW_QUADCHILD_GET_PRIVATE (child);

	g_mutex_lock (priv->mutex_texture_and_mesh);
	{
		if (priv->texture != NULL) {
			g_object_unref (priv->texture);
		}
		priv->texture = texture;
		priv->texture_initialized = TRUE;
		ww_mesh_change_type (priv->mesh, WW_MESH_TYPE_NORMAL);
	}
	g_mutex_unlock (priv->mutex_texture_and_mesh);

	g_object_unref (child);

	glarea_force_update_and_render ();
}

static void
ww_quadchild_texture_process (GObject * obj)
{
	WwQuadchild *child = WW_QUADCHILD (obj);
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);
	WwTexture *texture;

	/*
	   g_print ("ww_quadchild_texture_process:L:%d x:%d y:%d\n", priv->level, priv->x, priv->y);
	 */

	if (priv->data == NULL) {
		/* データがないといつまでたっても initialized は FALSE */
		return;
	}

	texture = ww_texture_new_data (priv->data, priv->size, priv->path_jpg, priv->path_dds);

	g_object_ref (obj);
	g_signal_connect (G_OBJECT (texture), "initialized", G_CALLBACK (quadchild_change_texture_cb), obj);
	mmap_idle_add_single (ww_texture_gen, G_OBJECT (texture));

	g_free (priv->data);
	priv->data = NULL;

	glarea_force_update_and_render ();
}

static void
ww_quadchild_texture_download (GObject * obj)
{
	WwQuadchild *child = WW_QUADCHILD (obj);
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);
	gint size;

	/*
	   g_print("ww_quadchild_texture_download:L:%d x:%d y:%d\n", priv->level, priv->x, priv->y);
	 */

	if (priv->is_detached == TRUE) {
		g_print ("ww_quadchild_texture_download:L:%d x:%d y:%d  is_detached\n", priv->level, priv->x, priv->y);
		return;
	}

	priv->data = mmapthread_download_get (priv->uri, &size);
	priv->size = size;

	mmapthread_add_power (ww_quadchild_texture_process, obj);
}

/* 画像の出力機能を付けた際に、render() をこまかく gl_begin end  するのでなく大きく区切るようにした。
 * その影響でテクスチャーの付け替えをロック内でするとデッドロックになるので、ロック前に作って、ロック中で
 * 付け替えのみを行うようにする。
 */
static void
create_texture_sub (GObject * obj)
{
	WwQuadchild *child = WW_QUADCHILD (obj);
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);
	WwTexture *texture;

	/*
	   g_print("create_texture_sub:L:%d x:%d y:%d\n", priv->level, priv->x, priv->y);
	 */

	if (priv->is_detached == TRUE) {
		g_print ("create_texture_sub:L:%d x:%d y:%d  is_detached\n", priv->level, priv->x, priv->y);
		return;
	}

	if (filetest (priv->path_dds) == TRUE) {
		texture = ww_texture_new_dds (priv->path_dds, 512, 512);

		g_object_ref (obj);
		g_signal_connect (G_OBJECT (texture), "initialized", G_CALLBACK (quadchild_change_texture_cb), obj);
		mmap_idle_add_single (ww_texture_gen, G_OBJECT (texture));

	} else if (filetest (priv->path_jpg) == TRUE) {
		texture = ww_texture_new_jpg (priv->path_jpg, priv->path_dds);

		g_object_ref (obj);
		g_signal_connect (G_OBJECT (texture), "initialized", G_CALLBACK (quadchild_change_texture_cb), obj);
		mmap_idle_add_single (ww_texture_gen, G_OBJECT (texture));

	} else {
		mmapthread_add_download2 (priv->uri, ww_quadchild_texture_download, (GObject *) child);
	}
}


static void
create_mesh_from_data (GObject * obj)
{
	WwQuadchild *child = WW_QUADCHILD (obj);
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);
	gdouble x0, y0, x1, y1;
	WwMesh *mesh;
	WwQuadtile *tile = priv->quadtile;
	gdouble above_surface = ww_quadtile_get_above_surface (tile);

	/*
	   g_print("create_mesh_from_data\n");
	 */
	if (priv->is_detached == TRUE) {
		g_print ("mesh_download:L:%d x:%d y:%d  is_detached\n", priv->level, priv->x, priv->y);
		return;
	}

	x0 = (priv->x + 0) * priv->tile_deg - 180.0;
	x1 = (priv->x + 1) * priv->tile_deg - 180.0;
	y0 = (priv->y + 0) * priv->tile_deg - 90.0;
	y1 = (priv->y + 1) * priv->tile_deg - 90.0;
	ww_terrainaccessor_read_data (priv->ta, x0, y0, x1, y1, SPLIT, SPLIT);

	mesh = ww_mesh_new_with_terrain (x1, x0, y1, y0, SPLIT, SPLIT, above_surface, priv->ta);

	g_mutex_lock (priv->mutex_texture_and_mesh);
	{
		WwMeshType old_type;

		if (priv->mesh != NULL) {
			/* update() は mesh の完了を待ってくれないので、すでに unref されている可能性がある。 */

			old_type = ww_mesh_get_mesh_type (priv->mesh);
			g_object_unref (priv->mesh);
			priv->mesh = mesh;
			priv->mesh_initialized = TRUE;
			/* これをしないと、texture より mesh が先にできた時に、mesh だけ先行して NORMAL になり同じ texture が
			 * ４枚表示されたりする。 */
			ww_mesh_change_type (priv->mesh, old_type);
		} else {
			;
		}

	}
	g_mutex_unlock (priv->mutex_texture_and_mesh);
}

static void
mesh_download (GObject * obj)
{
	WwQuadchild *child = WW_QUADCHILD (obj);
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);
	gdouble x0, y0, x1, y1;

	/*
	   g_print("mesh_download\n");
	 */
	if (priv->is_detached == TRUE) {
		g_print ("mesh_download:L:%d x:%d y:%d  is_detached\n", priv->level, priv->x, priv->y);
		return;
	}

	x0 = (priv->x + 0) * priv->tile_deg - 180.0;
	x1 = (priv->x + 1) * priv->tile_deg - 180.0;
	y0 = (priv->y + 0) * priv->tile_deg - 90.0;
	y1 = (priv->y + 1) * priv->tile_deg - 90.0;
	ww_terrainaccessor_data_prepare (priv->ta, x0, y0, x1, y1, SPLIT, SPLIT);

	mmapthread_add_power (create_mesh_from_data, obj);
}

static WwMesh *
create_mesh_sub_copy (WwQuadtile * tile, gint x, gint y, gint level, WwMesh * mesh_parent, WwMeshType type)
{
	WwMesh *mesh = NULL;
	gdouble top_tile_deg = ww_quadtile_get_level_zero_tile_size_degrees (tile);
	gdouble tile_deg = top_tile_deg / pow (2.0, level);
	gdouble x0, y0, x1, y1;
	gdouble above_surface = ww_quadtile_get_above_surface (tile);
	gboolean terrain_mapped = ww_quadtile_get_terrain_mapped (tile);


	x0 = (x + 0) * tile_deg - 180.0;
	x1 = (x + 1) * tile_deg - 180.0;
	y0 = (y + 0) * tile_deg - 90.0;
	y1 = (y + 1) * tile_deg - 90.0;

	mesh = ww_mesh_new_copy (x1, x0, y1, y0, SPLIT, SPLIT, above_surface, terrain_mapped, mesh_parent, type);

	return mesh;
}

static void
create_mesh_sub (GObject * obj)
{
	WwQuadchild *child = WW_QUADCHILD (obj);
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);
	WwQuadtile *tile = priv->quadtile;
	gint x = priv->x;
	gint y = priv->y;
	gint level = priv->level;
	gdouble top_tile_deg = ww_quadtile_get_level_zero_tile_size_degrees (tile);
	gdouble tile_deg = top_tile_deg / pow (2.0, level);
	gdouble x0, y0, x1, y1;
	gdouble above_surface = ww_quadtile_get_above_surface (tile);
	gboolean terrain_mapped = ww_quadtile_get_terrain_mapped (tile);
	WwMesh *mesh = NULL;

	/*
	   g_print("create_mesh_sub:L:%d x:%d y:%d\n", priv->level, priv->x, priv->y);
	 */

	if (priv->is_detached == TRUE) {
		/*
		   g_print("create_mesh_sub:L:%d x:%d y:%d  is_detached\n", priv->level, priv->x, priv->y);
		 */
		return;
	}

	x0 = (x + 0) * tile_deg - 180.0;
	x1 = (x + 1) * tile_deg - 180.0;
	y0 = (y + 0) * tile_deg - 90.0;
	y1 = (y + 1) * tile_deg - 90.0;

	if (terrain_mapped == TRUE) {
		const gchar *url;

		if (priv->mesh == NULL) {
			/* 新規作成なら */
			mesh = ww_mesh_new_simple (x1, x0, y1, y0, SPLIT, SPLIT, above_surface, terrain_mapped);
			g_mutex_lock (priv->mutex_texture_and_mesh);
			{
				priv->mesh = mesh;
				priv->mesh_initialized = FALSE;
			}
			g_mutex_unlock (priv->mutex_texture_and_mesh);

		} else {
			/* TEMP を ORIG に作り替えるなら */
			g_mutex_lock (priv->mutex_texture_and_mesh);
			{
				priv->mesh_initialized = FALSE;
			}
			g_mutex_unlock (priv->mutex_texture_and_mesh);
		}

		url = ww_terrainaccessor_get_server_url (priv->ta);

		mmapthread_add_download2 (url, mesh_download, obj);
	} else {
		if (priv->mesh == NULL) {
			/* 新規作成なら */
			mesh = ww_mesh_new_simple (x1, x0, y1, y0, SPLIT, SPLIT, above_surface, terrain_mapped);
			g_mutex_lock (priv->mutex_texture_and_mesh);
			{
				priv->mesh = mesh;
				priv->mesh_initialized = TRUE;
			}
			g_mutex_unlock (priv->mutex_texture_and_mesh);
		} else {
			/* TEMP を ORIG に作り替えるなら */
			g_mutex_lock (priv->mutex_texture_and_mesh);
			{
				priv->mesh_initialized = TRUE;
			}
			g_mutex_unlock (priv->mutex_texture_and_mesh);
		}
	}

	glarea_force_update_and_render ();
}

WwQuadchild *
ww_quadchild_new_child (WwQuadtile * tile, gint x, gint y, gint level, gdouble tile_deg, WwMeshType type,
						WwTexture * texture, WwMesh * mesh)
{
	WwQuadchild *child;
	WwQuadchildPrivate *priv;
	gchar *path;
	const gchar *server_url = ww_quadtile_get_server_url (tile);
	const gchar *data_set_name = ww_quadtile_get_data_set_name (tile);
	const gchar *world_name = mmap_get_world_name ();
	const gchar *ext = ww_quadtile_get_image_file_extension (tile);

	/*
	   g_print ("ww_quadchild_new_child:x:%d y:%d level:%d\n", x, y, level);
	 */

	child = ww_quadchild_new ();
	priv = WW_QUADCHILD_GET_PRIVATE (child);
	priv->state = WW_QUADCHILD_STATE_TEMP;	/* まずは流用したテクスチャーとメッシュにして、次のupdate で正式版にするか決める */
	priv->x = x;
	priv->y = y;
	priv->level = level;
	priv->tile_deg = tile_deg;
	priv->quadtile = tile;
	g_object_ref (tile);

	path = ww_object_get_path (WW_OBJECT (tile));
	priv->path_dds = g_strdup_printf ("%s%s/%s%d/%04d/%04d_%04d.dds", mmap_dir_cache, world_name, path, level, y, y, x);
	priv->path_jpg =
		g_strdup_printf ("%s%s/%s%d/%04d/%04d_%04d.%s", mmap_dir_cache, world_name, path, level, y, y, x, ext);
	priv->uri = g_strdup_printf ("%s?T=%s&L=%d&X=%d&Y=%d", server_url, data_set_name, level, x, y);

	priv->ta = mmap_get_terrainaccessor (x * tile_deg, y * tile_deg, (x + 1) * tile_deg, (y + 1) * tile_deg);

	priv->texture = ww_texture_copy (texture);
	priv->texture_initialized = TRUE;

	priv->mesh = create_mesh_sub_copy (tile, x, y, level, mesh, type);
	priv->mesh_initialized = TRUE;

	g_free (path);

	return child;
}

WwQuadchild *
ww_quadchild_new_root (WwQuadtile * tile, gint x, gint y, gint level, gdouble tile_deg)
{
	WwQuadchild *child;
	WwQuadchildPrivate *priv;
	gchar *path;
	const gchar *server_url = ww_quadtile_get_server_url (tile);
	const gchar *data_set_name = ww_quadtile_get_data_set_name (tile);
	const gchar *world_name = mmap_get_world_name ();
	const gchar *ext = ww_quadtile_get_image_file_extension (tile);
	/*
	   g_print("ww_quadchild_new_root:x:%d y:%d level:%d\n", x, y, level);
	 */
	child = ww_quadchild_new ();
	priv = WW_QUADCHILD_GET_PRIVATE (child);
	priv->state = WW_QUADCHILD_STATE_ORIG;
	priv->x = x;
	priv->y = y;
	priv->level = level;
	priv->tile_deg = tile_deg;
	priv->quadtile = tile;
	g_object_ref (tile);

	path = ww_object_get_path (WW_OBJECT (tile));
	priv->path_dds = g_strdup_printf ("%s%s/%s%d/%04d/%04d_%04d.dds", mmap_dir_cache, world_name, path, level, y, y, x);
	priv->path_jpg =
		g_strdup_printf ("%s%s/%s%d/%04d/%04d_%04d.%s", mmap_dir_cache, world_name, path, level, y, y, x, ext);
	priv->uri = g_strdup_printf ("%s?T=%s&L=%d&X=%d&Y=%d", server_url, data_set_name, level, x, y);
	/*
	   g_print ("ww_quadchild_new_root:path_dds:%s\n", priv->path_dds);
	 */

	priv->ta = mmap_get_terrainaccessor (x * tile_deg, y * tile_deg, (x + 1) * tile_deg, (y + 1) * tile_deg);

	mmapthread_add_power (create_texture_sub, (GObject *) child);
	mmapthread_add_power (create_mesh_sub, (GObject *) child);

	g_free (path);

	return child;
}

void
ww_quadchild_clear_flag (WwQuadchild * child)
{
	WwQuadchildPrivate *priv;

	if (child == NULL) {
		return;
	}

	priv = WW_QUADCHILD_GET_PRIVATE (child);

	priv->need_flag = FALSE;

	ww_quadchild_clear_flag (priv->north_west);
	ww_quadchild_clear_flag (priv->north_east);
	ww_quadchild_clear_flag (priv->south_west);
	ww_quadchild_clear_flag (priv->south_east);
}

static gboolean
is_in_view (gint x, gint y, gdouble tile_deg)
{
	gint x0, y0;
	gint x1, y1;
	gdouble north, south, west, east;

	camera_get_view (&west, &south, &east, &north);
	util_tile_get_view (west, south, east, north, tile_deg, &x0, &y0, &x1, &y1);

	/*
	   g_print("is_in_view:tile_deg:%.2f (%d,%d)::x %d-%d  y %d-%d\n", tile_deg, x,y,x0,x1,y0,y1);
	 */

	if (x0 < x1) {
		if (x >= x0 && x <= x1 && y >= y0 && y <= y1) {
			return TRUE;
		} else {
			return FALSE;
		}
	} else {
		if ((x >= x0 || x <= x1) && y >= y0 && y <= y1) {
			return TRUE;
		} else {
			return FALSE;
		}
	}
}

void ww_quadchild_set_need_flag (WwQuadchild * child, gboolean need_flag);

static void
child_flag_set (WwQuadchild * self, gboolean value)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (self);

	if (priv->north_west != NULL) {
		ww_quadchild_set_need_flag (priv->north_west, value);
	}
	if (priv->north_east != NULL) {
		ww_quadchild_set_need_flag (priv->north_east, value);
	}
	if (priv->south_west != NULL) {
		ww_quadchild_set_need_flag (priv->south_west, value);
	}
	if (priv->south_east != NULL) {
		ww_quadchild_set_need_flag (priv->south_east, value);
	}
}


/* メモ
 * ある QuadTileSet を表示しようとする時、まず level を計算し、そのレベルまでゼロから順に降りていく。
 * 最初は level0 を作って準備。level1 が必要でも level0 の作成完了後に準備を開始する。
 */

/* 
 * ここの updata() は、自分の子が必要か不要かを決める。自分自身の要、不要は一つ上の階層でされている点に注意。
 * 子が必要なら、それらを作成し、自分自身の処理をする。
 * 子が不要なら、それらを破棄し、自分自身の処理をする。
 *
 * ここくるということは、
 * ・この mesh は、作成中か作成済み、または空で下にまかしている
 * ・自分は視野範囲と重なる部分を持っているが、子は視野範囲と重なってないかもしれない（ここで処理する）
 * */

/*
#define DB_UP(x) (x)
*/
#define DB_UP(x)

void
ww_quadchild_update (WwQuadchild * child, gint level)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);
	gboolean in_view;

	/*
	   if ((priv->texture != NULL && ww_texture_is_initialized(priv->texture) == FALSE) ) {
	 */
	if (priv->texture_initialized == FALSE || priv->mesh_initialized == FALSE) {
		DB_UP (g_print ("ww_quadchild_update:not initialized\n"));
		/* テクスチャーかメッシュの作成完了待ちなら下のレベルを調べない。 */

		/* 作成が完了してから要不要を判定するので、作成中は必要。 */
		priv->need_flag = TRUE;

		/* 上昇中だと自分が表示されるまで、子に消えられると困るので、残す */
		child_flag_set (child, TRUE);
		return;
	}

	/* TEMP か ORIG で作成が完了した。または、下にまかせた状態。 */

	in_view = is_in_view (priv->x, priv->y, priv->tile_deg);

	DB_UP (g_print ("ww_quadchild_update:x:%d y:%d level:%d   :%d:   ", priv->x, priv->y, priv->level, level));


	if (priv->state == WW_QUADCHILD_STATE_INIT && in_view == TRUE) {
		DB_UP (g_print ("INIT  in_view:TRUE\n"));

		/* さらに目標レベルと同じなら ORIG で作成し、子は開放。同じでないなら下をチェックするだけ */
		if (priv->level == level) {
			g_mutex_lock (priv->mutex_texture_and_mesh);
			{
				priv->texture_initialized = FALSE;
				mmapthread_add_power (create_texture_sub, (GObject *) child);
				priv->mesh_initialized = FALSE;
				mmapthread_add_power (create_mesh_sub, (GObject *) child);
			}
			g_mutex_unlock (priv->mutex_texture_and_mesh);

			/* すぐに開放してしまうと、消えた後に表示されることになるので、次の update にまかす。 */
			priv->state = WW_QUADCHILD_STATE_ORIG;
			priv->need_flag = TRUE;

			/* 上昇中だと自分が表示されるまで、子に消えられると困るので、残す */
			child_flag_set (child, TRUE);

		} else {
			if (ww_quadchild_is_initialized (priv->north_west) == TRUE &&
				ww_quadchild_is_initialized (priv->north_east) == TRUE &&
				ww_quadchild_is_initialized (priv->south_west) == TRUE &&
				ww_quadchild_is_initialized (priv->south_east) == TRUE) {
				g_mutex_lock (priv->mutex_texture_and_mesh);
				{
					if (priv->texture != NULL) {
						g_object_unref (priv->texture);
					}
					if (priv->mesh != NULL) {
						g_object_unref (priv->mesh);
					}
					priv->texture = NULL;
					priv->mesh = NULL;
				}
				g_mutex_unlock (priv->mutex_texture_and_mesh);
			}

			ww_quadchild_update (priv->north_west, level);
			ww_quadchild_update (priv->north_east, level);
			ww_quadchild_update (priv->south_west, level);
			ww_quadchild_update (priv->south_east, level);
			priv->need_flag = TRUE;
		}

		return;

	} else if (priv->state == WW_QUADCHILD_STATE_INIT && in_view == FALSE) {
		/* ここを ORIG で作りなおす。子レベルは開放 */
		DB_UP (g_print ("INIT in_view:FALSE\n"));

		g_mutex_lock (priv->mutex_texture_and_mesh);
		{
			priv->texture_initialized = FALSE;
			mmapthread_add_power (create_texture_sub, (GObject *) child);
			priv->mesh_initialized = FALSE;
			mmapthread_add_power (create_mesh_sub, (GObject *) child);
		}
		g_mutex_unlock (priv->mutex_texture_and_mesh);

		/* すぐに開放してしまうと、消えた後に表示されることになるので、次の update にまかす。 */
		priv->state = WW_QUADCHILD_STATE_ORIG;
		priv->need_flag = TRUE;

		return;

	} else if (priv->state == WW_QUADCHILD_STATE_TEMP && in_view == TRUE) {
		/* ORIG に格上げする。子レベルはないはず。 */
		DB_UP (g_print ("TEMP  in_view:TRUE\n"));

		g_mutex_lock (priv->mutex_texture_and_mesh);
		{
			priv->texture_initialized = FALSE;
			mmapthread_add_power (create_texture_sub, (GObject *) child);
			priv->mesh_initialized = FALSE;
			mmapthread_add_power (create_mesh_sub, (GObject *) child);
		}
		g_mutex_unlock (priv->mutex_texture_and_mesh);

		priv->state = WW_QUADCHILD_STATE_ORIG;
		priv->need_flag = TRUE;

		return;

	} else if (priv->state == WW_QUADCHILD_STATE_TEMP && in_view == FALSE) {
		/* 現状維持。子レベルはないはず */
		DB_UP (g_print ("TEMP  in_view:FALSE\n"));

		priv->need_flag = TRUE;

		return;

	} else if (priv->state == WW_QUADCHILD_STATE_ORIG && in_view == TRUE) {
		/* さらに目標レベル前なら子レベルを TEMP で作って、自分のテクスチャーとメッシュは開放 */
		DB_UP (g_print ("ORIG  in_view:TRUE\n"));

		if (priv->level == level) {
			/* 高度を上げてきた時にここにくる */

			child_flag_set (child, FALSE);
			priv->need_flag = TRUE;

		} else {
			gint x2 = 2 * priv->x;
			gint y2 = 2 * priv->y;
			gint next_level = priv->level + 1;
			gdouble next_tile_deg = priv->tile_deg / 2.0;
			priv->north_west = ww_quadchild_new_child (priv->quadtile, x2 + 0, y2 + 1, next_level, next_tile_deg,
													   WW_MESH_TYPE_NORTH_WEST, priv->texture, priv->mesh);
			priv->north_east = ww_quadchild_new_child (priv->quadtile, x2 + 1, y2 + 1, next_level, next_tile_deg,
													   WW_MESH_TYPE_NORTH_EAST, priv->texture, priv->mesh);
			priv->south_west = ww_quadchild_new_child (priv->quadtile, x2 + 0, y2 + 0, next_level, next_tile_deg,
													   WW_MESH_TYPE_SOUTH_WEST, priv->texture, priv->mesh);
			priv->south_east = ww_quadchild_new_child (priv->quadtile, x2 + 1, y2 + 0, next_level, next_tile_deg,
													   WW_MESH_TYPE_SOUTH_EAST, priv->texture, priv->mesh);
			/* テクスチャーとメッシュの開放は、次のアップデートで子レベルの初期化完了を確認して行う */
			priv->state = WW_QUADCHILD_STATE_INIT;
			priv->need_flag = TRUE;

			glarea_force_update_and_render ();
		}

		return;

	} else if (priv->state == WW_QUADCHILD_STATE_ORIG && in_view == FALSE) {
		DB_UP (g_print ("ORIG  in_view:FALSE\n"));

		/* 現状維持。子レベルは開放。 */

		child_flag_set (child, FALSE);
		priv->need_flag = TRUE;

		return;

	} else {
		g_print ("ww_quadchild_update:????????:\n");
		exit (1);
	}
}

void
ww_quadchild_remove_child (WwQuadchild * child)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);

	if (priv->north_west != NULL) {
		if (ww_quadchild_get_need_flag (priv->north_west) == FALSE) {
			ww_quadchild_is_detached_from_render_list (priv->north_west);
			g_object_unref (priv->north_west);
			priv->north_west = NULL;
		} else {
			ww_quadchild_remove_child (priv->north_west);
		}
	}

	if (priv->north_east != NULL) {
		if (ww_quadchild_get_need_flag (priv->north_east) == FALSE) {
			ww_quadchild_is_detached_from_render_list (priv->north_east);
			g_object_unref (priv->north_east);
			priv->north_east = NULL;
		} else {
			ww_quadchild_remove_child (priv->north_east);
		}
	}

	if (priv->south_west != NULL) {
		if (ww_quadchild_get_need_flag (priv->south_west) == FALSE) {
			ww_quadchild_is_detached_from_render_list (priv->south_west);
			g_object_unref (priv->south_west);
			priv->south_west = NULL;
		} else {
			ww_quadchild_remove_child (priv->south_west);
		}
	}

	if (priv->south_east != NULL) {
		if (ww_quadchild_get_need_flag (priv->south_east) == FALSE) {
			ww_quadchild_is_detached_from_render_list (priv->south_east);
			g_object_unref (priv->south_east);
			priv->south_east = NULL;
		} else {
			ww_quadchild_remove_child (priv->south_east);
		}
	}

}

/*
#define DB_RE(x) (x)
*/
#define DB_RE(x)

void
ww_quadchild_render (WwQuadchild * child)
{
	WwQuadchildPrivate *priv;

	if (child == NULL) {
		return;
	}

	priv = WW_QUADCHILD_GET_PRIVATE (child);

	DB_RE (g_print ("ww_quadchild_render:L:%d x:%d y:%d\n", priv->level, priv->x, priv->y));

	g_mutex_lock (priv->mutex_texture_and_mesh);
	{
		gboolean textured;
		gboolean is_in_progress;

		textured = ww_texture_bind (priv->texture);

		is_in_progress = !(priv->texture_initialized);

		if (textured == TRUE) {
			ww_mesh_render (priv->mesh, WW_MESH_RENDER_TYPE_TEXTURE, is_in_progress, priv->level);
		} else {
			ww_mesh_render (priv->mesh, WW_MESH_RENDER_TYPE_BOX, is_in_progress, priv->level);
		}
	}
	g_mutex_unlock (priv->mutex_texture_and_mesh);

	if (priv->north_west != NULL) {
		ww_quadchild_render (priv->north_west);
	}
	if (priv->north_east != NULL) {
		ww_quadchild_render (priv->north_east);
	}
	if (priv->south_west != NULL) {
		ww_quadchild_render (priv->south_west);
	}
	if (priv->south_east != NULL) {
		ww_quadchild_render (priv->south_east);
	}
}

gboolean
ww_quadchild_get_need_flag (WwQuadchild * child)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);

	return priv->need_flag;
}

void
ww_quadchild_set_need_flag (WwQuadchild * child, gboolean need_flag)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);

	priv->need_flag = need_flag;
}

gboolean
ww_quadchild_check_xy (WwQuadchild * child, gint x, gint y)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);

	if (priv->x == x && priv->y == y) {
		return TRUE;
	} else {
		return FALSE;
	}
}

gboolean
ww_quadchild_is_initialized (WwQuadchild * child)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);
	gboolean initialized;

	initialized = ww_texture_is_initialized (priv->texture) && priv->mesh_initialized;

	return initialized;

}

void
ww_quadchild_is_detached_from_render_list (WwQuadchild * child)
{
	WwQuadchildPrivate *priv = WW_QUADCHILD_GET_PRIVATE (child);

	priv->is_detached = TRUE;
}

void
ww_quadchild_change_vertical_exaggeration (WwQuadchild * child)
{
	WwQuadchildPrivate *priv;

	if (child == NULL) {
		return;
	}

	priv = WW_QUADCHILD_GET_PRIVATE (child);
	/*
	   g_print ("ww_quadchild_change_vertical_exaggeration:\n");
	 */

	ww_mesh_change_vertical_exaggeration (priv->mesh);

	if (priv->north_west != NULL) {
		ww_quadchild_change_vertical_exaggeration (priv->north_west);
	}
	if (priv->north_east != NULL) {
		ww_quadchild_change_vertical_exaggeration (priv->north_east);
	}
	if (priv->south_west != NULL) {
		ww_quadchild_change_vertical_exaggeration (priv->south_west);
	}
	if (priv->south_east != NULL) {
		ww_quadchild_change_vertical_exaggeration (priv->south_east);
	}
}
