/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <stdlib.h>

#include <gtk/gtkgl.h>
#include <GL/gl.h>
#include <GL/glext.h>

#include "config.h"

#include "glarea.h"
#include "disk.h"
#include "ww_texture.h"
#include "util.h"
#include "mmap_marshal.h"

typedef enum _WwTextureDataType WwTextureDataType;
enum _WwTextureDataType {
	WW_TEXTURE_DATA_NOT_COMPRESSED,
	WW_TEXTURE_DATA_COMPRESSED,
};

typedef struct _WwTexturePrivate WwTexturePrivate;
struct _WwTexturePrivate {
	gboolean dispose_has_run;

	gint w;
	gint h;

	WwTextureDataType type;		/* *data が画像の生データか圧縮画像のデータか */
	gint size;					/* *data のサイズ */
	void *data;
	gchar *savepath_dds;

	guint t_num;
};

enum {
	INITIALIZED,
	LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

#define WW_TEXTURE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), WW_TYPE_TEXTURE, WwTexturePrivate))

static GObjectClass *parent_class = NULL;

static void ww_texture_class_init (WwTextureClass * klass);
static void ww_texture_init (WwTexture * object);
static void ww_texture_finalize (GObject * object);
static void ww_texture_dispose (GObject * object);
static GObject *ww_texture_constructor (GType type, guint n_props, GObjectConstructParam * props);

/*****************************************************************************/

static void ww_texture_real_initialized (WwTexture * texture);

/*****************************************************************************/

GType
ww_texture_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (WwTextureClass),
			NULL,				/* base_init */
			NULL,				/* base_finalize */
			(GClassInitFunc) ww_texture_class_init,
			NULL,				/* class_finalize */
			NULL,				/* class_data */
			sizeof (WwTexture),
			0,					/* n_preallocs */
			(GInstanceInitFunc) ww_texture_init
		};

		type = g_type_register_static (G_TYPE_OBJECT, "WwTexture", &info, 0);
	}

	return type;
}

static void
ww_texture_class_init (WwTextureClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	/*
	   g_print ("ww_texture_class_init:c:%p:\n", klass);
	 */
	g_type_class_add_private (klass, sizeof (WwTexturePrivate));

	parent_class = g_type_class_peek_parent (klass);
	object_class->constructor = ww_texture_constructor;
	object_class->finalize = ww_texture_finalize;
	object_class->dispose = ww_texture_dispose;

	klass->initialized = ww_texture_real_initialized;

	signals[INITIALIZED] = g_signal_new ("initialized",
										 G_OBJECT_CLASS_TYPE (object_class),
										 G_SIGNAL_RUN_FIRST,
										 G_STRUCT_OFFSET (WwTextureClass, initialized),
										 NULL, NULL, mmap_marshal_VOID__VOID, G_TYPE_NONE, 0);

}

static void
ww_texture_init (WwTexture * self)
{
	WwTexturePrivate *priv = WW_TEXTURE_GET_PRIVATE (self);
	/*
	   g_print ("ww_texture_init:o:%p:\n", self);
	 */

	priv->dispose_has_run = FALSE;
	priv->w = 0;
	priv->h = 0;
	priv->type = WW_TEXTURE_DATA_NOT_COMPRESSED;
	priv->size = 0;
	priv->data = NULL;
	priv->savepath_dds = NULL;
	priv->t_num = 0;
}

static void
ww_texture_dispose (GObject * obj)
{
	WwTexture *self = WW_TEXTURE (obj);
	WwTexturePrivate *priv = WW_TEXTURE_GET_PRIVATE (self);
	/*
	   g_print ("ww_texture_dispose\n");
	 */
	if (priv->dispose_has_run) {
		return;
	}
	priv->dispose_has_run = TRUE;
	/*
	   g_object_unref(self->***);
	 */
	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

/**********************/

static gboolean
idle_func_texture_free (gpointer data)
{
	guint *t_num = (guint *) data;
	/*
	   g_print ("idle_func_texture_free\n");
	 */
	glarea_gl_begin ();
	glDeleteTextures (1, t_num);
	glarea_gl_end ();

	g_free (t_num);

	return FALSE;				/* FALSE を返すと単発で終了 */
}

static void
ww_texture_finalize (GObject * obj)
{
	WwTexture *self = WW_TEXTURE (obj);
	WwTexturePrivate *priv = WW_TEXTURE_GET_PRIVATE (self);
	guint *t_num;
	/*
	   g_print ("ww_texture_finalize\n");
	 */

	t_num = g_new (guint, 1);
	*t_num = priv->t_num;

	/* テクスチャーの開放を色々なスレッドでしようとすると面倒なので、idle ループ内でする。  */
	/* 全ての OpenGL 関係を idle ループ内でする。 */
	g_idle_add (idle_func_texture_free, t_num);

	if (priv->data != NULL) {
		g_free (priv->data);
	}
	if (priv->savepath_dds != NULL) {
		g_free (priv->savepath_dds);
	}

	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

/**********************/

static GObject *
ww_texture_constructor (GType type, guint n_props, GObjectConstructParam * props)
{
	GObject *object;
	GObjectClass *object_class = G_OBJECT_CLASS (parent_class);
	/*
	   g_print ("ww_texture_constructor\n");
	 */
	object = object_class->constructor (type, n_props, props);

	return object;
}

WwTexture *
ww_texture_new (void)
{
	GObject *object;

	object = g_object_new (WW_TYPE_TEXTURE, NULL);
	/*
	   g_print ("ww_texture_new:o:%p\n", object);
	 */
	return WW_TEXTURE (object);
}

/********************************************************************/

static void
ww_texture_real_initialized (WwTexture * texture)
{
/*
    g_print("ww_texture_real_initialized\n");
*/
}

/********************************************************************/


void
ww_texture_gen (WwTexture * tex)
{
	WwTexturePrivate *priv;
	/*
	   g_print ("ww_texture_gen\n");
	 */

	g_return_if_fail (WW_TEXTURE (tex));

	priv = WW_TEXTURE_GET_PRIVATE (tex);

	glarea_gl_begin ();
	{
		guint num;

		glGenTextures (1, &num);
		priv->t_num = num;
		glBindTexture (GL_TEXTURE_2D, num);
		/*
		   glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
		   glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
		 */
		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);

		if (priv->type == WW_TEXTURE_DATA_NOT_COMPRESSED) {
			if (glarea_is_support_compress () == TRUE && glarea_is_direct () == TRUE) {
				gint size;
				gint format;

				/*
				   g_print("before compress\n");
				 */
				glTexImage2D (GL_TEXTURE_2D, 0, GL_COMPRESSED_RGB, priv->w, priv->h, 0, GL_RGB, GL_UNSIGNED_BYTE,
							  priv->data);
				/*
				   g_print("after compress\n");
				 */

				if (priv->savepath_dds != NULL) {
					gchar *data = NULL;

					glGetTexLevelParameteriv (GL_TEXTURE_2D, 0, GL_TEXTURE_COMPRESSED_IMAGE_SIZE, &size);
					glGetTexLevelParameteriv (GL_TEXTURE_2D, 0, GL_TEXTURE_INTERNAL_FORMAT, &format);
					/*
					   g_print ("internal_format:%x compressed_texture_size:%d\n", format, size);
					 */
					data = g_new (gchar, size);
					glGetCompressedTexImage (GL_TEXTURE_2D, 0, data);

					disk_save_bin (priv->savepath_dds, data, size);

					g_free (data);
				}
			} else {
				g_print ("before glteximage2d\n");
				glTexImage2D (GL_TEXTURE_2D, 0, GL_RGB, priv->w, priv->h, 0, GL_RGB, GL_UNSIGNED_BYTE, priv->data);
			}

		} else {
			gint texture_format;

			texture_format = glarea_get_texture_format ();
			glCompressedTexImage2D (GL_TEXTURE_2D, 0, texture_format, priv->w, priv->h, 0, priv->size, priv->data);
		}

	}
	glarea_gl_end ();

	g_free (priv->data);
	priv->data = NULL;

	g_signal_emit (tex, signals[INITIALIZED], 0);
}

static void
texture_pixbuf_to_t_num (WwTexture * tex, GdkPixbuf * pixbuf, const gchar * savepath)
{
	WwTexturePrivate *priv;
	guchar *pixels;
	gint rowstride;
	gint size;

	if (pixbuf == NULL) {
		return;
	}

	pixels = gdk_pixbuf_get_pixels (pixbuf);

	priv = WW_TEXTURE_GET_PRIVATE (tex);

	priv->w = gdk_pixbuf_get_width (pixbuf);
	priv->h = gdk_pixbuf_get_height (pixbuf);

	rowstride = gdk_pixbuf_get_rowstride (pixbuf);
	size = rowstride * priv->h;

	priv->type = WW_TEXTURE_DATA_NOT_COMPRESSED;
	priv->size = size;
	priv->data = g_memdup (pixels, size);
	priv->savepath_dds = g_strdup (savepath);
}

WwTexture *
ww_texture_new_dds (const gchar * path_dds, gint width, gint height)
{
	WwTexture *tex;
	WwTexturePrivate *priv;
	gint size;
	/*
	   g_print ("ww_texture_new_dds:%s\n", path_dds);
	 */
	tex = ww_texture_new ();
	priv = WW_TEXTURE_GET_PRIVATE (tex);

	priv->w = width;
	priv->h = height;
	priv->type = WW_TEXTURE_DATA_COMPRESSED;
	priv->data = disk_load_bin (path_dds, &size);
	priv->size = size;
	priv->savepath_dds = g_strdup (path_dds);

	return tex;
}

WwTexture *
ww_texture_new_jpg (const gchar * path_jpg, const gchar * path_dds_save)
{
	WwTexture *tex;
	WwTexturePrivate *priv;
	GdkPixbuf *pixbuf;
	/*
	   g_print ("ww_texture_new_jpg:jpg:%s dds:%s\n", path_jpg, path_dds_save);
	 */
	tex = ww_texture_new ();
	priv = WW_TEXTURE_GET_PRIVATE (tex);

	pixbuf = disk_load_jpg (path_jpg);
	/* テクスチャーを作って ddspath で圧縮済テクスチャーを保存 */
	texture_pixbuf_to_t_num (tex, pixbuf, path_dds_save);
	g_object_unref (pixbuf);

	return tex;
}


WwTexture *
ww_texture_new_pixbuf (GdkPixbuf * pixbuf, const gchar * path_dds_save)
{
	WwTexture *texture;
	WwTexturePrivate *priv;

	texture = ww_texture_new ();
	priv = WW_TEXTURE_GET_PRIVATE (texture);

	texture_pixbuf_to_t_num (texture, pixbuf, path_dds_save);

	return texture;
}

WwTexture *
ww_texture_new_data (const gchar * data, gint size, const gchar * path_jpg_save, const gchar * path_dds_save)
{
	WwTexture *texture;
	WwTexturePrivate *priv;

	texture = ww_texture_new ();
	priv = WW_TEXTURE_GET_PRIVATE (texture);

	if (data == NULL) {
		priv->t_num = 0;
	} else {
		GdkPixbuf *pixbuf;

		disk_save_bin (path_jpg_save, (gchar *) data, size);
		pixbuf = util_bin_to_pixbuf (data, size);

		texture_pixbuf_to_t_num (texture, pixbuf, path_dds_save);
		g_object_unref (pixbuf);
	}

	return texture;
}

WwTexture *
ww_texture_copy (WwTexture * texture)
{
	g_object_ref (texture);

	return texture;
}

gboolean
ww_texture_is_initialized (WwTexture * texture)
{
	if (texture == NULL) {
		return FALSE;
	}

	WwTexturePrivate *priv;

	priv = WW_TEXTURE_GET_PRIVATE (texture);

	if (priv->t_num == 0) {
		return FALSE;

	} else {
		return TRUE;
	}
}

gboolean
ww_texture_bind (WwTexture * texture)
{
	WwTexturePrivate *priv;

	/*
	   g_print("ww_texture_bind\n");
	 */

	if (texture == NULL) {
		glBindTexture (GL_TEXTURE_2D, 0);

		return FALSE;
	}

	priv = WW_TEXTURE_GET_PRIVATE (texture);

	if (priv->t_num == 0) {
		glBindTexture (GL_TEXTURE_2D, 0);

		return FALSE;

	} else {
		glBindTexture (GL_TEXTURE_2D, priv->t_num);

		return TRUE;
	}
}
