/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
#include <gtk/gtk.h>
#include <math.h>

#include "simplemath.h"

/********************************************************************************/

static gdouble
det (gdouble m[])
{
	gdouble det;

	det = m[0] * (m[5] * (m[10] * m[15] - m[14] * m[11])
				  - m[9] * (m[6] * m[15] - m[14] * m[7])
				  + m[13] * (m[6] * m[11] - m[10] * m[7]));
	det -= m[4] * (m[1] * (m[10] * m[15] - m[14] * m[11])
				   - m[9] * (m[2] * m[15] - m[14] * m[3])
				   + m[13] * (m[2] * m[11] - m[10] * m[3]));
	det += m[8] * (m[1] * (m[6] * m[15] - m[14] * m[7])
				   - m[5] * (m[2] * m[15] - m[14] * m[3])
				   + m[13] * (m[2] * m[7] - m[6] * m[3]));
	det -= m[12] * (m[1] * (m[6] * m[11] - m[10] * m[7])
					- m[5] * (m[2] * m[11] - m[10] * m[3])
					+ m[9] * (m[2] * m[7] - m[6] * m[3]));

	return det;
}

void
matrix4_inverse (gdouble m[], gdouble ret[])
{
	gdouble idet = 1.0 / det (m);

	ret[0] =
		(m[5] * (m[10] * m[15] - m[14] * m[11]) - m[9] * (m[6] * m[15] - m[14] * m[7]) +
		 m[13] * (m[6] * m[11] - m[10] * m[7])) * idet;
	ret[1] =
		-(m[1] * (m[10] * m[15] - m[14] * m[11]) - m[9] * (m[2] * m[15] - m[14] * m[3]) +
		  m[13] * (m[2] * m[11] - m[10] * m[3])) * idet;
	ret[2] =
		(m[1] * (m[6] * m[15] - m[14] * m[7]) - m[5] * (m[2] * m[15] - m[14] * m[3]) +
		 m[13] * (m[2] * m[7] - m[6] * m[3])) * idet;
	ret[3] =
		-(m[1] * (m[6] * m[11] - m[10] * m[7]) - m[5] * (m[2] * m[11] - m[10] * m[3]) +
		  m[9] * (m[2] * m[7] - m[6] * m[3])) * idet;
	ret[4] =
		-(m[4] * (m[10] * m[15] - m[14] * m[11]) - m[8] * (m[6] * m[15] - m[14] * m[7]) +
		  m[12] * (m[6] * m[11] - m[10] * m[7])) * idet;
	ret[5] =
		(m[0] * (m[10] * m[15] - m[14] * m[11]) - m[8] * (m[2] * m[15] - m[14] * m[3]) +
		 m[12] * (m[2] * m[11] - m[10] * m[3])) * idet;
	ret[6] =
		-(m[0] * (m[6] * m[15] - m[14] * m[7]) - m[4] * (m[2] * m[15] - m[14] * m[3]) +
		  m[12] * (m[2] * m[7] - m[6] * m[3])) * idet;
	ret[7] =
		(m[0] * (m[6] * m[11] - m[10] * m[7]) - m[4] * (m[2] * m[11] - m[10] * m[3]) + m[8] * (m[2] * m[7] - m[6] * m[3])) * idet;
	ret[8] =
		(m[4] * (m[9] * m[15] - m[13] * m[11]) - m[8] * (m[5] * m[15] - m[13] * m[7]) +
		 m[12] * (m[5] * m[11] - m[9] * m[7])) * idet;
	ret[9] =
		-(m[0] * (m[9] * m[15] - m[13] * m[11]) - m[8] * (m[1] * m[15] - m[13] * m[3]) +
		  m[12] * (m[1] * m[11] - m[9] * m[3])) * idet;
	ret[10] =
		(m[0] * (m[5] * m[15] - m[13] * m[7]) - m[4] * (m[1] * m[15] - m[13] * m[3]) +
		 m[12] * (m[1] * m[7] - m[5] * m[3])) * idet;
	ret[11] =
		-(m[0] * (m[5] * m[11] - m[9] * m[7]) - m[4] * (m[1] * m[11] - m[9] * m[3]) + m[8] * (m[1] * m[7] - m[5] * m[3])) * idet;
	ret[12] =
		-(m[4] * (m[9] * m[14] - m[13] * m[10]) - m[8] * (m[5] * m[14] - m[13] * m[6]) +
		  m[12] * (m[5] * m[10] - m[9] * m[6])) * idet;
	ret[13] =
		(m[0] * (m[9] * m[14] - m[13] * m[10]) - m[8] * (m[1] * m[14] - m[13] * m[2]) +
		 m[12] * (m[1] * m[10] - m[9] * m[2])) * idet;
	ret[14] =
		-(m[0] * (m[5] * m[14] - m[13] * m[6]) - m[4] * (m[1] * m[14] - m[13] * m[2]) +
		  m[12] * (m[1] * m[6] - m[5] * m[2])) * idet;
	ret[15] =
		(m[0] * (m[5] * m[10] - m[9] * m[6]) - m[4] * (m[1] * m[10] - m[9] * m[2]) + m[8] * (m[1] * m[6] - m[5] * m[2])) * idet;
}

void
mul_mat4_mat4 (gdouble m0[], gdouble m1[], gdouble ret[])
{
	ret[0] = m0[0] * m1[0] + m0[4] * m1[1] + m0[8] * m1[2] + m0[12] * m1[3];
	ret[1] = m0[1] * m1[0] + m0[5] * m1[1] + m0[9] * m1[2] + m0[13] * m1[3];
	ret[2] = m0[2] * m1[0] + m0[6] * m1[1] + m0[10] * m1[2] + m0[14] * m1[3];
	ret[3] = m0[3] * m1[0] + m0[7] * m1[1] + m0[11] * m1[2] + m0[15] * m1[3];
	ret[4] = m0[0] * m1[4] + m0[4] * m1[5] + m0[8] * m1[6] + m0[12] * m1[7];
	ret[5] = m0[1] * m1[4] + m0[5] * m1[5] + m0[9] * m1[6] + m0[13] * m1[7];
	ret[6] = m0[2] * m1[4] + m0[6] * m1[5] + m0[10] * m1[6] + m0[14] * m1[7];
	ret[7] = m0[3] * m1[4] + m0[7] * m1[5] + m0[11] * m1[6] + m0[15] * m1[7];
	ret[8] = m0[0] * m1[8] + m0[4] * m1[9] + m0[8] * m1[10] + m0[12] * m1[11];
	ret[9] = m0[1] * m1[8] + m0[5] * m1[9] + m0[9] * m1[10] + m0[13] * m1[11];
	ret[10] = m0[2] * m1[8] + m0[6] * m1[9] + m0[10] * m1[10] + m0[14] * m1[11];
	ret[11] = m0[3] * m1[8] + m0[7] * m1[9] + m0[11] * m1[10] + m0[15] * m1[11];
	ret[12] = m0[0] * m1[12] + m0[4] * m1[13] + m0[8] * m1[14] + m0[12] * m1[15];
	ret[13] = m0[1] * m1[12] + m0[5] * m1[13] + m0[9] * m1[14] + m0[13] * m1[15];
	ret[14] = m0[2] * m1[12] + m0[6] * m1[13] + m0[10] * m1[14] + m0[14] * m1[15];
	ret[15] = m0[3] * m1[12] + m0[7] * m1[13] + m0[11] * m1[14] + m0[15] * m1[15];
}

void
mul_mat4_vec4 (gdouble m[], gdouble v[], gdouble ret[])
{
	ret[0] = m[0] * v[0] + m[4] * v[1] + m[8] * v[2] + m[12] * v[3];
	ret[1] = m[1] * v[0] + m[5] * v[1] + m[9] * v[2] + m[13] * v[3];
	ret[2] = m[2] * v[0] + m[6] * v[1] + m[10] * v[2] + m[14] * v[3];
	ret[3] = m[3] * v[0] + m[7] * v[1] + m[11] * v[2] + m[15] * v[3];
}

/********************************************************************************/


/* 正規化する。上書きして返す。*/
void
quaternion_normalize (gdouble q[])
{
	gdouble l;

	l = sqrt (q[0] * q[0] + q[1] * q[1] + q[2] * q[2] + q[3] * q[3]);
	/*
	   g_print("q normalize before:l:%.10f\n", l);
	 */
	q[0] /= l;
	q[1] /= l;
	q[2] /= l;
	q[3] /= l;

	/*
	   l = sqrt(q[0] * q[0] + q[1] * q[1] + q[2] * q[2] + q[3] * q[3]);
	   g_print("q normalize after:l:%.10f\n", l);
	 */
}

/* クォータニオンの積 p x q -> r */
void
qmul (const gdouble p[], const gdouble q[], gdouble r[])
{
	r[0] = p[0] * q[0] - p[1] * q[1] - p[2] * q[2] - p[3] * q[3];
	r[1] = p[0] * q[1] + p[1] * q[0] + p[2] * q[3] - p[3] * q[2];
	r[2] = p[0] * q[2] - p[1] * q[3] + p[2] * q[0] + p[3] * q[1];
	r[3] = p[0] * q[3] + p[1] * q[2] - p[2] * q[1] + p[3] * q[0];
}

/* クォータニオン q を回転行列 m に変換 */
void
quaternion_to_matrix (const gdouble q[], gdouble m[])
{
	double x2 = q[1] * q[1] * 2.0;
	double y2 = q[2] * q[2] * 2.0;
	double z2 = q[3] * q[3] * 2.0;
	double xy = q[1] * q[2] * 2.0;
	double yz = q[2] * q[3] * 2.0;
	double zx = q[3] * q[1] * 2.0;
	double xw = q[1] * q[0] * 2.0;
	double yw = q[2] * q[0] * 2.0;
	double zw = q[3] * q[0] * 2.0;

	m[0] = 1.0 - y2 - z2;
	m[1] = xy + zw;
	m[2] = zx - yw;
	m[4] = xy - zw;
	m[5] = 1.0 - z2 - x2;
	m[6] = yz + xw;
	m[8] = zx + yw;
	m[9] = yz - xw;
	m[10] = 1.0 - x2 - y2;
	m[3] = m[7] = m[11] = m[12] = m[13] = m[14] = 0.0;
	m[15] = 1.0;
}


/* クォータニオンをオイラー角に変換
 * yaw,pitch,roll はラジアン */
void
quaternion_to_euler (const gdouble q[], gdouble * yaw, gdouble * pitch, gdouble * roll)
{
	gdouble q0 = q[0];
	gdouble q1 = q[1];
	gdouble q2 = q[2];
	gdouble q3 = q[3];

	*yaw = atan2 (2 * (q2 * q3 + q0 * q1), (q0 * q0 - q1 * q1 - q2 * q2 + q3 * q3));
	*pitch = asin (-2 * (q1 * q3 - q0 * q2));
	*roll = atan2 (2 * (q1 * q2 + q0 * q3), (q0 * q0 + q1 * q1 - q2 * q2 - q3 * q3));

	/*
	   gdouble q1,q2,q3,q4;

	   q4 = qw;
	   q1 = qx;
	   q2 = qy;
	   q3 = qz;

	   *yaw = atan2( 2.0*(qy*qw - qx*qz), (1.0 - 2.0 * qy*qy - 2.0 * qz * qz));
	   *pitch = asin(2.0 * (qx*qy + qz*qw));
	   *roll = atan2( 2.0*(qx*qw - qy*qz), (1.0 - 2.0 * qx*qx - 2.0 * qz*qz));
	 */

}

/* オイラー角をクォータニオンに変換
 * yaw,pitch,roll はラジアン */
void
euler_to_quaternion (gdouble yaw, gdouble pitch, gdouble roll, gdouble q[])
{
	gdouble cy = cos (yaw * 0.5);
	gdouble cp = cos (pitch * 0.5);
	gdouble cr = cos (roll * 0.5);
	gdouble sy = sin (yaw * 0.5);
	gdouble sp = sin (pitch * 0.5);
	gdouble sr = sin (roll * 0.5);

	q[0] = cy * cp * cr + sy * sp * sr;
	q[1] = sy * cp * cr - cy * sp * sr;
	q[2] = cy * sp * cr + sy * cp * sr;
	q[3] = cy * cp * sr - sy * sp * cr;
}

/********************************************************************************/

/* ２点の経度緯度から２点間の角度をもとめる */
gdouble
distance_degree (gdouble lon0, gdouble lat0, gdouble lon1, gdouble lat1)
{
	gdouble d;

	d = acos (cos (RAD (lat0)) * cos (RAD (lat1)) * cos (RAD (lon0) - RAD (lon1)) + sin (RAD (lat0)) * sin (RAD (lat1)));

	return DEG (d);
}

/********************************************************************************/

void
debug_print_matrix (gdouble m[])
{
	g_print ("matrix\n");
	g_print ("%10.2f  %10.2f  %10.2f  %10.2f\n", m[0], m[4], m[8], m[12]);
	g_print ("%10.2f  %10.2f  %10.2f  %10.2f\n", m[1], m[5], m[9], m[13]);
	g_print ("%10.2f  %10.2f  %10.2f  %10.2f\n", m[2], m[6], m[10], m[14]);
	g_print ("%10.2f  %10.2f  %10.2f  %10.2f\n", m[3], m[7], m[11], m[15]);
}
