/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <gtk/gtk.h>
#include <glib/gprintf.h>
#include <gtk/gtkgl.h>

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glext.h>

#include "simplemath.h"
#include "mmap.h"
#include "util.h"
#include "info.h"
#include "glarea.h"
#include "camera.h"

/********************************************/

/* 現在位置と視線 */
/* 名前に target がついているものは、最終目的地。ないものはそのときの瞬間の位置 */
typedef struct {
	CameraType type;

	gdouble q[4];				/* 回転の状態 (quaternion) */
	gdouble tilt;				/* 0 〜 90 地球の中心を向いているのが 0 */
	gdouble dist;				/* 凝視している地表までの距離(m) */
	/* world wind では地表からの距離になっているが、ここでは標高ゼロの点からの距離 */

	gdouble q_target[4];		/* 回転の状態 (quaternion) */
	gdouble tilt_target;		/* 0 〜 90 地球の中心を向いているのが 0 */
	gdouble dist_target;		/* 凝視している地表までの距離(m) */

	/*** 下は上の値から計算される ***/

	gdouble rt[16];				/* q[4] のマトリックス版 */

	gdouble lon;				/* 経度(-180 --- 180) */
	gdouble lat;				/* 緯度(-90 --- 90) */
	gdouble lon_target;			/* 経度:(-180 --- 180) */
	gdouble lat_target;			/* 緯度(-90 --- 90) */
	gdouble heading;			/* 視線の地軸方向からの傾き(-180 <-> 0 <-> 180：北向きがゼロ) */

	gdouble alt;				/* 高度(m) */

	/* tilt なしの状態で真上から見たときに見える範囲(degree) */
	/* 地平線近くまで含めた広い版 */
	gdouble deg_view_lon_l;
	gdouble deg_view_lat_l;
	/* 真上から見たときに画面に収まる程度＋αという狭い版 */
	gdouble deg_view_lon_s;
	gdouble deg_view_lat_s;

	gboolean inertia;
} Camera;

static Camera *camera;

/**********************************************************/

/* Camera の既知の値(上半分)から下半分の値を計算する */
static void
calc_camera (void)
{
	gdouble yaw;
	gdouble pitch;
	gdouble roll;
	gdouble yaw_target;
	gdouble pitch_target;
	gdouble roll_target;
	gdouble r;

	quaternion_to_matrix (camera->q, camera->rt);	/* クォータニオンから回転の変換行列を求める */

	quaternion_to_euler (camera->q, &yaw, &pitch, &roll);
	camera->lon = DEG (yaw);
	camera->lat = DEG (pitch);

	/* camera_get_view で返すのは目的地にしたので作る。 */
	quaternion_to_euler (camera->q_target, &yaw_target, &pitch_target, &roll_target);
	camera->lon_target = DEG (yaw_target);
	camera->lat_target = DEG (pitch_target);

	camera->heading = DEG (roll) + 90.0;	/* よくわからないが帳尻をあわせて使っている */
	if (camera->heading > 180.0) {
		camera->heading -= 360.0;
	}

	r = mmap_get_world_radius ();

	if (camera->type == CAMERA_TYPE_NWW) {
		camera->alt = sqrt (camera->dist * camera->dist + r * r + 2.0 * camera->dist * r * cos (RAD (camera->tilt))) - r;
	} else {
		camera->alt = camera->dist;
	}
	/*
	   {
	   gdouble q[4];

	   euler_to_quaternion(yaw, pitch, roll, q);

	   g_print("quaternio     orig:%.2f %.2f %.2f %.2f\n", camera->q[0], camera->q[1], camera->q[2], camera->q[3]);
	   g_print("euler to quaternio:%.2f %.2f %.2f %.2f\n", q[0], q[1], q[2], q[3]);
	   }
	   g_print("yaw:%.2f  pitch:%.2f  roll:%.2f\n", camera->lon, camera->lat, camera->heading);
	 */

	{
		/* 視野範囲は、最終目的地 (**_target) から計算する。 */
		gdouble l;
		gdouble m;
		gdouble k;
		gdouble angle_v;		/* 地平線まで全部表示するための角度 */
		gdouble angle;

		angle_v = acos (r / (r + camera->dist_target));
		/*
		   g_print("angle:%.2f\n", DEG(angle_v));
		 */

		/* 緯度が上がるにしたがい経度方向の幅を増やすための係数 */
		m = cos (RAD (camera->lat_target));
		if (m < 0.2) {
			m = 0.2;
		} else {
			;
		}

		/* 高度を下げるにしたがい、視野範囲をしぼるための係数 */
		{
			gdouble a = 0.8;	/* 高度 3*r の時の係数 */
			gdouble b = 0.3;	/* 高度 0 の時の係数 */
			if (camera->dist_target > 3.0 * r) {
				k = a;
			} else {
				k = (a - b) / (3.0 * r) * camera->dist_target + b;
			}
		}
		camera->deg_view_lat_l = 2.0 * DEG (angle_v) * k;
		camera->deg_view_lon_l = camera->deg_view_lat_l / m;

		angle = 65.0;
		/*
		   g_print ("angle:%.2f  m:%.2f\n", angle, m);
		 */
		/* 真下を見下ろした時の視界の距離。ものすごくテキトーな計算 */
		l = 2.0 * camera->dist_target * tan (RAD (angle / 2.0));
		camera->deg_view_lat_s = 2.0 * DEG (atan (l / 2.0 / r));
		camera->deg_view_lon_s = camera->deg_view_lat_s / m;

		if (camera->deg_view_lon_l >= 360.0) {
			/* 360 だと幅ゼロだと見られてしまうの少し小さくする */
			camera->deg_view_lon_l = 359.9;
		}
		if (camera->deg_view_lon_s >= 360.0) {
			/* 360 だと幅ゼロだと見られてしまうの少し小さくする */
			camera->deg_view_lon_s = 359.9;
		}
		/*
		   g_print ("deg_view:l:lon:%.2f  lat:%.2f\n", camera->deg_view_lon_l, camera->deg_view_lat_l);
		   g_print ("deg_view:s:lon:%.2f  lat:%.2f\n", camera->deg_view_lon_s, camera->deg_view_lat_s);
		 */
	}
}

/*************************************************/

void
camera_init (void)
{

	camera = g_new (Camera, 1);

	camera->type = CAMERA_TYPE_NWW;

	/*
	   euler_to_quaternion (0.0, 0.0, RAD (-90.0), camera->q);
	 */
	euler_to_quaternion (RAD (135.0), RAD (35.0), RAD (-90.0), camera->q);
	/*
	   g_print ("camera_init:%.2f %.2f %.2f %.2f\n", camera->q[0], camera->q[1], camera->q[2], camera->q[3]);
	 */
	camera->tilt = 0.0;			/* 真下 */
	camera->dist = 20000000.0;

	camera->q_target[0] = camera->q[0];
	camera->q_target[1] = camera->q[1];
	camera->q_target[2] = camera->q[2];
	camera->q_target[3] = camera->q[3];

	camera->tilt_target = camera->tilt;
	camera->dist_target = camera->dist;

	camera->inertia = TRUE;

	calc_camera ();
}

void
camera_set_position (gdouble lon, gdouble lat)
{
	euler_to_quaternion (RAD (lon), RAD (lat), RAD (-90.0), camera->q_target);

	if (camera->inertia == FALSE) {
		camera->q[0] = camera->q_target[0];
		camera->q[1] = camera->q_target[1];
		camera->q[2] = camera->q_target[2];
		camera->q[3] = camera->q_target[3];
	}

	calc_camera ();
}

/* tilt と heading をゼロにする */
void
camera_reset (void)
{
	euler_to_quaternion (RAD (camera->lon_target), RAD (camera->lat_target), RAD (-90.0), camera->q_target);
	camera->tilt_target = 0.0;

	if (camera->inertia == FALSE) {
		camera->q[0] = camera->q_target[0];
		camera->q[1] = camera->q_target[1];
		camera->q[2] = camera->q_target[2];
		camera->q[3] = camera->q_target[3];

		camera->tilt = 0.0;
	}

	calc_camera ();
}

/*****************************************************/

/*
#define DB_SUB(x) (x)
*/
#define DB_SUB(x)

/* target:目標点
 * current:現在値--この中で書き換えられる
 * limit:収束判定値
 * point:減速点
 * step:減速前の単位速度変化
 * multi:減速後に差分の何割ずつ寄っていくか(0.0 < multi < 1.0)
 */
static gboolean
motion_sub (gdouble * target, gdouble * current, gdouble limit, gdouble point, gdouble step, gdouble multi)
{
	gboolean finished = FALSE;	/* 目的の範囲に入ったら TRUE */
	gdouble diff;

	diff = *target - *current;
	if (diff == 0.0) {
		return TRUE;
	}

	DB_SUB (g_print ("motion_SUB:target:%.2f current:%.2f diff:%.2f\t", camera->tilt_target, camera->tilt, diff));

	if (diff > limit) {
		if (diff < -point) {
			*current += step;

			DB_SUB (g_print (":%.2f:", step));
		} else {
			*current += diff * multi;

			DB_SUB (g_print (":diff:"));
		}

		finished = FALSE;
	} else if (diff < -limit) {
		if (diff < -point) {
			*current -= step;

			DB_SUB (g_print (":%.2f:", step));
		} else {
			*current += diff * multi;

			DB_SUB (g_print (":diff:"));
		}

		finished = FALSE;
	} else {
		*target = *current;

		DB_SUB (g_print (":    :"));
		finished = TRUE;
	}
	DB_SUB (g_print ("\n"));

	return finished;
}

/*****************************************************/


/*****************************************************/

static gboolean
motion_tilt (void)
{
	gboolean finished = FALSE;	/* 目的の範囲に入ったら TRUE */

	gdouble limit = 0.1;
	gdouble point = 15.0;		/* 減速の開始位置 */
	gdouble step = 1.5;
	gdouble multi = 0.1;

	finished = motion_sub (&(camera->tilt_target), &(camera->tilt), limit, point, step, multi);

	return finished;
}

static gdouble
tilt_check (gdouble tilt)
{
	if (tilt >= 90.0) {
		tilt = 90.0;
	} else if (tilt <= 0.0) {
		tilt = 0.0;
	}

	return tilt;
}

void
camera_set_tilt (gdouble tilt)
{
	camera->tilt_target = tilt_check (tilt);
	if (camera->inertia == FALSE) {
		camera->tilt = tilt_check (tilt);
	}
}

gdouble
camera_get_tilt (void)
{
	return camera->tilt;
}

void
camera_tilt_set_diff (gdouble diff)
{
	camera->tilt_target = tilt_check (camera->tilt_target + diff);

	if (camera->inertia == FALSE) {
		camera->tilt = camera->tilt_target;
	}
}

/*****************************************************/

#define DB_ROT(x) (x)
/*
#define DB_ROT(x)
*/

/*
 * http://marina.sys.wakayama-u.ac.jp/~tokoi/?date=20040430
 * 床井研究室 球面線形補間 のコードを参考にした。
 */

#include <errno.h>

/* クォータニオン q から r への途中の点を補間する。t は粒度。結果を p に入れて返す。
 *  ######### 注意 #######
 *  q, r は正規化をしてないと計算で例外が出る 
 *  #####################
 * 返り値
 * TRUE :計算が成功した
 * FALSE:計算が失敗した。結果は、q をそのまま返す。 */
gboolean
slerp (gdouble p[], const gdouble q[], const gdouble r[], const gdouble t)
{
	/* qr: q, r 間の回転角を ph とした時の cos(ph) */
	double qr = q[0] * r[0] + q[1] * r[1] + q[2] * r[2] + q[3] * r[3];
	double ss = 1.0L - qr * qr;	/* sin(ph) をもとめる準備 */

	/*
	   g_print("qr:%.10f\n", qr);
	   g_print("ss:%.10f\n", ss);
	 */
	/*
	   if (qr > 1.0) {
	   qr = 1.0;
	   } else if (qr < -1.0) {
	   qr = -1.0;
	   }
	 */
	if (ss < 0.0) {
		ss = 0.0;
	}

	if (ss == 0.0) {
		/* この場合、永遠に収束しない */
		p[0] = q[0];
		p[1] = q[1];
		p[2] = q[2];
		p[3] = q[3];

		return FALSE;

	} else {
		gdouble ph = acos (qr);	/* q, r 間の回転角 */
		gdouble sin_ph = sqrt (ss);	/* sin(ph) */
		gdouble t0 = sin (ph * (1.0 - t)) / sin_ph;
		gdouble t1 = sin (ph * t) / sin_ph;
		/*
		   g_print ("slerp:ph:%f\n", ph);
		   g_print ("slerp:sin_ph:%.2f\n", sin_ph);
		   g_print ("slerp:q %.2f %.2f %.2f %.2f\n", q[0], q[1], q[2], q[3]);
		   g_print ("slerp:r %.2f %.2f %.2f %.2f\n", r[0], r[1], r[2], r[3]);
		 */
		p[0] = q[0] * t0 + r[0] * t1;
		p[1] = q[1] * t0 + r[1] * t1;
		p[2] = q[2] * t0 + r[2] * t1;
		p[3] = q[3] * t0 + r[3] * t1;

		quaternion_normalize (p);

		return TRUE;
	}
}

static gboolean
motion_rotate (void)
{
	gboolean finished = FALSE;	/* 目的の範囲に入ったら TRUE */
	gdouble q_new[4];

	if (camera->q[0] == camera->q_target[0] && camera->q[1] == camera->q_target[1] &&
		camera->q[2] == camera->q_target[2] && camera->q[3] == camera->q_target[3]) {
		return TRUE;
	}

	slerp (q_new, camera->q, camera->q_target, 0.09);
	camera->q[0] = q_new[0];
	camera->q[1] = q_new[1];
	camera->q[2] = q_new[2];
	camera->q[3] = q_new[3];

	{
		gdouble lon0, lat0;
		gdouble lon1, lat1;
		gdouble yaw0, pitch0, roll0;
		gdouble yaw1, pitch1, roll1;
		gdouble d;
		gdouble d_deg;
		gdouble d_roll;

		quaternion_to_euler (camera->q, &yaw0, &pitch0, &roll0);
		quaternion_to_euler (camera->q_target, &yaw1, &pitch1, &roll1);
		lon0 = DEG (yaw0);
		lat0 = DEG (pitch0);
		lon1 = DEG (yaw1);
		lat1 = DEG (pitch1);
		d_roll = fabs (DEG (roll0 - roll1));

		d = distance_degree (lon0, lat0, lon1, lat1);
		d_deg = 1.0 / 3600.0 * camera->dist / 100000.0;	/* 高度10万メートルで１秒のズレを許容する */

/*
g_print("motion_rotate:q:%.2f,%.2f,%.2f,%.2f\n", camera->q[1], camera->q[2], camera->q[3], camera->q[0]);
g_print("motion_rotate:t:%.2f,%.2f,%.2f,%.2f\n", camera->q_target[1], camera->q_target[2], camera->q_target[3], camera->q_target[0]);
g_print("motion_rotate:y:%.2f p:%.2f r:%.2f\n", yaw0, pitch0, roll0);
g_print("motion_rotate:(%.2f, %.2f)  (%.2f, %.2f)   d:%.2f d_deg:%.2f d_roll:%.2f\n", lon0, lat0, lon1, lat1, d, d_deg, d_roll);
*/

		if (d < d_deg && d_roll < 0.1) {
			/* 現在値が目的地に十分近付いたので、目的地を現在値で上書きする */
			camera->q_target[0] = camera->q[0];
			camera->q_target[1] = camera->q[1];
			camera->q_target[2] = camera->q[2];
			camera->q_target[3] = camera->q[3];

			finished = TRUE;
		} else {
			finished = FALSE;
		}
	}

	return finished;
}

void
camera_mult_quaternion (gdouble q[])
{
	gdouble q_new[4];

	qmul (q, camera->q_target, q_new);

	quaternion_normalize (q_new);

	camera->q_target[0] = q_new[0];
	camera->q_target[1] = q_new[1];
	camera->q_target[2] = q_new[2];
	camera->q_target[3] = q_new[3];

	if (camera->inertia == FALSE) {
		camera->q[0] = q_new[0];
		camera->q[1] = q_new[1];
		camera->q[2] = q_new[2];
		camera->q[3] = q_new[3];
	}
	/*
	   g_print ("camera_mult_quaternion:%.2f %.2f %.2f %.2f\n",
	   camera->q_target[0], camera->q_target[1], camera->q_target[2], camera->q_target[3]);
	 */
}

void
camera_rotate (gdouble yaw, gdouble pitch, gdouble roll)
{
	gdouble q_new[4];
	gdouble q_rot[4];
	/*
	   g_print("camera_rotate:yaw:%.2f pitch:%.2f roll:%.2f\n",yaw, pitch, roll);
	 */
	euler_to_quaternion (yaw, pitch, roll, q_rot);

	qmul (q_rot, camera->q_target, q_new);

	quaternion_normalize (q_new);

	camera->q_target[0] = q_new[0];
	camera->q_target[1] = q_new[1];
	camera->q_target[2] = q_new[2];
	camera->q_target[3] = q_new[3];

	if (camera->inertia == FALSE) {
		camera->q[0] = q_new[0];
		camera->q[1] = q_new[1];
		camera->q[2] = q_new[2];
		camera->q[3] = q_new[3];
	}
}

/*****************************************************/

/*
#define DB_DIST(x) (x)
*/
#define DB_DIST(x)

static gboolean
motion_dist (void)
{
	gboolean finished = FALSE;	/* 目的の範囲に入ったら TRUE */
	gdouble diff = 0.0;

	DB_DIST (g_print ("motion_dist:new_dist:%.2f  current_dist:%.2f \t", camera->dist_target, camera->dist));

	diff = camera->dist_target - camera->dist;
	if (diff == 0.0) {
		return TRUE;
	}

	if (diff > 0.001 * camera->dist_target) {
		if (diff * 0.3 > 0.03 * camera->dist) {
			camera->dist *= 1.03;
			DB_DIST (g_print (":*=    :"));
		} else {
			camera->dist += diff * 0.3;
			DB_DIST (g_print (":+=diff:"));
		}
		finished = FALSE;

		DB_DIST (g_print (":finished:diff:%.2f\n", diff));
	} else if (diff < -0.001 * camera->dist_target) {
		if (diff * 0.3 < -0.03 * camera->dist) {
			camera->dist *= 0.97;
			DB_DIST (g_print (":*=    :"));
		} else {
			camera->dist += diff * 0.3;
			DB_DIST (g_print (":+=diff:"));
		}
		finished = FALSE;

		DB_DIST (g_print (":finished:diff:%.2f\n", diff));
	} else {
		camera->dist_target = camera->dist;
		finished = TRUE;

		DB_DIST (g_print (":  finished:diff:%.2f\n", diff));
	}


	return finished;
}

void
camera_set_dist (gdouble dist)
{
	camera->dist_target = dist;

	if (camera->inertia == FALSE) {
		camera->dist = camera->dist_target;
	}
}

/* world が変わった時などの、のんびり上がってられない時にはこちらを使う */
void
camera_set_dist_direct (gdouble dist)
{
	camera->dist_target = dist;
	camera->dist = camera->dist_target;
}

void
camera_dist_up (void)
{
	camera->dist_target *= 1.1;
	if (camera->inertia == FALSE) {
		camera->dist = camera->dist_target;
	}
}

void
camera_dist_down (void)
{
	camera->dist_target *= 0.9;
	if (camera->inertia == FALSE) {
		camera->dist = camera->dist_target;
	}
}

/*****************************************************/

gboolean
camera_next_step (void)
{
	gboolean finished_dist = FALSE;	/* 目的の範囲に入ったら TRUE */
	gboolean finished_tilt = FALSE;
	gboolean finished_rotate = FALSE;

	finished_dist = motion_dist ();
	finished_tilt = motion_tilt ();
	finished_rotate = motion_rotate ();

	calc_camera ();

	/*  
	   util_print_bool("dist:", finished_dist);
	   util_print_bool("tilt:", finished_tilt);
	   util_print_bool("rotate:", finished_rotate);
	 */

	/* 全部が目的の範囲に入ったら TRUE を返す */
	return finished_dist && finished_tilt && finished_rotate;
}

/*****************************************************/

void
camera_matrix_prepare (void)
{
	gdouble r = mmap_get_world_radius ();

	glMatrixMode (GL_MODELVIEW);
	glLoadIdentity ();

	if (camera->type == CAMERA_TYPE_NWW) {
		glTranslated (0.0, 0.0, -1.0 * camera->dist);

		glRotated (-1.0 * camera->tilt, 1.0, 0.0, 0.0);	/* 視線の方向をあわす */
		glTranslated (0.0, 0.0, -r);
	} else {
		glRotated (-1.0 * camera->tilt, 1.0, 0.0, 0.0);
		glTranslated (0.0, 0.0, -(r + camera->dist));
	}
	glMultMatrixd (camera->rt);	/* 回転 */
	glRotated (-90.0, 0.0, 1.0, 0.0);

	/*
	   {
	   glGetDoublev (GL_MODELVIEW_MATRIX, m_m);
	   glGetDoublev (GL_PROJECTION_MATRIX, m_p);
	   mul_mat4_mat4 (m_p, m_m, m);
	   debug_print_matrix(m_p);
	   debug_print_matrix(m_m);
	   debug_print_matrix(m);
	   }
	 */
}


/*****************************************************/

gdouble
camera_get_lon (void)
{
	return camera->lon;
}

gdouble
camera_get_lat (void)
{
	return camera->lat;
}

gdouble
camera_get_alt (void)
{
	return camera->alt;
}

gdouble
camera_get_dist (void)
{
	return camera->dist;
}

gdouble
camera_get_dist_target (void)
{
	return camera->dist_target;
}

/* 注視点を真上からみた時に見えている範囲。単位 degree
 * 左下(x0,y0) 右上(x1,y1) */
/* 現在の場所の範囲でなく、目的の場所での範囲を返す */
void
camera_get_view_narrow (gdouble * x0, gdouble * y0, gdouble * x1, gdouble * y1)
{
	gdouble west;
	gdouble south;
	gdouble east;
	gdouble north;

	west = camera->lon_target - camera->deg_view_lon_s / 2.0;
	east = camera->lon_target + camera->deg_view_lon_s / 2.0;
	north = camera->lat_target + camera->deg_view_lat_s / 2.0;
	south = camera->lat_target - camera->deg_view_lat_s / 2.0;

	if (west < -180.0) {
		west += 360.0;
	}
	if (east > 180.0) {
		east -= 360.0;
	}
	if (north > 90.0) {
		north = 90.0;
	}
	if (south < -90.0) {
		south = -90.0;
	}

	*x0 = west;
	*y0 = south;
	*x1 = east;
	*y1 = north;
}

void
camera_get_view_wide (gdouble * x0, gdouble * y0, gdouble * x1, gdouble * y1)
{
	gdouble west;
	gdouble south;
	gdouble east;
	gdouble north;

	west = camera->lon_target - camera->deg_view_lon_l / 2.0;
	east = camera->lon_target + camera->deg_view_lon_l / 2.0;
	north = camera->lat_target + camera->deg_view_lat_l / 2.0;
	south = camera->lat_target - camera->deg_view_lat_l / 2.0;

	if (west < -180.0) {
		west += 360.0;
	}
	if (east > 180.0) {
		east -= 360.0;
	}
	if (north > 90.0) {
		north = 90.0;
	}
	if (south < -90.0) {
		south = -90.0;
	}

	*x0 = west;
	*y0 = south;
	*x1 = east;
	*y1 = north;
}



void
camera_render_info (void)
{
	gint w, h;
	gint w_font, h_font;
	gint x, y;
	gchar buf[100];

	glarea_get_size (&w, &h);
	info_get_font_size (&w_font, &h_font);

	/*
	   g_print ("camera_render_info:w:%d h:%d\n", w, h);
	 */
	x = w - 25 * w_font;
	y = h - 2 * h_font;

	g_sprintf (buf, "Lon     : %10.5f", camera->lon);
	info_render (buf, x, y);

	y -= h_font;
	g_sprintf (buf, "Lat     : %10.5f", camera->lat);
	info_render (buf, x, y);

	y -= h_font;
	g_sprintf (buf, "Heading : %10.5f", camera->heading);
	info_render (buf, x, y);

	y -= h_font;
	g_sprintf (buf, "Tilt    : %10.5f", camera->tilt);
	info_render (buf, x, y);

	y -= h_font;
	if (camera->dist > 100000) {
		g_sprintf (buf, "        : %10.2f km", camera->dist / 1000.0);
	} else {
		g_sprintf (buf, "        : %10.2f m", camera->dist);
	}
	info_render (buf, x, y);

	y -= h_font;
	if (camera->alt > 100000) {
		g_sprintf (buf, "        : %10.2f km", camera->alt / 1000.0);
	} else {
		g_sprintf (buf, "        : %10.2f m", camera->alt);
	}
	info_render (buf, x, y);
}

#define N_DEBUG 10
void
camera_debug_view_render (void)
{
	gdouble west_l, east_l, south_l, north_l;
	gdouble west_s, east_s, south_s, north_s;
	gdouble xyz_l[N_DEBUG][N_DEBUG][3];
	gdouble xyz_s[N_DEBUG][N_DEBUG][3];
	gint i, j;
	gdouble n = N_DEBUG;

	west_l = camera->lon_target - camera->deg_view_lon_l / 2.0;
	east_l = camera->lon_target + camera->deg_view_lon_l / 2.0;
	north_l = camera->lat_target + camera->deg_view_lat_l / 2.0;
	south_l = camera->lat_target - camera->deg_view_lat_l / 2.0;

	west_s = camera->lon_target - camera->deg_view_lon_s / 2.0;
	east_s = camera->lon_target + camera->deg_view_lon_s / 2.0;
	north_s = camera->lat_target + camera->deg_view_lat_s / 2.0;
	south_s = camera->lat_target - camera->deg_view_lat_s / 2.0;

	for (j = 0; j < n; ++j) {
		gdouble lat_l = south_l + (north_l - south_l) / (n - 1.0) * j;
		gdouble lat_s = south_s + (north_s - south_s) / (n - 1.0) * j;

		for (i = 0; i < n; ++i) {
			gdouble lon_l = west_l + (east_l - west_l) / (n - 1.0) * i;
			gdouble lon_s = west_s + (east_s - west_s) / (n - 1.0) * i;
			gdouble x_l, y_l, z_l;
			gdouble x_s, y_s, z_s;

			if (lon_l < -180.0) {
				lon_l += 360.0;
			}
			if (lon_l > 180.0) {
				lon_l -= 360.0;
			}
			if (lat_l > 90.0) {
				lat_l = 90.0;
			}
			if (lat_l < -90.0) {
				lat_l = -90.0;
			}

			if (lon_s < -180.0) {
				lon_s += 360.0;
			}
			if (lon_s > 180.0) {
				lon_s -= 360.0;
			}
			if (lat_s > 90.0) {
				lat_s = 90.0;
			}
			if (lat_s < -90.0) {
				lat_s = -90.0;
			}

			mmap_deg_to_xyz (lon_l, lat_l, 100.0, &x_l, &y_l, &z_l);
			mmap_deg_to_xyz (lon_s, lat_s, 100.0, &x_s, &y_s, &z_s);
			xyz_l[i][j][0] = x_l;
			xyz_l[i][j][1] = y_l;
			xyz_l[i][j][2] = z_l;
			xyz_s[i][j][0] = x_s;
			xyz_s[i][j][1] = y_s;
			xyz_s[i][j][2] = z_s;
		}
	}

	{
		gint i, j;
		gdouble color_l[3] = { 1.0, 1.0, 1.0 };
		gdouble color_s[3] = { 0.5, 0.5, 0.5 };

		glDisable (GL_DEPTH_TEST);
		for (j = 0; j < n; ++j) {
			for (i = 0; i < n - 1; ++i) {
				glBegin (GL_LINE_STRIP);
				glColor3dv (color_l);
				glVertex3dv (xyz_l[i + 0][j]);
				glVertex3dv (xyz_l[i + 1][j]);
				glEnd ();

				glBegin (GL_LINE_STRIP);
				glColor3dv (color_s);
				glVertex3dv (xyz_s[i + 0][j]);
				glVertex3dv (xyz_s[i + 1][j]);
				glEnd ();
			}
		}
		for (j = 0; j < n; ++j) {
			for (i = 0; i < n - 1; ++i) {
				glBegin (GL_LINE_STRIP);
				glColor3dv (color_l);
				glVertex3dv (xyz_l[j][i + 0]);
				glVertex3dv (xyz_l[j][i + 1]);
				glEnd ();

				glBegin (GL_LINE_STRIP);
				glColor3dv (color_s);
				glVertex3dv (xyz_s[j][i + 0]);
				glVertex3dv (xyz_s[j][i + 1]);
				glEnd ();
			}
		}
		glEnable (GL_DEPTH_TEST);
	}
}

gboolean
camera_check_overlap (gdouble north, gdouble south, gdouble west, gdouble east)
{
	gdouble cwest;
	gdouble csouth;
	gdouble ceast;
	gdouble cnorth;
	/*
	   g_print("camera_check_overlap:n:%.2f s:%.2f w:%.2f e:%.2f\n", north, south, west, east);
	 */
	cwest = camera->lon - camera->deg_view_lon_s / 2.0;
	ceast = camera->lon + camera->deg_view_lon_s / 2.0;
	cnorth = camera->lat + camera->deg_view_lat_s / 2.0;
	csouth = camera->lat - camera->deg_view_lat_s / 2.0;

	if (cwest < -180.0) {
		cwest += 360.0;
	}
	if (ceast > 180.0) {
		ceast -= 360.0;
	}
	if (cnorth > 90.0) {
		cnorth = 90.0;
	}
	if (csouth < -90.0) {
		csouth = -90.0;
	}
	/*
	   g_print("camera_check_overlap:camera:n:%.2f s:%.2f w:%.2f e:%.2f\n", cnorth, csouth, cwest, ceast);
	 */
	return util_check_overlap (cwest, csouth, ceast, cnorth, west, south, east, north);
}

void
camera_set_type (CameraType type)
{
	camera->type = type;
}

void
camera_set_inertia (gboolean bool)
{
	camera->inertia = bool;
}

gboolean
camera_get_inertia (void)
{
	return camera->inertia;
}
