/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <glib.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <glib/gprintf.h>

#include "disk.h"
#include "data.h"
#include "gsi.h"

void
edge_create (const gchar * name, Slp * slp, GIOChannel * ch_edge, GIOChannel * ch_node)
{
	gchar *fullpath;
	gchar *line;
	GIOChannel *ch;
	GIOStatus status;
	GError *err = NULL;

	fullpath = g_strconcat (name, "dk.sal", NULL);
	ch = disk_channel_open (fullpath, READ);
	g_free (fullpath);

	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		const gchar *p;
		Data25k top;
		gchar fi = 0;			/* fi の２文字目の文字が入る */
		gboolean kb = FALSE;
		gboolean multi = FALSE;
		gboolean flag_next = FALSE;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("err:edge_create:%s:\n", name);
			exit (1);
		}
		line = g_strchomp (line);

		gsi25k_get_data (line, &top);

		p = top.value;
		do {
			Data25k d25k;

			p = gsi25k_get_data (p, &d25k);
			/*
			g_print("id:%s: value:%s\n", d25k.id, d25k.value);
			 */

			if (d25k.id[0] == 'J' && d25k.id[1] == 'T') {
				if (d25k.value[1] == '2') {     /* 建設中なら捨てる */
					flag_next = TRUE;
				}

			} else if (d25k.id[0] == 'Y' && d25k.id[1] == 'U') {
				if (d25k.value[1] == 'C') {
					multi = TRUE;	            /* 有料道路は２倍速 */
				}

			} else if (d25k.id[0] == 'S' && d25k.id[1] == 'B') {
				if (d25k.value[1] == '3' || d25k.value[1] == '4') {
					flag_next = TRUE;           /* 庭園路か石段なら捨てる */
				} else {
					if (d25k.value[1] == '6') {
						multi = TRUE;	        /* 高速道路は２倍速 */
					}
				}

			} else if (d25k.id[0] == 'F' && d25k.id[1] == 'I') {
				fi = d25k.value[1];

			} else if (d25k.id[0] == 'K' && d25k.id[1] == 'B') {
				kb = TRUE;

			} else if (d25k.id[0] == 'C' && d25k.id[1] == 'V') {
				if (multi == TRUE) {                                /* 高速か有料道路なら幅員に関係なく２倍速 */
					gsi25k_print_curve (d25k.value, slp, ch_edge, ch_node, 5);

				} else if (fi == 'A' || fi == 'B' || fi == 'E') {   /* 幅員 5.5m 以上は１倍速 */
					gsi25k_print_curve (d25k.value, slp, ch_edge, ch_node, 10);

				} else if (fi == '9') {                             /* 幅員 3.0-5.5m は３分の１倍速 */
					gsi25k_print_curve (d25k.value, slp, ch_edge, ch_node, 30);

				} else if (fi == '8' && kb == TRUE) {               /* 幅員 1.5-3.0m でも国道なら３分の１倍速 */
					gsi25k_print_curve (d25k.value, slp, ch_edge, ch_node, 30);

				} else {                                            /* 幅員 1.5m 未満は通らない */
					;
				}

			} else {
				;
			}

			g_free (d25k.name);
			g_free (d25k.value);
			if (flag_next == TRUE) {
				break;
			}

		} while (*p != '\0');
		g_free (top.name);
		g_free (top.value);

		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_io_channel_unref (ch);
}

/*****************************************/

static gchar *
read_line(GIOChannel *ch)
{
	GIOStatus status;
	GError *err = NULL;
	gchar *line = NULL;

	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	if (status == G_IO_STATUS_ERROR) {
		g_print ("error:slm read_line:%s\n", err->message);
		g_free (line);
		exit (1);
	}

    return line;
}

/* name:11111 */
void
slm (const gchar * name, gint64 * x, gint64 * y, gint * n)
{
	gchar *fullpath;
	gchar *line = NULL;
	GIOChannel *ch;

	fullpath = g_strconcat (name, ".slm", NULL);
	ch = disk_channel_open (fullpath, READ);
	g_free (fullpath);

    line = read_line(ch);       /* 左下の経度と緯度 */
	line = g_strchomp (line);
	sscanf (line, "%Ld,%Ld", x, y);
	g_free (line);

    line = read_line(ch);
	g_free (line);

    line = read_line(ch);       /* 頂点数 */
	line = g_strchomp (line);
	sscanf (line, "%d", n);
	g_free (line);

	g_io_channel_unref (ch);

	/*
	g_print ("slm :%Ld:%Ld :%d:\n", *x, *y, *n);
	*/
}

/*****************************************/

void
slp_free (Slp * slp)
{
	gint i;

	for (i = 0; i < slp->n; ++i) {
		g_free (slp->v[i]);
	}
	g_free (slp->v);
	g_free (slp);
}

Slp *
slp_create (const gchar * name, gint64 x0, gint64 y0, gint n)
{
	gint i;
	gchar *fullpath;
	gchar *bin;
	gchar *p_bin;
	gint size;
	Slp *slp;
	gint line_n;

	fullpath = g_strconcat (name, ".slp", NULL);
	bin = disk_bin_from_disk (fullpath, &size);
	g_free (fullpath);

	line_n = size / n;

	slp = g_new (Slp, 1);
	slp->n = n;
	slp->v = g_new (gchar *, slp->n);

	p_bin = bin;

	for (i = 0; i < slp->n; ++i, p_bin += line_n) {
		gint64 x_offset = 0.0;
		gint64 y_offset = 0.0;
		gint64 lon, lat;
		gchar line[30];

		x_offset += (p_bin[0] - '0') * 10000000.0;
		x_offset += (p_bin[1] - '0') * 1000000.0;
		x_offset += (p_bin[2] - '0') * 100000.0;
		x_offset += (p_bin[3] - '0') * 10000.0;
		x_offset += (p_bin[4] - '0') * 1000.0;
		x_offset += (p_bin[5] - '0') * 100.0;
		x_offset += (p_bin[6] - '0') * 10.0;
		x_offset += (p_bin[7] - '0') * 1.0;

		y_offset += (p_bin[8] - '0') * 10000000.0;
		y_offset += (p_bin[9] - '0') * 1000000.0;
		y_offset += (p_bin[10] - '0') * 100000.0;
		y_offset += (p_bin[11] - '0') * 10000.0;
		y_offset += (p_bin[12] - '0') * 1000.0;
		y_offset += (p_bin[13] - '0') * 100.0;
		y_offset += (p_bin[14] - '0') * 10.0;
		y_offset += (p_bin[15] - '0') * 1.0;

		lon = x0 + x_offset;
		lat = y0 + y_offset;

		g_sprintf (line, "%010Ld%010Ld", lon, lat);
		/*
		   g_print("slp:%d:%s\n",i, line);
		 */

		slp->v[i] = g_strdup (line);
	}

	g_free (bin);

	return slp;
}
