/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <glib/gprintf.h>
#include <math.h>

#include "config.h"

#include "mmap.h"
#include "gsi.h"
#include "ww_data.h"
#include "ww_object.h"
#include "ww_mmap.h"
#include "ww_smap.h"
#include "ww_layerset.h"
#include "util.h"
#include "camera.h"
#include "color.h"

typedef struct _WwMmapPrivate WwMmapPrivate;
struct _WwMmapPrivate {
	gboolean dispose_has_run;

	/*****/

	gdouble max_alt;

	gdouble north;
	gdouble south;
	gdouble west;
	gdouble east;

	GSList *sl_smap;			/* WwSmap */
	GMutex *mutex_sl_smap;

	gdouble old_vertical_exaggeration;
};

#define WW_MMAP_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), WW_TYPE_MMAP, WwMmapPrivate))

static WwObjectClass *parent_class = NULL;

static void ww_mmap_class_init (WwMmapClass * klass);
static void ww_mmap_init (WwMmap * object);
static void ww_mmap_dispose (GObject * object);
static void ww_mmap_finalize (GObject * object);
static GObject *ww_mmap_constructor (GType type, guint n_props, GObjectConstructParam * props);
static void ww_mmap_interface_init (gpointer g_iface, gpointer iface_data);

/*****************************************************************************/

static void ww_mmap_set_attribute (WwData * data, const gchar ** name, const gchar ** value);
static void ww_mmap_set_element (WwData * data, const gchar * element0, const gchar * element1, const gchar * value);
static void ww_mmap_set_parent (WwData * data, WwData * parent);
static void ww_mmap_set_child (WwData * data, WwData * child);

static void ww_mmap_update (WwObject * self);
static void ww_mmap_render (WwObject * self);
static void ww_mmap_set_on (WwObject * self, gboolean);
static void ww_mmap_debug_print (WwObject * self);

/*****************************************************************************/

GType
ww_mmap_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (WwMmapClass),
			NULL,				/* base_init */
			NULL,				/* base_finalize */
			(GClassInitFunc) ww_mmap_class_init,
			NULL,				/* class_finalize */
			NULL,				/* class_data */
			sizeof (WwMmap),
			0,					/* n_preallocs */
			(GInstanceInitFunc) ww_mmap_init
		};

		static const GInterfaceInfo ww_data_info = {
			(GInterfaceInitFunc) ww_mmap_interface_init,	/* interface_init */
			NULL,				/* interface_finalize */
			NULL				/* interface_data */
		};

		type = g_type_register_static (WW_TYPE_OBJECT, "WwMmap", &info, 0);

		g_type_add_interface_static (type, WW_TYPE_DATA, &ww_data_info);
		/*
		   g_print ("ww_mmap_get_type\n");
		 */
	}

	return type;
}

static void
ww_mmap_class_init (WwMmapClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	WwObjectClass *wwobject_class = WW_OBJECT_CLASS (klass);

	/*
	   g_print ("ww_mmap_class_init:c:%p:\n", klass);
	 */

	g_type_class_add_private (klass, sizeof (WwMmapPrivate));

	parent_class = g_type_class_peek_parent (klass);
	object_class->constructor = ww_mmap_constructor;
	object_class->dispose = ww_mmap_dispose;
	object_class->finalize = ww_mmap_finalize;

	wwobject_class->update = ww_mmap_update;
	wwobject_class->render = ww_mmap_render;
	wwobject_class->set_on = ww_mmap_set_on;
	wwobject_class->debug_print = ww_mmap_debug_print;
}

static void
ww_mmap_init (WwMmap * self)
{
	WwMmapPrivate *priv = WW_MMAP_GET_PRIVATE (self);

	/*
	   g_print ("ww_mmap_init:o:%p:\n", self);
	 */

	priv->dispose_has_run = FALSE;
	priv->max_alt = 15000;
	priv->north = 45.0 + 33.0 / 60.0;
	priv->south = 20.0 + 25.0 / 60.0;
	priv->west = 122.0 + 56.0 / 60.0;
	priv->east = 153.0 + 59.0 / 60.0;

	priv->sl_smap = NULL;
	priv->mutex_sl_smap = g_mutex_new ();

	priv->old_vertical_exaggeration = mmap_get_vertical_exaggeration ();

    /* セットはしてるが削除をしてない。mmap は一つしかないはずなので、このまま放置する */
	color_set_mmap (WW_MMAP (self));
}

static void
ww_mmap_dispose (GObject * obj)
{
	WwMmap *self = WW_MMAP (obj);
	WwMmapPrivate *priv = WW_MMAP_GET_PRIVATE (self);
	GSList *sl;

	g_print ("ww_mmap_dispose\n");

	if (priv->dispose_has_run) {
		return;
	}
	priv->dispose_has_run = TRUE;

	g_mutex_lock (priv->mutex_sl_smap);

	for (sl = priv->sl_smap; sl != NULL; sl = sl->next) {
		WwSmap *smap = (WwSmap *) sl->data;

		g_object_unref (smap);
	}
	g_slist_free (priv->sl_smap);
	priv->sl_smap = NULL;

	g_mutex_unlock (priv->mutex_sl_smap);

	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void
ww_mmap_finalize (GObject * obj)
{
	WwMmap *self = WW_MMAP (obj);
	WwMmapPrivate *priv = WW_MMAP_GET_PRIVATE (self);
	/*
	   g_print ("finalize\n");
	 */
	g_mutex_free (priv->mutex_sl_smap);


	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static GObject *
ww_mmap_constructor (GType type, guint n_props, GObjectConstructParam * props)
{
	GObject *object;
	GObjectClass *object_class = G_OBJECT_CLASS (parent_class);

	/*
	   g_print ("constructor\n");
	 */

	object = object_class->constructor (type, n_props, props);

	return object;
}

static void
ww_mmap_interface_init (gpointer g_iface, gpointer iface_data)
{
	WwDataInterface *iface = (WwDataInterface *) g_iface;

	iface->set_attribute = ww_mmap_set_attribute;
	iface->set_element = ww_mmap_set_element;
	iface->set_parent = ww_mmap_set_parent;
	iface->set_child = ww_mmap_set_child;
}

WwObject *
ww_mmap_new (void)
{
	GObject *object;

	object = g_object_new (WW_TYPE_MMAP, NULL);

	/*
	   g_print ("ww_mmap_new:o:%p\n", object);
	 */


	return WW_OBJECT (object);
}

/********************************************************************/

static void
remove_top_level (WwMmap * tile)
{
	WwMmapPrivate *priv = WW_MMAP_GET_PRIVATE (tile);
	GSList *sl;
	GSList *prev;
	GSList *next;

	/*
	   g_print ("remove_top_level\n");
	 */

	for (sl = priv->sl_smap, prev = NULL; sl != NULL; sl = next) {
		WwSmap *smap = sl->data;

		next = sl->next;

		if (ww_smap_get_need_flag (smap) == FALSE) {


			ww_smap_is_detached_from_render_list (smap);
			g_object_unref (smap);

			if (prev == NULL) {
				priv->sl_smap = next;
			} else {
				prev->next = next;
			}
			g_slist_free_1 (sl);

		} else {
			prev = sl;
		}
	}
}

static void
ww_mmap_clear_flag (WwMmap * tile)
{
	WwMmapPrivate *priv = WW_MMAP_GET_PRIVATE (tile);
	GSList *sl;

	for (sl = priv->sl_smap; sl != NULL; sl = sl->next) {
		WwSmap *smap = (WwSmap *) sl->data;

		/*
		   g_print ("ww_mmap_clear_flag:%s\n", ww_smap_get_name (smap));
		 */

		ww_smap_clear_flag (smap);
	}
}

static gboolean
is_created (GSList * sl_smap, const gchar * name)
{
	GSList *sl;

	for (sl = sl_smap; sl != NULL; sl = sl->next) {
		WwSmap *smap = sl->data;
		const gchar *name_cmp;

		name_cmp = ww_smap_get_name (smap);

		if (g_strcasecmp (name_cmp, name) == 0) {
			ww_smap_set_need_flag (smap, TRUE);

			ww_smap_update (smap);

			return TRUE;
		}
	}
	return FALSE;
}

/* 11112233 のような地図名の左下の経度緯度を求める */
static void
gsi_name_to_keidoido (const gchar * name, gint * x1, gint * y1)
{
	/* 注意：並び方は (**,**,*,*,*,*) = (ly,lx,my,mx,sy,sx) */
	gint lx, ly;				/* ????**** */
	gint mx, my;				/* ****??** */
	gint sx, sy;				/* ******?? */

	DB (g_print ("gsi_name_to_keidoido:%s:\n", name));

	sscanf (name, "%2d%2d%1d%1d%1d%1d", &ly, &lx, &my, &mx, &sy, &sx);

	DB (g_print ("%d %d %d %d %d %d\n", ly, lx, my, mx, sy, sx));

	*x1 = 3600 * (100 + lx) + mx * W_SECOND * 2 + sx / 5 * W_SECOND;
	*y1 = 2400 * ly + my * H_SECOND * 2 + sy / 5 * H_SECOND;
	/*
	 *y1 = 3600 * (ly * 2 / 3) + my * H_SECOND * 2 + sy / 5 * H_SECOND;
	 */

	DB (g_print ("name_to_keidoido:%s:%d %d\n", name, *x1, *y1));
}

/* 経度ど緯度を秒でもらって地図の名前 11112233 を得る */
static gchar *
gsi_keidoido_to_name (gdouble x, gdouble y)
{
	gint lx, ly;				/* ????**** */
	gint mx, my;				/* ****??** */
	gint sx, sy;				/* ******?? */
	gchar name[10];
	/*
	   g_print ("gsi_keidoido_to_name:%.1f %.1f\n", x, y);
	 */

	lx = (gint) x / 3600 - 100;
	ly = (gint) y / 2400;

	mx = ((gint) x % 3600) / (W_SECOND * 2);
	my = ((gint) y % 2400) / (H_SECOND * 2);

	sx = 5 * (gint) (((gint) x % (W_SECOND * 2)) / W_SECOND);
	sy = 5 * (gint) (((gint) y % (H_SECOND * 2)) / H_SECOND);

	/* l,m,s ともに緯度 経度の順になることに注意 */
	g_sprintf (name, "%2d%2d%1d%1d%1d%1d", ly, lx, my, mx, sy, sx);

	/*
	   g_print ("%s:%2d %2d %d %d %d %d\n", name, ly, lx, my, mx, sy, sx);
	 */

	return g_strdup (name);
}


static void
update_top_level (WwMmap * tile)
{
	WwMmapPrivate *priv = WW_MMAP_GET_PRIVATE (tile);
	gint i, j;
	gdouble north, south;
	gdouble west, east;
	gchar *name;
	gint x0, y0;
	gint x1, y1;
	gdouble diff;
	gdouble lon, lat;

	camera_get_view_narrow (&west, &south, &east, &north);	/* 見える範囲。要修正。極付近は大間違い。 */

	/*
	g_print ("update_top_level:old w:%.2f e:%.2f s:%.2f n:%.2f\n", west, east, south, north);
	*/

	lon = (east + west) / 2.0;
	lat = (north + south) / 2.0;
	diff = east - west;
	if (diff < 2.0 * W_SECOND / 3600.0) {
		west = lon - 1.0 * W_SECOND / 3600.0;
		east = lon + 1.0 * W_SECOND / 3600.0;
		south = lat - 1.0 * H_SECOND / 3600.0;
		north = lat + 1.0 * H_SECOND / 3600.0;
	}
	/*
	   g_print ("update_top_level:new w:%.2f e:%.2f s:%.2f n:%.2f   lon:%.2f lat:%.2f\n", west, east, south, north, lon, lat);
	 */

	/* 一番左下の地図の名前を調べる */
	name = gsi_keidoido_to_name (west * 3600.0, south * 3600.0);
	/* その地図の左下角の経度と緯度を調べる */
	gsi_name_to_keidoido (name, &x0, &y0);
	g_free (name);

	x1 = (gint) (east * 3600.0);
	y1 = (gint) (north * 3600.0);

	for (j = y0; j < y1; j += H_SECOND) {
		for (i = x0; i < x1; i += W_SECOND) {
			gchar *name;

			name = gsi_keidoido_to_name (i, j);

			/*
			   g_print ("update_top_level:check:%s (%d, %d)\n", name, i, j);
			 */

			/* is_created() の中で smap をアップデート */
			if (is_created (priv->sl_smap, name) == FALSE) {
				WwSmap *smap;

				/*
				   g_print ("new_smap:###### ok ##### name:%s x:%d y:%d\n", name, i, j);
				 */

				smap = ww_smap_new_with_name (tile, i, j, name);
				priv->sl_smap = g_slist_prepend (priv->sl_smap, smap);

			}
			g_free (name);
		}
	}
}

static void
vertical_exaggeration_is_changed (WwMmap * mmap)
{
	GSList *sl;
	WwMmapPrivate *priv = WW_MMAP_GET_PRIVATE (mmap);

	for (sl = priv->sl_smap; sl != NULL; sl = sl->next) {
		WwSmap *smap = sl->data;

		ww_smap_change_vertical_exaggeration (smap);
	}
}

static void
ww_mmap_update (WwObject * self)
{
	WwMmap *mmap;
	WwMmapPrivate *priv;
	gdouble lon;
	gdouble lat;
	gdouble alt;
	gdouble new_vertical_exaggeration;

	/*
	   g_print ("ww_mmap_update:%s\n", self->name);
	 */

	if (self == NULL) {
		return;
	}

	lon = camera_get_lon ();
	lat = camera_get_lat ();
	alt = camera_get_dist_target ();	/* 操作方法かえた時に見直すこと */

	mmap = WW_MMAP (self);
	priv = WW_MMAP_GET_PRIVATE (mmap);

	ww_mmap_clear_flag (mmap);

	if (self->is_on == TRUE) {
	    if (alt < priv->max_alt && camera_check_overlap (priv->north, priv->south, priv->west, priv->east) == TRUE) {
		    update_top_level (mmap);
	    }
    }

	g_mutex_lock (priv->mutex_sl_smap);
	remove_top_level (mmap);	/* need_flag の落ちたトップレベルの child を削除する。 */
	g_mutex_unlock (priv->mutex_sl_smap);

	new_vertical_exaggeration = mmap_get_vertical_exaggeration ();

	if (priv->old_vertical_exaggeration != new_vertical_exaggeration) {
		vertical_exaggeration_is_changed (mmap);

		priv->old_vertical_exaggeration = new_vertical_exaggeration;
	}
}

static void
ww_mmap_render (WwObject * self)
{
	GSList *sl;
	WwMmapPrivate *priv = WW_MMAP_GET_PRIVATE (self);

	if (self->is_on == FALSE) {
		return;
	}
	/*
	   g_print ("ww_mmap_render:%s\n", self->name);
	 */

	g_mutex_lock (priv->mutex_sl_smap);
	for (sl = priv->sl_smap; sl != NULL; sl = sl->next) {
		WwSmap *smap = sl->data;

		ww_smap_render (smap);
	}
	g_mutex_unlock (priv->mutex_sl_smap);

	/*
	   g_print ("ww_mmap_render:%s:end\n", self->name);
	 */
}

static void
ww_mmap_set_attribute (WwData * data, const gchar ** name, const gchar ** value)
{
	WwDataInterface *parent_wwdata_iface = g_type_interface_peek (parent_class, WW_TYPE_DATA);

	/*
	   g_print ("ww_mmap_set_attribute:o:%p\n", data);
	 */

	/* 専用の attribute はない */

	parent_wwdata_iface->set_attribute (data, name, value);
}

static void
ww_mmap_set_element (WwData * data, const gchar * element0, const gchar * element1, const gchar * value)
{
	WwDataInterface *parent_wwdata_iface = g_type_interface_peek (parent_class, WW_TYPE_DATA);

	/*
	   g_print ("ww_mmap_set_element:o:%p e0:%s e1:%s value:%s\n", data, element0, element1, value);
	 */

	/* 独自の element なし */

	parent_wwdata_iface->set_element (data, element0, element1, value);
}

static void
ww_mmap_set_parent (WwData * data, WwData * parent)
{
	WwDataInterface *parent_wwdata_iface = g_type_interface_peek (parent_class, WW_TYPE_DATA);

	/* 親のままでいい */
	parent_wwdata_iface->set_parent (data, parent);
}

static void
ww_mmap_set_child (WwData * data, WwData * child)
{
	/* なにもしない */
}

static void
ww_mmap_set_on (WwObject * obj, gboolean is_on)
{
	WwObject *obj_parent = obj->wwparent;

	if (is_on == TRUE && WW_IS_LAYERSET (obj_parent)) {
		/* is_on == TRUE の時のみでないと、ループする */
		ww_layerset_check_show_only_one_layer (WW_LAYERSET (obj_parent));
	}

	parent_class->set_on (obj, is_on);
}

static void
ww_mmap_debug_print (WwObject * obj)
{
	WwMmapPrivate *priv = WW_MMAP_GET_PRIVATE (obj);

	g_print ("ww_mmap_debug_print##########:o:%p\n", obj);

	parent_class->debug_print (obj);

	g_print ("\t north:%.2f\n", priv->north);
	g_print ("\t south:%.2f\n", priv->south);
	g_print ("\t west:%.2f\n", priv->west);
	g_print ("\t east:%.2f\n", priv->east);
}

void
ww_mmap_color_change_all (WwMmap * mmap)
{
	WwMmapPrivate *priv;
	GSList *sl;

	g_return_if_fail (mmap != NULL);

	priv = WW_MMAP_GET_PRIVATE (mmap);

	for (sl = priv->sl_smap; sl != NULL; sl = sl->next) {
		WwSmap *smap = sl->data;

		ww_smap_color_set (smap);
	}
}
