/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <math.h>

#include "config.h"

#include "mmap.h"
#include "ww_data.h"
#include "ww_object.h"
#include "ww_quadtile.h"
#include "ww_layerset.h"
#include "ww_quadchild.h"
#include "util.h"
#include "camera.h"

typedef struct _WwQuadtilePrivate WwQuadtilePrivate;
struct _WwQuadtilePrivate {
	gboolean dispose_has_run;

	/*****/

	gdouble distance_above_surface;

	gdouble north;
	gdouble south;
	gdouble west;
	gdouble east;

	gboolean terrain_mapped;

	/* 下の２つは使われてない？ */
	gdouble tile_draw_distance_factor;
	gdouble tile_spread_factor;

	/* ImageAccessor 関係 */
	gdouble level_zero_tile_size_degrees;
	gint number_levels;
	gint texture_size_pixels;
	gchar *image_file_extension;
	gchar *permanent_directory;	/* 使われてない */

	gchar *server_url;
	gchar *data_set_name;
	/*
	   cache_expiration_time;
	 */
	guint days;
	guint hours;
	guint mins;
	guint seconds;
	gchar *server_logo_file_path;

	GSList *sl_child;			/* WwQuadchild * */
	GMutex *mutex_sl_child;

	gdouble old_vertical_exaggeration;
};

#define WW_QUADTILE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), WW_TYPE_QUADTILE, WwQuadtilePrivate))

static WwObjectClass *parent_class = NULL;

static void ww_quadtile_class_init (WwQuadtileClass * klass);
static void ww_quadtile_init (WwQuadtile * object);
static void ww_quadtile_dispose (GObject * object);
static void ww_quadtile_finalize (GObject * object);
static GObject *ww_quadtile_constructor (GType type, guint n_props, GObjectConstructParam * props);
static void ww_quadtile_interface_init (gpointer g_iface, gpointer iface_data);

/*****************************************************************************/

static void ww_quadtile_set_attribute (WwData * data, const gchar ** name, const gchar ** value);
static void ww_quadtile_set_element (WwData * data, const gchar * element0, const gchar * element1, const gchar * value);
static void ww_quadtile_set_parent (WwData * data, WwData * parent);
static void ww_quadtile_set_child (WwData * data, WwData * child);

static void ww_quadtile_update (WwObject * self);
static void ww_quadtile_render (WwObject * self);
static void ww_quadtile_set_on (WwObject * self, gboolean);
static void ww_quadtile_debug_print (WwObject * self);

/*****************************************************************************/

GType
ww_quadtile_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (WwQuadtileClass),
			NULL,				/* base_init */
			NULL,				/* base_finalize */
			(GClassInitFunc) ww_quadtile_class_init,
			NULL,				/* class_finalize */
			NULL,				/* class_data */
			sizeof (WwQuadtile),
			0,					/* n_preallocs */
			(GInstanceInitFunc) ww_quadtile_init
		};

		static const GInterfaceInfo ww_data_info = {
			(GInterfaceInitFunc) ww_quadtile_interface_init,	/* interface_init */
			NULL,				/* interface_finalize */
			NULL				/* interface_data */
		};

		type = g_type_register_static (WW_TYPE_OBJECT, "WwQuadtile", &info, 0);

		g_type_add_interface_static (type, WW_TYPE_DATA, &ww_data_info);
		/*
		   g_print ("ww_quadtile_get_type\n");
		 */
	}

	return type;
}

static void
ww_quadtile_class_init (WwQuadtileClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	WwObjectClass *wwobject_class = WW_OBJECT_CLASS (klass);

	/*
	   g_print ("ww_quadtile_class_init:c:%p:\n", klass);
	 */

	g_type_class_add_private (klass, sizeof (WwQuadtilePrivate));

	parent_class = g_type_class_peek_parent (klass);
	object_class->constructor = ww_quadtile_constructor;
	object_class->dispose = ww_quadtile_dispose;
	object_class->finalize = ww_quadtile_finalize;

	wwobject_class->update = ww_quadtile_update;
	wwobject_class->render = ww_quadtile_render;
	wwobject_class->set_on = ww_quadtile_set_on;
	wwobject_class->debug_print = ww_quadtile_debug_print;
}

static void
ww_quadtile_init (WwQuadtile * self)
{
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (self);

	/*
	   g_print ("ww_quadtile_init:o:%p:\n", self);
	 */

	priv->dispose_has_run = FALSE;
	priv->distance_above_surface = 0.0;
	priv->north = 0.0;
	priv->south = 0.0;
	priv->west = 0.0;
	priv->east = 0.0;
	priv->terrain_mapped = FALSE;
	priv->tile_draw_distance_factor = 0.0;
	priv->tile_spread_factor = 0.0;

	priv->level_zero_tile_size_degrees = 0.0;
	priv->number_levels = 0;
	priv->texture_size_pixels = 0;
	priv->image_file_extension = NULL;
	priv->permanent_directory = NULL;

	priv->server_url = NULL;
	priv->data_set_name = NULL;
	priv->days = 0;
	priv->hours = 0;
	priv->mins = 0;
	priv->seconds = 0;
	priv->server_logo_file_path = NULL;

	priv->sl_child = NULL;
	priv->mutex_sl_child = g_mutex_new ();
	priv->old_vertical_exaggeration = mmap_get_vertical_exaggeration ();
}

static void
ww_quadtile_dispose (GObject * obj)
{
	WwQuadtile *self = WW_QUADTILE (obj);
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (self);
	GSList *sl;

	/*
	   g_print ("ww_quadtile_dispose\n");
	 */

	if (priv->dispose_has_run) {
		return;
	}
	priv->dispose_has_run = TRUE;

	g_mutex_lock (priv->mutex_sl_child);
	for (sl = priv->sl_child; sl != NULL; sl = sl->next) {
		WwQuadchild *child = (WwQuadchild *) sl->data;

		g_object_unref (child);
	}
	g_slist_free (priv->sl_child);
	priv->sl_child = NULL;
	g_mutex_unlock (priv->mutex_sl_child);
	/*
	   g_object_unref(priv->image_accessor);
	 */

	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void
ww_quadtile_finalize (GObject * obj)
{
	WwQuadtile *self = WW_QUADTILE (obj);
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (self);
	/*
	   g_print ("finalize\n");
	 */
	g_free (priv->image_file_extension);
	g_free (priv->permanent_directory);
	g_free (priv->server_url);
	g_free (priv->data_set_name);
	g_free (priv->server_logo_file_path);
	g_mutex_free (priv->mutex_sl_child);


	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static GObject *
ww_quadtile_constructor (GType type, guint n_props, GObjectConstructParam * props)
{
	GObject *object;
	GObjectClass *object_class = G_OBJECT_CLASS (parent_class);

	/*
	   g_print ("constructor\n");
	 */

	object = object_class->constructor (type, n_props, props);

	return object;
}

static void
ww_quadtile_interface_init (gpointer g_iface, gpointer iface_data)
{
	WwDataInterface *iface = (WwDataInterface *) g_iface;

	iface->set_attribute = ww_quadtile_set_attribute;
	iface->set_element = ww_quadtile_set_element;
	iface->set_parent = ww_quadtile_set_parent;
	iface->set_child = ww_quadtile_set_child;
}


WwObject *
ww_quadtile_new (void)
{
	GObject *object;

	object = g_object_new (WW_TYPE_QUADTILE, NULL);

	/*
	   g_print ("ww_quadtile_new:o:%p\n", object);
	 */

	return WW_OBJECT (object);
}

/********************************************************************/

/* 視点からの距離とレベルゼロの tile_deg から、今どのレベルを表示するべきかを返す。
 * ####### 物凄いテキトーな計算 ####### 要修正 ###########
 * 表示するべきでないなら -1 を返す。*/
static gint
get_level (gdouble level_zero_tile_deg, gint number_levels)
{
	gint i;
	gdouble l;
	gdouble dist = camera_get_dist_target ();

	/* 視野角55度の時の注視点付近での視野範囲の距離 */
	l = 2.0 * dist * tan (2.0 * G_PI * 55.0 / 2.0 / 360.0);

	for (i = 0; i < number_levels; ++i) {
		gdouble ll;
		gdouble deg = level_zero_tile_deg / pow (2.0, i);
		gdouble radius = mmap_get_world_radius ();

		/* 今のレベルでのタイル１枚の辺の距離 */
		ll = 2.0 * radius * tan (2.0 * G_PI * deg / 2.0 / 360.0);

		/*
		   g_print("get_level:dist:%.2f   l:%.2f  ll:%.2f\n", dist, l, ll);
		 */

		if (i == 0 && l > 3.0 * ll) {
			return -1;
		}

		if (l > 2.5 * ll) {
			/*
			   g_print("get_level:%d\n", i);
			 */
			return i;
		}
	}
	return number_levels - 1;
}

static void
remove_child (WwQuadtile * tile, gint level)
{
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (tile);
	GSList *sl;
	/*
	   g_print ("quadtile:remove_child\n");
	 */
	for (sl = priv->sl_child; sl != NULL; sl = sl->next) {
		WwQuadchild *child = sl->data;
		/*  
		   util_print_ref_count ("remove_child: ref_count:%d\n", child);
		 */
		ww_quadchild_remove_child (child, level);
	}
}

static void
remove_top_level (WwQuadtile * tile, gint level)
{
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (tile);
	GSList *sl;
	GSList *prev;
	GSList *next;
	/*
	   g_print ("remove_top_level\n");
	 */
	for (sl = priv->sl_child, prev = NULL; sl != NULL; sl = next) {
		WwQuadchild *child = sl->data;

		next = sl->next;
		/*
		   util_print_ref_count ("remove_top_level: ref_count:%d\n", child);
		 */

		if (ww_quadchild_get_need_flag (child) == FALSE || level < 0) {
			g_object_unref (child);
			/*
			   util_print_ref_count ("               removed: ref_count:%d\n", child);
			 */

			if (prev == NULL) {
				priv->sl_child = next;
			} else {
				prev->next = next;
			}
			g_slist_free_1 (sl);

		} else {
			prev = sl;
		}
	}
}

static void
ww_quadtile_clear_flag (WwQuadtile * tile)
{
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (tile);
	GSList *sl;

	for (sl = priv->sl_child; sl != NULL; sl = sl->next) {
		WwQuadchild *child = (WwQuadchild *) sl->data;

		ww_quadchild_clear_flag (child);
	}
}

static gboolean
is_created (GSList * sl_child, gint x, gint y)
{
	GSList *sl;

	for (sl = sl_child; sl != NULL; sl = sl->next) {
		WwQuadchild *child = sl->data;

		if (ww_quadchild_check_xy (child, x, y) == TRUE) {
			ww_quadchild_set_need_flag (child, TRUE);
			return TRUE;
		}
	}
	return FALSE;
}

static void
update_top_level (WwQuadtile * tile)
{
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (tile);
	gint i, j;
	gint x0, y0;
	gint x1, y1;
	gdouble north, south, west, east;
	gdouble tile_deg = priv->level_zero_tile_size_degrees;

	camera_get_view_wide (&west, &south, &east, &north);	/* 見える範囲。要修正。極付近は大間違い。 */

	util_tile_get_view (west, south, east, north, tile_deg, &x0, &y0, &x1, &y1);
	/*
	g_print ("update_top_level:w:%.2f e:%.2f s:%.2f n:%.2f\n", west, east, south, north);
	g_print ("update_top_level:(x0,y0)-(x1,y1):(%d,%d)-(%d,%d)\n", x0, y0, x1, y1);
	*/

	/* 範囲が東経180度をまたぐ時は、下の for で使えないので下駄をかます */
	if (x0 > x1) {
		x1 += 360.0 / (tile_deg);
	}

	for (j = y0; j <= y1; ++j) {
		for (i = x0; i <= x1; ++i) {
			gint k;

			if (i > round (360.0 / tile_deg) - 1) {
				k = i - round (360.0 / tile_deg);
			} else {
				k = i;
			}

			if (is_created (priv->sl_child, k, j) == FALSE) {
				WwQuadchild *new_child;

				new_child = ww_quadchild_new_simple (tile, NULL, k, j, 0, tile_deg);
				priv->sl_child = g_slist_prepend (priv->sl_child, new_child);
			}
		}
	}
}


static void
update_child (WwQuadtile * tile, gint level)
{
	GSList *sl;
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (tile);

	for (sl = priv->sl_child; sl != NULL; sl = sl->next) {
		WwQuadchild *child = sl->data;

		ww_quadchild_update (child, level);
	}
}

static void
vertical_exaggeration_is_changed (WwQuadtile * tile)
{
	GSList *sl;
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (tile);

	for (sl = priv->sl_child; sl != NULL; sl = sl->next) {
		WwQuadchild *child = sl->data;

		ww_quadchild_change_vertical_exaggeration (child);
	}
}

static void
ww_quadtile_update (WwObject * self)
{
	WwQuadtile *tile;
	WwQuadtilePrivate *priv;
	gint level;
	gdouble new_vertical_exaggeration;

	if (self == NULL) {
		return;
	}

	tile = WW_QUADTILE (self);
	priv = WW_QUADTILE_GET_PRIVATE (tile);

	level = get_level (priv->level_zero_tile_size_degrees, priv->number_levels);

    /*
	g_print ("ww_quadtile_update:level(target level:%d):%s\n", level, self->name);
    */

	ww_quadtile_clear_flag (tile);

	if (self->is_on == TRUE) {
	    /*
	    g_print("wwquadtile_updat:%.2f %.2f %.2f %.2f\n", priv->west, priv->south, priv->east, priv->north);
	    */
	    if (level >= 0 && camera_check_overlap (priv->north, priv->south, priv->west, priv->east) == TRUE) {
		    /* レベル0 だけを作りなおす。無いものを作り、不要なものの need フラグを落とす。削除はしない。 */
		    update_top_level (tile);
	    }
    }
	g_mutex_lock (priv->mutex_sl_child);
	remove_top_level (tile, level);	/* need_flag の落ちたトップレベルの child を削除する。 */
	g_mutex_unlock (priv->mutex_sl_child);

	if (self->is_on == TRUE) {
	    /* level1 以上の need_flag はここで付け直す */
	    update_child (tile, level);
    }
	g_mutex_lock (priv->mutex_sl_child);
	/* need_flag の落ちた child を削除するとともに、不要になったテクスチャーとメッシュを削除する。 */
	remove_child (tile, level);
	g_mutex_unlock (priv->mutex_sl_child);

	new_vertical_exaggeration = mmap_get_vertical_exaggeration ();

	if (priv->old_vertical_exaggeration != new_vertical_exaggeration) {
		vertical_exaggeration_is_changed (tile);

		priv->old_vertical_exaggeration = new_vertical_exaggeration;
	}

	/*
	   g_print ("ww_quadtile_update::end\n");
	 */
}

static void
ww_quadtile_render (WwObject * obj)
{
	GSList *sl;
	WwQuadtilePrivate *priv;

	if (obj == NULL) {
		return;
	}

	priv = WW_QUADTILE_GET_PRIVATE (obj);

	if (obj->is_on == FALSE) {
		return;
	}
	/*
	   g_print ("ww_quadtile_render:%s\n", obj->name);
	 */

	g_mutex_lock (priv->mutex_sl_child);
	for (sl = priv->sl_child; sl != NULL; sl = sl->next) {
		WwQuadchild *child = sl->data;

		ww_quadchild_render (child);
	}
	g_mutex_unlock (priv->mutex_sl_child);

	/*
	   g_print ("ww_quadtile_render:%s:end\n", self->name);
	 */
}

static void
ww_quadtile_set_attribute (WwData * data, const gchar ** name, const gchar ** value)
{
	WwDataInterface *parent_wwdata_iface = g_type_interface_peek (parent_class, WW_TYPE_DATA);
	/*
	   g_print ("ww_quadtile_set_attribute:o:%p\n", obj);
	 */

	/* 専用の attribute はない */

	parent_wwdata_iface->set_attribute (data, name, value);
}

static void
ww_quadtile_set_element (WwData * data, const gchar * element0, const gchar * element1, const gchar * value)
{
	WwQuadtile *obj = WW_QUADTILE (data);
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (obj);
	WwDataInterface *parent_wwdata_iface = g_type_interface_peek (parent_class, WW_TYPE_DATA);
	/*
	   g_print ("ww_quadtile_set_element:o:%p e0:%s e1:%s value:%s\n", obj, element0, element1, value);
	 */

	if (g_ascii_strcasecmp (element0, "DistanceAboveSurface") == 0) {
		priv->distance_above_surface = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "Value") == 0 && g_ascii_strcasecmp (element1, "North") == 0) {
		priv->north = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "Value") == 0 && g_ascii_strcasecmp (element1, "South") == 0) {
		priv->south = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "Value") == 0 && g_ascii_strcasecmp (element1, "West") == 0) {
		priv->west = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "Value") == 0 && g_ascii_strcasecmp (element1, "East") == 0) {
		priv->east = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "TerrainMapped") == 0) {
		priv->terrain_mapped = util_char_to_boolean (value);

	} else if (g_ascii_strcasecmp (element0, "TileDrawDistanceFactor") == 0) {
		priv->tile_draw_distance_factor = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "TileSpreadFactor") == 0) {
		priv->tile_spread_factor = util_char_to_double (value);

/************/
	} else if (g_ascii_strcasecmp (element0, "LevelZeroTileSizeDegrees") == 0) {
		priv->level_zero_tile_size_degrees = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "NumberLevels") == 0) {
		priv->number_levels = util_char_to_int (value);

	} else if (g_ascii_strcasecmp (element0, "TextureSizePixels") == 0) {
		priv->texture_size_pixels = util_char_to_int (value);

	} else if (g_ascii_strcasecmp (element0, "ImageFileExtension") == 0) {
		priv->image_file_extension = g_strdup (value);

	} else if (g_ascii_strcasecmp (element0, "PermanantDirectory") == 0) {	/* typo ? 未使用なので未確認 */
		priv->permanent_directory = g_strdup (value);

	} else if (g_ascii_strcasecmp (element0, "ServerUrl") == 0) {
		priv->server_url = g_strdup (value);

	} else if (g_ascii_strcasecmp (element0, "DataSetName") == 0) {
		priv->data_set_name = g_strdup (value);

	} else if (g_ascii_strcasecmp (element0, "Days") == 0) {
		priv->days = util_char_to_int (value);

	} else if (g_ascii_strcasecmp (element0, "Hours") == 0) {
		priv->hours = util_char_to_int (value);

	} else if (g_ascii_strcasecmp (element0, "Mins") == 0) {
		priv->mins = util_char_to_int (value);

	} else if (g_ascii_strcasecmp (element0, "Seconds") == 0) {
		priv->seconds = util_char_to_int (value);

	} else if (g_ascii_strcasecmp (element0, "ServerLogoFilePath") == 0) {
		gchar *tmp = g_strdup (value);
		util_separator_dos_to_unix (tmp);
		priv->server_logo_file_path = tmp;

	} else {
		;
	}

	parent_wwdata_iface->set_element (data, element0, element1, value);
}

static void
ww_quadtile_set_parent (WwData * data, WwData * parent)
{
	WwDataInterface *parent_wwdata_iface = g_type_interface_peek (parent_class, WW_TYPE_DATA);

	/* 親のままでいい */
	parent_wwdata_iface->set_parent (data, parent);
}

static void
ww_quadtile_set_child (WwData * data, WwData * child)
{
	/* なにもしない */
}

static void
ww_quadtile_set_on (WwObject * obj, gboolean is_on)
{
	WwObject *obj_parent = obj->wwparent;

	if (is_on == TRUE && WW_IS_LAYERSET (obj_parent)) {
		/* is_on == TRUE の時のみでないと、ループする */
		ww_layerset_check_show_only_one_layer (WW_LAYERSET (obj_parent));
	}

	parent_class->set_on (obj, is_on);
}

static void
ww_quadtile_debug_print (WwObject * obj)
{
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (obj);

	g_print ("ww_quadtile_debug_print##########:o:%p\n", obj);

	parent_class->debug_print (obj);

	g_print ("\t distance_above_surface:%.2f\n", priv->distance_above_surface);
	g_print ("\t north:%.2f\n", priv->north);
	g_print ("\t south:%.2f\n", priv->south);
	g_print ("\t west:%.2f\n", priv->west);
	g_print ("\t east:%.2f\n", priv->east);
	util_print_bool ("\t terrain_mapped", priv->terrain_mapped);
	g_print ("\t tile_draw_distance_factor:%.2f\n", priv->tile_draw_distance_factor);
	g_print ("\t tile_spread_factor:%.2f\n", priv->tile_spread_factor);
}

gboolean
ww_quadtile_get_terrain_mapped (WwQuadtile * tile)
{
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (tile);

	return priv->terrain_mapped;
}

gdouble
ww_quadtile_get_above_surface (WwQuadtile * tile)
{
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (tile);

	return priv->distance_above_surface;
}

const gchar *
ww_quadtile_get_server_url (WwQuadtile * tile)
{
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (tile);

	return priv->server_url;
}

const gchar *
ww_quadtile_get_data_set_name (WwQuadtile * tile)
{
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (tile);

	return priv->data_set_name;
}

gdouble
ww_quadtile_get_level_zero_tile_size_degrees (WwQuadtile * tile)
{
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (tile);

	return priv->level_zero_tile_size_degrees;
}

gint
ww_quadtile_get_number_levels (WwQuadtile * tile)
{
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (tile);

	return priv->number_levels;
}

gint
ww_quadtile_get_texture_size_pixels (WwQuadtile * tile)
{
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (tile);

	return priv->texture_size_pixels;
}

const gchar *
ww_quadtile_get_image_file_extension (WwQuadtile * tile)
{
	WwQuadtilePrivate *priv = WW_QUADTILE_GET_PRIVATE (tile);

	return priv->image_file_extension;
}
