/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <gtk/gtk.h>
#include <gtk/gtkgl.h>

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glext.h>


#include "glarea.h"
#include "util.h"
#include "ww_textureloader.h"
#include "mmapthread.h"
#include "mmap.h"
#include "disk.h"

typedef struct _WwTextureloaderPrivate WwTextureloaderPrivate;
struct _WwTextureloaderPrivate {
	gboolean dispose_has_run;

	gchar *path_tex;			/* 圧縮テクスチャーのバイナリーデータのパス */
	gchar *path_img;			/* jpg や png などの画像ファイルのパス */
	gchar *uri;

	gint texture_width;
	gint texture_height;

	ImageManipulateFunc image_manipulate_func;
	GObject *func_data;

	void *data_tex;				/* 圧縮テクスチャーのデータ */
	gint size_tex;				/* 同サイズ */

	void *data_pixels;			/* jpg や png などの画像のピクセルデータ(gdkpixbuf から取った物 */
	gint size_pixels;			/* 同サイズ */

	void *data_img;				/* jpg や png などの画像のデータ(ダウンロードしたデータ） */
	gint size_img;				/* 同サイズ */

	WwTexture *texture;
};

#define WW_TEXTURELOADER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), WW_TYPE_TEXTURELOADER, WwTextureloaderPrivate))

static GObjectClass *parent_class = NULL;

static void ww_textureloader_class_init (WwTextureloaderClass * klass);
static void ww_textureloader_init (WwTextureloader * object);
static void ww_textureloader_finalize (GObject * object);
static void ww_textureloader_dispose (GObject * object);
static GObject *ww_textureloader_constructor (GType type, guint n_props, GObjectConstructParam * props);

/*****************************************************************************/

static GHashTable *hash_loader = NULL;

static void
textureloader_add (const gchar * uri, WwTextureloader * loader)
{
	g_hash_table_insert (hash_loader, (gchar *) uri, loader);
}

static void
textureloader_del (const gchar * uri)
{
	WwTextureloader *loader = NULL;

	loader = g_hash_table_lookup (hash_loader, uri);
	if (g_hash_table_remove (hash_loader, uri) == FALSE) {
		g_print ("textureloader_del: fail\n");
	}
}


WwTextureloader *
ww_textureloader_lookup (const gchar * uri)
{
	WwTextureloader *loader = NULL;

	if (hash_loader == NULL) {
		hash_loader = g_hash_table_new (g_str_hash, g_str_equal);
	}

	loader = g_hash_table_lookup (hash_loader, uri);
	if (loader != NULL) {
        /*
		g_print ("############# ww_textureloader_lookup: not NULL\n");
        */
	}


	return loader;
}

/*****************************************************************************/

GType
ww_textureloader_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (WwTextureloaderClass),
			NULL,				/* base_init */
			NULL,				/* base_finalize */
			(GClassInitFunc) ww_textureloader_class_init,
			NULL,				/* class_finalize */
			NULL,				/* class_data */
			sizeof (WwTextureloader),
			0,					/* n_preallocs */
			(GInstanceInitFunc) ww_textureloader_init
		};

		type = g_type_register_static (G_TYPE_OBJECT, "WwTextureloader", &info, 0);
	}

	return type;
}

static void
ww_textureloader_class_init (WwTextureloaderClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	/*
	   g_print ("ww_textureloader_class_init:c:%p:\n", klass);
	 */
	g_type_class_add_private (klass, sizeof (WwTextureloaderPrivate));

	parent_class = g_type_class_peek_parent (klass);
	object_class->constructor = ww_textureloader_constructor;
	object_class->finalize = ww_textureloader_finalize;
	object_class->dispose = ww_textureloader_dispose;

}

static void
ww_textureloader_init (WwTextureloader * self)
{
	WwTextureloaderPrivate *priv = WW_TEXTURELOADER_GET_PRIVATE (self);
	/*
	   g_print ("ww_textureloader_init:o:%p:\n", self);
	 */

	priv->dispose_has_run = FALSE;

	priv->path_tex = NULL;
	priv->path_img = NULL;
	priv->uri = NULL;
	priv->texture_width = 0;
	priv->texture_height = 0;
	priv->data_tex = NULL;
	priv->size_tex = 0;
	priv->data_pixels = NULL;
	priv->size_pixels = 0;
	priv->data_img = NULL;
	priv->size_img = 0;

	priv->texture = NULL;
	priv->image_manipulate_func = NULL;
	priv->func_data = NULL;
}

static void
ww_textureloader_dispose (GObject * obj)
{
	WwTextureloader *self = WW_TEXTURELOADER (obj);
	WwTextureloaderPrivate *priv = WW_TEXTURELOADER_GET_PRIVATE (self);

    /*
	g_print ("ww_textureloader_dispose\n");
    */

	/************************/
	textureloader_del (priv->uri);
	/************************/

	if (priv->dispose_has_run) {
		return;
	}
	priv->dispose_has_run = TRUE;

	if (priv->texture != NULL) {
		g_object_unref (priv->texture);
		priv->texture = NULL;
	}
	if (priv->func_data != NULL) {
		g_object_unref (priv->func_data);
		priv->func_data = NULL;
	}

	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void
ww_textureloader_finalize (GObject * obj)
{
	WwTextureloader *self = WW_TEXTURELOADER (obj);
	WwTextureloaderPrivate *priv = WW_TEXTURELOADER_GET_PRIVATE (self);
	/*
	   g_print ("ww_textureloader_finalize\n");
	 */

	if (priv->path_tex != NULL) {
		g_free (priv->path_tex);
	}
	if (priv->path_img != NULL) {
		g_free (priv->path_img);
	}
	if (priv->uri != NULL) {
		g_free (priv->uri);
	}
	if (priv->data_tex != NULL) {
		g_free (priv->data_tex);
	}
	if (priv->data_pixels != NULL) {
		g_free (priv->data_pixels);
	}
	if (priv->data_img != NULL) {
		g_free (priv->data_img);
	}

	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static GObject *
ww_textureloader_constructor (GType type, guint n_props, GObjectConstructParam * props)
{
	GObject *object;
	GObjectClass *object_class = G_OBJECT_CLASS (parent_class);
	/*
	   g_print ("ww_textureloader_constructor\n");
	 */
	object = object_class->constructor (type, n_props, props);

	return object;
}

WwTextureloader *
ww_textureloader_new (void)
{
	GObject *object;

	object = g_object_new (WW_TYPE_TEXTURELOADER, NULL);
	/*
	   g_print ("ww_textureloader_new:o:%p\n", object);
	 */
	return WW_TEXTURELOADER (object);
}

/********************************************************************/

static void
get_data_from_pixbuf (WwTextureloader * loader, GdkPixbuf * pixbuf)
{
	WwTextureloaderPrivate *priv = WW_TEXTURELOADER_GET_PRIVATE (loader);
	guchar *pixels;
	gint width;
	gint height;
	gint rowstride;
	gint size;

	if (pixbuf == NULL) {
		g_print ("error:::get_data_from_pixbuf: pixbuf == NULL\n");
		return;
	}

	if (priv->image_manipulate_func != NULL) {
		GdkPixbuf *pixbuf_new;

		pixbuf_new = priv->image_manipulate_func (priv->func_data, pixbuf);
		g_object_unref (priv->func_data);
		priv->func_data = NULL;

		pixels = gdk_pixbuf_get_pixels (pixbuf_new);
		width = gdk_pixbuf_get_width (pixbuf_new);
		height = gdk_pixbuf_get_height (pixbuf_new);
		rowstride = gdk_pixbuf_get_rowstride (pixbuf_new);
		size = rowstride * height;

		priv->data_pixels = g_memdup (pixels, size);
		priv->size_pixels = size;

		g_object_unref (pixbuf_new);
	} else {
		pixels = gdk_pixbuf_get_pixels (pixbuf);
		width = gdk_pixbuf_get_width (pixbuf);
		height = gdk_pixbuf_get_height (pixbuf);
		rowstride = gdk_pixbuf_get_rowstride (pixbuf);
		size = rowstride * height;

		priv->data_pixels = g_memdup (pixels, size);
		priv->size_pixels = size;
	}

}

static void
textureloader_texgen (GObject * obj)
{
	WwTextureloader *loader = WW_TEXTURELOADER (obj);
	WwTextureloaderPrivate *priv;
	/*
	   g_print ("textureloaer_start\n");
	 */

	priv = WW_TEXTURELOADER_GET_PRIVATE (loader);

	glarea_gl_begin ();
	{
		guint num;

		glGenTextures (1, &num);
		glBindTexture (GL_TEXTURE_2D, num);
		/*
		   glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
		   glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
		 */
		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);

		if (priv->data_tex != NULL) {
			gint texture_format;

			texture_format = glarea_get_texture_format ();
			glCompressedTexImage2D (GL_TEXTURE_2D, 0, texture_format, priv->texture_width, priv->texture_height, 0,
									priv->size_tex, priv->data_tex);

			g_free (priv->data_tex);
			priv->data_tex = NULL;
			priv->size_tex = 0;

		} else if (priv->data_pixels != NULL) {
			if (glarea_is_support_compress () == TRUE && glarea_is_direct () == TRUE) {
				gint size;
				gint format;
				/*
				   g_print("before compress\n");
				 */
				glTexImage2D (GL_TEXTURE_2D, 0, GL_COMPRESSED_RGB, priv->texture_width, priv->texture_height, 0, GL_RGB,
							  GL_UNSIGNED_BYTE, priv->data_pixels);
				/*
				   g_print("after compress\n");
				 */

				if (priv->path_tex != NULL) {
					gchar *data = NULL;

					glGetTexLevelParameteriv (GL_TEXTURE_2D, 0, GL_TEXTURE_COMPRESSED_IMAGE_SIZE, &size);
					glGetTexLevelParameteriv (GL_TEXTURE_2D, 0, GL_TEXTURE_INTERNAL_FORMAT, &format);
					/*
					   g_print ("internal_format:%x compressed_texture_size:%d\n", format, size);
					 */
					data = g_new (gchar, size);
					glGetCompressedTexImage (GL_TEXTURE_2D, 0, data);

					disk_save_bin (priv->path_tex, data, size);

					g_free (data);
				}
			} else {
				g_print ("before glteximage2d\n");
				glTexImage2D (GL_TEXTURE_2D, 0, GL_RGB, priv->texture_width, priv->texture_height, 0, GL_RGB, GL_UNSIGNED_BYTE,
							  priv->data_pixels);
			}

			g_free (priv->data_pixels);
			priv->data_pixels = NULL;
			priv->size_pixels = 0;

		} else {
			g_print ("textureloader_texgen:???????\n");
		}

		ww_texture_set_id (priv->texture, num);
	}
	glarea_gl_end ();

	glarea_force_update ();
}

static void
textureloader_download_to_data (GObject * obj)
{
	WwTextureloader *loader = WW_TEXTURELOADER (obj);
	WwTextureloaderPrivate *priv = WW_TEXTURELOADER_GET_PRIVATE (loader);
	GdkPixbuf *pixbuf;

	disk_save_bin (priv->path_img, priv->data_img, priv->size_img);
	pixbuf = util_bin_to_pixbuf (priv->data_img, priv->size_img);
	get_data_from_pixbuf (loader, pixbuf);

	g_free (priv->data_img);
	priv->data_img = NULL;
	priv->size_img = 0;

	g_object_unref (pixbuf);

	mmap_idle_add_single (textureloader_texgen, (GObject *) loader);
}


static void
textureloader_download (GObject * obj)
{
	WwTextureloader *loader = WW_TEXTURELOADER (obj);
	WwTextureloaderPrivate *priv = WW_TEXTURELOADER_GET_PRIVATE (loader);
	gint size;
	/*
	   g_print("textureloader_donwload\n");
	 */
	priv->data_img = mmapthread_download_get (priv->uri, &size);
	priv->size_img = size;

	if (priv->data_img == NULL) {
		return;
	} else {
		mmapthread_add_power (textureloader_download_to_data, obj);
	}
}

/**********************************/

void
ww_textureloader_set_source (WwTextureloader * loader,
							 const gchar * path_tex, const gchar * path_img, const gchar * uri, gint tex_w, gint tex_h)
{
	WwTextureloaderPrivate *priv;

	priv = WW_TEXTURELOADER_GET_PRIVATE (loader);

	priv->path_tex = g_strdup (path_tex);
	priv->path_img = g_strdup (path_img);
	priv->uri = g_strdup (uri);
	priv->texture_width = tex_w;
	priv->texture_height = tex_h;

	/************************/
	textureloader_add (priv->uri, loader);
	/************************/
}

void
ww_texture_loader_set_image_manipulate_func (WwTextureloader * loader, ImageManipulateFunc func, GObject * data)
{
	WwTextureloaderPrivate *priv;

	priv = WW_TEXTURELOADER_GET_PRIVATE (loader);

	priv->image_manipulate_func = func;
	priv->func_data = data;
	g_object_ref (data);
}


void
ww_textureloader_start (WwTextureloader * loader)
{
	WwTextureloaderPrivate *priv;

	priv = WW_TEXTURELOADER_GET_PRIVATE (loader);

	if (filetest (priv->path_tex) == TRUE) {
		gint size = 0;

		priv->data_tex = disk_load_bin (priv->path_tex, &size);
		priv->size_tex = size;

		mmap_idle_add_single (textureloader_texgen, (GObject *) loader);

	} else if (filetest (priv->path_img) == TRUE) {
		GdkPixbuf *pixbuf;

		pixbuf = disk_load_jpg (priv->path_img);
		get_data_from_pixbuf (loader, pixbuf);

		g_object_unref (pixbuf);

		mmap_idle_add_single (textureloader_texgen, (GObject *) loader);

	} else {
		mmapthread_add_download2 (priv->uri, textureloader_download, (GObject *) loader);
	}
}

void
ww_textureloader_set_texture (WwTextureloader * loader, WwTexture * texture)
{
	WwTextureloaderPrivate *priv;

	priv = WW_TEXTURELOADER_GET_PRIVATE (loader);

	priv->texture = texture;
}

WwTexture *
ww_textureloader_get_texture (WwTextureloader * loader)
{
	WwTextureloaderPrivate *priv;

	priv = WW_TEXTURELOADER_GET_PRIVATE (loader);

	if (ww_texture_is_ok (priv->texture) == TRUE) {
		return priv->texture;
	} else {
		return NULL;
	}
}

/* ローダーの使い方メモ
 *
 * 使う側は、new した後、パスや uri を指定し、もし画像の変形をするならその func を指定た後に、start させる。
 * 使う側は、読み込みが完了したことを知ることができないが、読み込み完了後に update が走るので、その中で
 * get してデータが取れているか確認する。NULL ならまだだったということ。
 * もし、データが取れなかったら、いつまでたっても get に NULL が返る。画像のない場所では問題があるので要修正。
 */
