/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.midiMotion.midi;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.TreeMap;
import java.util.TreeSet;
import javax.sound.midi.*;

/**
 * Standard MIDI ファイルを整理するためのクラス.
 * チャンネルごとにメッセージを分ける.
 * @author nazo
 */
public class MidiResolver {
    private String name;
    private Sequence midiSeq;
    private Track [] tracks;
    /** ticks for quota note */
    private int reso;
    /** mil sec for quota note */
    private int tempo=500000;
    private TreeSet<MidiEvent> [] channelList;
    private TreeMap<MidiEvent,MidiRelation> map_relation;
    private TreeMap<Byte,MidiRelation> map_note;
//    private long end=0;

    /** 空のMidiResolver を作る.
     * sequence も同時に作られる. 基本的に SMF Format 0 になる.
     * @param beats 終点の長さ.
     */
    public MidiResolver(int beats){
        name="EMPTY";
        try {
            midiSeq=new Sequence(Sequence.PPQ,96);
            reso=midiSeq.getResolution();
            tracks=new Track[1];
            tracks[0] =midiSeq.createTrack();
            tracks[0].add(new MidiEvent(
                    new MetaMessage(0x2f, new byte[0], 0), beats*96)
            );
            channelList=new TreeSet[16];
            map_relation=new TreeMap<MidiEvent, MidiRelation>(new MidiComparator());
        } catch (InvalidMidiDataException ex) { // 無いはず
            System.err.println(ex);
            System.exit(-1);
        }
    }

    /**
     * Standard MIDI File をモーション生成用に重要な情報に分ける.
     * SMFはこんな構造をしている
     * <ul>
     * <li>resolution や ファイルサイズ などメタ情報</li>
     * <li>Track[1+] (format 0 なら 1つしかない)<ul>
     *   <li>MidiEvents<ul>
     *     <li>tics</li>
     *     <li>MidiMessage</li>
     *   </ul></li>
     * </ul></li>
     * </ul>
     * MidiMessage は ShortMessage と MetaMessage と SystemMessageに分かれるが、
     * ShortMessage 以外は、モーションに関係ないので、ShortMessageのみを
     * チャンネルごとに分け、分類をする。 MetaMessage からは Tempo情報を読み取る.
     * @param f SMF のファイル
     * @throws InvalidMidiDataException MIDI ファイルとして異常なとき.
     * @throws IOException 読み込みエラー.
     */
    public MidiResolver(File f) throws InvalidMidiDataException,IOException {
        int i,j;
        MidiEvent me;
        MidiMessage mm;
        Track t;

        midiSeq=MidiSystem.getSequence(f);
        name=f.getName();
        tracks=midiSeq.getTracks();
        reso=midiSeq.getResolution();
        channelList=new TreeSet[16];
        map_relation=new TreeMap<MidiEvent, MidiRelation>(new MidiComparator());

        for(i=0;i<tracks.length;i++){
            t=tracks[i];
            map_note = new TreeMap<Byte, MidiRelation>();
            for(j=0;j<t.size();j++){
                me=t.get(j);
                if(me==null)break;
                if((mm=me.getMessage())==null)continue;
                if(mm instanceof ShortMessage){
                    resolve(me);
                } else if(mm instanceof MetaMessage){
                    MetaMessage mem=((MetaMessage)mm);
                    if(mem.getType()==0x51){ // set temp
                        byte []tempB=mem.getData();
                        tempo=(((int)tempB[0]&0xff)<<16)
                                +(((int)tempB[1]&0xff)<<8)
                                +((int)tempB[2]&0xff);
/*                        }else if(mem.getType()==0x2f){ // end of track
                            long end=me.getTick();
                            System.out.println("end: " + end);*/
                    }
                }
            } // in a track
        } // track
    }

    /**
     * SMF から指譜やらを作るのに便利な情報処理.
     * @param sm 演奏と直接関係のある {@link ShortMessage} を持つ
     * {@link MidiEvent}のみ.
     */
    private void resolve(MidiEvent me) {
        ShortMessage sm =(ShortMessage)me.getMessage();
        int ch=sm.getChannel();
        int com=sm.getCommand();

        if(com==ShortMessage.NOTE_ON){
            if(sm.getData2()==0){ // velocity==0
                com=ShortMessage.NOTE_OFF;
                try {
                    sm.setMessage(com, ch, sm.getData1(), 0);
                } catch (InvalidMidiDataException ex){ // おきないと思う.
                    System.err.println("IvalidMidi: "+ex);
                }
            } else {
                MidiRelation rel=new MidiRelation();
                rel.setNoteOn(me);
                map_relation.put(me, rel);
                map_note.put((byte)sm.getData1(),rel);
            }
        }

        if(com==ShortMessage.NOTE_OFF){
            byte note=(byte)sm.getData1();
            MidiRelation rel=map_note.get(note);
            if(rel==null){
                rel=new MidiRelation();
            } else {
                map_note.remove(note);
            }
            rel.setNoteOff(me);
            map_relation.put(me, rel);
        }

        if(channelList[ch]==null){
            channelList[ch]=new TreeSet<MidiEvent>(new MidiComparator());
        }
        channelList[ch].add(me);
    }
    
    /**
     * そのチャンネルのMidiメッセージが tics 順に帰ってくる.
     * @param ch 指定チャンネル
     * @return ch の全Midiメッセージイベント. このメッセージは全部{@link ShortMessage}.
     */
    public MidiEvent [] getChannelMessage(int ch){
        if(channelList[ch]==null)
            return null;
        return channelList[ch].toArray(new MidiEvent[channelList[ch].size()]);
    }

    /**
     * @return the midiSeq
     */
    public Sequence getMidiSeq() {
        return midiSeq;
    }

    /**
     * tics per beat.
     * @return the reso
     */
    public int getReso() {
        return reso;
    }

    /**
     * mili seconds per beat
     * @return the tempo
     */
    public int getTemp() {
        return tempo;
    }

    /**
     * {@link MidiResolver} をつなぐ.
     * テンポとレゾリューションは共通であると想定. 終点はビート区切りになる.
     * @param mr 後に続く {@link MidiResolver}.
     */
    public void add(MidiResolver mr){
        Track t=tracks[0];
        long length=midiSeq.getTickLength();
        if(length%reso >0){
            length=(length/reso+1)*reso;
        }

        MidiEvent [] mms;
        int ch;
        for(ch=0;ch<16;ch++){
            mms=mr.getChannelMessage(ch);
            if(mms!=null){
                if(channelList[ch]==null){
                    channelList[ch]=new TreeSet<MidiEvent>(new MidiComparator());                    
                }
                for(MidiEvent me:mms){
                    me.setTick(me.getTick()+length);
                    t.add(me);
                    channelList[ch].add(me);
                }
            }
        }
        long length2=mr.getMidiSeq().getTickLength();
        if(length2 % reso >0){
            length2=(length2/reso+1)*reso;
        }
        try {
            t.add(new MidiEvent(
                    new MetaMessage(0x2f, new byte[0], 0),
                    length+length2)
            );
        } catch (InvalidMidiDataException ex) { // ありえない
            System.out.println("InvalidMidi in MidiResolver.add:"+ex);
        }
    }

    /**
     * シーケンスの出力. Starndard MIDI File Format 0 でシーケンスを書き出し.
     * Output Sequence. 
     * @param os 書き込み先
     * @throws IOException 書き込みエラー
     */
    public void write(OutputStream os) throws IOException{
        MidiSystem.write(midiSeq, 0, os);
    }
    
    /**
     * 文字列化.
     * @return 元ファイル名 か EMPTY.
     */
    @Override
    public String toString(){
        return name;
    }

    /**
     * note On/Off イベントを与えると、note on と note off
     * の対が入った MidiRelation が得られる.
     * @param me note On/Off イベント
     * @return その MiidiRelation.
     */
    public MidiRelation getRelation(MidiEvent me) {
        return map_relation.get(me);
    }
}
