/*
 * AbstractHttpClient.java
 *
 * Copyright 2014 the Monache.org.
 * http://www.monache.org/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.monache.lib.httpclient;

import java.io.IOException;

import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;

/**
 * 
 * 
 * @author monache.org
 * @since 1.0
 */
public abstract class ApacheHttpClient implements HttpClient {

	private String host = null;
	private int port = 0;

	/**
	 * 
	 * @since 1.0
	 */
	public ApacheHttpClient() {
	}

	protected abstract org.apache.http.client.HttpClient getHttpClient() throws HttpClientException;

	public HttpClientResult post(JsonPostParams postParams) throws HttpClientException {
		try {
			// ApacheのHttpClientを取得
			org.apache.http.client.HttpClient httpClient = getHttpClient();
			// JSONデータを設定
			StringEntity entity = new StringEntity(postParams.getData());
			entity.setContentType("application/json");
			// POST処理を実行
			return postExecute(postParams, httpClient, entity);
		} catch (Exception e) {
			if (e instanceof HttpClientException) {
				throw (HttpClientException)e;
			}
			throw new HttpClientException("POST error: " + e.getMessage(), e);
		}
	}
	
	protected HttpClientResult postExecute(AbstractPostParams postParams, org.apache.http.client.HttpClient httpClient, HttpEntity entity) throws HttpClientException {

		HttpEntity resEntity = null;
		
		try {
			// ウォレットのとの通信内容
			HttpPost request = new HttpPost(postParams.getUri());
			HttpHost target = new HttpHost(getHost(), getPort(), postParams.getSchema());
			request.setEntity(entity);

			// 通信実行
			HttpResponse response = httpClient.execute(target, request);

			// 通信結果を返す
			HttpClientResult result = new HttpClientResult();
			resEntity = response.getEntity();
			result.setContent(EntityUtils.toString(resEntity, postParams.getResponseCharset()));
			return result;

		} catch (Exception e) {
			throw new HttpClientException("POST error: " + e.getMessage(), e);
		} finally {
			if (resEntity != null) {
				try {
					EntityUtils.consume(resEntity);
				} catch (IOException e) {
				}
			}
		}
	}
	
	/**
	 * @return host
	 * @since 1.0
	 */
	public String getHost() {
		return host;
	}

	/**
	 * @param host セットする host
	 * @since 1.0
	 */
	public void setHost(String host) {
		this.host = host;
	}

	/**
	 * @return port
	 * @since 1.0
	 */
	public int getPort() {
		return port;
	}

	/**
	 * @param port セットする port
	 * @since 1.0
	 */
	public void setPort(int port) {
		this.port = port;
	}

}
