/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.bean.workflow.WorkflowReferenceBeanInterface;
import jp.mosp.platform.bean.workflow.WorkflowRegistBeanInterface;
import jp.mosp.platform.bean.workflow.impl.WorkflowReferenceBean;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dao.workflow.WorkflowDaoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeBean;
import jp.mosp.time.bean.CutoffUtilBeanInterface;
import jp.mosp.time.bean.HolidayRequestReferenceBeanInterface;
import jp.mosp.time.bean.HolidayRequestRegistBeanInterface;
import jp.mosp.time.bean.RequestUtilBeanInterface;
import jp.mosp.time.bean.ScheduleUtilBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dao.settings.AttendanceDaoInterface;
import jp.mosp.time.dao.settings.HolidayDataDaoInterface;
import jp.mosp.time.dao.settings.HolidayRequestDaoInterface;
import jp.mosp.time.dto.settings.AttendanceDtoInterface;
import jp.mosp.time.dto.settings.HolidayDataDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.impl.TmdHolidayRequestDto;

/**
 * 休暇申請登録クラス。
 */
public class HolidayRequestRegistBean extends TimeBean implements HolidayRequestRegistBeanInterface {
	
	/**
	 * 休暇申請DAOクラス。<br>
	 */
	HolidayRequestDaoInterface				dao;
	/**
	 * 休暇申請参照インターフェース。<br>
	 */
	HolidayRequestReferenceBeanInterface	holidayReference;
	/**
	 * ワークフローDAOクラス。<br>
	 */
	private WorkflowDaoInterface			workflowDao;
	
	/**
	 * ワークフロー登録クラス。<br>
	 */
	private WorkflowRegistBeanInterface		workflowRegist;
	/**
	 * ワークフロー参照クラス。<br>
	 */
	private WorkflowReferenceBeanInterface	workflowReference;
	
	/**
	 * 締日ユーティリティ。<br>
	 */
	private CutoffUtilBeanInterface			cutoffUtil;
	
	/**
	 * カレンダユーティリティ。
	 */
	private ScheduleUtilBeanInterface		scheduleUtil;
	
	/**
	 * 申請ユーティリティ。
	 */
	private RequestUtilBeanInterface		requestUtil;
	
	private AttendanceDaoInterface			attendanceDao;
	private HolidayDataDaoInterface			holidayDataDao;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public HolidayRequestRegistBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス
	 * @param connection DBコネクション
	 */
	public HolidayRequestRegistBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		// DAO準備
		dao = (HolidayRequestDaoInterface)createDao(HolidayRequestDaoInterface.class);
		holidayReference = (HolidayRequestReferenceBeanInterface)createBean(HolidayRequestReferenceBeanInterface.class);
		workflowDao = (WorkflowDaoInterface)createDao(WorkflowDaoInterface.class);
		workflowReference = (WorkflowReferenceBean)createBean(WorkflowReferenceBean.class);
		attendanceDao = (AttendanceDaoInterface)createDao(AttendanceDaoInterface.class);
		holidayDataDao = (HolidayDataDaoInterface)createDao(HolidayDataDaoInterface.class);
		cutoffUtil = (CutoffUtilBeanInterface)createBean(CutoffUtilBeanInterface.class);
		scheduleUtil = (ScheduleUtilBeanInterface)createBean(ScheduleUtilBeanInterface.class);
		requestUtil = (RequestUtilBeanInterface)createBean(RequestUtilBeanInterface.class);
	}
	
	@Override
	public HolidayRequestDtoInterface getInitDto() {
		return new TmdHolidayRequestDto();
	}
	
	@Override
	public void insert(HolidayRequestDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 新規登録情報の検証
		checkInsert(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdHolidayRequestId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	@Override
	public void update(HolidayRequestDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 履歴更新情報の検証
		checkUpdate(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// 論理削除
		logicalDelete(dao, dto.getTmdHolidayRequestId());
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdHolidayRequestId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	@Override
	public void update(long[] idArray) throws MospException {
		if (idArray == null || idArray.length == 0) {
			// 表示例 必ず一件はチェックボックスを選択してください。
			mospParams.addErrorMessage(PlatformMessageConst.MSG_CHECK, null);
			return;
		}
		// Bean初期化
		workflowRegist = (WorkflowRegistBeanInterface)createBean(WorkflowRegistBeanInterface.class);
		// 処理ワークフロー情報リスト準備
		List<WorkflowDtoInterface> workflowList = new ArrayList<WorkflowDtoInterface>();
		for (long id : idArray) {
			// DTOの準備
			BaseDtoInterface baseDto = findForKey(dao, id, true);
			checkExclusive(baseDto);
			if (mospParams.hasErrorMessage()) {
				continue;
			}
			HolidayRequestDtoInterface dto = (HolidayRequestDtoInterface)baseDto;
			// 申請時の確認
			checkAppli(dto);
			if (mospParams.hasErrorMessage()) {
				continue;
			}
			// ワークフローDTOの準備
			WorkflowDtoInterface workflowDto = workflowDao.findForKey(dto.getWorkflow());
			// 申請
			workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
					PlatformConst.WORKFLOW_TYPE_TIME, null);
			// 処理ワークフロー情報リストへ追加
			if (workflowDto != null) {
				workflowList.add(workflowDto);
			}
		}
	}
	
	@Override
	public void regist(HolidayRequestDtoInterface dto) throws MospException {
		if (dao.findForKey(dto.getTmdHolidayRequestId(), false) == null) {
			// 新規登録
			insert(dto);
		} else {
			// 履歴追加
			add(dto);
		}
	}
	
	@Override
	public void add(HolidayRequestDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 履歴追加情報の検証
		checkAdd(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// 論理削除
		logicalDelete(dao, dto.getTmdHolidayRequestId());
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdHolidayRequestId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	@Override
	public void delete(HolidayRequestDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 論理削除
		logicalDelete(dao, dto.getTmdHolidayRequestId());
		// ロック解除
		unLockTable();
	}
	
	/**
	 * 新規登録時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkInsert(HolidayRequestDtoInterface dto) throws MospException {
		// 対象レコードの有効日が重複していないかを確認
		checkDuplicateInsert(dao.findForKeyOnWorkflow(dto.getPersonalId(), dto.getRequestStartDate(), dto
			.getHolidayType1(), dto.getHolidayType2(), dto.getHolidayRange(), dto.getStartTime()));
	}
	
	/**
	 * 履歴更新時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkUpdate(HolidayRequestDtoInterface dto) throws MospException {
		// 対象レコード識別IDのデータが削除されていないかを確認
		checkExclusive(dao, dto.getTmdHolidayRequestId());
	}
	
	/**
	 * 履歴追加時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkAdd(HolidayRequestDtoInterface dto) throws MospException {
		// 対象レコード識別IDのデータが削除されていないかを確認
		checkExclusive(dao, dto.getTmdHolidayRequestId());
	}
	
	@Override
	public void validate(HolidayRequestDtoInterface dto) throws MospException {
		// 基本情報のチェック
		holidayReference.chkBasicInfo(dto.getPersonalId(), dto.getRequestStartDate());
	}
	
	@Override
	public void checkDraft(HolidayRequestDtoInterface dto) throws MospException {
		checkTemporaryClosingFinal(dto);
		// 他の申請チェック。
		checkRequest(dto);
		checkLimitDate(dto);
	}
	
	@Override
	public void checkAppli(HolidayRequestDtoInterface dto) throws MospException {
		// 下書き同様の処理を行う。
		checkDraft(dto);
		// 休暇申請の申請期間チェック。
		checkPeriod(dto);
		// 休暇申請の重複チェック。
		checkHolidayOverlap(dto);
		// 勤怠の申請チェック。
		checkAttendance(dto);
		// 休暇申請の項目の必須チェック。
		checkRequired(dto);
	}
	
	@Override
	public void checkWithdrawn(HolidayRequestDtoInterface dto) {
		// 現在処理無し。処理が必要になった場合追加される予定。
	}
	
	@Override
	public void checkApproval(HolidayRequestDtoInterface dto) throws MospException {
		// 申請時と同様の処理を行う
		checkAppli(dto);
	}
	
	@Override
	public void checkCancel(HolidayRequestDtoInterface dto) {
		// 現在処理無し。処理が必要になった場合追加される予定。
	}
	
	@Override
	public void checkHolidayOverlap(HolidayRequestDtoInterface dto) throws MospException {
		if (dto.getRequestStartDate().after(dto.getRequestEndDate())) {
			// 開始日が終了日より後の場合
			// 休暇年月日の入力内容が不正です。終了日は開始日よりも後になるよう入力してください。
			addHolidayRequestDateErrorMessage();
			return;
		}
		Date requestDate = dto.getRequestStartDate();
		while (!requestDate.after(dto.getRequestEndDate())) {
			boolean holidayRangeAm = dto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_AM;
			boolean holidayRangePm = dto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_PM;
			// 休暇申請リスト取得
			List<HolidayRequestDtoInterface> list = dao.findForList(dto.getPersonalId(), requestDate);
			for (HolidayRequestDtoInterface requestDto : list) {
				// ワークフローの取得
				WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
				if (workflowDto == null) {
					continue;
				}
				if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
					// 取下の場合
					continue;
				}
				if (dto.getWorkflow() == workflowDto.getWorkflow()) {
					// ワークフロー番号が同じ場合は同じ申請
					continue;
				}
				if (dto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_ALL
						|| requestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_ALL) {
					// 全休の場合
					// 表示例 休暇内容が重複しています。休暇年月日を選択し直してください。
					addHolidayOverlapRange1ErrorMessage();
					return;
				}
				if (dto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_AM
						|| dto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_PM) {
					// 午前休又は午後休の場合
					if (dto.getHolidayRange() == requestDto.getHolidayRange()) {
						// 表示例 休暇内容が重複しています。休暇年月日または休暇範囲を選択し直してください。
						addHolidayOverlapRange2ErrorMessage();
						return;
					}
					if (requestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_AM) {
						// 午前休の場合
						holidayRangeAm = true;
					}
					if (requestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_PM) {
						// 午後休の場合
						holidayRangePm = true;
					}
				}
			}
			if (holidayRangeAm || holidayRangePm) {
				// 午前休又は午後休の場合
				// 各種申請情報取得
				requestUtil.setRequests(dto.getPersonalId(), requestDate);
				int subHolidayRange = requestUtil.checkHolidayRangeSubHoliday(requestUtil.getSubHolidayList(false));
				boolean allHoliday = (holidayRangeAm && holidayRangePm)
						|| (holidayRangeAm && subHolidayRange == TimeConst.CODE_HOLIDAY_RANGE_PM)
						|| (holidayRangePm && subHolidayRange == TimeConst.CODE_HOLIDAY_RANGE_AM)
						|| (subHolidayRange == TimeConst.CODE_HOLIDAY_RANGE_AM && subHolidayRange == TimeConst.CODE_HOLIDAY_RANGE_PM);
				if (allHoliday && !requestUtil.getOverTimeList(false).isEmpty()) {
					// 残業申請がある場合
					addOthersRequestErrorMessage(requestDate, mospParams.getName("OvertimeWork"));
					return;
				}
//				if (allHoliday && requestUtil.getDifferenceDto(false) != null) {
//					// 時差出勤申請がある場合
//					StringBuffer sb = new StringBuffer();
//					sb.append(mospParams.getName("TimeDifference"));
//					sb.append(mospParams.getName("GoingWork"));
//					addOthersRequestErrorMessage(requestDate, sb.toString());
//					return;
//				}
			}
			// 1日加算
			requestDate = addDay(requestDate, 1);
		}
	}
	
	@Override
	public void checkRequest(HolidayRequestDtoInterface dto) throws MospException {
		if (dto.getRequestStartDate().after(dto.getRequestEndDate())) {
			// 休暇申請の重複チェックでメッセージを追加するため、ここのメッセージ設定は無し。
			return;
		}
		Date requestDate = dto.getRequestStartDate();
		while (!requestDate.after(dto.getRequestEndDate())) {
			// 各種申請情報取得
			requestUtil.setRequests(dto.getPersonalId(), requestDate);
			// 勤務形態コードを取得
			String workTypeCode = scheduleUtil.getScheduledWorkTypeCode(dto.getPersonalId(), requestDate);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			// カレンダの勤務形態が法定休日又は所定休日の場合
			if (TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(workTypeCode)
					|| TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(workTypeCode)) {
				// 下書、取下でない休日出勤申請情報取得
				WorkOnHolidayRequestDtoInterface workOnHolidayRequestDto = requestUtil.getWorkOnHolidayDto(false);
				// 休日出勤申請の場合
				if (workOnHolidayRequestDto != null
						&& workOnHolidayRequestDto.getSubstitute() == TimeConst.CODE_WORK_ON_HOLIDAY_SUBSTITUTE_OFF) {
					// （日付）は休日出勤が申請されています。休暇年月日を選択し直してください。
					addOthersRequestErrorMessage(requestDate, mospParams.getName("WorkingHoliday"));
					return;
				}
				// 承認済かつ振替出勤申請の休日出勤申請情報がない場合
				if (requestUtil.getWorkOnHolidayDto(true) == null
						&& (requestDate.equals(dto.getRequestStartDate()) || requestDate
							.equals(dto.getRequestEndDate()))) {
					// 法定休日又は所定休日の場合
					// 休暇開始日又は休暇終了日の場合はエラーとする
					addHolidayTargetWorkDateHolidayErrorMessage(requestDate);
					return;
				}
			}
			// 振替休日
			if (requestUtil.checkHolidayRangeSubstitute(requestUtil.getSubstituteList(false)) == TimeConst.CODE_HOLIDAY_RANGE_ALL) {
				addOthersRequestErrorMessage(requestDate, mospParams.getName("Transfer")
						+ mospParams.getName("GoingWork"));
				return;
			}
			// 代休申請
			int subHolidayRange = requestUtil.checkHolidayRangeSubHoliday(requestUtil.getSubHolidayList(false));
			if (subHolidayRange == TimeConst.CODE_HOLIDAY_RANGE_ALL
					|| subHolidayRange == TimeConst.CODE_HOLIDAY_RANGE_AM + TimeConst.CODE_HOLIDAY_RANGE_PM
					|| (dto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_ALL && (subHolidayRange == TimeConst.CODE_HOLIDAY_RANGE_ALL
							|| subHolidayRange == TimeConst.CODE_HOLIDAY_RANGE_AM
							|| subHolidayRange == TimeConst.CODE_HOLIDAY_RANGE_PM || subHolidayRange == TimeConst.CODE_HOLIDAY_RANGE_AM
							+ TimeConst.CODE_HOLIDAY_RANGE_PM))
					|| (dto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_AM && subHolidayRange == TimeConst.CODE_HOLIDAY_RANGE_AM)
					|| (dto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_PM && subHolidayRange == TimeConst.CODE_HOLIDAY_RANGE_PM)) {
				addOthersRequestErrorMessage(requestDate, mospParams.getName("CompensatoryHoliday"));
				return;
			}
			// 残業申請
			if (dto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_ALL
					&& !requestUtil.getOverTimeList(false).isEmpty()) {
				// 表示例 <日付>は既に他の申請が行われています。休暇年月日を選択し直してください。
				addOthersRequestErrorMessage(requestDate, mospParams.getName("OvertimeWork"));
				return;
			}
			// 時差出勤申請
			if ((dto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_ALL || dto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_AM)
					&& requestUtil.getDifferenceDto(false) != null) {
				// 全休又は午前休の場合
				StringBuffer sb = new StringBuffer();
				sb.append(mospParams.getName("TimeDifference"));
				sb.append(mospParams.getName("GoingWork"));
				addOthersRequestErrorMessage(requestDate, sb.toString());
			}
			// 1日加算
			requestDate = addDay(requestDate, 1);
		}
	}
	
	@Override
	public void checkAttendance(HolidayRequestDtoInterface dto) throws MospException {
		if (dto.getRequestStartDate().after(dto.getRequestEndDate())) {
			// 開始日が終了日より後の場合
			return;
		}
		Date requestDate = dto.getRequestStartDate();
		while (!requestDate.after(dto.getRequestEndDate())) {
			Date targetDate = requestDate;
			// 1日加算
			requestDate = addDay(requestDate, 1);
			AttendanceDtoInterface attendanceDto = attendanceDao.findForKey(dto.getPersonalId(), targetDate, 1);
			if (attendanceDto == null) {
				continue;
			}
			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(attendanceDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 取下の場合
				continue;
			}
			if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())) {
				// 下書の場合
				continue;
			}
			if (workflowDto.getWorkflowStage() == 0
					&& PlatformConst.CODE_STATUS_REVERT.equals(workflowDto.getWorkflowStatus())) {
				// 1次戻の場合
				continue;
			}
			// （日付）は既に勤怠の申請が行われています。（申請区分毎の日付名称）を選択し直してください。
			addHolidayTargetWorkDateAttendanceRequestErrorMessage(targetDate);
			return;
		}
	}
	
	@Override
	public void checkRequired(HolidayRequestDtoInterface dto) {
		if (dto.getHolidayType1() != 1 && dto.getHolidayType1() != 4) {
			// 有給休暇でなく且つ欠勤でない場合
			return;
		}
		// 有給休暇又は欠勤の場合
		if (dto.getRequestReason().isEmpty()) {
			addHolidayRequestReasonErrorMessage();
		}
	}
	
	@Override
	public void checkLimitDate(HolidayRequestDtoInterface dto) throws MospException {
		if (dto == null) {
			return;
		}
		int holidayType = dto.getHolidayType1();
		if (holidayType != TimeConst.CODE_HOLIDAYTYPE_SPECIAL && holidayType != TimeConst.CODE_HOLIDAYTYPE_OTHER) {
			// 特別休暇でなく且つその他休暇でない場合
			return;
		}
		HolidayDataDtoInterface holidayDataDto = holidayDataDao.findForInfo(dto.getPersonalId(), dto
			.getHolidayAcquisitionDate(), dto.getHolidayType2(), dto.getHolidayType1());
		if (holidayDataDto == null) {
			return;
		}
		if (holidayDataDto.getHolidayLimitDate().before(dto.getRequestEndDate())) {
			// 取得期限が終了日より前の場合
			addHolidayLimitDateErrorMessage();
		}
	}
	
	@Override
	public void checkPeriod(HolidayRequestDtoInterface dto) {
		if (dto.getRequestEndDate().after(DateUtility.addMonth(getSystemDate(), 6))) {
			// 終了日が6ヶ月後より後の場合
			addHolidayPeriodErrorMessage();
		}
	}
	
	@Override
	public void checkTemporaryClosingFinal(HolidayRequestDtoInterface dto) throws MospException {
		// 対象個人IDにつき対象日付が未締であるかの確認
		cutoffUtil.checkTighten(dto.getPersonalId(), dto.getRequestStartDate(), getNameVacationDay());
	}
	
	/**
	 * 休暇年月日名称を取得する。
	 * @return 休暇年月日名称
	 */
	protected String getNameVacationDay() {
		return mospParams.getName("Vacation") + mospParams.getName("Year") + mospParams.getName("Month")
				+ mospParams.getName("Day");
	}
	
}
