/*
 * signal.c
 *
 * Copyright 2004, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include"types.h"
#include"lib.h"
#include"proc.h"
#include"interrupt.h"
#include"mm.h"
#include"errno.h"
#include"signal.h"


/****************************************************************************************
 *
 * < signal default actions >
 *
 *****************************************************************************************/

/*
 * PRIVATE
 * Default aactinos.
 */
/* terminate process. */
static void terminate(int stat)
{
	sys_exit(0);
}

/* abnormal terminate process. */
static void abnormalTerminate(int stat)
{
	sys_exit(1);
}

/* stop process. */
static void stopProcess(int stat)
{
	del_from_schedule(TASK_SIGNAL_WAIT);
	wait_task();
}


/*
 * PUBLIC
 * signal actions.
 */
static SIG_ACTION defaultAction[]={
	NULL,
	abnormalTerminate,	/* SIGABRT		A */
	terminate,			/* SIGALRM		T */
	abnormalTerminate,	/* SIGBUS		A */
	SIG_IGN,			/* SIGCHLD		I */
	SIG_DFL,			/* SIGCONT		C */
	abnormalTerminate,	/* SIGFPE		A */
	terminate,			/* SIGHUP		T */
	abnormalTerminate,	/* SIGILL		A */
	terminate,			/* SIGINT		T */
	terminate,			/* SIGKILL		T */
	terminate,			/* SIGPIPE		T */
	terminate,			/* SIGPOLL		T */
	terminate,			/* SIGPROF		T */
	abnormalTerminate,	/* SIGQUIT		A */
	abnormalTerminate,	/* SIGSEGV		A */
	stopProcess,		/* SIGSTOP		S */
	abnormalTerminate,	/* SIGSYS		A */
	terminate,			/* SIGTERM		T */
	abnormalTerminate,	/* SIGTRAP		A */
	stopProcess,		/* SIGTSTP		S */
	stopProcess,		/* SIGTTIN		S */
	stopProcess,		/* SIGTTOU		S */
	SIG_IGN,			/* SIGURG		I */
	terminate,			/* SIGUSR1		T */
	terminate,			/* SIGUSR2		T */
	terminate,			/* SIGVTALRM	T */
	abnormalTerminate,	/* SIGXCPU		A */
	abnormalTerminate,	/* SIGXFSZ		A */
};


/****************************************************************************************
 *
 * < ʥ >
 *
 * 桼ϥɥ¹Ԥȼѹ
 *
 *****************************************************************************************/

/*
 * PRIVATE
 * ʥ빽¤Υɥ쥹顢ץɥ쥹롣
 * parameters : ʥ빽¤
 * return : ץ
 */
static inline PROC *sigToProc(SIGNAL *sig_st)
{
	return (PROC*)((uint)sig_st-(uint)&((PROC*)0)->signal_struct);
}


/*
 * PRIVATE
 * ʥ¹ԡ
 * parameters : user esp address
 */
static inline void doSignal(uint *user_esp)
{
	SIGNAL *sig_st=(SIGNAL*)get_current_task()->signal_struct;
	int i;


	if(sig_st->signal==0)return;

	/* ʥϥɥμ¹ԡ */
	for(i=0;sig_st->signal!=0;++i)
	{
		cli();

		if(sig_st->signal&(1<<i))
		{
			sig_st->signal^=(1<<i);
			sti();

			if((uint)sig_st->action[i]>=USER_BEG)
			{
				uint before_mask;

				/* ʥޥ촹 */
				before_mask=sig_st->mask;
				sig_st->mask|=sig_st->actMask[i]|(1<<i);

				/* 桼ϥɥ¹ԡ */
				doUserHandler(i,sig_st->action[i],*user_esp,&sig_st->beforeEsp,&sig_st->beforeStackEntry);
				sig_st->beforeEsp=0;

				/* ʥޥ᤹ */
				sig_st->mask=before_mask;
			}
			else if(sig_st->action[i]!=SIG_DFL)
				/* ǥեȥϥɥ¹ԡ */
				sig_st->action[i](i);
		}
		else sti();
	}
}


/*
 * PRIVATE
 * ̾ߤΥʥ¹ԡ
 * parameters : user esp
 */
void doSignalFromTrap(TRAP_FRAME frame)
{
	doSignal(&frame.user_esp);
}


/*
 * PRIVATE
 * 㳰ߤΥʥ¹ԡ
 * parameters : user esp
 */
void doSignalFromExcept(EXCEPT_FRAME frame)
{
	doSignal(&frame.user_esp);
}


/*
 * GLOBAL
 * ʥ
 * ǥեȥΡcontinue processפϡʥץ
 * ư롣
 * parameters : 襷ʥ빽¤,ʥ
 */
void sendSignal(SIGNAL *sig_st,int signal)
{
	int sig;


	/* ʥ뤬̵ˤƤʤ */
	if(sig_st->action[signal]==SIG_IGN)return ;

	sig_st->reserve|=(1<<signal)&sig_st->mask;		/* αΥʥ롣 */
	sig=((1<<signal)|sig_st->mask)^sig_st->mask;	/* αʤʥ롣 */

	if(sig!=0)
	{
		/* ڥɥʥԤʤ鵯 */
		add_to_schedule(sigToProc(sig_st),TASK_SIGSUS_WAIT);

		/* SIGCONTʤץư롣 */
		if(sig==(1<<SIGCONT))
			add_to_schedule(sigToProc(sig_st),TASK_SIGNAL_WAIT);
		else sig_st->signal|=sig;
	}
}


/*
 * GLOBAL
 * ʥ
 * parameters : 襷ʥ빽¤,ʥ
 */
void forceSendSignal(SIGNAL *sig_st,int signal)
{
	/* SIGCONTʤץư롣 */
	if(sig_st->signal==(1<<SIGCONT))
		add_to_schedule(sigToProc(sig_st),TASK_WAIT);
	else sig_st->signal|=1<<signal;
}


/*
 * GLOBAL
 * fork()Υʥ뵡ꡣ
 * parameters : ƥץΥʥ빽¤Ρ
 * return : 0 or error number
 */
SIGNAL *setSignal(SIGNAL *parent_sig)
{
	SIGNAL *sig_st;


	if((sig_st=kmalloc(sizeof(SIGNAL)))==NULL)return NULL;

	memcpy(sig_st->action,parent_sig->action,sizeof(SIGNAL));
	sig_st->signal=0;
	sig_st->reserve=0;
	sig_st->mask=parent_sig->mask;

	return sig_st;
}


/*
 * GLOBAL
 * exit()Υʥ뵡γ
 * parameters : ʥ빽¤
 */
void releaseSignal(SIGNAL *sig_st)
{
	kfree(sig_st);
}


/*
 * GLOBAL
 * ʥ뵡Υꥻåȡ
 * parameters : ʥ빽¤
 */
void resetSignal(SIGNAL *sig_st)
{
	/* ǥեȥ򥳥ԡ */
	memcpy(sig_st->action,defaultAction,SIG_ACTIONS*sizeof(SIG_ACTION));

	memset(sig_st->actMask,0,sizeof(uint)*SIG_ACTIONS);
	sig_st->signal=0;
	sig_st->reserve=0;
	sig_st->beforeEsp=0;
}


/*
 * GLOBAL
 * espͤ롣
 */
uint getBeforeEsp()
{
	return ((SIGNAL*)get_current_task()->signal_struct)->beforeEsp;
}


/*
 * GLOBAL
 * Υͥ륹åʪɥ쥹롣
 */
uint getBeforeStackEntry()
{
	return ((SIGNAL*)get_current_task()->signal_struct)->beforeStackEntry;
}


/*
 * GLOBAL
 * ʥ뵡ν
 * return : SIGNAL struct or NULL
 */
SIGNAL *initSignal()
{
	SIGNAL *sig_st;


	if((sig_st=kmalloc(sizeof(SIGNAL)))==NULL)return NULL;

	/*  */
	resetSignal(sig_st);
	sig_st->mask=0;

	return sig_st;
}


/****************************************************************************************
 *
 * < ƥॳ >
 *
 *****************************************************************************************/

int sys_sigaction(int signum,const struct sigaction *act,struct sigaction *oldact)
{
	SIGNAL *signal;
	struct sigaction tmp_act;


	if(signum==0)return 0;
	if((signum>=SIG_ACTIONS)||(signum<0))return -EINVAL;

	signal=(SIGNAL*)get_current_task()->signal_struct;
	tmp_act=*act;

	if(oldact!=NULL)
	{
		if(signal->action[signum]==SIG_IGN)oldact->sa_handler=SIG_IGN;
		else if(signal->action[signum]==defaultAction[signum])oldact->sa_handler=SIG_DFL;
		else oldact->sa_handler=signal->action[signum];

		oldact->sa_mask=signal->actMask[signum];
	}

	if(act!=NULL)
	{
		if((signum==SIGKILL)||(signum==SIGSTOP))return -EINVAL;		/* SIGKILLSIGSTOPѹԲġ */


		if(tmp_act.sa_handler==SIG_DFL)signal->action[signum]=defaultAction[signum];
		else if(tmp_act.sa_handler==SIG_IGN)signal->action[signum]=SIG_IGN;
		else if((uint)tmp_act.sa_handler>=USER_BEG)signal->action[signum]=tmp_act.sa_handler;
		else return -EINVAL;		/* ϥɥ饢ɥ쥹 */

		signal->actMask[signum]=tmp_act.sa_mask;
	}

	return 0;
}


int sys_sigprocmask(int how,uint *set,uint *oldset)
{
	uint tmp_mask=*set;
	SIGNAL *sig_st;


	sig_st=get_current_task()->signal_struct;

	if(oldset!=NULL)*oldset=sig_st->mask;

	switch(how)
	{
		case SIG_BLOCK:
			if(tmp_mask&((1<<SIGKILL)|(1<<SIGSTOP)))return -EINVAL;	/* SIGKILLSIGSTOPϥ֥åԲġ */
			sig_st->mask|=tmp_mask;
			break;
		case SIG_UNBLOCK:
			sig_st->mask&=~tmp_mask;
			break;
		case SIG_SETMASK:
			if(tmp_mask&((1<<SIGKILL)|(1<<SIGSTOP)))return -EINVAL;	/* SIGKILLSIGSTOPϥ֥åԲġ */
			sig_st->mask=tmp_mask;
			break;
	}

	return 0;
}


int sys_sigpending(sigset_t *set)
{
	SIGNAL *sig_st;


	if((uint)set<USER_BEG)return -EINVAL;

	sig_st=get_current_task()->signal_struct;
	*set=sig_st->reserve;

	return 0;
}


int sys_sigsuspend(const sigset_t *mask)
{
	uint before_mask;
	SIGNAL *sig_st;


	/* SIGKILLSIGSTOPϥ֥åԲġ */
	if(*mask&((1<<SIGKILL)|(1<<SIGSTOP)))return -EINVAL;

	sig_st=get_current_task()->signal_struct;
	before_mask=sig_st->mask;
	sig_st->mask=*mask;

	/* suspend. */
	del_from_schedule(TASK_SIGSUS_WAIT);
	wait_task();

	sig_st->mask=before_mask;

	return -EINTR;
}
