/*
 * console.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include"types.h"
#include"lib.h"
#include"lock.h"
#include"device.h"
#include"proc.h"
#include"interrupt.h"
#include"keymap.h"
#include"term.h"
#include"console.h"


/*****************************************************
static void testPrint(int uvalue)
{
	char tmp_buf[12];
	int i,j,last;


	for(i=12-1;;--i)
	{
		tmp_buf[i]=uvalue%10+'0';
		if(!(uvalue/=10))break;
	}

	last=i;
	j=ROUNDUP(scrn.start,COLUMNS)-1;
	for(i=12-1;i>=last;--i,--j)vram[j*2]=tmp_buf[i];
}
*******************************************************/


/***************************************************************************************
 *
 * ü饹
 *
 ***************************************************************************************/

/*
 * PUBLIC
 * Console termios struct.
 */
static struct termios termio={
	ICRNL|IXOFF|IXON,	/* c_iflag. */
	OPOST,				/* c_oflag. */
	CS8|CREAD,			/* c_cflag. */
	ISIG|ICANON|ECHO,	/* c_lflag. */
};


/*
 * PUBLIC
 * Init terminal.
 */
static inline void initTerm()
{
	/* termiosꡣ */
	termio.c_cc[VINTR]='c'&0x1f;
	termio.c_cc[VQUIT]='/'&0x1f;
	termio.c_cc[VSTART]='q'&0x1f;
	termio.c_cc[VSTOP]='s'&0x1f;
	termio.c_cc[VSUSP]='z'&0x1f;
	termio.c_cc[VEOF]='d'&0x1f;
	termio.c_cc[VEOL]=0;
	termio.c_cc[VERASE]='h'&0x1f;
	termio.c_cc[VKILL]='u'&0x1f;
	termio.c_cc[VEOF]=0;
	termio.c_cc[VEOL]=0;
}


/***************************************************************************************
 *
 * 󥽡Хåե饹
 *
 ***************************************************************************************/

enum{
	BUF_SIZE=240,	/* 󥽡ХåեХȥ */
};


typedef struct{
	char buf[BUF_SIZE];	/* 󥽡Хåե */
	int crt;			/* ȥݥ */
	int start;			/* ɽȥݥ */
	int last;			/* 饹ȥݥ */
	int newLine;		/* Ƭݥ */
}CONSOLE_BUF;


/*
 * PUBLIC
 */
static CONSOLE_BUF cbuf;


/*
 * PUBLIC
 * 󥽡Хåե
 */
/*
 * 1ʸɲá
 * return : ɲ=1,ɲäʤ=0
 */
static inline int inputBuf(char ch)
{
	int i;


	if(cbuf.crt<BUF_SIZE-2)
	{
		/* ʸΰư */
		if(cbuf.last==BUF_SIZE-2)cbuf.buf[--cbuf.last]='\0';
		for(i=cbuf.last;i>=cbuf.crt;--i)cbuf.buf[i+1]=cbuf.buf[i];
		cbuf.buf[cbuf.crt++]=ch;
		++cbuf.last;
		return 1;
	}
	return 0;
}

/* Delete */
static inline void deleteBuf()
{
	/* ʸ˰ư */
	if(cbuf.last>cbuf.crt)
	{
		memcpy(&cbuf.buf[cbuf.crt],&cbuf.buf[cbuf.crt+1],cbuf.last-cbuf.crt);
		--cbuf.last;
	}

}

/*
 * BackSpace
 * return : =1,ʤ=0
 */
static inline int backSpaceBuf()
{
	if(cbuf.crt>cbuf.newLine)
	{
		/* ʸ˰ư */
		if(cbuf.start>--cbuf.crt)cbuf.start=cbuf.crt;
		memcpy(&cbuf.buf[cbuf.crt],&cbuf.buf[cbuf.crt+1],cbuf.last-cbuf.crt);
		--cbuf.last;
		return 1;
	}
	return 0;
}

/*
 * Left cursor.
 * return : ư=1,ưʤ=0
 */
static inline int leftCursorBuf()
{
	if(cbuf.crt>cbuf.newLine)
	{
		cbuf.start=--cbuf.crt;
		return 1;
	}
	return 0;
}

/*
 * Right cursor.
 * return : ư=1,ưʤ=0
 */
static inline int rightCursorBuf()
{
	if(cbuf.crt<cbuf.last)
	{
		cbuf.start=++cbuf.crt;
		return 1;
	}
	return 0;
}

/*
 * Carriage retun.
 * return : Хȿ
 */
static inline int carriageRetunBuf()
{
	int len=cbuf.crt-cbuf.newLine;


	cbuf.start=cbuf.crt=cbuf.newLine;
	return len;
}

/* ԡ */
static inline void newLineBuf()
{
	cbuf.buf[cbuf.last]='\n';
	if(cbuf.last<BUF_SIZE-2)++cbuf.last;
	cbuf.buf[cbuf.last]='\0';
	cbuf.start=cbuf.newLine=cbuf.crt=cbuf.last;
}


/*
 * PUBLIC
 * 󥽡Хåեν
 */
static void initConsoleBuf()
{
	cbuf.newLine=cbuf.crt=cbuf.start=cbuf.last=0;
	cbuf.buf[0]='\0';
}


/*
 * PUBLIC
 * Check console buffer.
 * parameters : user buffer,max copy size
 * return : ϥХȿ
 */
static int readConsoleBuf(uchar *user_buf,size_t size)
{
	int i;


	if(cbuf.last==0)return 0;

	if((termio.c_lflag&ICANON)&&(cbuf.newLine>0))
	{
		i=0;
		for(;cbuf.buf[i]!='\n';++i);
		memcpy(user_buf,cbuf.buf,(size>i)?i:size);
		memcpy(cbuf.buf,&cbuf.buf[i+1],cbuf.last-i);
		cbuf.start-=i+1;
		cbuf.crt-=i+1;
		cbuf.last-=i+1;
		cbuf.newLine-=i+1;
	}
	else
	{
		memcpy(user_buf,cbuf.buf,cbuf.last);
		i=cbuf.last;
		initConsoleBuf();
	}

	return (size>i)?i:size;
}


/***************************************************************************************
 *
 * ǥץ쥤饹
 *
 ***************************************************************************************/

enum{
	BASE_ADDR=0xb8000,		/* Start address of video memory */
	VMEM_SIZE=0x8000,		/* Size of video memory */

	/* CRT IO registers */
	CTL_ADDR=0x3d4,			/* Addres register */
	CTL_DATA=0x3d5,			/* Data IO register */
	START_ADDR_HIGH=0xc,	/* High byte of print start address */
	START_ADDR_LOW=0xd,		/* Low  byte of print start address  */
	CURSOR_ADDR_HIGH=0xe,	/* High byte of cursor address */
	CURSOR_ADDR_LOW=0xf,	/* Low  byte of cursor address */

	LINES=25,				/* Character lines */
	COLUMNS=80,				/* Character columns */
	LAST_POS=VMEM_SIZE/2-LINES*COLUMNS-COLUMNS*3,
};


typedef struct{
	int start;		/* ӥǥ꡼賫ϰ֡ */
	int last;		/* ӥǥ꡼轪λ֡ */
	int top;		/* ӥǥ꡼β̤γϰ֡ */
	int cursor;		/* ֡ */
}SCREEN;


static char *vram=(char*)BASE_ADDR;	/* Video ram. */
static SCREEN scrn;


/*
 * PRIVATE
 * 
 */
/* ʸ */
static inline void input(char ch)
{
	scrn.cursor+=inputBuf(ch);
}

/* Backspace */
static inline void backSpace()
{
	if(scrn.start>(scrn.cursor-=backSpaceBuf()))
		scrn.start=scrn.cursor;
}

/* Left cursor. */
static inline void leftCursor()
{
	scrn.start=scrn.cursor-=leftCursorBuf();
}

/* Right cursor. */
static inline void rightCursor()
{
	scrn.start=scrn.cursor+=rightCursorBuf();
}

/* Carriage retun. */
static inline void carriageRetun()
{
	scrn.start=scrn.cursor-=carriageRetunBuf();
}

/* Print screen from buffer. */
static inline void printScreen()
{
	int i,j;


	for(i=cbuf.start,j=scrn.start*2;i<=cbuf.last;++i,j+=2)
		vram[j]=cbuf.buf[i];
	scrn.last=j/2-1;
	cbuf.start=cbuf.crt;
	scrn.start=scrn.cursor;
}

/*  */
static inline void newLine()
{
	printScreen();
	newLineBuf();
	scrn.cursor=scrn.start=ROUNDUP(scrn.last+1,COLUMNS);
}

/* Update cursor position. */
static inline void updateCursor()
{
	outb(CTL_ADDR,CURSOR_ADDR_HIGH);
	outb(CTL_DATA,scrn.cursor>>8);
	outb(CTL_ADDR,CURSOR_ADDR_LOW);
	outb(CTL_DATA,scrn.cursor);
}

/* Hard scroll up. */
static void scrollup()
{
	char *p;
	int i,last;


	if(scrn.last-scrn.top>=COLUMNS*LINES)
	{
		scrn.top=ROUNDUP(scrn.last-COLUMNS*LINES+1,COLUMNS);

		if(scrn.top>=LAST_POS)
		{
			/* Video ramƬ˰ư */
			p=&vram[scrn.top*2];
			last=(scrn.last-scrn.top)*2;
			for(i=0;i<last;i+=2)vram[i]=p[i];

			/* Video ram0ꥢ */
			for(;i<(LAST_POS+COLUMNS*LINES)*2;i+=2)vram[i]=0;

			scrn.start=scrn.cursor=scrn.last-scrn.top;
			scrn.top=0;
		}

		/* Hard scroll up */
		outb(CTL_ADDR,START_ADDR_HIGH);
		outb(CTL_DATA,scrn.top>>8);
		outb(CTL_ADDR,START_ADDR_LOW);
		outb(CTL_DATA,scrn.top);
	}
}


/*
 * GLOBAL
 * Print buf
 * parameters : Begin position,Print bytes
 * returns : Number of printing characters
 */
int writeVram(const uchar *buf,size_t n,size_t m)
{
	int i;


	for(i=0;i<n;++i)
	{
    	if(buf[i]=='\n')newLine();
    	else if(buf[i]=='\r')carriageRetun();
		else if(buf[i]=='\b')backSpace();
    	else if(buf[i]==MAP_DEL)deleteBuf();
    	else if(buf[i]==MAP_LFT)leftCursor();
    	else if(buf[i]==MAP_RHT)rightCursor();
    	else if(buf[i]>=' ')input(buf[i]);
    }

    /* ̤ɽ */
    printScreen();

    /* ɬפʤ饹륢åס */
    scrollup();

	/* ư */
    updateCursor();

	return n;
}


/*
 * GLOBAL
 * Init console
 */
int initConsole()
{
	short *p;


	/* Start position 0 */
	outb(CTL_ADDR,START_ADDR_HIGH);
	outb(CTL_DATA,0);
	outb(CTL_ADDR,START_ADDR_LOW);
	outb(CTL_DATA,0);

	/* Init screen state. */
	scrn.start=scrn.top=scrn.cursor=0;

	/* Clear screen */
	for(p=(short*)BASE_ADDR;p<(short*)(BASE_ADDR+VMEM_SIZE);++p)*p=0x700;

	/* 󥽡Хåեν */
	initConsoleBuf();

	/* üǽν */
	initTerm();

	return 0;
}


/***************************************************************************************
 *
 * ܡɥ饹
 *
 ***************************************************************************************/

enum{
	IRQ_NO=1,		/* IRQ number */
};


/* GLOBAL */
static PROC_LINK waitLink={&waitLink,&waitLink};	/* Ԥץ󥯡 */


/*
 * PRIVATE
 * Turn on/off LED.
 */
static inline void switchLed(uchar value)
{
	outb(KEYBOARD_OUTPUT_REG,0xed);		/* Command turn on/of LED. */
	while(inb(KEYBOARD_STAT_REG)&2);
	outb(KEYBOARD_OUTPUT_REG,value);	/* LED on/off. */
}


/*
 * PRIVATE
 * keyboard main handler.
 * parameters : scan key coad
 * return : task switch on=1 or off=0
 */
static int keyboardHnadler(uchar key_coad)
{
	enum{
		KEY_BREAK=0x80,	/* key break bit. */

		/* Add key bit. */
		SHT_BIT=1<<0,
		CTR_BIT=1<<1,

		/* LED bit. */
		LED_SCRL=1<<0,	/* ScrollLock LED. */
		LED_NMLC=1<<1,	/* NumLock LED. */
		LED_CPSL=1<<2,	/* CapsLock LED. */
	};

	static int add_key=0;				/* normal=0,add shift=1,add ctrl=2. */
	static int func_key=0;				/* ǽե饰(󥳡0xe0) */
	static int shift_key=0;				/* Shift key flag. */
	static int led=0;					/* LED flag, */
	static int ctr_key=0;				/* Ctrl key. */
	static uchar add_map[]={0,1,2,2};	/* ղåѴޥåס */

	uchar map_coad;
	int rest=0;


	/* ǽν */
	if(func_key)
	{
		if((key_coad>=VMAP_START_COAD)&&(key_coad<=VMAP_END_COAD))
			key_coad+=VMAP_ADD;
		func_key=0;
	}
	else if(key_coad==KEY_FUNC)
	{
		func_key=1;
		return 0;
	}

	/* ޥåץɤθ */
	if((map_coad=keymap[key_coad&~KEY_BREAK][add_key])==0)return 0;
	handleTermCChar(&termio,map_coad);
	if(map_coad>=MIN_ADDKEY)
	{
		switch(map_coad)
		{
			case MAP_SHT:
				if((key_coad&KEY_BREAK)==0)shift_key=SHT_BIT;
				else shift_key=0;
				break;
			case MAP_CPSL:
				if((key_coad&KEY_BREAK)==0)
				{
					led^=LED_CPSL;
					switchLed(led);
				}
				break;
			case MAP_CTR:
				if((key_coad&KEY_BREAK)==0)ctr_key=CTR_BIT;
				else ctr_key=0;
				break;
		}
		add_key=add_map[(shift_key|ctr_key)^(led>>2)];
	}
	else if((key_coad&KEY_BREAK)==0)
	{
		switch(map_coad)
		{
			case '\n':
				if(termio.c_iflag&INLCR)map_coad='\r';
				else if(termio.c_lflag&ICANON)rest=1;
				break;
			case '\r':
				if(termio.c_iflag&IGNCR)map_coad='\0';
				else if(termio.c_iflag&ICRNL)
				{
					map_coad='\n';
					if(termio.c_lflag&ICANON)rest=1;
				}
		}
		writeVram(&map_coad,1,0);
	}

	return rest;
}


/*
 * GLOBAL
 * keyboard interrupt handler.
 * return : task switch on off
 */
static int keyboardIntrHandler()
{
	if(keyboardHnadler(inb(KEYBOARD_OUTPUT_REG)))
	{
		wakeProc(&waitLink);
		return 1;
	}
	return 0;
}


/*
 * GLOBAL
 * init keyboard driver
 */
void initKeyboard()
{


	/* Turn off LED. */
	switchLed(0);

	/* ߤ */
	irq_entry[IRQ1]=keyboardIntrHandler;

	release_irq_mask(IRQ_NO);

	/* clear data register */
	inb(KEYBOARD_OUTPUT_REG);
}


/***************************************************************************************
 *
 * ƥॳ륯饹
 *
 ***************************************************************************************/

static int open()
{
	return 0;
}

/* size=žǽХȿ */
static int read(void *buf,size_t size,size_t start)
{
	int len;


	/* Ϥ뤫󥽡Хåեǧ */
	set_irq_mask(IRQ_NO);		/* ߥޥ */
	if((len=readConsoleBuf(buf,size))==0)
	{
		/* Ԥ */
		waitProc(&waitLink,IRQ_NO);

		len=readConsoleBuf(buf,size);
	}
	else
	{
		release_irq_mask(IRQ_NO);
		inb(KEYBOARD_OUTPUT_REG);	/* clear data register */
	}

	return len;
}

int writeConsole(void *buf,size_t size,size_t start)
{
	int num;


	num=writeVram((const uchar*)buf,size,0);
	initConsoleBuf();

	return num;
}


/***************************************************************************************
 *
 * 饹
 *
 ***************************************************************************************/

/* PRIVATE,Device interface */
static DEV_INFO devInfo={
	"console",0,0,0,open,read,writeConsole,NULL
};


/*
 * CLOBAL
 * ǥХϿ򤹤롣
 */
int registConsole()
{
	return regist_device(&devInfo);
}
