/*
 * fs.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * ۥե륷ƥࡣ
 * ե륷ƥඦ̽
 */


#include"types.h"
#include"config.h"
#include"lib.h"
#include"mm.h"
#include"proc.h"
#include"lock.h"
#include"fs.h"
#include"test.h"
#include"errno.h"
#include"interrupt.h"



/* ޥȥե륷ƥ।ǥå */
typedef struct{
	ushort fs_num;			/* ե륷ƥʥС */
	ushort ref_count;		/* ȥȡ */
}MOUNT_FS;


static FS *fs_info[MAX_REGIST_FS];
static MOUNT_FS mount_fs[MAX_DEVICE_OPEN];


/***************************************************************************************
 *
 * < ۥե륷ƥ९饹 >
 *
 *  ޥȻȥǥ쥯ȥѹե륪ץե¹Իɲä롣
 *  ȥ꡼ϡ"."+".."+̾ͽ¤٤롣
 *  ȥ꡼󥯤κǸϺǽΥȥ꡼Ϣ뤹롣
 *  ǥ쥯ȥϥޥȤ줿ΰʳϥǥХ֥åǤ϶
 *  ƥॳǸüȥ꡼ϡΥƥॳ˺ʤ褦󥿤䤷
 *   ΥƥॳκǸǥ󥿤򸺤餹
 *
 ***************************************************************************************/

enum{
	VFS_MAX_NAME_SIZE=255,		/* ۥե륷ƥκ̾ */
};


/* ۥȥ꡼ */
typedef struct VENTRY{
	struct VENTRY *next;		/* ȥ꡼󥯥ͥ */
	struct VENTRY *prev;		/* ȥ꡼󥯥ץӥ塼 */
	uint block;					/* ǥХ֥å */
	struct VENTRY *parent;		/* ƥȥ꡼ */
	struct VENTRY *child;		/* ΥΡɤΥȥåץȥ꡼ */
	union{
		struct VENTRY *mount;	/* ޥȥȥ꡼ */
		void *exe_page;			/* ¹ԥե륤᡼ɤ߼ڡ */
	}opt;
	int busy;					/* busy flag. */
	uchar refCount;				/* ȥ󥿡ޥȤ䥫ȥǥ쥯ȥ¹ԥե¹ԥ */
	uchar din;					/* ǥХinode */
	uchar type;					/* ե륿 */
	uchar len;					/* Name size. */
	char name[0];				/* ѥ͡ */
}VENTRY;

typedef struct{
	const char *path;
	VENTRY *vent;
}SEARCH_PARAM;


/* PUBLIC 롼ȥȥ꡼ */
static VENTRY rootEnt;

/* PUBLIC devǥ쥯ȥꥨȥ꡼ */
static char dev[sizeof(VENTRY)+3];
static VENTRY *devEnt=(VENTRY*)dev;

/* PUBLIC ۥե륷ƥॹԥåȡ */
static int spinLock=0;


/*
 * PRIVATE
 * ȥ꡼󥯤饨ȥ꡼򸡺롣
 * Υȥ꡼ɥ쥹ݥ󥿡ˤϸĤäȥ꡼ޤϡ
 * ǸΥȥ꡼ɥ쥹Хåե롣
 * parameters : ѥɥ쥹,ǥ쥯ȥꥨȥ꡼ɥ쥹ݥ󥿡,entry buffer
 * return : next path or failed=NULL
 */
static const char *searchEntry(const char *path,VENTRY *entry,VENTRY **buf)
{
	int rest;
	VENTRY *last;


	if(cmpPathLength(".",path,1)==0)
	{
		*buf=entry;
		if(path[1]=='/')return &path[2];
		else return &path[1];
	}
	if(cmpPathLength("..",path,2)==0)
	{
		*buf=entry->parent;
		if(path[2]=='/')return &path[3];
		else return &path[2];
	}

	if((last=entry=entry->child)==NULL)
	{
		*buf=NULL;
		return NULL;
	}
	switch(rest=cmpPathLength(entry->name,path,entry->len))
	{
		case 1:
			*buf=NULL;
			return NULL;
		case -1:
			for(entry=entry->next;entry!=last;entry=entry->next)
				if((rest=cmpPathLength(entry->name,path,entry->len))!=-1)break;
	}

	*buf=entry;

	if(rest==0)
	{
		if(path[entry->len]=='/')return &path[entry->len+1];
		return &path[entry->len];
	}

	return NULL;
}


/*
 * PRIVATE
 * ȥ꡼롣
 * parameters : entry addres
 * return : 0 or error number
 */
static int delEntry(VENTRY *entry)
{
	VENTRY *parent;


	if(entry->refCount>0)return -EBUSY;
	if(entry->child!=NULL)return -EBUSY;

	parent=entry->parent;

	if(entry->next==entry)parent->child=NULL;
	else
	{
		if(parent->child==entry)parent->child=entry->next;
		entry->prev->next=entry->next;
		entry->next->prev=entry->prev;
	}

	kfree(entry);

	return 0;
}


/*
 * PRIVATE
 * ȥ꡼̤äƺ롣
 * parameters : ȥ꡼
 */
static inline void delUpEntry(VENTRY *entry)
{
	VENTRY *parent;

	for(;;)
	{
		parent=entry->parent;
		if(delEntry(entry)!=0)break;
		entry=parent;
	}
}


/*
 * PUBLIC
 * parameters : entry
 */
/* ӥե饰ΩƤ롣 */
static inline void setBusy(VENTRY *entry)
{
	entry->busy=1;
}

/* ӥե饰ݤ */
static inline void resetBusy(VENTRY *entry)
{
	entry->busy=0;
}

/* ӥե饰򸡺롣 */
static inline int isBusy(VENTRY *entry)
{
	return entry->busy;
}


/*
 * PUBLIC
 * ѥ̾Υȥ꡼򸡺롣
 * ǽΰˤϸƤӽФͤͿ졢ޥȤ줿ǥ쥯ȥ꤬Ф롣
 * ΰϥѥκǽΥȥ꡼롣
 * parameters : start search parmeters struct,last search parmeters struct
 * return : 0 or error number
 */
static int searchPath(SEARCH_PARAM *start,SEARCH_PARAM *end)
{
	const char *path,*pt;
	VENTRY *entry=start->vent;
	VENTRY *buf;


	for(path=start->path;*path!='\0';)
	{
		if(entry->type!=DIRECTORY)return -ENOTDIR;
		if(entry->opt.mount!=NULL)
		{
			start->path=path;
			start->vent=entry->opt.mount;
			entry=entry->opt.mount;
		}

		if((pt=searchEntry(path,entry,&buf))==NULL)break;
		path=pt;
		entry=buf;
	}

	end->path=path;
	end->vent=entry;

	return 0;
}


/*
 * PUBLIC
 * ѥƬΡ̾Υȥ꡼󥯤ɲä롣
 * parameters : path pointer,parent entry,file type
 * return : new entry address
 */
static VENTRY *addEntry(const char *path,VENTRY *entry,int type)
{
	int len;
	VENTRY *next_ent,*new_ent;


	/* ȥ꡼򸡺 */
	if(searchEntry(path,entry,&next_ent)!=NULL)return NULL;

	/* ȥ꡼κ */
	len=pathLen(path);
	if((new_ent=(VENTRY*)kmalloc(sizeof(VENTRY)+len))==NULL)return NULL;
	memset(new_ent,0,sizeof(VENTRY));
	new_ent->din=entry->din;
	new_ent->type=type;
	new_ent->parent=entry;
	new_ent->len=len;
	memcpy(new_ent->name,path,len);

	/* 󥯤ɲä롣 */
	if(next_ent!=NULL)
	{
		new_ent->prev=next_ent->prev;
		new_ent->next=next_ent;
		next_ent->prev->next=new_ent;
		next_ent->prev=new_ent;
	}
	else if(entry->child!=NULL)
	{
		next_ent=entry->child;
		new_ent->prev=next_ent->prev;
		new_ent->next=next_ent;
		next_ent->prev->next=new_ent;
		next_ent->prev=new_ent;
		entry->child=new_ent;
	}
	else
	{
		new_ent->prev=new_ent;
		new_ent->next=new_ent;
		entry->child=new_ent;
	}

	return new_ent;
}


/*
 * PUBLIC
 * ȥ꡼λȥ󥿤1䤹
 * parameters : ۥȥ꡼
 */
static inline void linkVentry(VENTRY *vent)
{
	++vent->refCount;
}


/*
 * PUBLIC
 * ȥ꡼λȥ󥿤1餷0ʤ̥Ρɥȥ꡼̤äƺ롣
 * parameters : ۥȥ꡼
 */
static inline void unlinkVentry(VENTRY *vent)
{
	if(--vent->refCount<=0)
	{
		vent->refCount=0;
		delUpEntry(vent);
	}
}


/*
 * PUBLIC
 * ȥ꡼ѥɲä롣
 * ա pathκǽ餬NULLʸǤʤȡ
 * parameters : path,ۥȥ꡼
 * return : last entry or error number
 */
static inline int mkEntry(const char *path,VENTRY *vent)
{
	for(;;++path)
	{
		if((vent=addEntry(path,vent,DIRECTORY))==NULL)
		{
			unlinkVentry(vent);
			return -ENOMEM;
		}
		path+=vent->len;

		if(*path=='\0')break;
	}

	return (int)vent;
}


/*
 * PUBLIC
 * ȥ꡼˿ȥ꡼ޥȤ롣
 * parameters : ޥȥȥ꡼,ǥХinode,롼ȥǥ쥯ȥ֥å
 * return : 0 or error=-1
 */
static int mountEntry(VENTRY *entry,int din,uint block)
{
	VENTRY *mount_ent;


	/* Ǥ˥ޥȤƤ뤫 */
	if(entry->opt.mount!=NULL)return -EBUSY;

	/* ޥȥȥ꡼κ */
	if((mount_ent=(VENTRY*)kmalloc(sizeof(VENTRY)))==NULL)return -ENOMEM;
	memset(mount_ent,0,sizeof(VENTRY)+1);
	mount_ent->din=din;
	mount_ent->block=block;
	mount_ent->parent=entry->parent;
	linkVentry(mount_ent);

	/* ޥȥȥ꡼˥ޥȤ롣 */
	++entry->refCount;
	entry->opt.mount=mount_ent;

	return 0;
}


/*
 * PUBLIC
 * ޥȡ
 * parameters : ޥȥȥ꡼
 * return : 0 or error number
 */
static int umountEntry(VENTRY *entry)
{
	int rest;
	VENTRY *mount_ent;


	if((mount_ent=entry->opt.mount)==NULL)return -ENOENT;
	if(mount_ent->refCount>1)return -EBUSY;
	if(mount_ent->child!=mount_ent->child->next->next)return -EBUSY;

	/* ޥȥȥ꡼κ */
	mount_ent->refCount=0;
	if((rest=delEntry(mount_ent))<0)return rest;
	--entry->refCount;

	/* ޥȥȥ꡼ʤ */
	unlinkVentry(entry);

	return 0;
}


/*
 * PUBLIC
 * ѥ롼ȤޤäƥХåե˥ԡ롣
 * parameters : buffer,buffer size,ۥȥ꡼
 * return : write size or error=-1
 */
static int copyCurrentPath(char *buf,size_t size,VENTRY *vent)
{
	int len;


	if(vent==&rootEnt)
	{
		buf[0]='/';

		return 1;
	}
	else
	{
		if((len=copyCurrentPath(buf,size,vent->parent))==-1)return -1;
		if(len+vent->len+1>size)return -1;
		memcpy(&buf[len],vent->name,vent->len);
		buf[len+vent->len]='/';

		return len+vent->len+1;
	}
}


/************************************************************************************
 *
 * < ե륷ƥ饹 >
 *
 ************************************************************************************/

enum{
	MOUNT_DEV=0,		/* ǥХե륷ƥdevice inodeɬ0 */
};


/* եǥץ */
typedef struct{
	uchar ref_count;	/* ȥ */
	uchar din;			/* device inode number. */
	ushort accessMode;	/* access mode. */
	uint inode;			/* inode֥å */
	size_t offset;		/* ɤ߽񤭳ϥեå */
	VENTRY *vent;		/* ۥȥ꡼ */
}F_DSC;

/* ץѥե빽¤ */
typedef struct{
	VENTRY *current_dir;		/* ȥǥ쥯ȥ */
	F_DSC *fd[MAX_FILE_OPEN];
	uchar num[MAX_FILE_OPEN];
	uchar fd_count;				/* եǥץץ */
	uchar next_fd;				/* ζեǥץ */
}FILE_STRUCT;


/*
 * PUBLIC
 * ޥȥǥХ
 * parameters : device name,filesystem number
 * return : device inode or error number
 */
static inline int mountDevice(const char *dev_path,int fs)
{
	enum{DEV_NAME_SIZE=5};		/* "/dev/"Υ */

	int mount_din;
	int rest;


	/* ǥХץ */
	if(cmpPathNull("/dev",dev_path)==NULL)return -EINVAL;
	if((rest=fs_info[mount_fs[MOUNT_DEV].fs_num]->open(dev_path+DEV_NAME_SIZE,0,0,0))<0)return rest;
	mount_din=rest;

	/* ǥХ˥ޥȤƤ뤫 */
	if((++mount_fs[mount_din].ref_count>1)&&(mount_fs[mount_din].fs_num!=fs))goto ERR;
	else mount_fs[mount_din].fs_num=fs;

	return mount_din;

ERR:
	fs_info[mount_fs[MOUNT_DEV].fs_num]->close(MOUNT_DEV,mount_din);

	return -EBUSY;
}


/*
 * PUBLIC
 * ȥǥ쥯ȥ롣
 * parameters : ѥɥ쥹ݥ
 * return : Ȳۥǥ쥯ȥꥨȥ꡼
 */
static inline VENTRY *getCurrentDir(const char **path)
{
	if(**path=='/')
	{
		*path+=1;
		return &rootEnt;
	}
	else
		return ((FILE_STRUCT*)get_current_task()->file_struct)->current_dir;
}


/*
 * GLOBAL
 * Register filesystem
 * parameters : Filesystem struct pinter
 * return : 0 or Error=-1
 */
int regist_fs(FS *fs)
{
	int i;


	for(i=0;i<MAX_REGIST_FS;++i)
		if(fs_info[i]==NULL)
		{
			fs_info[i]=fs;
			return 0;
		}

	return -1;
}


/*
 * GLOBAL
 * Search file system
 * parameters : file system name
 * return : file system number or noshing=-1
 */
int search_fs(const char *name)
{
	int i;


	for(i=0;fs_info[i]!=NULL;++i)
		if(strcmp(fs_info[i]->name,name)==0)return i;

	return -1;
}


/*
 * GLOBAL
 * եǥץinode֤
 * parameters : file descriptor
 * return : inode block or error=-1
 */
int get_inode(int fd)
{
	F_DSC *fdsc;


	if(fd>MAX_FILE_OPEN)return -1;

	if((fdsc=((FILE_STRUCT*)(get_current_task()->file_struct))->fd[fd])==NULL)return -1;

	return fdsc->inode;
}


/*
 * GLOBAL
 * fork˥եطι¤Τƥץ饳ԡ롣
 * parameters : parent file structure
 * return : child file structure or error=NULL
 */
void *cpy_file_struct(void *fst)
{
	int i,fd_count;
	FILE_STRUCT *cpfst;


	if((cpfst=kmalloc(sizeof(FILE_STRUCT)))==NULL)return NULL;
	memcpy(cpfst,fst,sizeof(FILE_STRUCT));

	/* եǥץλȥ󥿤1䤹 */
	fd_count=((FILE_STRUCT*)fst)->fd_count;
	for(i=0;;++i)
		if(cpfst->fd[i]!=NULL)
		{
			++cpfst->fd[i]->ref_count;
			if(--fd_count<=0)break;
		}

	return cpfst;
}


/*
 * GLOBAL
 * եطι¤Τ롣
 * parameters : FILE_STRUCT address
 * return : 0 or error=-1
 */
void releaseFileStruct(void *fstruct)
{
	int i;


	for(i=0;i<MAX_FILE_OPEN;++i)
	{
		if(((FILE_STRUCT*)fstruct)->fd[i]!=NULL)
		{
			if(sys_close(i)==-1)return;
			if(((FILE_STRUCT*)fstruct)->fd_count<=0)return;
		}
	}

	kfree(fstruct);
}


/*
 * GLOBAL
 * ǽΥץȤ˸ƤӽФ
 * parameters : file structure address
 */
void setFileStruct(void *fst)
{
	((FILE_STRUCT*)fst)->current_dir=&rootEnt;
}


/*
 * GLOBAL
 * ɥץѥե빽¤Τν
 * parameters : process
 * return : 0 or error=-1
 */
int init_file_struct(PROC *proc)
{
	int i;
	FILE_STRUCT *p;


	if((p=(FILE_STRUCT*)kmalloc(sizeof(FILE_STRUCT)))==NULL)return -1;
	p->fd_count=0;
	p->next_fd=0;
	memset(p->fd,0,sizeof(F_DSC*)*MAX_FILE_OPEN);
	for(i=0;i<MAX_FILE_OPEN;++i)p->num[i]=i+1;
	p->current_dir=NULL;
	proc->file_struct=(void*)p;

	return 0;
}


/************************************************************************************
 *
 * ե¹
 *
 ************************************************************************************/

/*
 * GLOBAL
 * ¹ԥե륪ץ̾Υץδάؿ
 * parameters : path,open file struct
 * return : 0 or error number
 */
int exec_open(const char *path,OPEN_F *open_f)
{
	int block;
	int rest;
	SEARCH_PARAM start,end;
	VENTRY *vent;


	if(*path=='\0')return -ENOENT;

	/* եõơۥȥ꡼롣 */
	start.vent=getCurrentDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if((*end.path=='\0')&&(end.vent->type==DIRECTORY))
	{
		rest=-EISDIR;
		goto END;
	}

	if(*end.path=='\0')
	{
		linkVentry(end.vent);
		vent=end.vent;
	}
	else
	{
		if((rest=fs_info[mount_fs[start.vent->din].fs_num]->open(start.path,start.vent->din,start.vent->block,0))<0)goto END;
		block=rest;

		enter_spinlock(&spinLock);
		{
			rest=mkEntry(end.path,end.vent);
		}
		exit_spinlock(&spinLock);
		if(rest<0)
		{
			fs_info[mount_fs[start.vent->din].fs_num]->close(start.vent->din,start.vent->block);
			goto END;
		}

		vent=(VENTRY*)rest;
		vent->block=block;
		vent->type=NORMAL_FILE;
	}

	open_f->din=vent->din;
	open_f->inode=vent->block;
	rest=0;
END:
	resetBusy(end.vent);

	return rest;
}


/*
 * GLOBAL
 * ¹ԥեɤ߹ߡ̾Υ꡼ɤδάؿ
 * parameters : open file struct,buffer,read size
 * return : read byte or error=-1
 */
int exec_read(OPEN_F *open_f,void *buf,size_t size)
{
	return fs_info[mount_fs[open_f->din].fs_num]->read(open_f->din,open_f->inode,buf,size,open_f->offset);
}


/************************************************************************************
 *
 * System call interface
 *
 ************************************************************************************/

int sys_mount(const char *dev_path,const char *fs_name,const char *path)
{
	int fs;
	int mount_din;
	int rest;
	uint root_blk;
	SEARCH_PARAM start,end;
	VENTRY *vent;
	DIR dir;


	/* Search file system number */
	if((fs=search_fs(fs_name))==-1)return -ENOTSUP;

	/* ǥХץ */
	if((rest=mountDevice(dev_path,fs))<0)return rest;
	mount_din=rest;

	/* ޥȴؿƤӽФ */
	if((rest=fs_info[fs]->mount(mount_din))<0)goto ERR;
	root_blk=rest;

	/* ۥե륷ƥΥǥ쥯ȥ򸡺롣 */
	start.vent=getCurrentDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			goto ERR;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(*end.path!='\0')
	{
		if((rest=fs_info[mount_fs[start.vent->din].fs_num]->opendir(start.path,start.vent->din,start.vent->block,&dir))<0)goto ERR2;
		if((rest=mkEntry(end.path,end.vent))<0)goto ERR2;
		vent=(VENTRY*)rest;
	}
	else vent=end.vent;

	/* ޥȤ벾ۥǥ쥯ȥ롣 */
	if((rest=mountEntry(vent,mount_din,root_blk))<0)goto ERR2;

	resetBusy(end.vent);

	return 0;
ERR2:
	resetBusy(end.vent);
ERR:
	--mount_fs[mount_din].ref_count;
	fs_info[mount_fs[MOUNT_DEV].fs_num]->close(MOUNT_DEV,mount_din);

	return rest;
}


int sys_umount(const char *path)
{
	int fs;
	int rest;
	VENTRY *mount_ent;
	SEARCH_PARAM start,end;


	/* ǥ쥯ȥõ */
	start.vent=getCurrentDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if((*end.path!='\0')||(end.vent->type!=DIRECTORY))
	{
		rest=-ENOENT;
		goto END;
	}

	/* ޥȡ */
	mount_ent=end.vent->opt.mount;
	if((rest=umountEntry(end.vent))<0)goto END;

	fs=mount_fs[mount_ent->din].fs_num;
	if((rest=fs_info[fs]->umount(mount_ent->din))<0)goto END;
	--mount_fs[mount_ent->din].ref_count;
END:
	resetBusy(end.vent);

	return rest;
}


int sys_lseek(int fd_num,int offset,int flag)
{
	int rest=0;
	FSTAT fstat;
	F_DSC *fd=((FILE_STRUCT*)get_current_task()->file_struct)->fd[fd_num];


	if(fd==NULL)return -EBADF;

	fs_info[mount_fs[fd->din].fs_num]->stat(fd->inode,NULL,0,0,&fstat);
	switch(flag)
	{
		case SEEK_CUR:
			rest=fd->offset;
		case SEEK_SET:
			if((rest+=offset)>fstat.st_size)return -EOVERFLOW;
			break;
		case SEEK_END:
			rest=fstat.st_size;
			if((rest-=offset)<0)return -EINVAL;
			break;
		default:
			return -EINVAL;
	}

	return fd->offset=rest;
}


int sys_creat(const char *path,int flag,int mode)
{
	int fd_num;
	int rest;
	int block;
	SEARCH_PARAM start,end;
	FILE_STRUCT *fstruct;
	F_DSC *fd;
	VENTRY *vent;


	if(*path=='\0')return -ENOENT;

	/* ץΥå */
	fstruct=(FILE_STRUCT*)get_current_task()->file_struct;
	if(fstruct->fd_count>=MAX_FILE_OPEN)return -EMFILE;

	/* եǥץϿ롣 */
	if((fd=(F_DSC*)kmalloc(sizeof(F_DSC)))==NULL)return -ENOMEM;

	/* եõơۥȥ꡼롣 */
	start.vent=getCurrentDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(*end.path=='\0')
	{
		rest=-EEXIST;
		goto END;
	}

	if((rest=fs_info[mount_fs[start.vent->din].fs_num]->creat(start.path,start.vent->din,start.vent->block,mode))<0)goto END;
	block=rest;

	enter_spinlock(&spinLock);
	{
		rest=mkEntry(end.path,end.vent);
	}
	exit_spinlock(&spinLock);
	if(rest<0)
	{
		fs_info[mount_fs[start.vent->din].fs_num]->close(start.vent->din,start.vent->block);
		goto END;
	}

	vent=(VENTRY*)rest;
	vent->block=block;
	vent->type=NORMAL_FILE;

	/* եǥץν */
	fd->din=vent->din;
	fd->accessMode=flag;
	fd->inode=vent->block;
	fd->offset=0;
	fd->ref_count=1;
	fd->vent=vent;
	fstruct->fd[fstruct->next_fd]=fd;
	fd_num=fstruct->next_fd;
	fstruct->next_fd=fstruct->num[fstruct->next_fd];
	++fstruct->fd_count;
	rest=fd_num;
END:
	resetBusy(end.vent);

	return rest;
}


/*
 * parameters : path,flag,mode
 * return : file descriptor number or error=-1
 */
int sys_open(const char *path,int flag,int mode)
{
	int fd_num;
	int rest;
	int block;
	SEARCH_PARAM start,end;
	FILE_STRUCT *fstruct;
	F_DSC *fd;
	VENTRY *vent;


	if(*path=='\0')return -ENOENT;

	if((flag&(O_CREAT|O_EXCL))==(O_CREAT|O_EXCL))return sys_creat(path,flag,mode);

	/* ץΥå */
	fstruct=(FILE_STRUCT*)get_current_task()->file_struct;
	if(fstruct->fd_count>=MAX_FILE_OPEN)return -EMFILE;

	if((fd=(F_DSC*)kmalloc(sizeof(F_DSC)))==NULL)return -ENOMEM;

	/* եõơۥȥ꡼롣 */
	start.vent=getCurrentDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(*end.path=='\0')
	{
		if(end.vent->type!=DIRECTORY)
		{
			linkVentry(end.vent);
			vent=end.vent;
		}
		else
		{
			rest=-EISDIR;
			goto END;
		}
	}
	else
	{
		rest=fs_info[mount_fs[start.vent->din].fs_num]->open(start.path,start.vent->din,start.vent->block,flag&O_TRUNC);
		if((rest==-ENOENT)&&(flag&O_CREAT))
			rest=fs_info[mount_fs[start.vent->din].fs_num]->creat(start.path,start.vent->din,start.vent->block,mode);
		if(rest<0)goto END;
		block=rest;

		enter_spinlock(&spinLock);
		{
			rest=mkEntry(end.path,end.vent);
		}
		exit_spinlock(&spinLock);
		if(rest<0)
		{
			fs_info[mount_fs[start.vent->din].fs_num]->close(start.vent->din,start.vent->block);
			goto END;
		}

		vent=(VENTRY*)rest;
		vent->block=block;
		vent->type=NORMAL_FILE;
	}

	/* եǥץϿ롣 */
	fd->din=vent->din;
	fd->accessMode=flag;
	fd->inode=vent->block;
	fd->offset=0;
	fd->ref_count=1;
	fd->vent=vent;
	fstruct->fd[fstruct->next_fd]=fd;
	fd_num=fstruct->next_fd;
	fstruct->next_fd=fstruct->num[fstruct->next_fd];
	++fstruct->fd_count;
	rest=fd_num;

	if(flag&O_APPEND)sys_lseek(fd_num,0,SEEK_END);
END:
	resetBusy(end.vent);

	return rest;
}


/*
 * parameters : file descriptor number
 * return : 0 or error=-1
 */
int sys_close(int fd_num)
{
	FILE_STRUCT *fstruct;
	F_DSC *fd;
	int rest;


	fstruct=(FILE_STRUCT*)get_current_task()->file_struct;
	if((fd=fstruct->fd[fd_num])==NULL)return -EBADF;

	fstruct->fd[fd_num]=NULL;
	fstruct->num[fd_num]=fstruct->next_fd;
	fstruct->next_fd=fd_num;
	--fstruct->fd_count;

	unlinkVentry(fd->vent);
	if(--fd->ref_count>0)return 0;			/* ȥ󥿤1餹 */
	rest=fs_info[mount_fs[fd->din].fs_num]->close(fd->din,fd->inode);
	kfree(fd);

	return rest;
}


/*
 * parameters : file descriptor number,baffer,read bytes
 * return : read bytes or error=-1
 */
int sys_read(int fd_num,void *buf,size_t size)
{
	FILE_STRUCT *fstruct;
	F_DSC *fd;
	int rest;


	fstruct=(FILE_STRUCT*)get_current_task()->file_struct;
	if((fd=fstruct->fd[fd_num])==NULL)return -EBADF;
	if((fd->accessMode&(O_RDONLY|O_RDWR))==0)return -EBADF;		/* ⡼ɤΥå */

	/* readؿƤӽФ */
	if((rest=fs_info[mount_fs[fd->din].fs_num]->read(fd->din,fd->inode,buf,size,fd->offset))>0)
		fd->offset+=rest;

	return rest;
}


int sys_write(int fd_num,void *buf,size_t size)
{
	FILE_STRUCT *fstruct;
	F_DSC *fd;
	int rest;


	fstruct=(FILE_STRUCT*)get_current_task()->file_struct;
	if((fd=fstruct->fd[fd_num])==NULL)return -EBADF;
	if((fd->accessMode&(O_WRONLY|O_RDWR))==0)return -EBADF;		/* ⡼ɤΥå */

	/* writeؿƤӽФ */
	rest=fs_info[mount_fs[fd->din].fs_num]->write(fd->din,fd->inode,buf,size,fd->offset);
	if(rest==-1)return -1;
	fd->offset+=rest;

	return rest;
}


int sys_mkdir(const char *path,int mode)
{
	int rest;
	SEARCH_PARAM start,end;


	if(*path=='\0')return -ENOENT;

	start.vent=getCurrentDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(*end.path=='\0')
	{
		rest=-EEXIST;
		goto END;
	}

	rest=fs_info[mount_fs[start.vent->din].fs_num]->mkdir(start.path,start.vent->din,start.vent->block,mode);
END:
	resetBusy(end.vent);

	return rest;
}


int sys_unlink(const char *path)
{
	int rest;
	SEARCH_PARAM start,end;


	if(*path=='\0')return -ENOENT;

	start.vent=getCurrentDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(*end.path!='\0')
		rest=fs_info[mount_fs[start.vent->din].fs_num]->delete(start.path,start.vent->din,start.vent->block,NORMAL_FILE);
	else rest=-EBUSY;

	resetBusy(end.vent);

	return rest;
}


int sys_rmdir(const char *path)
{
	int rest;
	SEARCH_PARAM start,end;


	if(*path=='\0')return -ENOENT;

	start.vent=getCurrentDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(*end.path!='\0')
		rest=fs_info[mount_fs[start.vent->din].fs_num]->delete(start.path,start.vent->din,start.vent->block,DIRECTORY);
	else rest=-EBUSY;

	resetBusy(end.vent);

	return rest;
}


int sys_opendir(const char *path,DIR *dir)
{
	int rest;
	SEARCH_PARAM start,end;


	if(*path=='\0')return -ENOENT;

	/* եõ */
	start.vent=getCurrentDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(end.vent->type==DIRECTORY)
	{
		if((rest=fs_info[mount_fs[start.vent->din].fs_num]->opendir(start.path,start.vent->din,start.vent->block,dir))>0)
		{
			dir->dinode=start.vent->din;
			dir->inode=rest;
			rest=0;
		}
	}
	else rest=-ENOENT;

	resetBusy(end.vent);

	return rest;
}


int sys_closedir(DIR *dir)
{
	return fs_info[mount_fs[dir->dinode].fs_num]->close(dir->dinode,dir->inode);
}


int sys_readdir(DIR *dir,char *name)
{
	return fs_info[mount_fs[dir->dinode].fs_num]->readdir(dir->dinode,dir->inode,dir,name);
}


/*
 * old pathnew pathƱ졢ͭå롣
 */
int sys_rename(const char *old_path,const char *new_path)
{
	int rest;
	SEARCH_PARAM old_start,old_end;
	SEARCH_PARAM new_start,new_end;


	if(strcmp(old_path,new_path)==0)return 0;

	/* եõ */
	old_start.vent=getCurrentDir(&old_path);
	old_start.path=old_path;
	new_start.vent=getCurrentDir(&new_path);
	new_start.path=new_path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if(((rest=searchPath(&old_start,&old_end))<0)||((rest=searchPath(&new_start,&new_end))<0))
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if((isBusy(old_end.vent)==0)&&(isBusy(new_end.vent)==0))
		{
			setBusy(old_end.vent);
			setBusy(new_end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(old_start.vent->din==new_start.vent->din)
	{
		if((*old_end.path!='\0')&&(*new_end.path!='\0'))
			rest=fs_info[mount_fs[old_start.vent->din].fs_num]->
				rename(old_start.vent->din,old_start.vent->block,old_start.path,new_start.vent->block,new_start.path);
		else rest=-EBUSY;
	}
	else rest=-EXDEV;

	resetBusy(old_end.vent);
	if(old_end.vent!=new_end.vent)resetBusy(new_end.vent);

	return rest;
}


int sys_chdir(const char *path)
{
	uint block;
	int rest=0;
	SEARCH_PARAM start,end;
	VENTRY *vent;
	FILE_STRUCT *fstruct;
	DIR dir;


	if(*path=='\0')return -ENOENT;

	/* ǥ쥯ȥõ */
	start.vent=getCurrentDir(&path);
    start.path=path;
    for(;;)
    {
        enter_spinlock(&spinLock);

        if((rest=searchPath(&start,&end))<0)
        {
            exit_spinlock(&spinLock);
            return rest;
        }
        if(isBusy(end.vent)==0)
        {
            setBusy(end.vent);
            exit_spinlock(&spinLock);
            break;
        }
        exit_spinlock(&spinLock);
        wait_task();
    }

	if(end.vent->type==DIRECTORY)
	{
		if(*end.path=='\0')
		{
			linkVentry(end.vent);
			vent=end.vent;
		}
		else
		{
			if((rest=fs_info[mount_fs[start.vent->din].fs_num]->opendir(start.path,start.vent->din,start.vent->block,&dir))>0)
			{
				block=rest;
				enter_spinlock(&spinLock);
				{
					rest=mkEntry(end.path,end.vent);
				}
				exit_spinlock(&spinLock);

				if(rest>0)
				{
					vent=(VENTRY*)rest;
					vent->block=block;
					vent->type=DIRECTORY;
				}
				else goto ERR;
			}
			else goto ERR;
		}
	}
	else
	{
		rest=-ENOTDIR;
		goto ERR;
	}

	resetBusy(end.vent);

	/* ե빽¤Τ˿ȥǥ쥯ȥϿ */
	fstruct=(FILE_STRUCT*)get_current_task()->file_struct;
	unlinkVentry(fstruct->current_dir);
	fstruct->current_dir=vent;

	return 0;
ERR:
	resetBusy(end.vent);
	
	return rest;
}


int sys_getcwd(char *buf,size_t size)
{
	int len;
	VENTRY *vent;


	if(checkMem(buf)==-1)return -EFAULT;
	if(size==0)return -EINVAL;

	vent=((FILE_STRUCT*)get_current_task()->file_struct)->current_dir;
	if((len=copyCurrentPath(buf,size,vent))==-1)return -ERANGE;
	if(len==1)buf[1]='\0';
	else buf[len-1]='\0';

	return 0;
}


/*
 * pathNULLʤfile descriptorõ
 * parameters : file descriptor,path,stat buffer
 */
int sys_stat(int fd_num,const char *path,struct stat *buf)
{
	int rest;
	SEARCH_PARAM start,end;
	F_DSC *fd;


	if(checkMem(buf)==-1)return -EFAULT;

	if(path==NULL)
	{
		if((fd=((FILE_STRUCT*)get_current_task()->file_struct)->fd[fd_num])==NULL)return -EBADF;
		return fs_info[mount_fs[fd->din].fs_num]->stat(fd->inode,NULL,0,0,buf);
	}
	else
	{
		start.vent=getCurrentDir(&path);
		start.path=path;
		for(;;)
		{
			enter_spinlock(&spinLock);

			if((rest=searchPath(&start,&end))<0)
        	{
				exit_spinlock(&spinLock);
				return rest;
			}
			if(isBusy(end.vent)==0)
			{
				setBusy(end.vent);
				exit_spinlock(&spinLock);
				break;
			}
			exit_spinlock(&spinLock);
			wait_task();
		}

		if((*end.path=='\0')&&(end.vent->block!=0))
			rest=fs_info[mount_fs[end.vent->din].fs_num]->stat(end.vent->block,NULL,0,0,buf);
		else 
			rest=fs_info[mount_fs[start.vent->din].fs_num]->stat(0,path,start.vent->din,start.vent->block,buf);
		
		resetBusy(end.vent);

		return rest;
	}
} 

/************************************* ̤ ******************************************/
int sys_lock()
{
	return 0;
}


/************************************************************************************
 *
 * 
 *
 ************************************************************************************/

/*
 * GLOBAL
 * Init filesystem
 */
void init_fs()
{
	/* ե륷ƥν */
	memset(fs_info,0,sizeof(FS*)*MAX_REGIST_FS);
	memset(mount_fs,0,sizeof(MOUNT_FS)*MAX_DEVICE_OPEN);

	/* ۥե륷ƥν */
	memset(&rootEnt,0,sizeof(rootEnt));
	rootEnt.parent=&rootEnt;
	rootEnt.refCount=1;
	rootEnt.type=DIRECTORY;

	/* deviceȥ꡼κ */
	memset(devEnt,0,sizeof(*devEnt));
	devEnt->prev=devEnt;
	devEnt->next=devEnt;
	devEnt->parent=&rootEnt;
	devEnt->refCount=1;
	devEnt->type=DIRECTORY;
	devEnt->len=3;
	memcpy(devEnt->name,"dev",3);
	rootEnt.child=devEnt;

	/* deviceȥ꡼˥ޥȤ롣 */
	if(mountEntry(devEnt,MOUNT_DEV,0)<0)
		printk("Failed mount device filesystem!\n");
}


/*
 * GLOBAL
 * mount filesystem
 * parameters : device name,filesystem name
 * return : 0 or error=-1
 */
int mountRoot(const char *dev_path,const char *fs_name)
{
	int fs,mount_din;
	uint root_blk;

	/* Search file system number */
	if((fs=search_fs(fs_name))==-1)return -1;

	/* ǥХץ */
	if((mount_din=mountDevice(dev_path,fs))==-1)return -1;

	/* ޥȴؿƤӽФ */
	if((root_blk=fs_info[fs]->mount(mount_din))==-1)goto ERR;

	/* 롼Ȳۥǥ쥯ȥꡣ */
	rootEnt.din=mount_din;
	rootEnt.block=root_blk;

	return 0;

ERR:
	fs_info[mount_fs[MOUNT_DEV].fs_num]->close(MOUNT_DEV,mount_din);

	return -1;
}


/*
 * GLOBAL
 * ޥȥڡХååץǥХ
 * parameters : device name
 * return : device inode or error number
 */
int mountPageBackupDev(const char *dev_path)
{
	enum{PAGE_BACKUP_FS=MAX_REGIST_FS};


	return mountDevice(dev_path,PAGE_BACKUP_FS);
}


/*
 * GLOBAL
 * Mount device file
 * parameters : file system register number
 * return : 0 or error=-1
 */
int mount_dev_fs(int fs)
{
	if(mount_fs[MOUNT_DEV].ref_count!=0)return -1;
	mount_fs[MOUNT_DEV].ref_count=1;
	mount_fs[MOUNT_DEV].fs_num=fs;
	devEnt->din=MOUNT_DEV;

	return 0;
}

/*************************************************************************************************/
void test_fs()
{
	sys_mkdir("/home",0);
}
/**************************************************************************************************/

