/*
 * unistd.c
 *
 * Copyright 2003, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include<sys/types.h>
#include<stddef.h>
#include<callgate.h>
#include<system.h>
#include<share/syscall.h>
#include<errno.h>
#include<string.h>
#include<unistd.h>


int chdir(const char *path)
{
	int rest;


	if((rest=syscall2(SYS_CHDIR,(int)path))<0)
	{
		errno=-rest;
		return -1;
	}

	return 0;
}


/*----------------------------------------------------------------------------------
 *
 * exec
 *
 *----------------------------------------------------------------------------------*/

extern char **environ;


/*
 * PATH環境変数文字列を取得する。
 */
static inline const char *getPathEnv()
{
	char **env=environ;
	char *pathstr="PATH=";
	int i;
	
	
	for(;env!=NULL;++env)
		for(i=0;(*env)[i]==pathstr[i];)
			if(pathstr[++i]=='\0')return &(*env)[i];
	
	return ":/ bin:/usr/bin";
}


/*
 * PATH環境変数文字列をコピーする。
 * return : pathの次の文字のポインタ(':' or '\0')
 */
static inline const char *cpyPathEnv(char *buf,const char *env)
{
	int i;
	
	
	for(i=0;env[i]!=':';++i)
		if((buf[i]=env[i])=='\0')return &env[i];
	buf[i]='\0';
	
	return &env[i];
}


int execve(const char *path, char *const argv[], char *const envp[])
{
	errno=syscall4(SYS_EXEC,(int)path,(int)argv,(int)envp);
	return -1;
}


int execl(const char *path, const char *arg0, ...)
{
	return execve(path,(char**)&arg0,environ);
}


int execle(const char *path, const char *arg0, ...)
{
	char **env;


	env=(char**)&arg0;
	while(*env++!=NULL);
	return execve(path,(char**)&arg0,env);
}


int execv(const char *path, char *const argv[])
{
	return execve(path,argv,environ);
}


int execlp(const char *file, const char *arg0, ...)
{
	const char *epath;
	char buf[_PC_PATH_MAX];
	
	
	if(strchr(file,'/')==NULL)
	{
		/* path環境変数を探す。 */
		epath=getPathEnv();
		
		while(*epath!='0')
		{
			epath=cpyPathEnv(buf,epath);
			execve(strcat(buf,file),(char**)&arg0,environ);
			if(errno!=ENOENT)break;
			if(*epath==':')++epath;
		}
		
		return -1;
	}
	else return execve(file,(char**)&arg0,environ);
}


int execvp(const char *file, char *const argv[])
{
	const char *epath;
	char buf[_PC_PATH_MAX];
	
	
	if(strchr(file,'/')==NULL)
	{
		/* path環境変数を探す。 */
		epath=getPathEnv();
		
		while(*epath!='0')
		{
			epath=cpyPathEnv(buf,epath);
			execve(strcat(buf,file),argv,environ);
			if(errno!=ENOENT)break;
			if(*epath==':')++epath;
		}
		
		return -1;
	}
	else return execve(file,argv,environ);
}


/*-----------------------------------------------------------------------------------*/


void _exit(int state)
{
	syscall2(SYS_EXIT,(int)state);
}


pid_t fork(void)
{
	return syscall1(SYS_FORK);
}


char *getcwd(char *buf, size_t size)
{
	int rest;


	if((rest=syscall3(SYS_GETCWD,(int)buf,size))<0)
	{
		errno=-rest;
		return NULL;
	}

	return buf;
}


off_t lseek(int fildes,off_t offset,int whence)
{
	int rest;


	if((rest=syscall4(SYS_LSEEK,fildes,offset,whence))<0)
	{
		errno=-rest;
		return -1;
	}

	return rest;
}


ssize_t read(int fildes,void *buf,size_t nbyte)
{
	int rest;


	if((rest=syscall4(SYS_READ,fildes,(int)buf,(int)nbyte))<0)
	{
		errno=-rest;
		return -1;
	}

	return rest;
}


int rmdir(const char *path)
{
	int rest;


	if((rest=syscall2(SYS_RMDIR,(int)path))<0)
	{
		errno=-rest;
		return -1;
	}

	return 0;
}


ssize_t write(int fildes,const void *buf,size_t nbyte)
{
	int rest;


	if((rest=syscall4(SYS_WRITE,fildes,(int)buf,(int)nbyte))<0)
	{
		errno=-rest;
		return -1;
	}

	return rest;
}


int close(int fildes)
{
	int rest;


	if((rest=syscall2(SYS_CLOSE,fildes))<0)
	{
		errno=-rest;
		return -1;
	}

	return 0;
}


int brk(void *end_segment)
{
	static uint allocLast=0;
	uint alloc_last;


	if((alloc_last=ROUNDUP((uint)end_segment,PAGE_SIZE)-1)>allocLast)
	{
		if(syscall2(SYS_BRK,alloc_last)!=0)return -1;
		allocLast=alloc_last;
	}

	return 0;
}


void *sbrk(ptrdiff_t increment)
{
	void *begin;


	begin=(void*)(sys_info->lastAddr+1);

	if(increment>0)
	{
		if(brk((void*)(sys_info->lastAddr+increment))!=0)return NULL;
		sys_info->lastAddr+=increment;
	}

	return begin;
}


pid_t setsid(void)
{
	int rest;


	if((rest=syscall1(SYS_SETGID))<0)
	{
		errno=-rest;
		return -1;
	}
	else return rest;
};


void sync()
{
	syscall1(SYS_SYNC);
}


int unlink(const char *path)
{
	int rest;


	if((rest=syscall2(SYS_UNLINK,(int)path))<0)
	{
		errno=-rest;
		return -1;
	}
	else return 0;
}


int reboot(int flag)
{
	int rest;


	if((rest=syscall2(SYS_REBOOT,flag))<0)
	{
		errno=-rest;
		return -1;
	}
	else return 0;
}

