/*
 * time.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include"types.h"
#include"lib.h"
#include"errno.h"
#include"interrupt.h"
#include"proc.h"
#include"sp.h"
#include"mp.h"
#include"lock.h"
#include"mm.h"
#include"signal.h"
#include"time.h"


enum{
	MAX_TIMER=4,		/* ݻޡ */

	/* Timer flags. */
	TYP_INTERVAL=	1<<0,	/* Interval timer. */
	TYP_REAL=		1<<1,	/* Realclock timer. */
	TIMER_WAKE=		1<<2,	/* Process wake timer. */
	TIMER_ALARM=	1<<3,	/* Alarm timer. */
};


typedef struct TIMER_LINK{
	struct TIMER_LINK *next;
	struct TIMER_LINK *prev;
	int count;					/* Number of used real timer. */
	int lock_gate;				/* ԥåȡ */
}TIMER_LINK;


int clock_1micro;		/* 1micros */
int clock_1m;			/* 1ms */


/*
 * PUBLIC
 * TIMER_LINK¤Υɥ쥹PROCɥ쥹롣
 */
static inline PROC *timerLinkToProc(TIMER_LINK *p)
{
	return (PROC*)((uint)p-(uint)&((PROC*)0)->wait_next);
}


/***************************************************************************************************
 *
 * 8254 interval timer
 *
 ***************************************************************************************************/

static uint tasking_count=				/* 󥰥ץåѥޡ */
	PIT_HZ/(1000/TASK_TIME);
static TIMER_LINK intervalTimer=		/* Interval timer link queue */
	{&intervalTimer,&intervalTimer,0,0};
static int interval_current_count=0;	/* Interval timer current count */


/*
 * GLOBAL
 * 8254 interval timer handler
 */
static int interval_timer_handler()
{
	PROC *proc;
	TIMER_LINK *p;
	TIMER *timer;

/****************************************************************/
	printk("Interrupt IRQ0\n");
/*****************************************************************/
	enter_spinlock(&intervalTimer.lock_gate);
	{
		/* ॢȤΥץ褵 */
		for(p=intervalTimer.next;;p=p->next)
		{
			proc=timerLinkToProc(p);
			timer=proc->timer_struct;
			if(timer->count!=interval_current_count)break;

			timer->count=0;
			p->prev->next=p->next;
			p->next->prev=p->prev;
			add_to_schedule(proc,TASK_DEVICE_WAIT);

			--intervalTimer.count;
		}

		/* No timer */
		if(intervalTimer.count==0)
		{
			if(MFPS_addres==0)
			{
				tasking_count=PIT_HZ/(1000/TASK_TIME);
				init_sp_task();
			}
		}
		else
		{
			/* ƥץΥȤ򸺤餹 */
			for(p=intervalTimer.next;p!=&intervalTimer;p=p->next)
			{
				proc=timerLinkToProc(p);
				timer=proc->timer_struct;
				timer->count-=interval_current_count;
			}

			proc=timerLinkToProc(intervalTimer.prev);
			timer=proc->timer_struct;
			if(timer->count>0x10000)interval_current_count=0x10000;
			else interval_current_count=timer->count;

			if(MFPS_addres==0)
			{
				if((tasking_count-=interval_current_count)==0)
					tasking_count=PIT_HZ/(1000/TASK_TIME);

				if(tasking_count<interval_current_count)interval_current_count=tasking_count;
			}

			/* ޡꤹ */
			outb(PIT_CNR0,interval_current_count&0xff);
			outb(PIT_CNR0,interval_current_count>>8);
		}
	}
	exit_spinlock(&intervalTimer.lock_gate);

	return 1;
}


/*
 * PUBLIC
 * Set interval timer
 * parameters : time(msñ),timer flag
 * rturn : Task switch flag or error number
 */
static int set_interval_timer(int time,int flag)
{
	uint count;
	PROC *proc,*pr;
	TIMER_LINK *p,*q;
	TIMER *timer,*timer2;


	if(intervalTimer.count==MAX_TIMER)return -ENOTIMER;

	time=PIT_HZ*time/1000;				/* åѴ */

	proc=get_current_task();
	p=(TIMER_LINK*)&proc->wait_next;
	timer=proc->timer_struct;

	enter_spinlock(&intervalTimer.lock_gate);
	{
		if(intervalTimer.count==0)
		{
			timer->count=time;
			p->next=p->prev=&intervalTimer;
			intervalTimer.next=intervalTimer.prev=p;

			if(MFPS_addres==0)
			{
				irq_entry[IRQ0]=interval_timer_handler;
				outb(PIT_CTR,0x30);					/* One shot timer(mode 0) */
				if(time>PIT_HZ/(1000/TASK_TIME))
					time=PIT_HZ/(1000/TASK_TIME);
			}

			/* ޡ */
			if(time>0x10000)interval_current_count=0x10000;
			else interval_current_count=time;
			outb(PIT_CNR0,interval_current_count&0xff);
			outb(PIT_CNR0,interval_current_count>>8);
		}
		else
		{
			count=inb(PIT_CNR0);
			count+=inb(PIT_CNR0)<<8;
			timer->count=time+count;

			/* ޡ󥯤³ */
			for(q=intervalTimer.next;q!=&intervalTimer;q=q->next)
			{
				pr=timerLinkToProc(q);
				timer2=pr->timer_struct;
				if(timer2->count>=timer->count)break;
			}
			p->next=q;
			p->prev=q->prev;
			p->prev->next=q->prev=p;

			if(timer->count<interval_current_count)
			{
				/* ޡ */
				interval_current_count=timer->count;
				outb(PIT_CNR0,time&0xff);
				outb(PIT_CNR0,time>>8);
			}
		}
	}
	exit_spinlock(&intervalTimer.lock_gate);

	++intervalTimer.count;
	timer->flag=TYP_INTERVAL|flag;

	return 0;
}


/*
 * PUBLIC
 * Delete interval timer from link
 * parameters : Process number
 */
static void del_interval_timer(PROC *proc)
{
	TIMER_LINK *p;
	TIMER *timer;


	p=(TIMER_LINK*)&proc->wait_next;
	timer=proc->timer_struct;

	enter_spinlock(&intervalTimer.lock_gate);
	{
		if(timer->count!=0)
		{
			p->prev->next=p->next;
			p->next->prev=p->prev;
			p->next=p->prev=p;
			timer->count=0;
		}

		if(--intervalTimer.count==0)
		{
			if(MFPS_addres==0)
			{
				tasking_count=PIT_HZ/(1000/TASK_TIME);
				init_sp_task();
			}
			else outb(PIT_CTR,0x30);		/* ޡȥå */
		}
	}
	exit_spinlock(&intervalTimer.lock_gate);
}


/***************************************************************************************************
 *
 * MC146818 realclock timer
 *
 ***************************************************************************************************/

enum{
	REAL_TIMER_RATE=500,	/* Real-clock timer rate(ms) */
};


static TIMER_LINK realTimer=		/* Real timer link queue */
	{&realTimer,&realTimer,0,0};
static uint utc;					/* current UTC(number of seconds since 00:00:00.1.1.1970). */


/*
 * PRIVATE
 * ʿʿѴ롣
 */
static inline uchar bcd_to_bin(uchar value)
{
	return value-6*(value>>4);
}


/*
 * PUBLIC
 * Set realclock timer
 * parameters : time(msñ),timer flag
 * rturn : 0 or error number
 */
static int set_real_timer(int time,int flag)
{
	PROC *proc;
	TIMER_LINK *tl;
	TIMER *timer;
	int tm;
	TIMER_LINK *p;


	if(realTimer.count==MAX_TIMER)return -ENOTIMER;

	proc=get_current_task();

	/* 󥯤³ */
	tl=(TIMER_LINK*)&proc->wait_next;
	tm=time;
	enter_spinlock(&realTimer.lock_gate);
	{
		for(p=realTimer.next;p!=&realTimer;p=p->next)
		{
			timer=timerLinkToProc(p)->timer_struct;
			if(tm<timer->count)
			{
				timer->count-=tm;
				break;
			}
			else tm-=timer->count;
		}
		tl->prev=p->prev;
		tl->next=p;
		p->prev=tl;
		tl->prev->next=tl;

		timer=proc->timer_struct;
		timer->count=tm;
		timer->flag=TYP_REAL|flag;
	}
	exit_spinlock(&realTimer.lock_gate);

	++realTimer.count;

	return 0;
}


/*
 * PUBLIC
 * Delete realclock timer
 * prameters : process
 */
static void del_real_timer(PROC *proc)
{
	TIMER_LINK *p;
	TIMER *timer;


	p=(TIMER_LINK*)&proc->wait_next;
	timer=proc->timer_struct;

	enter_spinlock(&realTimer.lock_gate);
	{
		if(timer->count!=0)
		{
			p->prev->next=p->next;
			p->next->prev=p->prev;
			p->next=p->prev=p;
		}
	}
	exit_spinlock(&realTimer.lock_gate);

	timer->count=0;
	--realTimer.count;
}


/*
 * GLOBAL
 * MC146818 real timer handler
 * return : ץåʤ
 */
static int real_timer_handler()
{
	static int half=0;
	PROC *proc;
	TIMER_LINK *p;
	TIMER *timer;
	int tm;


	/* Status register C ɤǳߤꥻåȤ롣 */
	read_cmos(CMOS_STRC);

	/* ॢȥץĴ٤ */
	tm=REAL_TIMER_RATE;
	enter_spinlock(&realTimer.lock_gate);
	{
		for(p=realTimer.next;p!=&realTimer;p=p->next)
		{
			timer=timerLinkToProc(p)->timer_struct;
			if((timer->count-tm)<=0)		/* ॢ */
			{
#ifdef DEBUG
				printk("time out\n");
#endif
				p->prev->next=p->next;
				p->next->prev=p->prev;

				proc=timerLinkToProc(p);
				if(timer->flag&TIMER_WAKE)
					add_to_schedule(proc,TASK_DEVICE_WAIT|TASK_SIGNAL_WAIT);
				else if(timer->flag&TIMER_ALARM)
					sendSignal(proc->signal_struct,SIGALRM);

				--realTimer.count;
				tm-=timer->count;
			}
			else break;
		}
	}
	exit_spinlock(&realTimer.lock_gate);

	/* UTC򹹿롣 */
	half^=1;				/* 500msȹ */
	utc+=half;

	return 0;
}


/*
 * GLOBAL
 * get current time from cmos
 * parameters : time structure
 */
static void get_time_from_cmos()
{
	static ushort sumMonthDay[13]=				/* ιס */
		{0,0,31,59,90,120,151,181,212,243,273,304,334};

	struct TIME{
		uchar second;		/* seconds */
		uchar minute;		/* minutes */
		uchar hour;			/* hours */
		uchar day;			/* days */
		uchar month;		/* months */
		ushort year;		/* years */
	}time;
	int days;


	do
	{
		time.second=bcd_to_bin(read_cmos(CMOS_SCD));
		time.minute=bcd_to_bin(read_cmos(CMOS_MNT));
		time.hour=bcd_to_bin(read_cmos(CMOS_HOR));
		time.day=bcd_to_bin(read_cmos(CMOS_DAY));
		time.month=bcd_to_bin(read_cmos(CMOS_MTH));
		time.year=bcd_to_bin(read_cmos(CMOS_YER));
	}while(bcd_to_bin(read_cmos(CMOS_SCD))<time.second);

	time.year+=(time.year<70)?2000:1900;

	/* UTCη׻ */
	days=(time.year-1970)*365;
	days+=(time.year-1-1970)/4;
	days+=sumMonthDay[time.month]+time.day-1;
	if((time.year%4==0)&&(time.month>2))++days;
	utc=((days*24+time.hour)*60+time.minute)*60+time.second;
	utc+=LOCAL_TIME_TOKYO;								/* 륿֤Ĵ */
}


/***************************************************************************************************
 *
 * ޡ
 *
 ***************************************************************************************************/

/*
 * GLOBAL
 * cpuå¬
 * ֤ : å
 */
int count_cpu_clock()
{
	uint cpu_clock;
	int low,high;
	long long r_beg,r_end;
	double time;


 	outb(PIT_CTR,0x30);	/* CLK0,LSB.MSB,⡼0,binary */
	outb(PIT_CNR0,0);	/* LSB */
	outb(PIT_CNR0,0);	/* MSB */

 	r_beg=rdtsc();		/* clock counter */

	for(;;)if(rdtsc()-r_beg>0x400000)break;		/* ٱ */

	r_end=rdtsc();

	low=inb(PIT_CNR0);
	high=inb(PIT_CNR0);

	time=(0xffff-low-(high<<8))/(double)PIT_HZ;
	cpu_clock=(r_end-r_beg)/time;

 	clock_1micro=cpu_clock/1000000+1;		/* 1micros */
 	clock_1m=cpu_clock/1000+1;				/* 1ms */

	return cpu_clock;
}


/*
 * GLOBAL
 * Set timer
 * ա
 *  δؿɬ del_from_schedule() θǻȤȡ⤷˥ॢȤ
 *  ץʤ롣
 * parameters : time(msñ)
 * rturn : Task switch flag or error number
 */
int set_timer(int time)
{
	if(time==0)return 0;

	if(time<REAL_TIMER_RATE)return set_interval_timer(time,TIMER_WAKE);
	else return set_real_timer(time,TIMER_WAKE);
}


/*
 * GLOBAL
 * Delete timer from timer link.
 * parameters : Process number
 */
void del_timer(PROC *proc)
{
	TIMER *timer;


	timer=proc->timer_struct;
	if(timer->flag&TYP_INTERVAL)del_interval_timer(proc);
	if(timer->flag&TYP_REAL)del_real_timer(proc);
}


/*
 * GLOBAL
 * set time struct.
 * return : time struct address or error=NULL
 */
void *setTimerStruct()
{
	TIMER *timer;


	if((timer=kmalloc(sizeof(TIMER)))==NULL)return NULL;
	memset(timer,0,sizeof(TIMER));

	return timer;
}


/*
 * GLOBAL
 * release time struct.
 */
void releaseTimer(TIMER *timer)
{
	if(timer->count!=0)del_timer(get_current_task());
	kfree(timer);
}


/***************************************************************************************************
 *
 * < ƥॳ >
 *
 ***************************************************************************************************/

/*
 * get Coordinated Universal Time(number of seconds since 00:00:00.1.1.1970)
 */
int sys_time(int *tp)
{
	if(tp!=NULL)
	{
		if(checkMem(tp)==-1)return -EFAULT;
		*tp=utc;
	}

	return (int)utc;
}


int sys_sleep(uint seconds)
{
	TIMER *timer;


	set_real_timer(seconds,TIMER_WAKE);
	del_from_schedule(TASK_SIGNAL_WAIT);
	wait_task();
	timer=get_current_task()->timer_struct;

	return timer->count;
}


int sys_alarm(uint seconds)
{
	TIMER *timer;


	if(seconds==0)return 0;

	timer=get_current_task()->timer_struct;
	if((timer->count>0)&&(timer->flag&TIMER_ALARM))
		return 	timer->count;
	set_real_timer(seconds,TIMER_ALARM);

	return 0;
}


/***************************************************************************************************
 *
 * 
 *
 ***************************************************************************************************/

/*
 * GLOBAL
 * Init time
 */
void init_time()
{
	/* ޥץå */
	if(MFPS_addres)
	{
		irq_entry[IRQ0]=interval_timer_handler;
		outb(PIT_CTR,0x30);						/* One shot timer(mode 0). */
		release_irq_mask(0);
	}

	/* Init MC146818 real time clock */
	irq_entry[IRQ8]=real_timer_handler;
	get_time_from_cmos();				/* get current time from cmos. */
	write_cmos(CMOS_STRA,0x2f);			/* time bese 0x010(default)|ޡ졼65536/(20xf)=2HZ(500ms) */
	read_cmos(CMOS_STRC);				/* Status register C ɤǳߤꥻåȤ롣 */
	write_cmos(CMOS_STRB,0x42);			/* 24hour,ߥ */
	release_irq_mask(8);
}

/*************************************************************************************/
void test_time()
{

}
/**************************************************************************************/
