/*-
 * Copyright (c) 1991, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * 2004: modified by minoru murashma.
 */


#include <errno.h>
#include <math.h>
#include "mathimpl.h"


double frexp(double value,int *eptr)
{
	union {
		double v;
		struct{
			unsigned u_mant2 : 32;
			unsigned u_mant1 : 20;
			unsigned   u_exp : 11;
			unsigned  u_sign :  1;
		}s;
	} u;


	if (value)
	{
		u.v = value;
		*eptr = u.s.u_exp - 1022;
		u.s.u_exp = 1022;

		return(u.v);
	}
	else
	{
		*eptr = 0;

		return((double)0);
	}
}


/*
 * ldexp(value, exp): return value * (2 ** exp).
 *
 * Written by Sean Eric Fagan (sef@kithrup.COM)
 * Sun Mar 11 20:27:09 PST 1990
 */
/*
 * We do the conversion in C to let gcc optimize it away, if possible.
 * The "fxch ; fstp" stuff is because value is still on the stack
 * (stupid 8087!).
 */
double ldexp (double value, int exp)
{
	double temp, texp, temp2;


	texp = exp;
#ifdef __GNUC__
#if    __GNUC__ >= 2
	asm ("fscale "
		: "=u" (temp2), "=t" (temp)
		: "0" (texp), "1" (value));
#else
	asm ("fscale ; fxch %%st(1) ; fstp%L1 %1 "
		: "=f" (temp), "=0" (temp2)
		: "0" (texp), "f" (value));
#endif
#else
error unknown asm
#endif

	return (temp);
}


/* SINH(X)
 * RETURN THE HYPERBOLIC SINE OF X
 * DOUBLE PRECISION (VAX D format 56 bits, IEEE DOUBLE 53 BITS)
 * CODED IN C BY K.C. NG, 1/8/85;
 * REVISED BY K.C. NG on 2/8/85, 3/7/85, 3/24/85, 4/16/85.
 *
 * Required system supported functions :
 *	copysign(x,y)
 *	scalb(x,N)
 *
 * Required kernel functions:
 *	expm1(x)	...return exp(x)-1
 *
 * Method :
 *	1. reduce x to non-negative by sinh(-x) = - sinh(x).
 *	2.
 *
 *	                                      expm1(x) + expm1(x)/(expm1(x)+1)
 *	    0 <= x <= lnovfl     : sinh(x) := --------------------------------
 *			       		                      2
 *     lnovfl <= x <= lnovfl+ln2 : sinh(x) := expm1(x)/2 (avoid overflow)
 * lnovfl+ln2 <  x <  INF        :  overflow to INF
 *
 *
 * Special cases:
 *	sinh(x) is x if x is +INF, -INF, or NaN.
 *	only sinh(0)=0 is exact for finite argument.
 *
 * Accuracy:
 *	sinh(x) returns the exact hyperbolic sine of x nearly rounded. In
 *	a test run with 1,024,000 random arguments on a VAX, the maximum
 *	observed error was 1.93 ulps (units in the last place).
 *
 * Constants:
 * The hexadecimal values are the intended ones for the following constants.
 * The decimal values may be used, provided that the compiler will convert
 * from decimal to binary accurately enough to produce the hexadecimal values
 * shown.
 */
vc(mln2hi, 8.8029691931113054792E1   ,0f33,43b0,2bdb,c7e2,   7, .B00F33C7E22BDB)
vc(mln2lo,-4.9650192275318476525E-16 ,1b60,a70f,582a,279e, -50,-.8F1B60279E582A)
vc(lnovfl, 8.8029691931113053016E1   ,0f33,43b0,2bda,c7e2,   7, .B00F33C7E22BDA)

ic(mln2hi, 7.0978271289338397310E2,    10, 1.62E42FEFA39EF)
ic(mln2lo, 2.3747039373786107478E-14, -45, 1.ABC9E3B39803F)
ic(lnovfl, 7.0978271289338397310E2,     9, 1.62E42FEFA39EF)


static int max = 1023;


double sinh(double x)
{
	static const double  one=1.0, half=1.0/2.0 ;
	double t, sign;


	if(x!=x) return(x);	/* x is NaN */

	sign=copysign(one,x);
	x=copysign(x,one);
	if(x<lnovfl)
	{
		t=expm1(x);
		return(copysign((t+t/(one+t))*half,sign));
	}

	else if(x <= lnovfl+0.7)
		/* subtract x by ln(2^(max+1)) and return 2^max*exp(x) to avoid unnecessary overflow */
	    return(copysign(scalb(one+expm1((x-mln2hi)-mln2lo),max),sign));

	else  /* sinh(+-INF) = +-INF, sinh(+-big no.) overflow to +-INF */
	    return( expm1(x)*sign );
}


/* COSH(X)
 * RETURN THE HYPERBOLIC COSINE OF X
 * DOUBLE PRECISION (VAX D format 56 bits, IEEE DOUBLE 53 BITS)
 * CODED IN C BY K.C. NG, 1/8/85;
 * REVISED BY K.C. NG on 2/8/85, 2/23/85, 3/7/85, 3/29/85, 4/16/85.
 *
 * Required system supported functions :
 *	copysign(x,y)
 *	scalb(x,N)
 *
 * Required kernel function:
 *	exp(x)
 *	exp__E(x,c)	...return exp(x+c)-1-x for |x|<0.3465
 *
 * Method :
 *	1. Replace x by |x|.
 *	2.
 *		                                        [ exp(x) - 1 ]^2
 *	    0        <= x <= 0.3465  :  cosh(x) := 1 + -------------------
 *			       			           2*exp(x)
 *
 *		                                   exp(x) +  1/exp(x)
 *	    0.3465   <= x <= 22      :  cosh(x) := -------------------
 *			       			           2
 *	    22       <= x <= lnovfl  :  cosh(x) := exp(x)/2
 *	    lnovfl   <= x <= lnovfl+log(2)
 *				     :  cosh(x) := exp(x)/2 (avoid overflow)
 *	    log(2)+lnovfl <  x <  INF:  overflow to INF
 *
 *	Note: .3465 is a number near one half of ln2.
 *
 * Special cases:
 *	cosh(x) is x if x is +INF, -INF, or NaN.
 *	only cosh(0)=1 is exact for finite x.
 *
 * Accuracy:
 *	cosh(x) returns the exact hyperbolic cosine of x nearly rounded.
 *	In a test run with 768,000 random arguments on a VAX, the maximum
 *	observed error was 1.23 ulps (units in the last place).
 *
 * Constants:
 * The hexadecimal values are the intended ones for the following constants.
 * The decimal values may be used, provided that the compiler will convert
 * from decimal to binary accurately enough to produce the hexadecimal values
 * shown.
 */
double cosh(double x)
{
	static const double half=1.0/2.0,one=1.0, small=1.0E-18; /* fl(1+small)==1 */
	double t;


	if(x!=x) return(x);	/* x is NaN */

	if((x=copysign(x,one)) <= 22)
	{
	    if(x<0.3465)
		if(x<small) return(one+x);
		else {t=x+__exp__E(x,0.0);x=t+t; return(one+t*t/(2.0+x)); }

	    else /* for x lies in [0.3465,22] */
	        { t=exp(x); return((t+one/t)*half); }
	}
	if( lnovfl <= x && x <= (lnovfl+0.7))
        /* for x lies in [lnovfl, lnovfl+ln2], decrease x by ln(2^(max+1)) and return 2^max*exp(x) to avoid unnecessary overflow. */
	    return(scalb(exp((x-mln2hi)-mln2lo), max));
	else return(exp(x)*half);	/* for large x,  cosh(x)=exp(x)/2 */
}


/* TANH(X)
 * RETURN THE HYPERBOLIC TANGENT OF X
 * DOUBLE PRECISION (VAX D FORMAT 56 BITS, IEEE DOUBLE 53 BITS)
 * CODED IN C BY K.C. NG, 1/8/85;
 * REVISED BY K.C. NG on 2/8/85, 2/11/85, 3/7/85, 3/24/85.
 *
 * Required system supported functions :
 *	copysign(x,y)
 *	finite(x)
 *
 * Required kernel function:
 *	expm1(x)	...exp(x)-1
 *
 * Method :
 *	1. reduce x to non-negative by tanh(-x) = - tanh(x).
 *	2.
 *	    0      <  x <=  1.e-10 :  tanh(x) := x
 *					          -expm1(-2x)
 *	    1.e-10 <  x <=  1      :  tanh(x) := --------------
 *					         expm1(-2x) + 2
 *							  2
 *	    1      <= x <=  22.0   :  tanh(x) := 1 -  ---------------
 *						      expm1(2x) + 2
 *	    22.0   <  x <= INF     :  tanh(x) := 1.
 *
 *	Note: 22 was chosen so that fl(1.0+2/(expm1(2*22)+2)) == 1.
 *
 * Special cases:
 *	tanh(NaN) is NaN;
 *	only tanh(0)=0 is exact for finite argument.
 *
 * Accuracy:
 *	tanh(x) returns the exact hyperbolic tangent of x nealy rounded.
 *	In a test run with 1,024,000 random arguments on a VAX, the maximum
 *	observed error was 2.22 ulps (units in the last place).
 */

double tanh(double x)
{
	static double one=1.0, two=2.0, small = 1.0e-10/*, big = 1.0e10*/;
	double expm1(), t, copysign(), sign;
	int finite();


	if(x!=x) return(x);	/* x is NaN */

	sign=copysign(one,x);
	x=copysign(x,one);
	if(x < 22.0)
	{
	    if( x > one )return(copysign(one-two/(expm1(x+x)+two),sign));
	    else if ( x > small )
		{
			t= -expm1(-(x+x));
			return(copysign(t/(two-t),sign));
		}
	    else		/* raise the INEXACT flag for non-zero x */
		{
			/*big+x;*/
			return(copysign(x,sign));
		}
	}
	else if(finite(x))return (sign+1.0E-37); /* raise the INEXACT flag */
	else return(sign);	/* x is +- INF */
}


/*----------------------------------------- pow -----------------------------------------------------------*/

/* POW(X,Y)
 * RETURN X**Y
 * DOUBLE PRECISION (VAX D format 56 bits, IEEE DOUBLE 53 BITS)
 * CODED IN C BY K.C. NG, 1/8/85;
 * REVISED BY K.C. NG on 7/10/85.
 * KERNEL pow_P() REPLACED BY P. McILROY 7/22/92.
 * Required system supported functions:
 *      scalb(x,n)
 *      logb(x)
 *	copysign(x,y)
 *	finite(x)
 *	drem(x,y)
 *
 * Required kernel functions:
 *	exp__D(a,c)			exp(a + c) for |a| << |c|
 *	struct d_double dlog(x)		r.a + r.b, |r.b| < |r.a|
 *
 * Method
 *	1. Compute and return log(x) in three pieces:
 *		log(x) = n*ln2 + hi + lo,
 *	   where n is an integer.
 *	2. Perform y*log(x) by simulating muti-precision arithmetic and
 *	   return the answer in three pieces:
 *		y*log(x) = m*ln2 + hi + lo,
 *	   where m is an integer.
 *	3. Return x**y = exp(y*log(x))
 *		= 2^m * ( exp(hi+lo) ).
 *
 * Special cases:
 *	(anything) ** 0  is 1 ;
 *	(anything) ** 1  is itself;
 *	(anything) ** NaN is NaN;
 *	NaN ** (anything except 0) is NaN;
 *	+(anything > 1) ** +INF is +INF;
 *	-(anything > 1) ** +INF is NaN;
 *	+-(anything > 1) ** -INF is +0;
 *	+-(anything < 1) ** +INF is +0;
 *	+(anything < 1) ** -INF is +INF;
 *	-(anything < 1) ** -INF is NaN;
 *	+-1 ** +-INF is NaN and signal INVALID;
 *	+0 ** +(anything except 0, NaN)  is +0;
 *	-0 ** +(anything except 0, NaN, odd integer)  is +0;
 *	+0 ** -(anything except 0, NaN)  is +INF and signal DIV-BY-ZERO;
 *	-0 ** -(anything except 0, NaN, odd integer)  is +INF with signal;
 *	-0 ** (odd integer) = -( +0 ** (odd integer) );
 *	+INF ** +(anything except 0,NaN) is +INF;
 *	+INF ** -(anything except 0,NaN) is +0;
 *	-INF ** (odd integer) = -( +INF ** (odd integer) );
 *	-INF ** (even integer) = ( +INF ** (even integer) );
 *	-INF ** -(anything except integer,NaN) is NaN with signal;
 *	-(x=anything) ** (k=integer) is (-1)**k * (x ** k);
 *	-(anything except 0) ** (non-integer) is NaN with signal;
 *
 * Accuracy:
 *	pow(x,y) returns x**y nearly rounded. In particular, on a SUN, a VAX,
 *	and a Zilog Z8000,
 *			pow(integer,integer)
 *	always returns the correct integer provided it is representable.
 *	In a test run with 100,000 random arguments with 0 < x, y < 20.0
 *	on a VAX, the maximum observed error was 1.79 ulps (units in the
 *	last place).
 *
 * Constants :
 * The hexadecimal values are the intended ones for the following constants.
 * The decimal values may be used, provided that the compiler will convert
 * from decimal to binary accurately enough to produce the hexadecimal values
 * shown.
 */


#define _IEEE		1
#define endian		(((*(int *) &one)) ? 1 : 0)
#define TRUNC(x) 	*(((int *) &x)+endian) &= 0xf8000000
#define infnan(x)	0.0


const static double zero=0.0, one=1.0, two=2.0, negone= -1.0;


/* kernel function for x >= 0 */
static double pow_P(double x, double y)
{
	struct Double s, t, __log__D();
	double  __exp__D();
	volatile double huge = 1e300, tiny = 1e-300;


	if (x == zero)
	{
		if (y > zero)return (zero);
		else if (_IEEE)return (huge*huge);
		else return (infnan(ERANGE));
	}
	if (x == one)return (one);
	if (!finite(x))
	{
		if (y < zero)return (zero);
		else if (_IEEE)return (huge*huge);
		else return (infnan(ERANGE));
	}
	if (y >= 7e18)		/* infinity */
	{
		if (x < 1)return(tiny*tiny);
		else if (_IEEE)return (huge*huge);
		else return (infnan(ERANGE));
	}

	/* Return exp(y*log(x)), using simulated extended */
	/* precision for the log and the multiply.	  */

	s = __log__D(x);
	t.a = y;
	TRUNC(t.a);
	t.b = y - t.a;
	t.b = s.b*y + t.b*s.a;
	t.a *= s.a;
	s.a = t.a + t.b;
	s.b = (t.a - s.a) + t.b;

	return (__exp__D(s.a, s.b));
}

double pow(double x,double y)
{
	double t;


	if (y==zero)return (one);
	else if(y==one||(_IEEE && x != x))return (x);		/* if x is NaN or y=1 */
	else if (_IEEE && y!=y)return (y);		/* if y is NaN */
	else if (!finite(y))		/* if y is INF */
	{
		if ((t=fabs(x))==one)return (y - y);	/* +-1 ** +-INF is NaN */
		else if (t>one)return ((y<0)? zero : ((x<zero)? y-y : y));
		else return ((y>0)? zero : ((x<0)? y-y : -y));
	}
	else if (y==two)return (x*x);
	else if (y==negone)return (one/x);
    /* x > 0, x == +0 */
	else if (copysign(one, x) == one)return (pow_P(x, y));
    /* sign(x)= -1 */
	/* if y is an even integer */
	else if ( (t=drem(y,two)) == zero)return (pow_P(-x, y));
	/* if y is an odd integer */
	else if (copysign(t,one) == one)return (-pow_P(-x, y));
	/* Henceforth y is not an integer */
	else if (x==zero)return ((y>zero)? -x : one/(-x));	/* x is -0 */
	else if (_IEEE)return (zero/zero);
	else return (infnan(EDOM));
}

/*--------------------------------------------- pow end ---------------------------------------------------*/
