#!/usr/bin/env bash

#*****************************************************************************
# xtbseq.sh
# 同一ブランド連続購入シーケンスの作成
# Y.Hamuro
# 1.0b 2004/03/08
# 1.0  -u の追加,トラップ対応 2004/11/26
#*****************************************************************************

#エラーメッセージ
function help {
cat >/dev/stderr <<EOF
----------------------
xtbseq.sh version 1.0
----------------------
概要) 同一ブランド連続購入シーケンスを作成する。

書式) xtbseq.sh -c 顧客項目名 -d 日付項目名 -b ブランド項目名
                -B 連続購入回数 [-u]
                [-A 選択する連続購入前の来店回数(default=0)]
                [-C 選択する連続購入後の来店回数(default=0)]
                [-iファイル] [-o出力ファイル名] [-T ワークディレクトリ名]
                [-V] [-h]
      -u : ブランドの連続購入をユニークな連続購入のみに限定する
      -V : コマンドの完了メッセージを表示させる。
      -h : ヘルプの表示

備考) 入力ファイルの項目名として"targetBrand","seqNo","term"の3つは利用不可。

例1) xtbseq.sh -c 顧客 -d 日付 -b ブランド -B 3 -i dat.xt -o rsl.xt
例2) xtbseq.sh -c 顧客 -d 日付 -b ブランド -B 3 -u -i dat.xt -o rsl.xt
例3) xtbseq.sh -c 顧客 -d 日付 -b ブランド -B 3 -A 5 -C 3 -i dat.xt -o rsl.xt
EOF
exit 1
}

#完了メッセージ用にコマンドラインを保存
cmdLine="$0 $*"

#デフォルトでは、コマンドのメッセージはOFF
export mssQuiet=1

#シグナルによる終了
function endByInt {
  rm -f $TD-xx*
  echo "#ERROR# $$" \"$cmdLine\" \""end by signal(ctr^C)"\" >/dev/stderr
  exit 1
}


brandUniq="0"
#パラメータのチェック＆セット
  while getopts ":c:d:b:B:A:C:i:o:uVh" opt; do
    case $opt in
      c  ) CUST=$OPTARG ;;
      d  ) DATE=$OPTARG ;;
      b  ) BRAND=$OPTARG ;;
      B  ) minSeqCnt=$OPTARG ;;
      A  ) aVisit=$OPTARG ;;
      C  ) cVisit=$OPTARG ;;
      i  ) input=$OPTARG ;;
      o  ) output=$OPTARG ;;
      T  ) tmpD=$OPTARG ;;
      u  ) brandUniq="1" ;;
      V  ) mssQuiet=0 ;;
      h  ) help ;;
      \? ) help ;;
    esac
  done
  shift $(($OPTIND -1 ))

  if [ "$tmpD" = "" ]; then
    tmpD=/tmp
  fi
  if [ "$input" = "" ]; then
    input=/dev/stdin
  fi
  if [ "$output" = "" ]; then
    output=/dev/stdout
  fi
  if [ "$CUST" = "" ]; then
    echo "#ERROR# $$ \"$0\" \"option -c is mandatory\"" >/dev/stderr
    exit 1
  fi
  if [ "$DATE" = "" ]; then
    echo "#ERROR# $$ \"$0\" \"option -d is mandatory\"" >/dev/stderr
    exit 1
  fi
  if [ "$BRAND" = "" ]; then
    echo "#ERROR# $$ \"$0\" \"option -b is mandatory\"" >/dev/stderr
    exit 1
  fi
  if [ "$minSeqCnt" = "" ]; then
    echo "#ERROR# $$ \"$0\" \"option -B is mandatory\"" >/dev/stderr
    exit 1
  fi
  if [ "$aVisit" = "" ]; then
    aVisit=0
  fi
  if [ "$cVisit" = "" ]; then
    cVisit=0
  fi

#ワークファイル名
TD=$tmpD/mss-xtbseq1-$$

#トラップ発動
trap endByInt INT QUIT TERM HUP

# ==============================================================================
# 手順０) このスクリプトで利用する項目以外の項目名リスト作成
# ------------------------------------------------------------------------------
# ex)
# 入力データの項目 : 顧客,日付,ブランド,金額,数量
#     利用する項目 : 顧客,日付,ブランド
# → OTHERFIELD=",金額,数量"
# ==============================================================================
echo "#MSG# $$ \"$0\" \"preprocessing\"" >/dev/stderr
mkfifo $TD-xxfifo

echo "$CUST,$DATE,$BRAND" |
tr ',' '\n' |
gawk '{print $1,NR}' |
sort >$TD-xxfld2

xtbest -R 0 <$TD-xxfifo |
xt2txt -F |
tr ' ' '\n' |
gawk '{print $1,NR}' |
sort >$TD-xxfld1 &

function setOtherField {
  xtcommon -t -r -k 1 -m $TD-xxfld2 -i $TD-xxfld1 -o $TD-xxfld3

  if [ -s  $TD-xxfld3 ] ; then
    xtsort -t -k 2%n -i $TD-xxfld3 |
    xtcut -t -f 1 |
    tr '\n' ',' |
    sed 's/,$//' >$TD-xxfld4
    OTHERFIELD=",`cat $TD-xxfld4`"
  else
    OTHERFIELD=""
  fi
}

# ==============================================================================
# 手順１) 日付連番(custDateSeqNo)の作成
# ------------------------------------------------------------------------------
# 作成されるデータイメージ
#  顧客,日付,ブランド,ブランド連番
# a 20030101 A 1
# a 20030104 A 2
# a 20030108 A 3
# a 20030112 B 4
# a 20030113 A 5 +
# a 20030113 B 5 +->同じ日付は同じ番号
# a 20030115 B 6
# a 20030118 B 7
# ==============================================================================
echo "#MSG# $$ \"$0\" \"creating date sequence number\"" >/dev/stderr

tee $TD-xxfifo <$input |

# 同一日同一ブランド複数購入は考慮に入れない(ユニークにする)
xtuniq -k $CUST,$DATE,$BRAND |

# 顧客日付で連番をふる
xtnumber -k $CUST,$DATE -B -a '##custDateSeqNo##' -o $TD-xxds

# ==============================================================================
# 手順２) ブランド連続購入範囲マスターの作成
# ------------------------------------------------------------------------------
#   前購入(aVisit回) → 同一ブランドminSeqCnt回以上連続購入 → 後購入(cVisit回)
#
# 同一ブランド連続購入とは
#   同一日に複数のブランドを購入していても連続とカウントする
#
# ex1)  AA(AB)(AC)A →Aを５回連続購入している(括弧内は同一日の購入)
#
# ------------------------------------------------------------------------------
# 作成されるデータイメージ
# 項目:1.顧客 2.開始日付連番 3.終了日付連番 4.ブランド 5.seqNo 6.term
# a -4 0 A 2 a   -> 前購入は-4から0回目まで(前購入はなかったということ)
# a  1 3 A 2 b   -> aさんの1から5回目来店にAブランドの連続購入があった
# a  4 8 A 2 c   -> 後購入は4から8回目まで
# ==============================================================================
echo "#MSG# $$ \"$0\" \"creating range master file\"" >/dev/stderr

# 顧客ブランド別に連続する日付(custDateSeqNo)に同じ番号をつける
# (custDateSeqNoがとんでるということは、同一ブランドの連続購入でない)
xtcal -k $CUST,$BRAND -s $DATE%r -c \
  'if(keyLine()==1,
     1,
     if((prvField($##custDateSeqNo##)-1)==$##custDateSeqNo##,
       prvResult(),
       prvResult()+1))' -a seqNo -i $TD-xxds |

# 同一ブランド連続購入には同じseqNoが付いているので、その行数が指定のminSeqCnt
# 以上のもののみ選択する(同一ブランド連続購入の選択)
xtsel -k $CUST,$BRAND,seqNo -c 'keyCnt()>='$minSeqCnt -o $TD-xxc

#上記で求めた同一ブランド連続購入の最初と最後のcustDateSeqNoを求める
xtbest -k $CUST,$BRAND,seqNo -s '##custDateSeqNo##%n'  -R 1 -i $TD-xxc -o $TD-xxmin
xtbest -k $CUST,$BRAND,seqNo -s '##custDateSeqNo##%nr' -R 1 -i $TD-xxc -o $TD-xxmax

# 同一ブランド連続購入の前aVisit回の範囲マスターを作成(term="a")
xtcal  -c '$##custDateSeqNo##-1'           -a to -i $TD-xxmin |
xtcal  -c '$##custDateSeqNo##-'$aVisit -a from |
xtcut  -f $CUST,from,to,$BRAND:targetBrand,seqNo |
xtsetchr -v a -a term -o $TD-xxd

# 同一ブランド連続購入の範囲マスターを作成(term="b")
xtjoin -k $CUST,$BRAND,seqNo -m $TD-xxmax -f '##custDateSeqNo##':to -i $TD-xxmin |
xtcut  -f $CUST,'##custDateSeqNo##':from,to,$BRAND:targetBrand,seqNo |
xtsetchr -v b -a term -o $TD-xxe

# 同一ブランド連続購入の後cVisit回の範囲マスターを作成(term="c")
xtcal  -c '$##custDateSeqNo##+1'           -a from -i $TD-xxmax |
xtcal  -c '$##custDateSeqNo##+'$cVisit -a to |
xtcut  -f $CUST,from,to,$BRAND:targetBrand,seqNo |
xtsetchr -v c -a term -o $TD-xxf

#上記３つのファイルを併合し、範囲マスターとする。
xtcat -i $TD-xxd,$TD-xxe,$TD-xxf |
xtsort -k $CUST,from,to,targetBrand,seqNo,term -o $TD-xxm

# ==============================================================================
# 手順３) 上記手順１と２で作成されたファイルを使い、
#         連続ブランドシーケンスデータを作成する
# ------------------------------------------------------------------------------
#  顧客,日付,ブランド,日付連番
# a 20030101 A 1
# a 20030104 A 2
# a 20030108 A 3
# a 20030112 B 4
# a 20030113 A 5
# a 20030113 B 5
# a 20030115 B 6
# a 20030118 B 7
#
# 上記のファイルの日付連番と下記のファイルの連番の範囲を突き合わせ、
# 範囲内に入っていれば(closed range)下記のファイルの4-6をjoinする。
# ブランド連番が二つの範囲に属する時は、２行出力される。
#
# 項目:1.顧客 2.開始日付連番 3.終了日付連番 4.ブランド 5.seqNo 6.term
# a -4 0 A 2 a
# a  1 3 A 2 b
# a  4 8 A 2 c
#
# 結果は以下の通り。あとは適当に並べ換える
#  
# a 20030101 A 1 A 2 b
# a 20030104 A 2 A 2 b
# a 20030108 A 3 A 2 b
# a 20030112 B 4 A 2 c
# a 20030113 A 5 A 2 c
# a 20030113 B 5 A 2 c
# a 20030115 B 6 A 2 c
# a 20030118 B 7 A 2 c
# ==============================================================================
echo "#MSG# $$ \"$0\" \"creating brand sequence pattern\"" >/dev/stderr

# 本スクリプトで利用していない項目名のセット
setOtherField

if [ "$brandUniq" = "0" ]; then

# ブランド連番を範囲で突き合わせスイッチ番号を結合する
xtnrjoin -k $CUST -v '##custDateSeqNo##%n' -m $TD-xxm -R from,to%e -f targetBrand,seqNo,term -i $TD-xxds |

# 必要項目を抜きだす
xtcut -f $CUST,targetBrand,seqNo,term,$DATE,$BRAND$OTHERFIELD |

# 見やすさのためのソート
xtsort -k $CUST,targetBrand,seqNo,term,$DATE >$output

else

# ブランド連番を範囲で突き合わせスイッチ番号を結合する
xtnrjoin -k $CUST -v '##custDateSeqNo##%n' -m $TD-xxm -R from,to%e -f targetBrand,seqNo,term -i $TD-xxds |

# 必要項目を抜きだす
xtcut -f $CUST,targetBrand,seqNo,term,$DATE,$BRAND$OTHERFIELD -o $TD-xxrsl

#ユニークでない連続購入マスターの作成
xtselstr -f term -v b -i $TD-xxrsl |
xtcut -f custID,targetBrand,seqNo,$BRAND |
xtuniq -k custID,targetBrand,seqNo,$BRAND |
xtcount -k custID,targetBrand,seqNo -a '##cnt##' |
xtsel -c '$##cnt##>1' -o $TD-xxbunq

xtcommon -k custID,targetBrand,seqNo -m $TD-xxbunq -r -i $TD-xxrsl |

# 見やすさのためのソート
xtsort -k $CUST,targetBrand,seqNo,term,$DATE >$output
fi

#ワークファイル削除
rm -f $TD-xx*

#完了メッセージ表示
echo "#END# $$" \"$cmdLine\" >/dev/stderr
exit 0
# ==============================================================================
