/* Copyright 2014 Akira Ohta (akohta001@gmail.com)
    This file is part of ntch.

    The ntch is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    The ntch is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ntch.  If not, see <http://www.gnu.org/licenses/>.
    
*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <wchar.h>

#include "env.h"
#include "nt_string.h"
#include "utils/nt_std_t.h"
#include "utils/text.h"
#include "_2ch/model_2ch.h"
#include "_2ch/search_2ch.h"
#include "ui/disp.h"
#include "ui/disp_string.h"
#include "usr/usr_db_t.h"

#define NT_CMD_ERR -1
#define NT_CMD_NONE 0
#define NT_CMD_LIMIT 1

#define LINE_HEIGHT 2

typedef struct tag_history_ctx_t *history_ctx_tp;
typedef struct tag_history_ctx_t {
	int prev_state;
	int cur_cursor;
	int scroll_pos;
	int query_offset;
	int query_limit;
	BOOL update_query;
	nt_enum_handle h_enum_disp_data;
}history_ctx_t;

typedef struct tag_disp_data_t *disp_data_tp;
typedef struct tag_disp_data_t{
	wchar_t *board_name;
	wchar_t *dat_name;
	wchar_t *title;
	int read_count;
	int res_count;
}disp_data_t;

static history_ctx_tp init_context();
static disp_data_tp disp_data_alloc(
		const wchar_t *board_name,  
		const wchar_t *dat_name, int read_count);
static void disp_data_free(void *ptr);
static BOOL set_model_data(nt_2ch_model_handle h_model, 
		nt_enum_handle h_enum_disp_data, 
		nt_link_tp *update_board_list);


static int adjust_scroll_pos(nt_window_tp wp, 
			history_ctx_tp ctxp);
static int parse_cmd1(const char *param, int *limit, int *offset);


static BOOL set_model_data(nt_2ch_model_handle h_model, 
		nt_enum_handle h_enum_disp_data, 
		nt_link_tp *update_board_list)
{
	nt_mutex_handle h_mutex;
	disp_data_tp disp_datap;
	nt_category_handle h_category;
	nt_board_handle h_board;
	nt_thread_handle h_thread;
	wchar_t *title;
	wchar_t *wc;
	nt_map_handle h_map;
	nt_2ch_selected_item_handle h_select;
	
	assert(h_model);
	assert(h_enum_disp_data);
	
	if(update_board_list)
		*update_board_list = NULL;
	
	h_map = NULL;
	
	h_mutex = nt_2ch_model_get_mutex(h_model);
	if(!h_mutex)
		return FALSE;
	if(!nt_mutex_lock(h_mutex)){
		return FALSE;
	}
	nt_enum_reset(h_enum_disp_data);
	while(NULL != (disp_datap = 
			(disp_data_tp)nt_enum_fetch(h_enum_disp_data))){
		if(disp_datap->title)
			continue;
		h_thread = nt_get_thread_by_board_and_dat_name(
					h_model, disp_datap->board_name, disp_datap->dat_name);
		title = NULL;
		if(h_thread){
			title = nt_w_trim(nt_thread_get_title(h_thread));
		}
		if(!title){
			h_board = nt_get_board_by_name(h_model,
					 disp_datap->board_name, &h_category);
			if(!h_board){
				if(h_thread)
					nt_thread_release_ref(h_thread);
				continue;
			}
			
			wc = nt_read_thread_title(
					h_board, disp_datap->dat_name);
			if(!wc){
				if(h_thread){
				/* Not in subject.txt */
					nt_board_release_ref(h_board);
					nt_category_release_ref(h_category);
					nt_thread_release_ref(h_thread);
				}else if(update_board_list){
				/* May contain subject.txt but yet to be downloaded. */
					if(!h_map){
						h_map = nt_map_alloc();
						if(!h_map){
							nt_board_release_ref(h_board);
							nt_category_release_ref(h_category);
							continue;
						}
					}else if(nt_map_find(h_map, disp_datap->board_name)){
						nt_board_release_ref(h_board);
						nt_category_release_ref(h_category);
						continue;
					}
					h_select = nt_2ch_selected_item_alloc();
					if(h_select){
						nt_set_selected_board(h_select, h_category, h_board);
						nt_board_release_ref(h_board);
						nt_category_release_ref(h_category);
						if(!nt_map_add_pair(h_map, disp_datap->board_name, h_select)){
							nt_2ch_selected_item_release_ref(h_select);
						}
					}else{
						nt_board_release_ref(h_board);
						nt_category_release_ref(h_category);
					}
				}else{
					nt_board_release_ref(h_board);
					nt_category_release_ref(h_category);
				}
				continue;
			}
			nt_board_release_ref(h_board);
			nt_category_release_ref(h_category);
			title = nt_w_trim(wc);
			free(wc);
		}
		if(h_thread){
			disp_datap->res_count = nt_thread_get_res_count(h_thread);
			nt_thread_release_ref(h_thread);
		}
		if(disp_datap->res_count < 0)
			disp_datap->res_count = 0;
		if(title){
			disp_datap->title = title;
		}
	}
	
	if(!nt_mutex_unlock(h_mutex)){
		assert(0);
	}
	if(h_map){
		*update_board_list = nt_map_get_values(h_map);
		nt_map_free(h_map, NULL);
	}
	return TRUE;
}


static history_ctx_tp init_context()
{
	history_ctx_tp ctxp;
	
	ctxp = malloc(sizeof(history_ctx_t));
	if(!ctxp)
		return NULL;
	
	ctxp->prev_state = DISP_STATE_HISTORY;
	ctxp->cur_cursor = 0;
	ctxp->scroll_pos = 0;
	ctxp->query_offset = 0;
	ctxp->query_limit = 100;
	ctxp->update_query = FALSE;
	ctxp->h_enum_disp_data = NULL;

	return ctxp;
}

void free_history_ctx(void *ptr)
{
	history_ctx_tp ctxp;
	
	assert(ptr);
	
	ctxp = (history_ctx_tp)ptr;
	if(ctxp->h_enum_disp_data){
		nt_enum_set_free_func(ctxp->h_enum_disp_data, disp_data_free);
		nt_enum_unset(ctxp->h_enum_disp_data);
	}
	free(ctxp);
}


static nt_link_tp set_disp_data(nt_link_tp slinkp)
{
	nt_link_tp rlinkp, wlinkp, dlinkp;
	disp_data_tp datap;
	nt_usr_db_thread_data_tp db_datap;
	
	if(!slinkp)
		return NULL;
	
	rlinkp = NULL;
	wlinkp = slinkp;
	do{
		db_datap = (nt_usr_db_thread_data_tp)wlinkp->data;
		datap = disp_data_alloc(
				db_datap->board_name, 
				db_datap->dat_name,
				db_datap->read_count);
		
		if(datap){
			dlinkp = nt_link_add_data(rlinkp, datap);
			if(!rlinkp)
				rlinkp = dlinkp;
		}
		wlinkp = wlinkp->next;
	}while(wlinkp != slinkp);
	
	return rlinkp;
}

int disp_history(nt_window_tp wp, 
		int prev_state, nt_2ch_model_handle h_model,
		nt_usr_db_handle h_usr_db,
		nt_searched_thread_handle *h_sel_threadp,
		nt_link_tp *update_board_list)
{
	int result_state;
	history_ctx_tp ctxp;
	nt_link_tp linkp, data_linkp;
	disp_data_tp disp_datap;
	int row, num, limit_row, nwrite, len;
	int limit, offset, cmd;
	attr_t attr;
	BOOL b_result;
	const wchar_t *title;
	wchar_t wbuf[256];
	wchar_t wbuf2[32];
	int read_count, res_count;

	
	result_state = DISP_STATE_HISTORY;
	*update_board_list = NULL;
	
	ctxp = (history_ctx_tp)wp->data;
	if(!ctxp){
		ctxp = init_context();
		if(!ctxp)
			return DISP_STATE_ERROR;
		
		wp->data = ctxp;
	}
	
	if(prev_state != DISP_STATE_HISTORY || ctxp->update_query){
		if(prev_state != DISP_STATE_HISTORY)
			ctxp->prev_state = prev_state;
		ctxp->update_query = FALSE;
		linkp = nt_usr_db_query_usr_table(h_usr_db, 
					ctxp->query_offset, 
					ctxp->query_limit);
		if(linkp){
			if(ctxp->h_enum_disp_data){
				nt_enum_set_free_func(ctxp->h_enum_disp_data, disp_data_free);
				nt_enum_unset(ctxp->h_enum_disp_data);
			}
			data_linkp = set_disp_data(linkp);
			ctxp->h_enum_disp_data = nt_enum_set(data_linkp);
			if(!ctxp->h_enum_disp_data){
				free_history_ctx(ctxp);
				return DISP_STATE_ERROR;
			}
			b_result = set_model_data(h_model, 
					ctxp->h_enum_disp_data, update_board_list);
			nt_all_link_free(linkp, nt_usr_db_thread_data_free);
			if(!b_result){
				free_history_ctx(ctxp);
				return DISP_STATE_ERROR;
			}
		}
	}
	
	switch(wp->key){
	case NT_KEY_CMD_BOARD_UPDATE:
		if(ctxp->h_enum_disp_data){
			set_model_data(h_model, ctxp->h_enum_disp_data, NULL);
		}
		break;
	case NT_KEY_CLOSE:
	case KEY_LEFT:
		return DISP_STATE_BOARDMENU;
	case NT_KEY_REFRESH:
		return DISP_CMD_REFRESH; 
	case NT_KEY_COMMAND1:
		assert(wp->cmd_param);
		cmd = parse_cmd1(wp->cmd_param, &limit, &offset);
		switch(cmd){
		case NT_CMD_LIMIT:
			if(limit >= 0)
				ctxp->query_limit = limit;
			if(offset >= 0)
				ctxp->query_offset = offset;
			if(limit < 0 && offset < 0){
				ctxp->query_offset += ctxp->query_limit;
			}
			ctxp->update_query = TRUE;
			ctxp->cur_cursor = 0;
			ctxp->scroll_pos = 0;
			return (DISP_STATE_RESLIST | DISP_CMD_REENTER);
		}
		break;
	case NT_KEY_RIGHT:
		return ctxp->prev_state;
	case NT_KEY_SELECT:
	case KEY_RIGHT:
		num = 0;
		nt_enum_reset(ctxp->h_enum_disp_data);
		while(NULL !=
				 (disp_datap = (disp_data_tp)
				 nt_enum_fetch(ctxp->h_enum_disp_data))){
			if(ctxp->cur_cursor != num++)
				continue;
			assert(disp_datap->board_name);
			assert(disp_datap->dat_name);
			(*h_sel_threadp) = nt_searched_thread_alloc(
					disp_datap->board_name, 
					disp_datap->dat_name, 
					disp_datap->title);
			if(*h_sel_threadp){
				return (DISP_STATE_RESLIST | DISP_CMD_SEL_THREAD);
			}
		}
		break;
	case NT_KEY_UP:
	case KEY_UP:
		ctxp->cur_cursor--;
		adjust_scroll_pos(wp, ctxp);
		break;
	case NT_KEY_DOWN: 
	case KEY_DOWN: 
		ctxp->cur_cursor++;
		adjust_scroll_pos(wp, ctxp);
		break;
	case NT_KEY_PAGEUP:
	case KEY_PPAGE:
		ctxp->cur_cursor -= wp->lines / LINE_HEIGHT;
		adjust_scroll_pos(wp, ctxp);
		break;
	case NT_KEY_PAGEDOWN:
	case KEY_NPAGE:
		ctxp->cur_cursor += wp->lines / LINE_HEIGHT;
		adjust_scroll_pos(wp, ctxp);
		break;
	}
	
	nt_enum_reset(ctxp->h_enum_disp_data);
	
	row = 0;
	num = 0;
	limit_row = ctxp->scroll_pos + wp->lines;
	while(NULL !=
			 (disp_datap = (disp_data_tp)
			 nt_enum_fetch(ctxp->h_enum_disp_data))){
		nwrite = -1;
		attr = (ctxp->cur_cursor == num) ? WA_BOLD : 0;
		num++;
		read_count = disp_datap->read_count;
		res_count = disp_datap->res_count;
		if(res_count == 0)
			res_count = read_count;
		title = disp_datap->title;
		if(!title)
			title = L"Unknown Title";
		swprintf(wbuf, sizeof(wbuf), L"%3d [%ls] %ls (%d)", num + ctxp->query_offset,
				disp_datap->board_name, title, res_count);
		if(read_count < res_count){
			swprintf(wbuf2, sizeof(wbuf2), L"未読%d", res_count - read_count);
			wcscat(wbuf, wbuf2);
		}
		if(row >= ctxp->scroll_pos){
			wmove(wp->wp, row - ctxp->scroll_pos, 0);
			nwrite = nt_add_wnstr(wp->wp, wbuf, attr, wp->cols);
		}else{
			nwrite = nt_get_wc_count_within_colmns(wbuf, wp->cols);
		}
		if(limit_row <= ++row)
			break;
		
		attr |= WA_UNDERLINE;
		if(row >= ctxp->scroll_pos){
			if(wcslen(wbuf+nwrite) > 0){
				len = nt_get_column_length(wbuf+nwrite);
				wmove(wp->wp, row - ctxp->scroll_pos, 0);
				nt_add_wnch(wp->wp, L' ', attr, 3);
				nt_add_wstr(wp->wp, wbuf+nwrite, attr);
				if(0 < (wp->cols - len))
					nt_add_wnch(wp->wp, L' ', attr, wp->cols - len);
			}else{
				wmove(wp->wp, row - ctxp->scroll_pos, 0);
				nt_add_wnch(wp->wp, L' ', attr, wp->cols);
			}
		}
		if(limit_row <= ++row)
			break;
	}
	
	return result_state;
}


static int adjust_scroll_pos(nt_window_tp wp, 
			history_ctx_tp ctxp)
{
	int num_data;
	int cursor_pos;
	int max_viaible_row;

	num_data = nt_enum_get_count(ctxp->h_enum_disp_data);
	if(num_data <= ctxp->cur_cursor){
		ctxp->cur_cursor = num_data - 1;
	}
	
	if(ctxp->cur_cursor <= 0){
		ctxp->cur_cursor = 0;
		ctxp->scroll_pos = 0;
		return 0;
	}
	
	cursor_pos = ctxp->cur_cursor * LINE_HEIGHT;
	
	if(cursor_pos < ctxp->scroll_pos){
		ctxp->scroll_pos = cursor_pos;
		return cursor_pos;
	}
	
	max_viaible_row = ctxp->scroll_pos + wp->lines - LINE_HEIGHT;
	if(max_viaible_row >= cursor_pos){
		return cursor_pos;
	}
	
	ctxp->scroll_pos = cursor_pos - wp->lines + LINE_HEIGHT;
	
	return cursor_pos;
}


static disp_data_tp disp_data_alloc(
		const wchar_t *board_name,  
		const wchar_t *dat_name, int read_count)
{
	disp_data_tp datap;
	
	if(read_count <= 0)
		return NULL;
	datap = malloc(sizeof(disp_data_t));
	if(!datap)
		return NULL;
	
	datap->board_name = nt_w_str_clone(board_name);
	if(!datap->board_name){
		free(datap);
		return NULL;
	}
	datap->dat_name = nt_w_str_clone(dat_name);
	if(!datap->dat_name){
		free(datap->board_name);
		free(datap);
		return NULL;
	}
	datap->title = NULL;
	datap->read_count = read_count;
	datap->res_count = 0;
	
	return datap;
}

static void disp_data_free(void *ptr)
{
	disp_data_tp  datap;
	assert(ptr);
	datap = (disp_data_tp)ptr;
	
	if(datap->title)
		free(datap->title);
	free(datap->board_name);
	free(datap->dat_name);
	free(datap);
}

static int parse_cmd1(const char *param, int *limit, int *offset)
{
	int len, i, state;
	const char *start, *end;
	char c;

	assert(param);
	
	if(!nt_strtok(param, ' ', &start, &end))
		return NT_CMD_NONE;
	
	len = end - start;
	if(len <= 0)
		return NT_CMD_NONE;
	
	if(0 == strncmp(NT_COMMAND1_LIMIT_1, start, len) ||
			0 == strncmp(NT_COMMAND1_LIMIT_2, start, len)){
	}else{
		return NT_CMD_NONE;
	}
	
	*limit = -1;
	*offset = -1;
	
	if(!nt_strtok(end, ' ', &start, &end))
		return NT_CMD_LIMIT;
		
	*limit = 0;
	state = 0;
	len = strlen(start);
	for(i = 0; i < len; i++){
		c = start[i];
		if(c >= '0' && c <= '9'){
			*limit *= 10;
			*limit += c - '0';
		}else if(c == ','){
			if(state == 0){
				state = 1;
				*offset = *limit;
				*limit = 0;
			}else{
				return NT_CMD_ERR;
			}
		}else if(c != ' '){
			break;
		}
	}
	if(*limit == 0){
		*limit = -1;
	}
	return NT_CMD_LIMIT;
}




