/* Copyright 2013 Akira Ohta (akohta001@gmail.com)
    This file is part of ntch.

    The ntch is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    The ntch is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ntch.  If not, see <http://www.gnu.org/licenses/>.
    
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <locale.h>
#include <assert.h>
#include <wchar.h>
#include <iconv.h>
#include <errno.h>

#define NT_CLOUD_PRIVATE

#include "env.h"
#include "utils/nt_std_t.h"
#include "utils/text.h"
#include "utils/file.h"
#include "utils/nt_pthread.h"
#include "utils/base64.h"
#include "utils/crypt.h"
#include "utils/nt_conv_char.h"
#include "net/nt_http.h"
#include "cloud/nt_cloud.h"

static BOOL arrange_res_number(
		int number, const char *seed, char *buf, const wchar_t *dat_name);
static int parse_res_number(const char *buf);
static nt_pthread_result_t update_read_count_fnc(void *param);
static nt_pthread_result_t delete_read_count_fnc(void *param);


void nt_cloud_update_read_count_async(nt_cloud_handle handle, 
		const wchar_t *_board_name, const wchar_t *_dat_name, 
		int res_number)
{
	nt_link_tp param;
	nt_pthread_handle h_pthread;
	wchar_t *board_name, *dat_name;
	
	board_name = nt_w_str_clone(_board_name);
	if(!board_name)
		return;
	dat_name = nt_w_str_clone(_dat_name);
	if(!dat_name){
		free(board_name);
		return;
	}
	param = nt_link_add_data(NULL, handle);
	if(!param){
		free(board_name);
		free(dat_name);
		return;
	}
	nt_cloud_add_ref(handle);
	if(!nt_link_add_data(param, board_name)){
		goto ERROR_TRAP;
	}
	if(!nt_link_add_data(param, dat_name)){
		goto ERROR_TRAP;
	}
	if(!nt_link_add_n_data(param, res_number)){
		goto ERROR_TRAP;
	}
	
	h_pthread = nt_pthread_alloc(
			update_read_count_fnc, param, NULL);
	if(!h_pthread){
		goto ERROR_TRAP;
	}
	
	if(!nt_pthread_put_que(h_pthread)){
		nt_pthread_release_ref(h_pthread);
		goto ERROR_TRAP;
	}
	
	nt_pthread_release_ref(h_pthread);
	return;
ERROR_TRAP:
	nt_cloud_release_ref(handle);
	nt_all_link_free(param, NULL);
	free(board_name);
	free(dat_name);

	
}

BOOL nt_cloud_update_read_count(nt_cloud_handle handle, 
		const wchar_t *board_name, const wchar_t *dat_name, 
		int res_number, int depth)
{
	char *url;
	char buf1[1024];
	char buf2[1024];
	char buf3[1024];
	char *post_data;
	nt_cloud_tp cloudp;
	int len;
	nt_crypt_handle h_crypt;
	iconv_t icd;
	static const int buf_delta = 1024;
	int result_code;
	
	assert(handle);
	assert(handle->chk_sum == NT_CLOUD_CHK_SUM);
	cloudp = (nt_cloud_tp)handle;
	assert(cloudp->usr_id);
	assert(cloudp->token);
	assert(cloudp->query_url);
	
	len = strlen(cloudp->query_url);
	len += strlen(NT_CLOUD_UPDATE_READ_CNT_PHP);
	url = malloc(len+1);
	if(!url)
		return FALSE;
	strcpy(url, cloudp->query_url);
	strcat(url, NT_CLOUD_UPDATE_READ_CNT_PHP);
	
	post_data = malloc(buf_delta);
	if(!post_data){
		free(url);
		return FALSE;
	}
	
	h_crypt = nt_crypt_get_handle();
	if(!h_crypt){
		free(url);
		free(post_data);
		return FALSE;
	}
	icd = iconv_open("cp932", "wchar_t");
	if(((iconv_t)-1) == icd){
		nt_crypt_release_ref(h_crypt);
		free(url);
		free(post_data);
		return FALSE;
	}
	sprintf(post_data, "user=%s&token=%s", 
				cloudp->usr_id, cloudp->token);
	

	if(!nt_wc_sjis_aes_b64_url_encode(icd, h_crypt,
				board_name,
				(unsigned char *)buf2, buf3, sizeof(buf3))){
		iconv_close(icd);
		nt_crypt_release_ref(h_crypt);
		free(url);
		free(post_data);
		return FALSE;
	}
	strcat(post_data, "&board_name=");
	strcat(post_data, buf3);
	
	if(!nt_wc_sjis_aes_b64_url_encode(icd, h_crypt,
				dat_name,
				(unsigned char *)buf2, buf3, sizeof(buf3))){
		iconv_close(icd);
		nt_crypt_release_ref(h_crypt);
		free(url);
		free(post_data);
		return FALSE;
	}
	strcat(post_data, "&dat_name=");
	strcat(post_data, buf3);
	
	assert(AES256_PASS);
	if(!arrange_res_number(res_number, AES256_PASS, buf1, dat_name)){
		iconv_close(icd);
		nt_crypt_release_ref(h_crypt);
		free(url);
		free(post_data);
		return FALSE;
	}
	if(!nt_ascii_aes_b64_url_encode(icd, h_crypt,
				buf1, (unsigned char *)buf2, buf3, sizeof(buf3))){
		iconv_close(icd);
		nt_crypt_release_ref(h_crypt);
		free(url);
		free(post_data);
		return FALSE;
	}
	iconv_close(icd);
	strcat(post_data, "&res_number=");
	strcat(post_data, buf3);
	if(!nt_http_post(url, post_data,
			buf1, sizeof(buf1), NULL, NULL, NULL, NULL)){
		free(post_data);
		free(url);
		return FALSE;
	}
	free(post_data);
	free(url);
	result_code = nt_get_cloud_result(buf1);
	if(result_code == 301 && depth == 0){
		if(!nt_cloud_chk_user(handle))
			return FALSE;
		return nt_cloud_update_read_count(handle, 
				board_name, dat_name, res_number, ++depth);
	}else if(result_code != 100){
		return FALSE;
	}
	return TRUE;
}

static nt_pthread_result_t update_read_count_fnc(void *param)
{
	nt_pthread_result_t result;
	nt_link_tp linkp;
	wchar_t *board_name, *dat_name;
	int res_num;
	nt_cloud_handle h_cloud;
	
	
	result.code = 0;
	result.data = NULL;
	
	linkp = (nt_link_tp)param;
	if(4 != nt_link_num(linkp)){
		nt_all_link_free(linkp, NULL);
		return result;
	}
	h_cloud = linkp->data;
	board_name = linkp->next->data;
	dat_name = linkp->next->next->data;
	res_num = linkp->next->next->next->n_data;
	nt_all_link_free(linkp, NULL);
	
	nt_cloud_update_read_count(h_cloud, 
				board_name, dat_name, res_num, 0);
	
	nt_cloud_release_ref(h_cloud);
	free(board_name);
	free(dat_name);
	return result;
}


BOOL nt_cloud_query_board_attributes(nt_cloud_handle handle, 
		const wchar_t *board_name, nt_usr_db_handle db_handle, int depth)
{
	char *url, *post_data, *line, *cptr;
	char buf1[1024];
	char buf2[1024];
	char buf3[1024];
	wchar_t wc[1024];
	nt_cloud_tp cloudp;
	nt_link_tp linkp, wrkp;
	wchar_t  *dat_name;
	char *last_query;
	int len, i, read_cnt, r_type;
	nt_crypt_handle h_crypt;
	iconv_t icd;
	static const int buf_delta = 1024;
	int result_code;
	
	assert(handle);
	assert(handle->chk_sum == NT_CLOUD_CHK_SUM);
	cloudp = (nt_cloud_tp)handle;
	assert(cloudp->usr_id);
	assert(cloudp->token);
	assert(cloudp->query_url);
	
	len = strlen(cloudp->query_url);
	len += strlen(NT_CLOUD_QUERY_BOARD_ATTRIBUTES_PHP);
	url = malloc(len+1);
	if(!url)
		return FALSE;
	strcpy(url, cloudp->query_url);
	strcat(url, NT_CLOUD_QUERY_BOARD_ATTRIBUTES_PHP);
	
	post_data = malloc(buf_delta);
	if(!post_data){
		free(url);
		return FALSE;
	}
	
	h_crypt = nt_crypt_get_handle();
	if(!h_crypt){
		free(url);
		free(post_data);
		return FALSE;
	}
	icd = iconv_open("cp932", "wchar_t");
	if(((iconv_t)-1) == icd){
		nt_crypt_release_ref(h_crypt);
		free(url);
		free(post_data);
		return FALSE;
	}
	sprintf(post_data, "user=%s&token=%s", 
				cloudp->usr_id, cloudp->token);
	

	if(!nt_wc_sjis_aes_b64_url_encode(icd, h_crypt,
				board_name,
				(unsigned char *)buf2, buf3, sizeof(buf3))){
		iconv_close(icd);
		nt_crypt_release_ref(h_crypt);
		free(url);
		free(post_data);
		return FALSE;
	}
	iconv_close(icd);
	
	strcat(post_data, "&board_name=");
	strcat(post_data, buf3);
	last_query = nt_usr_db_board_last_query(db_handle, board_name);
	if(last_query){
		strcat(post_data, "&last_query=");
		strcat(post_data, last_query);
	}
	
	if(!nt_http_post(url, post_data,
			buf1, sizeof(buf1), NULL, NULL, NULL, NULL)){
		nt_crypt_release_ref(h_crypt);
		if(last_query)
			free(last_query);
		free(post_data);
		free(url);
		return FALSE;
	}
	if(last_query)
		free(last_query);
	free(post_data);
	free(url);
	
	result_code = nt_get_cloud_result(buf1);
	if(result_code == 301 && depth == 0){
		if(!nt_cloud_chk_user(handle)){
			nt_crypt_release_ref(h_crypt);
			return FALSE;
		}
		return nt_cloud_query_board_attributes(handle, 
				board_name, db_handle, ++depth);
	}else if(result_code != 100){
		nt_crypt_release_ref(h_crypt);
		return FALSE;
	}
	
	linkp = nt_split_line(buf1);
	if(!linkp){
		nt_crypt_release_ref(h_crypt);
		return FALSE;
	}
	
	/* There is little data. */
	if(nt_link_num(linkp) < 4){
		nt_crypt_release_ref(h_crypt);
		return TRUE;
	}
	wrkp = linkp;
	for(i = 0; i < 2; i++)
		wrkp = wrkp->next;
	
	last_query = nt_str_clone((const char*)wrkp->data);
	if(!last_query){
		nt_crypt_release_ref(h_crypt);
		return FALSE;
	}
	wrkp = wrkp->next;
	icd = iconv_open("wchar_t", "cp932");
	if(((iconv_t)-1) == icd){
		nt_crypt_release_ref(h_crypt);
		free(last_query);
		return FALSE;
	}
	while(wrkp != linkp){
		line = (char*)wrkp->data;
		cptr = strchr(line, ',');
		if(!cptr)
			goto LABEL1;
		*cptr = '\0';
		if(!nt_url_b64_aes_sjis_wc_decode(icd, h_crypt,
					line,
					(unsigned char *)buf2, wc, sizeof(buf2))){
			goto LABEL1;
		}
		dat_name = wc;
		line = cptr+1;
		cptr = strchr(line, ',');
		if(!cptr)
			goto LABEL1;
		*cptr = '\0';
		if(!nt_url_b64_aes_ascii_decode(icd, h_crypt,
					line,
					(unsigned char *)buf2, buf3, sizeof(buf2))){
			goto LABEL1;
		}
		read_cnt = parse_res_number(buf3);
		if(read_cnt == 0)
			goto LABEL1;
		line = cptr+1;
		cptr = strchr(line, ',');
		if(!cptr)
			goto LABEL1;
		*cptr = '\0';
		r_type = atoi(line);
		line = cptr+1;
		if(1 == r_type)
			nt_usr_db_update_read_count(db_handle,
				board_name, dat_name, read_cnt, line);
		else if(4 == r_type)
			nt_usr_db_delete_thread_log(db_handle,
				board_name, dat_name, NULL);
LABEL1:
		wrkp = wrkp->next;
	}
	nt_usr_db_update_read_count(db_handle,
				board_name, L"query_time", 0, last_query);
	
	free(last_query);
	nt_all_link_free(linkp, NULL);
	iconv_close(icd);
	nt_crypt_release_ref(h_crypt);
	return TRUE;
}

static nt_pthread_result_t query_attributes_func(void* data)
{
	nt_pthread_result_t result;
	result.code = NT_PTHREAD_RESULT_NONE;
	result.data = NULL;
	nt_link_tp  linkp;
	nt_cloud_handle handle;
	nt_usr_db_handle db_handle;
	
	linkp = (nt_link_tp)data;
	assert(2 == nt_link_num(linkp));
	
	handle = (nt_cloud_handle)linkp->data;
	db_handle = (nt_usr_db_handle)linkp->next->data;
	
	if(nt_cloud_query_attributes(handle, db_handle, 0)){
		result.code = NT_PTHREAD_RESULT_UPDATE_FAVORITE;
	}
	nt_cloud_release_ref(handle);
	nt_all_link_free(linkp, NULL);
	return result;
}

static nt_pthread_result_t query_attributes_result_func(
		nt_pthread_result_t result)
{
	return result;
}

BOOL nt_cloud_query_favorite_attributes_async(nt_cloud_handle handle, 
		nt_usr_db_handle db_handle)
{
	nt_pthread_handle h_pthread;
	nt_link_tp  linkp;
	
	nt_cloud_add_ref(handle);
	linkp = nt_link_add_data(NULL, handle);
	if(!linkp){
		nt_cloud_release_ref(handle);
		return FALSE;
	}
	if(!nt_link_add_data(linkp, db_handle)){
		nt_cloud_release_ref(handle);
		free(linkp);
		return FALSE;
	}
	
	
	h_pthread = nt_pthread_alloc(
			query_attributes_func, linkp, 
			query_attributes_result_func);
	if(!h_pthread){
		nt_cloud_release_ref(handle);
		nt_all_link_free(linkp, NULL);
		return FALSE;
	}
	
	if(!nt_pthread_put_que(h_pthread)){
		nt_pthread_release_ref(h_pthread);
		nt_cloud_release_ref(handle);
		nt_all_link_free(linkp, NULL);
		return FALSE;
	}
	nt_pthread_release_ref(h_pthread);
	return TRUE;
}

BOOL nt_cloud_query_attributes(nt_cloud_handle handle, 
		nt_usr_db_handle db_handle, int depth)
{
	char *url, *post_data, *line, *cptr;
	char buf1[1024*16];
	char buf2[256];
	char buf3[256];
	wchar_t board_nm_buf[128];
	wchar_t dat_nm_buf[128];
	nt_cloud_tp cloudp;
	nt_link_tp linkp, wrkp;
	char *last_query;
	int len, i, read_cnt, r_type;
	nt_crypt_handle h_crypt;
	iconv_t icd;
	static const int buf_delta = 1024;
	int result_code;
	
	assert(handle);
	assert(handle->chk_sum == NT_CLOUD_CHK_SUM);
	cloudp = (nt_cloud_tp)handle;
	assert(cloudp->usr_id);
	assert(cloudp->token);
	assert(cloudp->query_url);
	
	len = strlen(cloudp->query_url);
	len += strlen(NT_CLOUD_QUERY_ATTRIBUTES_PHP);
	url = malloc(len+1);
	if(!url)
		return FALSE;
	strcpy(url, cloudp->query_url);
	strcat(url, NT_CLOUD_QUERY_ATTRIBUTES_PHP);
	
	post_data = malloc(buf_delta);
	if(!post_data){
		free(url);
		return FALSE;
	}
	
	h_crypt = nt_crypt_get_handle();
	if(!h_crypt){
		free(url);
		free(post_data);
		return FALSE;
	}
	sprintf(post_data, "user=%s&token=%s", 
				cloudp->usr_id, cloudp->token);
	

	last_query = nt_usr_db_board_last_query(db_handle, L"favorite");
	if(last_query){
		strcat(post_data, "&last_query=");
		strcat(post_data, last_query);
	}
	
	if(!nt_http_post(url, post_data,
			buf1, sizeof(buf1), NULL, NULL, NULL, NULL)){
		nt_crypt_release_ref(h_crypt);
		if(last_query)
			free(last_query);
		free(post_data);
		free(url);
		return FALSE;
	}
	if(last_query)
		free(last_query);
	free(post_data);
	free(url);
	
	result_code = nt_get_cloud_result(buf1);
	if(result_code == 301 && depth == 0){
		if(!nt_cloud_chk_user(handle)){
			nt_crypt_release_ref(h_crypt);
			return FALSE;
		}
		return nt_cloud_query_attributes(handle, 
				db_handle, ++depth);
	}else if(result_code != 100){
		nt_crypt_release_ref(h_crypt);
		return FALSE;
	}
	
	linkp = nt_split_line(buf1);
	if(!linkp){
		nt_crypt_release_ref(h_crypt);
		return FALSE;
	}
	
	/* There is little data. */
	if(nt_link_num(linkp) < 4){
		nt_crypt_release_ref(h_crypt);
		return TRUE;
	}
	wrkp = linkp;
	for(i = 0; i < 2; i++)
		wrkp = wrkp->next;
	
	last_query = nt_str_clone((const char*)wrkp->data);
	if(!last_query){
		nt_crypt_release_ref(h_crypt);
		return FALSE;
	}
	wrkp = wrkp->next;
	icd = iconv_open("wchar_t", "cp932");
	if(((iconv_t)-1) == icd){
		nt_crypt_release_ref(h_crypt);
		free(last_query);
		return FALSE;
	}
	while(wrkp != linkp){
		line = (char*)wrkp->data;
		cptr = strchr(line, ',');
		if(!cptr)
			goto LABEL1;
		*cptr = '\0';
		if(!nt_url_b64_aes_sjis_wc_decode(icd, h_crypt,
					line,
					(unsigned char *)buf2, board_nm_buf, sizeof(board_nm_buf))){
			goto LABEL1;
		}
		//board_name = wc;
		line = cptr+1;
		cptr = strchr(line, ',');
		if(!cptr)
			goto LABEL1;
		*cptr = '\0';
		if(!nt_url_b64_aes_sjis_wc_decode(icd, h_crypt,
					line,
					(unsigned char *)buf2, dat_nm_buf, sizeof(dat_nm_buf))){
			goto LABEL1;
		}
		//dat_name = wc;
		line = cptr+1;
		cptr = strchr(line, ',');
		if(!cptr)
			goto LABEL1;
		*cptr = '\0';
		if(!nt_url_b64_aes_ascii_decode(icd, h_crypt,
					line,
					(unsigned char *)buf2, buf3, sizeof(buf2))){
			goto LABEL1;
		}
		read_cnt = parse_res_number(buf3);
		if(read_cnt == 0)
			goto LABEL1;
		line = cptr+1;
		cptr = strchr(line, ',');
		if(!cptr)
			goto LABEL1;
		*cptr = '\0';
		r_type = atoi(line);
		line = cptr+1;
		if(1 == r_type)
			nt_usr_db_update_read_count(db_handle,
				board_nm_buf, dat_nm_buf, read_cnt, line);
		else if(4 == r_type)
			nt_usr_db_delete_thread_log(db_handle,
				board_nm_buf, dat_nm_buf, NULL);
LABEL1:
		wrkp = wrkp->next;
	}
	nt_usr_db_update_read_count(db_handle,
				L"favorite", L"query_time", 0, last_query);
	
	free(last_query);
	nt_all_link_free(linkp, NULL);
	iconv_close(icd);
	nt_crypt_release_ref(h_crypt);
	return TRUE;
}

void nt_cloud_delete_read_count_async(nt_cloud_handle handle, 
		const wchar_t *_board_name, const wchar_t *_dat_name)
{
	nt_link_tp param;
	nt_pthread_handle h_pthread;
	wchar_t *board_name, *dat_name;
	
	board_name = nt_w_str_clone(_board_name);
	if(!board_name)
		return;
	dat_name = nt_w_str_clone(_dat_name);
	if(!dat_name){
		free(board_name);
		return;
	}
	param = nt_link_add_data(NULL, handle);
	if(!param){
		free(board_name);
		free(dat_name);
		return;
	}
	nt_cloud_add_ref(handle);
	if(!nt_link_add_data(param, board_name)){
		goto ERROR_TRAP;
	}
	if(dat_name){
		if(!nt_link_add_data(param, dat_name)){
			goto ERROR_TRAP;
		}
	}
	h_pthread = nt_pthread_alloc(
			delete_read_count_fnc, param, NULL);
	if(!h_pthread){
		goto ERROR_TRAP;
	}
	
	if(!nt_pthread_put_que(h_pthread)){
		nt_pthread_release_ref(h_pthread);
		goto ERROR_TRAP;
	}
	
	nt_pthread_release_ref(h_pthread);
	return;
ERROR_TRAP:
	nt_cloud_release_ref(handle);
	nt_all_link_free(param, NULL);
	free(board_name);
	free(dat_name);

	
}

static nt_pthread_result_t delete_read_count_fnc(void *param)
{
	nt_pthread_result_t result;
	nt_link_tp linkp;
	wchar_t *board_name, *dat_name;
	int num;
	nt_cloud_handle h_cloud;
	
	
	result.code = 0;
	result.data = NULL;
	
	linkp = (nt_link_tp)param;
	num = nt_link_num(linkp);
	if(2 != num && 3 != num){
		nt_all_link_free(linkp, NULL);
		return result;
	}
	h_cloud = linkp->data;
	board_name = linkp->next->data;
	if(num == 3)
		dat_name = linkp->next->next->data;
	else
		dat_name = NULL;
	nt_all_link_free(linkp, NULL);
	
	nt_cloud_delete_read_count(h_cloud, 
				board_name, dat_name, 0);
	
	nt_cloud_release_ref(h_cloud);
	free(board_name);
	free(dat_name);
	return result;
}
BOOL nt_cloud_delete_read_count(nt_cloud_handle handle, 
		const wchar_t *board_name, const wchar_t *dat_name,
		int depth)
{
	char *url, *post_data;
	char buf1[1024];
	char buf2[1024];
	char buf3[1024];
	nt_cloud_tp cloudp;
	int len;
	nt_crypt_handle h_crypt;
	iconv_t icd;
	static const int buf_delta = 1024;
	int result_code;
	
	
	assert(handle);
	assert(handle->chk_sum == NT_CLOUD_CHK_SUM);
	cloudp = (nt_cloud_tp)handle;
	assert(cloudp->usr_id);
	assert(cloudp->token);
	assert(cloudp->query_url);
	
	len = strlen(cloudp->query_url);
	len += strlen(NT_CLOUD_DELETE_READ_CNT_PHP);
	url = malloc(len+1);
	if(!url)
		return FALSE;
	strcpy(url, cloudp->query_url);
	strcat(url, NT_CLOUD_DELETE_READ_CNT_PHP);
	
	post_data = malloc(buf_delta);
	if(!post_data){
		free(url);
		return FALSE;
	}
	
	h_crypt = nt_crypt_get_handle();
	if(!h_crypt){
		free(url);
		free(post_data);
		return FALSE;
	}
	icd = iconv_open("cp932", "wchar_t");
	if(((iconv_t)-1) == icd){
		nt_crypt_release_ref(h_crypt);
		free(url);
		free(post_data);
		return FALSE;
	}
	sprintf(post_data, "user=%s&token=%s", 
				cloudp->usr_id, cloudp->token);
	

	if(!nt_wc_sjis_aes_b64_url_encode(icd, h_crypt,
				board_name,
				(unsigned char *)buf2, buf3, sizeof(buf3))){
		iconv_close(icd);
		nt_crypt_release_ref(h_crypt);
		free(url);
		free(post_data);
		return FALSE;
	}
	strcat(post_data, "&board_name=");
	strcat(post_data, buf3);
	
	if(dat_name){
		if(!nt_wc_sjis_aes_b64_url_encode(icd, h_crypt,
					dat_name,
					(unsigned char *)buf2, buf3, sizeof(buf3))){
			iconv_close(icd);
			nt_crypt_release_ref(h_crypt);
			free(url);
			free(post_data);
			return FALSE;
		}
		strcat(post_data, "&dat_name=");
		strcat(post_data, buf3);
	}else{
		strcat(post_data, "&dat_name=*");
	}
	iconv_close(icd);
	nt_crypt_release_ref(h_crypt);
	
	if(!nt_http_post(url, post_data,
			buf1, sizeof(buf1), NULL, NULL, NULL, NULL)){
		free(post_data);
		return FALSE;
	}
	free(post_data);
	free(url);
	
	result_code = nt_get_cloud_result(buf1);
	if(result_code == 301 && depth == 0){
		if(!nt_cloud_chk_user(handle)){
			return FALSE;
		}
		return nt_cloud_delete_read_count(handle, 
				board_name, dat_name, ++depth);
	}else if(result_code != 100){
		return FALSE;
	}
	
	return TRUE;
}



static BOOL arrange_res_number(
		int number, const char *seed, char *buf, const wchar_t *dat_name)
{
	int i, n, len;
	char c;
	wchar_t *wptr;
	
	
	len = wcslen(dat_name);
	assert(len > 0);
	wptr = wcschr(dat_name, L'.');
	if(wptr && wptr > dat_name){
		c = (char)*(wptr-1);
	}else{
		c = (char)dat_name[len-1];
	}
	n = number;
	i = 0;
	do{
		buf[i] = '0' + n % 10;
		n /= 10;
		i += 2;
	}while(n);
	
	for( ;i < 15; i+=2)
		buf[i] = ' ';
	
	len = strlen(seed);
	
	for(i = 1; i < 15; i+=2){
		if(i/2 < len)
			buf[i] = seed[i/2];
		else
			buf[i] = ' ';
	}
	buf[13] = c;
	buf[15] = '\0';
	return TRUE;
}

static int parse_res_number(const char *buf)
{
	int i, n, len;
	int coefficient;
	char c;
	
	len = strlen(buf);
	assert(len == 15);
	
	coefficient = 1;
	n = 0;
	for(i = 0; i < 15; i+=2){
		c = buf[i];
		if(c < '0' || c > '9')
			break;
		n += (c - '0') * coefficient;
		coefficient *= 10;
	}
	return n;
}

