# -*- coding: utf-8 -*-
# app/models/display_to_list.rb のクラス DisplayToList をテストする。

require 'test_helper'

class DisplayToListTest < ActiveSupport::TestCase
  fixtures(:languages, :companies, :organizations, :groups,
           :displays, :items, :config_outputs, :config_output_items,
           :po_messages, :po_translations, :default_lists,
           :search_conditions, :search_condition_items, :products,
           :display_narrowings)

  def setup
    CacheEachRequest.clear
    @list = displays(:list1_1)
  end

  def teardown
    CacheEachRequest.clear
  end

  # DisplayToList.source_display_to_list の動作をテストする。
  def test_source_display_to_list
    display = DisplayToList.source_display_to_list(1)
    assert display
    assert_equal 4, display.id
    display = DisplayToList.source_display_to_list(888)
    assert display
    assert_equal 8884, display.id
    assert_nil DisplayToList.source_display_to_list(0)
  end

  def test_options_to_order_by
    options = @list.options_to_order_by
    assert options.empty? || options.key?(:order)
  end

  # Display#options_to_select の動作をテストする。
  def test_options_to_select
    assert_match /\bcode\b/, @list.options_to_select[:select]
  end

  # Display#options_to_select にブロックを与えた場合の動作をテストする。
  def test_options_to_select__block_given
    options = @list.options_to_select {|i| i.column_name != "code"}
    assert_equal "id", options[:select]
  end

  # DisplayToList#options_to_select が疑似項目が指定したカラム名をとってくる。
  def test_options_to_select__sharp789
    d = displays(:list8_1)
    assert_match /\bupdated_at\b/, d.options_to_select[:select]
  end

  # DisplayToListPrivate#share によって共有一覧を作成することをテストする。
  def test_display_to_list_private_share
    User.current = users(:admin)
    x = DisplayToListPrivate.find(:first)
    p = x.person
    s = x.items.size
    shared = x.share(Company.find(1))
    assert_kind_of SharedList, shared
    assert_equal x.id, shared.display_id
    assert_equal 1, shared.roleable_id
    assert_equal "Company", shared.roleable_type
  end

  # DisplayToListShared#export によって個人別一覧を作成することをテストする。
  def test_display_to_list_shared_export
    User.current = users(:long_user)
    x = DisplayToListPrivate.find(20001)
    s = x.items.size
    y = x.export(User.current.person_id)
    assert_kind_of DisplayToListPrivate, y
    assert_equal s, y.items.size
    assert_equal 2, y.person_id
    assert_not_equal x.name_po, y.name_po
  end

  # DisplayToList#private_copy によって個人利用のコピーを作成することをテストする。
  def test_private_copy__success
    User.current = users(:tesla)
    source = DisplayToList.find(20001)
    size = source.items.size
    display = source.private_copy
    assert_kind_of DisplayToListPrivate, display
    assert_equal 8, display.product_id
    assert_equal "display_to_list_private_1", display.code
    assert display.enabled?
    assert_equal User.current.person_id, display.person_id
    assert_equal size, display.items.size
  end

  # DisplayToList#private_copy は display_narrowing もコピーすることをテストする
  def test_private_copy__with_display_narrowing
    User.current = users(:admin)
    assert_difference('DisplayNarrowing.count', 1) do
      copied = displays(:demo_list1).private_copy
      assert copied.narrowing?
    end
    assert_difference('DisplayNarrowing.count', 3) do
      copied = displays(:product_detailed_0_list).private_copy
      assert copied.narrowing?
    end
  end

  # DisplayToList#options_to_extract が適切なハッシュを返すことをテストする。
  def test_options_to_extract
    options = @list.options_to_extract
    assert options.empty? || options.key?(:conditions)
    User.current = users(:tesla)
    d_name = Domain.quoted_table_name
    name = Display.quote_value('dummy%')
    code = Display.quote_value('DUMMY10050')
    options = {:conditions => "(#{d_name}.name LIKE #{name}) AND (#{d_name}.code <= #{code})"}
    assert_equal options, DisplayToList.find(84).options_to_extract
  end

  # DisplayToList#default_list? が既定の一覧かどうかを判定する。
  def test_default_list?
    User.current = users(:tesla)
    assert Display.find(20001).default_list?
    assert !Display.find(85).default_list?
  end
end
