/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.SystemParameter;
import org.opengion.fukurou.util.LogWriter;
import org.opengion.fukurou.util.HybsEntry ;
import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.model.Formatter;
import org.opengion.fukurou.db.ConnectionFactory;

import java.util.Map ;
import java.util.LinkedHashMap ;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ParameterMetaData;
import java.sql.SQLException;

/**
 * Process_DBMerge は、UPDATE と INSERT を指定し データベースを追加更新
 * する、ChainProcess インターフェースの実装クラスです。
 * 上流(プロセスチェインのデータは上流から下流へと渡されます。)から
 * 受け取った LineModel を元に、DBTableModel 形式ファイルを出力します。
 *
 * データベース接続先等は、ParamProcess のサブクラス(Process_DBParam)に
 * 設定された接続(Connection)を使用します。
 *
 * 引数文字列中にスペースを含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、スペースは挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * SQL文には、{&#064;DATE.YMDH}等のシステム変数が使用できます。
 *
 * @og.formSample
 *  Process_DBMerge -dbid=DBGE -insertTable=GE41
 *
 *   [ -dbid=DB接続ID           ] ： -dbid=DBGE (例: Process_DBParam の -configFile で指定する DBConfig.xml ファイルで規定)
 *   [ -update=検索SQL文        ] ： -update="UPDATE GE41 SET NAME_JA = [NAME_JA],LABEL_NAME = [LABEL_NAME]
 *                                         WHERE SYSTEM_ID = [SYSTEM_ID] AND CLM = [CLM]"
 *   [ -updateFile=登録SQLﾌｧｲﾙ  ] ： -updateFile=update.sql
 *                                ：   -update や -updateFile が指定されない場合は、エラーです。
 *   [ -update_XXXX=固定値      ] ： -update_SYSTEM_ID=GE
 *                                     SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。
 *                                     WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'
 *   [ -insertTable=登録ﾃｰﾌﾞﾙID ] ： INSERT文を指定する場合は不要。INSERT する場合のテーブルID
 *   [ -insert=検索SQL文        ] ： -insert="INSERT INTO GE41 (SYSTEM_ID,CLM,NAME_JA,LABEL_NAME)
 *                                         VALUES ([SYSTEM_ID],[CLM],[NAME_JA],[LABEL_NAME])"
 *   [ -insertFile=登録SQLﾌｧｲﾙ  ] ： -insertFile=insert.sql
 *                                ：   -insert や -insertFile や、-table が指定されない場合は、エラーです。
 *   [ -insert_XXXX=固定値      ] ： -insert_SYSTEM_ID=GE
 *                                     SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。
 *                                     WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'
 *   [ -const_XXXX=固定値       ] ： -const_FGJ=1
 *                                     LineModel のキー(const_ に続く文字列)の値に、固定値を設定します。
 *                                     キーが異なれば、複数のカラム名を指定できます。
 *   [ -commitCnt=commit処理指定] ： 指定数毎にコミットを発行します。0 の場合は、終了までコミットしません。
 *   [ -display=[false/true]    ] ： 結果を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *   [ -debug=[false/true]      ] ： デバッグ情報を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_DBMerge extends AbstractProcess implements ChainProcess {
	private static final String UPDATE_KEY	= "update_" ;
	private static final String INSERT_KEY	= "insert_" ;
	private static final String CNST_KEY	= "const_" ;

	private Connection	connection		;
	private PreparedStatement insPstmt	;
	private PreparedStatement updPstmt	;
	private ParameterMetaData insPmeta	;	// 5.1.2.0 (2010/01/01) setObject に、Type を渡す。(PostgreSQL対応)
	private ParameterMetaData updPmeta	;	// 5.1.2.0 (2010/01/01) setObject に、Type を渡す。(PostgreSQL対応)
	private boolean useParamMetaData	;	// 5.1.2.0 (2010/01/01) setObject に、Type を渡す。(PostgreSQL対応)

	private String		dbid		;
	private String		insert		;
	private String		update		;
	private String		insertTable	;
	private int[]		insClmNos	;		// insert 時のファイルのヘッダーのカラム番号
	private int[]		updClmNos	;		// update 時のファイルのヘッダーのカラム番号
	private int			commitCnt	;		// コミットするまとめ件数
	private boolean		display		;		// false:表示しない
	private boolean		debug		;		// 5.7.3.0 (2014/02/07) デバッグ情報

	private String[]	cnstClm		;		// 固定値を設定するカラム名
	private int[]		cnstClmNos	;		// 固定値を設定するカラム番号
	private String[]	constVal	;		// カラム番号に対応した固定値

	private boolean		firstRow	= true;		// 最初の一行目
	private int			count		;
	private int			insCount	;
	private int			updCount	;

	private static final Map<String,String> mustProparty   ;		// ［プロパティ］必須チェック用 Map
	private static final Map<String,String> usableProparty ;		// ［プロパティ］整合性チェック Map

	static {
		mustProparty = new LinkedHashMap<String,String>();

		usableProparty = new LinkedHashMap<String,String>();
		usableProparty.put( "dbid",	"Process_DBParam の -configFile で指定する DBConfig.xml ファイルで規定" );
		usableProparty.put( "update",	"更新SQL文(sql or sqlFile 必須)" +
									CR + "例: \"UPDATE GE41 " +
									CR + "SET NAME_JA = [NAME_JA],LABEL_NAME = [LABEL_NAME] " +
									CR + "WHERE SYSTEM_ID = [SYSTEM_ID] AND CLM = [CLM]\"" );
		usableProparty.put( "updateFile",	"更新SQLファイル(sql or sqlFile 必須)例: update.sql" );
		usableProparty.put( "update_",		"SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。" +
									CR + "WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'" );
		usableProparty.put( "insert",			"登録SQL文(sql or sqlFile 必須)" +
									CR + "例: \"INSERT INTO GE41 " +
									CR + "(SYSTEM_ID,CLM,NAME_JA,LABEL_NAME) " +
									CR + "VALUES ([SYSTEM_ID],[CLM],[NAME_JA],[LABEL_NAME])\"" );
		usableProparty.put( "insertFile",		"登録SQLファイル(sql or sqlFile 必須)例: insert.sql" );
		usableProparty.put( "insertTable",	"INSERT する場合のテーブルID SQL文を指定する場合は不要。" );
		usableProparty.put( "insert_",		"SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。" +
									CR + "WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'" );
		usableProparty.put( "const_",	"LineModel のキー(const_ に続く文字列)の値に、固定値を" +
									CR + "設定します。キーが異なれば、複数のカラム名を指定できます。" +
									CR + "例: -sql_SYSTEM_ID=GE" );
		usableProparty.put( "commitCnt",	"指定数毎にコミットを発行します。" +
									CR + "0 の場合は、終了までコミットしません(初期値: 0)" );
		usableProparty.put( "display",	"結果を標準出力に表示する(true)かしない(false)か" +
										CR + "(初期値:false:表示しない)" );
		usableProparty.put( "debug",	"デバッグ情報を標準出力に表示する(true)かしない(false)か" +
										CR + "(初期値:false:表示しない)" );		// 5.7.3.0 (2014/02/07) デバッグ情報
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_DBMerge() {
		super( "org.opengion.fukurou.process.Process_DBMerge",mustProparty,usableProparty );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理(ファイルオープン、ＤＢオープン等)に使用します。
	 *
	 * @og.rev 5.1.2.0 (2010/01/01) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
	 * @og.rev 5.3.8.0 (2011/08/01) useParamMetaData を ConnectionFactory経由で取得。(PostgreSQL対応)
	 *
	 * @param   paramProcess データベースの接続先情報などを持っているオブジェクト
	 */
	public void init( final ParamProcess paramProcess ) {
		final Argument arg = getArgument();

		insertTable	= arg.getProparty("insertTable");
		update		= arg.getFileProparty("update","updateFile",false);
		insert		= arg.getFileProparty("insert","insertFile",false);
		commitCnt	= arg.getProparty("commitCnt",commitCnt);
		display		= arg.getProparty("display",display);
		debug		= arg.getProparty("debug",debug);				// 5.7.3.0 (2014/02/07) デバッグ情報

		dbid		= arg.getProparty("dbid");
		connection	= paramProcess.getConnection( dbid );
		// 5.1.2.0 (2010/01/01) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
		useParamMetaData = ConnectionFactory.useParameterMetaData( dbid );	// 5.3.8.0 (2011/08/01)

		if( insert == null && insertTable == null ) {
			final String errMsg = "insert または、insertFile を指定しない場合は、insertTable を必ず指定してください。";
			throw new RuntimeException( errMsg );
		}

		if( insert != null && insertTable != null ) {
			final String errMsg = "insert または、insertFile と、insertTable は、両方同時に指定できません。["
								 + insert + "],[" + insertTable + "]";
			throw new RuntimeException( errMsg );
		}

		// 3.8.0.1 (2005/06/17) {@DATE.XXXX} 変換処理の追加
		// {@DATE.YMDH} などの文字列を、yyyyMMddHHmmss 型の日付に置き換えます。
		// SQL文の {@XXXX} 文字列の固定値への置き換え
		HybsEntry[] entry	=arg.getEntrys(UPDATE_KEY);		// 配列
		SystemParameter sysParam = new SystemParameter( update );
		update = sysParam.replace( entry );

		if( insert != null ) {
			entry	=arg.getEntrys(INSERT_KEY);		// 配列
			sysParam = new SystemParameter( insert );
			insert = sysParam.replace( entry );
		}

		final HybsEntry[] cnstKey = arg.getEntrys( CNST_KEY );		// 配列
		final int csize	= cnstKey.length;
		cnstClm		= new String[csize];
		constVal	= new String[csize];
		for( int i=0; i<csize; i++ ) {
			cnstClm[i]  = cnstKey[i].getKey();
			constVal[i] = cnstKey[i].getValue();
		}
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理(ファイルクローズ、ＤＢクローズ等)に使用します。
	 *
	 * @og.rev 4.0.0.0 (2007/11/27) commit,rollback,remove 処理を追加
	 * @og.rev 5.1.2.0 (2010/01/01) insPmeta , updPmeta のクリア
	 *
	 * @param   isOK トータルで、OKだったかどうか[true:成功/false:失敗]
	 */
	public void end( final boolean isOK ) {
		final boolean flag1 = Closer.stmtClose( updPstmt );
		final boolean flag2 = Closer.stmtClose( insPstmt );
		updPstmt = null;
		insPstmt = null;

		insPmeta	= null ;	// 5.1.2.0 (2010/01/01)
		updPmeta	= null ;	// 5.1.2.0 (2010/01/01)

		// close に失敗しているのに commit しても良いのか？
		if( isOK ) {
			Closer.commit( connection );
		}
		else {
			Closer.rollback( connection );
		}
		ConnectionFactory.remove( connection,dbid );

		if( ! flag1 ) {
			final String errMsg = "update ステートメントをクローズ出来ません。" + CR
								+ " update=[" + update + "] , commit=[" + isOK + "]" ;
			throw new RuntimeException( errMsg );
		}

		if( ! flag2 ) {
			final String errMsg = "insert ステートメントをクローズ出来ません。" + CR
								+ " insert=[" + insert + "] , commit=[" + isOK + "]" ;
			throw new RuntimeException( errMsg );
		}
	}

	/**
	 * 引数の LineModel を処理するメソッドです。
	 * 変換処理後の LineModel を返します。
	 * 後続処理を行わない場合(データのフィルタリングを行う場合)は、
	 * null データを返します。つまり、null データは、後続処理を行わない
	 * フラグの代わりにも使用しています。
	 * なお、変換処理後の LineModel と、オリジナルの LineModel が、
	 * 同一か、コピー(クローン)かは、各処理メソッド内で決めています。
	 * ドキュメントに明記されていない場合は、副作用が問題になる場合は、
	 * 各処理ごとに自分でコピー(クローン)して下さい。
	 *
	 * @og.rev 5.1.2.0 (2010/01/01) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
	 * @og.rev 5.3.8.0 (2011/08/01) useParamMetaData  setNull 対応(PostgreSQL対応)
	 * @og.rev 5.7.2.2 (2014/01/24) SQL実行エラーを少し詳細に出力します。
	 *
	 * @param	data オリジナルのLineModel
	 *
	 * @return	処理変換後のLineModel
	 */
	public LineModel action( final LineModel data ) {
		count++ ;
		int updCnt = 0;
		try {
			if( firstRow ) {
				makePrepareStatement( insertTable,data );

				final int size   = cnstClm.length;
				cnstClmNos = new int[size];
				for( int i=0; i<size; i++ ) {
					cnstClmNos[i] = data.getColumnNo( cnstClm[i] );
				}

				firstRow = false;
				if( display ) { println( data.nameLine() ); }		// 5.7.3.0 (2014/02/07) デバッグ情報
			}

			// 固定値置き換え処理
			for( int j=0; j<cnstClmNos.length; j++ ) {
				data.setValue( cnstClmNos[j],constVal[j] );
			}

			// 5.1.2.0 (2010/01/01) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
			if( useParamMetaData ) {
				for( int i=0; i<updClmNos.length; i++ ) {
					final int type = updPmeta.getParameterType( i+1 );
					// 5.3.8.0 (2011/08/01) setNull 対応
					final Object val = data.getValue(updClmNos[i]);
					if( val == null || ( val instanceof String && ((String)val).isEmpty() ) ) {
						updPstmt.setNull( i+1, type );
					}
					else {
						updPstmt.setObject( i+1, val, type );
					}
				}
			}
			else {
				for( int i=0; i<updClmNos.length; i++ ) {
					updPstmt.setObject( i+1,data.getValue(updClmNos[i]) );
				}
			}

			updCnt = updPstmt.executeUpdate();
			if( updCnt == 0 ) {
				// 5.1.2.0 (2010/01/01) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
				if( useParamMetaData ) {
					for( int i=0; i<insClmNos.length; i++ ) {
						final int type = insPmeta.getParameterType( i+1 );
						// 5.3.8.0 (2011/08/01) setNull 対応
						final Object val = data.getValue(insClmNos[i]);
						if( val == null || ( val instanceof String && ((String)val).isEmpty() ) ) {
							insPstmt.setNull( i+1, type );
						}
						else {
							insPstmt.setObject( i+1, val, type );
						}
					}
				}
				else {
					for( int i=0; i<insClmNos.length; i++ ) {
						insPstmt.setObject( i+1,data.getValue(insClmNos[i]) );
					}
				}
				final int insCnt = insPstmt.executeUpdate();
				if( insCnt == 0 ) {
					final String errMsg = "１件も追加されませんでした。[" + data.getRowNo() + "]件目" + CR
										+ " insert=[" + insert + "]" + CR
										+ " data=[" + data.dataLine() + "]" + CR ;	// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
					throw new RuntimeException( errMsg );
				}
				insCount++ ;
			}
			else if( updCnt > 1 ) {
				final String errMsg = "複数行(" + updCnt + ")が同時に更新されました。[" + data.getRowNo() + "]件目" + CR
									+ " update=[" + update + "]" + CR
									+ " data=[" + data.dataLine() + "]" + CR ;	// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
				throw new RuntimeException( errMsg );
			}
			else {
				updCount ++ ;
			}

			if( commitCnt > 0 && ( count%commitCnt == 0 ) ) {
				Closer.commit( connection );
			}
			if( display ) { println( data.dataLine() ); }			// 5.7.3.0 (2014/02/07) デバッグ情報
		}
		catch (SQLException ex) {
			final String errMsg = "登録処理でエラーが発生しました。[" + data.getRowNo() + "]件目" + CR
								+ ((updCnt == 1) ?
										" update=[" + update + "]"
									:	" insert=[" + insert + "]" + CR
										+ " insertTable=[" + insertTable + "]" )
								+ CR
								+ "errCode=[" + ex.getErrorCode() + "] State=[" + ex.getSQLState() + "]" + CR
								+ "data=[" + data.dataLine() + "]" + CR ;	// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
			throw new RuntimeException( errMsg,ex );
		}
		return data;
	}

	/**
	 * 内部で使用する PreparedStatement を作成します。
	 * 引数指定の SQL または、LineModel から作成した SQL より構築します。
	 *
	 * @og.rev 5.1.2.0 (2010/01/01) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
	 * @og.rev 5.7.2.2 (2014/01/24) SQL実行エラーを少し詳細に出力します。
	 * @og.rev 6.2.3.0 (2015/05/01) CSV形式の作成を、String#join( CharSequence , CharSequence... )を使用。
	 *
	 * @param	table	処理対象のテーブルID
	 * @param	data	処理対象のLineModel
	 */
	private void makePrepareStatement( final String table,final LineModel data ) {
		if( insert == null ) {
			final String[] names = data.getNames();
			final int size = names.length;

			// 6.2.3.0 (2015/05/01) CSV形式の作成を、String#join( CharSequence , CharSequence... )を使用。
			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
				.append( "INSERT INTO " ).append( table ).append( " (" )
				.append( String.join( "," , names ) );		// 6.2.3.0 (2015/05/01)
			buf.append( " ) VALUES ( ?" );
			for( int i=1; i<size; i++ ) {
				buf.append( ",?" );
			}
			buf.append( " )" );
			insert = buf.toString();

			// カラム番号を設定します。
			insClmNos = new int[size];
			for( int i=0; i<size; i++ ) {
				insClmNos[i] = i;
			}
		}
		else {
			final Formatter format = new Formatter( data );
			format.setFormat( insert );
			insert = format.getQueryFormatString();
			insClmNos = format.getClmNos();
		}

		final Formatter format = new Formatter( data );
		format.setFormat( update );
		update = format.getQueryFormatString();
		updClmNos = format.getClmNos();

		try {
			insPstmt = connection.prepareStatement( insert );
			updPstmt = connection.prepareStatement( update );
			// 5.1.2.0 (2010/01/01) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
			if( useParamMetaData ) {
				insPmeta = insPstmt.getParameterMetaData();
				updPmeta = updPstmt.getParameterMetaData();
			}
		}
		catch (SQLException ex) {
			// 5.7.2.2 (2014/01/24) SQL実行エラーを少し詳細に出力します。
			final String errMsg = "PreparedStatement を取得できませんでした。" + CR
								+ "errMsg=[" + ex.getMessage() + "]" + CR
								+ "errCode=[" + ex.getErrorCode() + "] State=[" + ex.getSQLState() + "]" + CR
								+ "insert=[" + insert + "]" + CR
								+ "update=[" + update + "]" + CR
								+ "table=[" + table + "]" + CR
								+ "nameLine=[" + data.nameLine() + "]" + CR
								+ "data=[" + data.dataLine() + "]" + CR ;
			throw new RuntimeException( errMsg,ex );
		}
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		final String report = "[" + getClass().getName() + "]" + CR
							+ TAB + "DBID         : " + dbid + CR
							+ TAB + "Input  Count : " + count + CR
							+ TAB + "Update Count : " + updCount + CR
							+ TAB + "Insert Count : " + insCount ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	このクラスの使用方法
	 * @og.rtnNotNull
	 */
	public String usage() {
		final StringBuilder buf = new StringBuilder( BUFFER_LARGE )
			.append( "Process_DBMerge は、UPDATE と INSERT を指定し データベースを追加更新"			).append( CR )
			.append( "する、ChainProcess インターフェースの実装クラスです。"						).append( CR )
			.append( "上流(プロセスチェインのデータは上流から下流へと渡されます。)から"				).append( CR )
			.append( "受け取った LineModel を元に、データベースの存在チェックを行い、"				).append( CR )
			.append( "下流への処理を振り分けます。"													).append( CR )
			.append( CR )
			.append( "データベース接続先等は、ParamProcess のサブクラス(Process_DBParam)に"			).append( CR )
			.append( "設定された接続(Connection)を使用します。"										).append( CR )
			.append( CR )
			.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR )
			.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR )
			.append( "繋げてください。"																).append( CR )
			.append( CR )
			.append( "SQL文には、{@DATE.YMDH}等のシステム変数が使用できます。"						).append( CR )
			.append( CR ).append( CR )
			.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_DBMerge().usage() );
	}
}
