/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.util.LogWriter;
import org.opengion.fukurou.util.HybsDateUtil;
import static org.opengion.fukurou.util.HybsConst.CR ;				// 6.1.0.0 (2014/12/26)
import static org.opengion.fukurou.util.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

import java.util.Calendar;
import java.util.Locale;
import java.text.DateFormat;
import java.text.SimpleDateFormat;

/**
 * データのコード情報を取り扱うクラスです。
 *
 * パラメータで指定した、開始、終了、ステップ、フォーマット、titleフォーマットの情報から、
 * HTMLのメニューやリストを作成するための オプション
 * タグを作成したり、与えられたキーをもとに、チェック済みのオプションタグを作成したりします。
 * パラメータの初期値は、開始(0)、終了(14)、ステップ(1)、フォーマット(MM/dd)、titleフォーマット(null) です。
 * 
 * ここでは、日付(年/月/日）の自動生成を行います。
 * 基準は、実行日に対して、＋－の数字を指定します。
 * 例：20130206 に対して、-2,3,1 を指定すると、20130204,20130205,20130206,20130207,20130208,20130209 となります。
 * 
 * 開始、終了に、特殊なコマンドを指定する事も可能です。
 * ・SD ：当月の最初の日付にセットします。(当月１日)
 * ・ED ：当月の最後の日付にセットします。(当月月末)
 * ・SW ：日付処理の週初め(月曜日)にセットします。日付は当日より前に移動します。
 * ・EW ：日付処理の週末(日曜日)にセットします。日付は当日より後ろに移動します。
 * ・D1 ～ DXXX ：日を指定の分だけ進めます。D1なら翌日、D200 なら200日後
 * ・M1 ～ MXXX ：月を指定の分だけ進めます。M1なら翌月、M6 なら半年後
 * ・BSD ：先月の最初の日付にセットします。(先月１日)
 * ・BED ：先月の最後の日付にセットします。(先月月末)
 * ・ASD ：翌月の最初の日付にセットします。(翌月１日)
 * ・AED ：翌月の最後の日付にセットします。(翌月月末)
 * ・M1  ：１か月先。数字部分は、任意に指定できるため、M5 とすれば、５か月先。M-2 は、２か月前
 *
 * ステップは、無指定の場合は、１日単位です。例えば、SW,M6,7 とすれば、毎週月曜日を６か月先まで作成します。
 * また、M1 は、１か月単位になります。ただし、開始日を SD などにしないと、大の月と小の月で、おかしくなるので
 * ご注意ください。
 *
 * キーは、８文字の yyyyMMdd 形式で与えられます。ラベルは、フォーマットの指定に準拠します。
 * フォーマットを指定する場合は、ステップは、必須となります。
 * フォーマットの初期値は、MM/dd です。
 * titleフォーマット は、tips表示する場合のフォーマットになります。初期値は、ありません。
 *
 * @og.group 選択データ制御
 * @og.rev 5.6.1.1 (2013/02/08) 新規追加
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Selection_YMD extends Selection_NULL {
	private final String   CACHE ;
	private final String   ST_ED_STEP ;

	private final DateFormat lblFmt  ;
	private final DateFormat tipsFmt ;

	private final long	maxCacheTime ;		// キャッシュの破棄タイミングを計るための最大有効時間

	/**
	 * コンストラクター
	 *
	 * 引数は、開始、終了、ステップ、フォーマット、titleフォーマットです。
	 * パラメータの初期値は、開始(0)、終了(14)、ステップ(1)、フォーマット(MM/dd)、titleフォーマット(null) です。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) type別Selectionの場合、ラベルリソースを使用する為、言語を引数で渡す。
	 *
	 * @param	editPrm	開始、終了、ステップ、フォーマット、titleフォーマットを表す引数(-2,3,1)
	 * @param	lang  言語(今は未使用)
	 */
	public Selection_YMD( final String editPrm,final String lang  ) {
	//	if( param.length < 2 ) {
	//	final String errMsg = "引数は、開始、終了、[ステップ]、[フォーマット]、 [titleフォーマット]です。最低でも２個必要です。";
	//		throw new IllegalArgumentException( errMsg );
	//	}

		final String[] param = (editPrm == null) ? new String[0] : editPrm.split( "," ) ;

		final String start   = (param.length > 0) ? param[0].trim() : "0" ;
		final String end     = (param.length > 1) ? param[1].trim() : "14" ;

		final String step    = (param.length > 2) ? param[2].trim() : "1" ;
		final String lblPrm  = (param.length > 3) ? param[3].trim() : "MM/dd" ;
		final String tipsPrm = (param.length > 4) ? param[4].trim() : "MM月dd日(EEE)" ;

		final Calendar cal = Calendar.getInstance();
		HybsDateUtil.calendarCalc( cal , start );

		final Calendar endCal = Calendar.getInstance();
		HybsDateUtil.calendarCalc( endCal , end );

		final DateFormat keyFmt  = new SimpleDateFormat( "yyyyMMdd",Locale.JAPAN );					// キーとなるフォーマット。yyyyMMdd 固定
		lblFmt  = new SimpleDateFormat( lblPrm,Locale.JAPAN );									// ラベルのフォーマット
		// 6.0.2.5 (2014/10/31) null でないことがわかっている値の冗長な null チェックがあります。
		tipsFmt = new SimpleDateFormat( tipsPrm,Locale.JAPAN );									// titleフォーマット

		ST_ED_STEP = "Start=" + lblFmt.format( cal.getTime() ) +
					 " , End=" + lblFmt.format( endCal.getTime() ) + " , Step=" + step ;

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		// cal.before( endCal ) では、同一時刻の場合に false になる為、ここの判定では使えません。
		// sign を掛け算しているのは、逆順対応
		final int sign = ( step.indexOf( '-' ) < 0 ) ? 1 : -1 ;	// ステップの符号。
		while( endCal.compareTo( cal ) * sign >= 0 ) {
			final String key  = keyFmt.format( cal.getTime() );
			final String val  = lblFmt.format( cal.getTime() );

			buf.append( '>' ).append( val )						// 6.0.2.5 (2014/10/31) char を append する。
				.append( "</option><option value=\"" ).append( key )
				.append("\" title=\"").append( tipsFmt.format( cal.getTime() ) )
				.append( "\">" ).append( val ).append( "</option>" );

			HybsDateUtil.calendarCalc( cal , step );
		}

		CACHE = buf.toString();

		// キャシュの有効期間を求めるための時刻を作成します。キャッシュは、当日のみ有効です。
		final Calendar now   = Calendar.getInstance();
		now.set( Calendar.HOUR   , 0 );				// 時、分、秒 をリセットします。
		now.set( Calendar.MINUTE , 0 );
		now.set( Calendar.SECOND , 0 );
		now.add( Calendar.DAY_OF_MONTH , 1 );		// 1日進めます。

		maxCacheTime = now.getTimeInMillis() ;
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * このメソッドでは、引数のuseShortLabelがtrueに指定された場合に、ラベル(短)をベースとした
	 * ツールチップ表示を行います。
	 *
	 * @param   selectValue  選択されている値
	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
	 * @param   useShortLabel ラベル(短)をベースとしたオプション表示を行うかどうか。(未使用)
	 *
	 * @return  オプションタグ
	 * @og.rtnNotNull
	 */
	@Override
	public String getOption( final String selectValue,final boolean seqFlag, final boolean useShortLabel ) {
		// マッチするアドレスを探す。
		final int selected = CACHE.indexOf( "\"" + selectValue + "\"" );

		if( selected < 0 ) {
			if( selectValue != null && selectValue.length() > 0 ) {
				final String errMsg = "年月日範囲に存在しない値が指定されました。"
							+ " value=[" + selectValue + "]"
							+ CR + ST_ED_STEP ;
				LogWriter.log( errMsg );
			}
			return CACHE;
		}
		else {
			// "年月日" 文字列の位置が、selected なので、年月日の文字数＋２までが、前半部分になる。(年月日の文字数は８固定のはず)
			final int indx = selected + selectValue.length() + 2 ;

			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
			// 3.6.0.6 (2004/10/22) シーケンスアクセス機能を指定する seqFlag を導入
			if( seqFlag ) {
				buf.append( "<option value=\"" ).append( selectValue ).append( '"' );		// 6.0.2.5 (2014/10/31) char を append する。
			}
			else {
				buf.append( CACHE.substring( 0,indx ) );
			}
			buf.append( " selected=\"selected\"" )
				.append( CACHE.substring( indx ) );
			return buf.toString() ;
		}
	}



	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * このメソッドでは、短縮ラベルを返すかどうかを指定するフラグを指定します。
	 * getValueLabel( XX,false ) は、getValueLabel( XX ) と同じです。
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) を追加
	 *
	 * @param	selectValue	選択肢の値
	 * @param	flag	短縮ラベルを [true:使用する/false:しない]
	 *
	 * @return  選択肢のラベル
	 * @og.rtnNotNull
	 * @see     #getValueLabel( String )
	 */
	@Override
	public String getValueLabel( final String selectValue,final boolean flag ) {
		// 選択肢があろうとなかろうと、フォーマット変換して値を返す。

		final Calendar cal = HybsDateUtil.getCalendar( selectValue );

		if( flag && tipsFmt != null ) {
			return "<span title=\"" + tipsFmt.format( cal.getTime() ) + "\">" + lblFmt.format( cal.getTime() ) + "</span>";
		}
		else {
			return lblFmt.format( cal.getTime() );
		}
	}

	/**
	 * オブジェクトのキャッシュが時間切れかどうかを返します。
	 * キャッシュが時間切れ(無効)であれば、true を、有効であれば、
	 * false を返します。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
	 *
	 * @return  キャッシュが時間切れなら true
	 */
	@Override
	public boolean isTimeOver() {
		return System.currentTimeMillis() > maxCacheTime ;
	}
}
