/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.SelectionCellEditor;					// 6.2.2.0 (2015/03/27)
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionFactory;
import org.opengion.fukurou.util.StringFormat;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.TagBuffer;

import static org.opengion.fukurou.util.StringUtil.isNull;				// 6.1.1.0 (2015/01/17)

/**
 * カラムの編集パラメーターのＳＱＬ文の実行結果より、プルダウンメニューを作成して
 * 編集する場合に使用するエディタークラスです。
 *
 * 編集パラメータには、プルダウンメニューを作成するための、SQL文を記述します。
 * このSQL文は、select KEY,LABEL from xx ･･･ という構文で、KEY部分とLABEL部分が
 * 選択されます。
 * 第一カラムはキー、第二カラムはラベルでこの２つは必須です。第三カラムは短縮ラベル、
 * 第四カラムはグループ(optgroup)、第五カラムは色付け等に使うクラスです。
 * 短縮ラベルが設定されている場合、一覧でこのエディタが適用されると短縮ラベル表示を
 * した上でマウスオーバー時はツールチップで通常のラベルを表示します。
 *
 * 各カラムの値(value値)に、AAA:BBB:CCC:DDD という値を設定できます。これは、
 * $1,$2,$3,$4 に割り当てなおして、QUERYを実行します。また、$1 は、本来の値として、
 * メニューの初期値設定等に使用します。上記の例では、AAA が値で、それ以降は、
 * 引数になります。
 * 又、$Cには自分自身のカラム名を割り当てます。
 * この機能を使用すれば、動的メニューを行ごとに条件を変えて作成することが
 * 可能になります。
 * 例：select KEY,LABEL from xx where KUBUN='$2' and CDK='$3'
 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
 * 変数は、""(ゼロ文字列)として、扱われます。
 *
 * 編集パラメータに"SEQ"と記述することで正方向にしか選べないシークメニューを実現できます。
 * これにより、シーケンスにステータスを順に挙げていくような、プルダウンメニュー
 * を作成することが出来ます。(逆に戻れないメニュー)
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.2.3.0 (2003/06/06) 新規作成
 * @og.rev 3.4.0.1 (2003/09/03) ＤＢ検索をリアルタイムに変更。
 * @og.rev 4.3.6.0 (2009/04/01) eventColumn対応
 * @og.rev 5.4.3.6 (2012/01/19) コメント変更
 * @og.rev 6.2.2.0 (2015/03/27) SelectionCellEditor I/Fを追加
 * @og.group データ編集
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_DBMENU extends AbstractEditor implements SelectionCellEditor {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.2.0 (2015/03/27)" ;

	private final String query ;
	private final String dbid ;
	private final String lang ;				// 4.0.0 (2006/11/15)
	private final boolean addNoValue ;		// 3.5.5.7 (2004/05/10)
	private final boolean seqFlag ;			// 3.6.0.6 (2004/10/22)
	private final String useSLabel ;		// 5.5.1.0 (2012/04/03)
	private final String addKeyLabel ;		// 6.2.0.0 (2015/02/27) キー:ラベル形式

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.4.0.1 (2003/09/03) 初期値でQUERY文をキープする。
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 5.5.1.0 (2012/04/03) Slabel対応
	 * @og.rev 6.2.0.0 (2015/02/27) キー:ラベル形式で表示するかどうかを、指定できるようにします。
	 */
	public Editor_DBMENU() {
		// 4.3.4.4 (2009/01/01)
		query	= null;
		dbid	= null;
		lang	= null;				// 4.0.0 (2006/11/15)
		addNoValue	= false;		// 3.5.5.7 (2004/05/10)
		seqFlag		= false;		// 3.6.0.6 (2004/10/22)
		useSLabel	= "auto";		// 5.5.1.0 (2012/04/03)
		addKeyLabel = null;			// 6.2.0.0 (2015/02/27) キー:ラベル形式
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) name , attributes 属性を final にする。
	 * @og.rev 3.4.0.1 (2003/09/03) 継承の親元の変更に伴う実装の移動。
	 * @og.rev 3.5.5.7 (2004/05/10) addNoValue 属性を追加します。
	 * @og.rev 3.5.5.9 (2004/06/07) editorParam 属性が null の場合は、エラーとします。
	 * @og.rev 3.5.6.0 (2004/06/18) XHTMLTag の 内部配列 SELECT_KEY を隠蔽します。
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 5.5.1.0 (2012/04/03) Slabel対応
	 * @og.rev 6.0.4.0 (2014/11/28) optionAttributes は、コンストラクタで設定します。
	 * @og.rev 6.0.4.0 (2014/11/28) useMultiSelect は、selection ではなく、colomn から取得する。
	 * @og.rev 6.2.0.0 (2015/02/27) キー:ラベル形式で表示するかどうかを、指定できるようにします。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_DBMENU( final DBColumn clm ) {
	//	super( clm );
		name		= clm.getName();
		addNoValue	= clm.isAddNoValue() ;		// 3.5.5.7 (2004/05/10)
		query		= clm.getEditorParam();
		dbid		= clm.getDbid();
		lang		= clm.getLang();			// 4.0.0.0 (2006/11/15)
		seqFlag		= false;					// 3.6.0.6 (2004/10/22)
		useSLabel = clm.getUseSLabel() ;		// 5.5.1.0 (2012/04/03)
		addKeyLabel = clm.getAddKeyLabel();		// 6.2.0.0 (2015/02/27) キー:ラベル形式

		// 3.5.5.9 (2004/06/07)
		if( query == null || query.isEmpty() ) {
			final String errMsg = "DBMENU Editor では、編集パラメータは必須です。"
					+ " name=[" + name + "]" + CR ;
			throw new HybsSystemException( errMsg );
		}

		final String disabled = clm.isWritable() ? null : "disabled" ;

		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		attributes = new Attributes()
					.set( "disabled"	,disabled )
					.set( clm.getEditorAttributes() );				// #addAttributes( Attributes ) の代替え

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		tagBuffer.add( XHTMLTag.selectAttri( attributes ) )
					.add( attributes.get( "optionAttributes" ) )						// 6.0.4.0 (2014/11/28)
					.add( "onkeydown" , "setKeySelect(this);" , clm.useMultiSelect() );	// 6.0.4.0 (2014/11/28)

	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_DBMENU( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。
	 * 又、$Cには自分自身のカラム名を割り当てます。
	 *
	 * @og.rev 3.4.0.1 (2003/09/03) リアルタイムで値を作成する様に変更。
	 * @og.rev 3.5.5.5 (2004/04/23) 新規に Attributes オブジェクトを作成する方式を止めます。
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 5.1.3.0 (2010/02/01) 一覧表示のみで、ツールチップ表示を行う。
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 5.5.1.0 (2012/04/03) Slabel対応
	 *
	 * @param	value 入力値
	 *
	 * @return	データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		final boolean useSlbl = "true".equalsIgnoreCase( useSLabel );		// 5.5.1.0 (2012/04/03)

		// select タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return getOption(
				new TagBuffer( "select" )
					.add( "name"	, name )									// 4.3.6.0 (2009/04/01)
					.add( "id"		, name , isNull( attributes.get( "id" ) ) )	// 4.3.7.2 (2009/06/15)
					.add( tagBuffer.makeTag() )
				, value
				, useSlbl
			).makeTag() ;

	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。
	 * 又、$Cには自分自身のカラム名を割り当てます。
	 *
	 * @og.rev 2.0.0.3 (2002/09/26) optionAttributes 属性に "$i" を使うとその行数に置き換る機能を追加。
	 * @og.rev 3.4.0.1 (2003/09/03) リアルタイムで値を作成する様に変更。
	 * @og.rev 3.5.5.0 (2004/03/12) 名前と行番号の区切り記号("__")を、HybsSystem.JOINT_STRING  に変更。
	 * @og.rev 3.5.5.5 (2004/04/23) 新規に Attributes オブジェクトを作成する方式を止めます。
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 5.1.3.0 (2010/02/01) 一覧表示のみで、ツールチップ表示を行う。
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 5.5.1.0 (2012/04/03) Slabel対応
	 *
	 * @param	row   行番号
	 * @param	value 入力値
	 *
	 * @return	データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		final boolean useSlbl = "auto".equalsIgnoreCase( useSLabel ) || "true".equalsIgnoreCase( useSLabel );		// 5.5.1.0 (2012/04/03)
		final String newName = name + HybsSystem.JOINT_STRING + row; // 4.3.6.0 (2009/04/01)

		// select タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return getOption(
				new TagBuffer( "select" )
					.add( "name"	, newName )										// 4.3.6.0 (2009/04/01)
					.add( "id"		, newName , isNull( attributes.get( "id" ) ) )	// 4.3.7.2 (2009/06/15)
					.add( tagBuffer.makeTag() )
				, value
				, useSlbl
			).makeTag( row,value ) ;		// ※ 元は、makeTag() でしたが、バグ？ の可能性がある。

	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)をTagBuffer に反映します。
	 * このオプションは、引数の値を初期値とするオプションタグ作成し、TagBuffer
	 * に値を設定して返します。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。
	 * 又、$Cには自分自身のカラム名を割り当てます。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) getOption( String value )の廃止を受けて、新規作成
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 4.0.0.0 (2006/11/15) SelectionFactory に lang 属性を追加します。
	 * @og.rev 4.3.4.0 (2008/12/01) $Cのカラム名置換えを追加
	 * @og.rev 5.1.3.0 (2010/02/01) 一覧表示のみで、ツールチップ表示を行う。
	 * @og.rev 6.0.4.0 (2014/11/28) useMultiSelect は、selection ではなく、colomn から取得する。
	 * @og.rev 6.2.0.0 (2015/02/27) キー:ラベル形式で表示するかどうかを、指定できるようにします。
	 *
	 * @param	buf    タグ文字列のバッファー
	 * @param	value  選択されている値
	 * @param   useSlbl ラベル(短)をベースとしたオプション表示を行うかどうか。
	 *
	 * @return	オプションタグ
	 * @og.rtnNotNull
	 */
	private TagBuffer getOption( final TagBuffer buf,final String value,final boolean useSlbl ) {

		// StringFormat format = new StringFormat( query,value);
		final StringFormat format = new StringFormat( query, value, name ); // 4.3.4.0 (2008/12/01)
		final String newQuery = format.format();
		final String newValue = format.getValue();

		// 6.2.0.0 (2015/02/27) キー:ラベル形式
		final Selection selection = SelectionFactory.newDBSelection( newQuery, dbid, lang, addKeyLabel );
		// 6.0.4.0 (2014/11/28) useMultiSelect は、selection ではなく、colomn から取得する。

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return buf.addBody( Selection.NO_VALUE_OPTION , addNoValue )			// 5.5.1.0 (2012/04/03)
				  .addBody( selection.getOption( newValue, seqFlag, useSlbl ) );

	}
}
