/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.PrintWriter;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * 【廃止予定】加工なしCSV形式ダブルクォートファイル(CSV)形式書き込みクラスです。
 * このクラスは、廃止予定です。代わりに、Data2 の セパレータをカンマに
 * 設定して使用してください。
 *
 * DefaultTableWriter を継承していますので，ラベル，名前，データの出力部のみ
 * オーバーライドして，可変長CSV形式ファイルの出力機能を実現しています。
 *
 * このクラスは、文字型のデータ出力時の、先頭が０の場合のシングルコーテーションを
 * 出さないように変更したクラスです。(専用特殊クラス)
 *
 * @og.formSample
 *
 * @og.rev 3.1.9.1 (2003/05/20) 新規作成
 * @og.group ファイル出力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableWriter_CSV2 extends TableWriter_Default {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.0.4.0 (2014/11/28)" ;

	/**
	 * DBTableModel から データを作成して,PrintWriter に書き出します。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 3.5.4.3 (2004/01/05) 引数に PrintWriter を受け取るように変更します。
	 *
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	public void writeDBTable( final PrintWriter writer )  {
		super.setSeparator( CSV_SEPARATOR );	// 3.5.6.0 (2004/06/18)
		super.writeDBTable( writer );
	}

	/**
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 * このクラスでは，データを ダブルコーテーション(")で囲みます。
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 *
	 * @og.rev 2.0.0.1 (2002/09/20) 先頭が0 でかつ数字タイプ(S9)でない場合に ' を出力するように修正
	 * @og.rev 2.0.0.5 (2002/09/30) 先頭が0 でかつ数字タイプ(S9 or R)でない場合に ' を出力するように修正。
	 * @og.rev 2.3.1.2 (2003/01/28) データ出力時に、改行が余分に出される箇所を修正。
	 * @og.rev 3.1.0.0 (2003/03/20) DBColumn から、getDbType() キーを直接取り出す
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 3.7.0.2 (2005/02/14) 行番号情報を、出力する(true)/しない(false)を指定
	 * @og.rev 3.8.0.1 (2005/06/17) DBTypeが NVAR の場合は、元のUnicodeに戻します。
	 * @og.rev 5.1.6.0 (2010/05/01) DbType の初期値(dbType)を利用する。
	 * @og.rev 5.2.1.0 (2010/10/01) useRenderer 対応
	 * @og.rev 6.0.1.2 (2014/08/08) カラム飛ばしできる機能を追加
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeData( final DBTableModel table,final PrintWriter writer ) {
		final int numberOfRows =	table.getRowCount();
		final boolean useNumber = isUseNumber();
		final boolean useRenderer = isUseRenderer();	// 5.2.1.0 (2010/10/01)

		for( int row=0; row<numberOfRows; row++ ) {
			if( useNumber ) {
				writer.print( quotation( String.valueOf( row+1 ) ) );
				writer.print( CSV_SEPARATOR );
			}

			for( int i=0; i<numberOfColumns; i++ ) {
				if( i != 0 ) { writer.print( CSV_SEPARATOR ); }
				final int clm = clmNo[i];
				if( clm < 0 ) { continue; }			// 6.0.1.2 (2014/08/08) カラム飛ばし

				String val = table.getValue(row,clm);
				if( dbType[i] == NVAR ) {
					val = StringUtil.getReplaceEscape( val );
				}
				// 5.2.1.0 (2010/10/01) useRenderer 対応
				else if( useRenderer ) {
					// 6.0.4.0 (2014/11/28) データ出力用のレンデラー
					val = dbColumn[clm].getWriteValue( val );
				}

				// 開始日などの 00000000 を文字列タイプで渡す
				writer.print( quotation( val ) );
			}
			writer.println();
		}
	}

	/**
	 * データを書き込む場合の,区切り文字をセットします。
	 * このクラスでは，CSV 固定の為,区切り文字のセットは無効になります。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 *
	 * @param	sprt 区切り文字
	 */
	@Override
	public void setSeparator( final String sprt ) {
		super.setSeparator( CSV_SEPARATOR ) ;	// 3.5.6.0 (2004/06/18)
	}
}
