/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.query;

import org.opengion.hayabusa.db.AbstractQuery;
import org.opengion.hayabusa.db.DBErrMsg;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.Closer;

import java.sql.Types;
import java.sql.Connection;
import java.sql.CallableStatement;
import java.sql.SQLException;
import java.sql.Array;								// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ) 対応。oracle.sql.ARRAY の置き換え
import oracle.jdbc.OracleConnection;				// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ) 対応

import java.util.Map;

/**
 * PL/SQL にエントリ情報を配列渡しする、Query 実行クラスです。
 *
 * java.sql.CallableStatement を用いて、データベース登録処理を行います。
 * 引数に、キーと値をセットで配列指定で渡すことが出来，エラー時には、
 * DBErrMsg オブジェクトにエラー情報を格納して返すことが可能です。
 * 内部変数の受け渡しのデフォルト実装は、AbstractQuery クラスを継承している
 * ため,ここでは、execute() メソッドを実装しています。
 * このクラスでは、ステートメント文を execute() する事により,データベースを
 * 検索した結果を DBTableModel に割り当てます。
 *
 * @og.formSample
 * 例：
 *     第一引数、第二引数は、結果(KEKKA)とエラーメッセージ配列を返します。
 *     キーエントリでは、エントリ(リクエスト情報)のキーと値をセットで
 *     引数の配列に設定します。
 *     キーを元に、値を利用する場合に使用します。
 *     下記の例は、動的カラムを実現しているPL/SQLの例です。
 *
 * &lt;og:entryQuery
 *     command    = &quot;NEW&quot;
 *     queryType  = &quot;JDBCKeyEntry&quot; &gt;
 *         { call DYNAMIC_PKG.DYNAMIC_TEST( ?,?,?,? ) }
 * &lt;/og:entryQuery&gt;
 *
 *    CREATE OR REPLACE PACKAGE DYNAMIC_PKG AS
 *        PROCEDURE DYNAMIC_TEST(
 *            P_KEKKA       OUT    NUMBER,
 *            P_ERRMSGS     OUT    ERR_MSG_ARRAY,
 *            P_KEYS        IN     ARG_ARRAY,
 *            P_ARGS        IN     ARG_ARRAY );
 *    END;
 *
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Query_JDBCKeyEntry extends AbstractQuery {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.0.0.0 (2014/04/11)" ;

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、PreparedQuery で使用する引数を配列でセットするものです。
	 * select * from emp where deptno = ? and job = ? などの PreparedQuery の
	 * ? 部分の引数を
	 * 順番にセットしていきます。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 3.3.3.1 (2003/07/18) ＤＢ登録時の後ろスペースを削除する。
	 * @og.rev 3.5.2.0 (2003/10/20) 内部オブジェクトタイプ名を システムパラメータ で定義します。
	 * @og.rev 3.5.6.0 (2004/06/18) nullに対する無駄な比較を削除します。
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 * @og.rev 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応
	 *
	 * @param   keys オブジェクトのキー配列
	 * @param   args オブジェクトの引数配列(可変長引数)
	 */
	@Override
	public void execute( final String[] keys, final String... args ) {			// 6.1.1.0 (2015/01/17) refactoring
		CallableStatement callStmt = null ;
		try {
			final Connection conn = getConnection();
			callStmt  = getConnection().prepareCall( getStatement() );
			callStmt.setQueryTimeout( DB_MAX_QUERY_TIMEOUT );
			final Map<String,Class<?>> map = conn.getTypeMap();
			map.put( ERR_MSG,DBErrMsg.class );	// 4.0.0 (2005/01/31)

			// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応 http://docs.oracle.com/cd/E28389_01/web.1111/b60995/thirdparty.htm
			final Array newArray_key = ((OracleConnection)conn).createOracleArray( ARG_ARRAY, keys );		// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応

			// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応 http://docs.oracle.com/cd/E28389_01/web.1111/b60995/thirdparty.htm
			final Array newArray_val = ((OracleConnection)conn).createOracleArray( ARG_ARRAY, StringUtil.rTrims( args ));		// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応

			callStmt.registerOutParameter(1, Types.INTEGER);
			callStmt.registerOutParameter(2, Types.ARRAY,ERR_MSG_ARRAY);				// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応
			callStmt.setArray( 3,newArray_key );										// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応
			callStmt.setArray( 4,newArray_val );										// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応

			callStmt.execute();

			final int rtnCode = callStmt.getInt(1);
			setErrorCode( rtnCode );
			if( rtnCode > ErrorMessage.OK ) {		// 正常以外の場合
				final Array rtn3 = callStmt.getArray(2);										// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応
				final Object[] rtnval3 = (Object[])rtn3.getArray();
				final ErrorMessage errMessage = new ErrorMessage( "Query_JDBCKeyEntry Error!!" );
				for( int i=0; i<rtnval3.length; i++ ) {
					final DBErrMsg er = (DBErrMsg)rtnval3[i];
					if( er == null ) { break; }
					errMessage.addMessage( er.getErrMsg() );
				}
				setErrorMessage( errMessage );
			}
		}
		catch (SQLException ex) {
			setErrorCode( ErrorMessage.EXCEPTION );
			final String errMsg = ex.getMessage() + ":" + ex.getSQLState() + CR
						+ getStatement() + CR ;
			rollback();
			realClose();
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		finally {
			Closer.stmtClose( callStmt );
		}
	}
}
