/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;

// import java.util.Map;

/**
 * TableFilter_MATCHES は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、指定のカラムに対して、正規表現でマッチングを行い、true のデータだけに絞ります。
 * つまり、マッチしなければ、そのデータを削除します。
 * 新しくテーブルを作成するのではなく、既存のテーブルのデータを物理削除しますので、ご注意ください。
 *
 * このフィルターでは、KEYSに指定するのは、カラム名で、自由に指定できます。
 * カラムが存在しない場合は、無視（削除しない）します。
 * エラーチェックは行いません。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 *
 * @og.formSample
 * ●形式：
 *      ① &lt;og:tableFilter classId="MATCHES" keys="CLM1,CLM2..." vals="正規表現1,正規表現2..." /&gt;
 *
 *      ② &lt;og:tableFilter classId="MATCHES" &gt;
 *               {
 *                   CLM1   : 正規表現1   ;
 *                   CLM2   : 正規表現2   ;
 *                   ･･･    : ･･･   ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 6.7.9.1 (2017/05/19) 新規追加
 *
 * @version  6.7  2017/05/19
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.8,
 */
public class TableFilter_MATCHES extends AbstractTableFilter {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.7.9.1 (2017/05/19)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.7.9.1 (2017/05/19) 新規追加
	 */
	public TableFilter_MATCHES() {
		super();
		// initSet(String,String) を指定しない場合は、KEYSに制限を行わない。
	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 6.7.9.1 (2017/05/19) 新規追加
	 *
	 * @return 処理結果のDBTableModel
	 */
	public DBTableModel execute() {
		final DBTableModel table = getDBTableModel();

		final String[] keys = getKeys();			// 判定対象のカラム名の配列
		final int      len  = keys.length;

		final String[] regex = new String[len];		// 判定に使う正規表現(valsの値)
		final int[]    clmNo = new int[len];		// カラムの番号

		for( int i=0; i<len; i++ ) {
			clmNo[i] = table.getColumnNo( keys[i],false );	// カラムが存在しなければ、-1
			regex[i] = getValue( keys[i] );
		}

		final int rowCnt = table.getRowCount();

		String[] data = null;	// エラー時に表示するため。
		// KBCLMに変更がなければ、レコードを削除します。よって、逆順にチェックします。
		for( int row=rowCnt-1; row>=0; row-- ) {
			try {
				data = table.getValues( row );

				for( int j=0; j<len; j++ ) {
					if( clmNo[j] >= 0 && ! data[clmNo[j]].matches( regex[j] ) ) {
						table.removeValue( row );
						break;							// 一つでもマッチしなければ、削除して抜ける。
					}
				}
			}
			catch( final RuntimeException ex ) {
				// 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
				makeErrorMessage( "TableFilter_MATCHES Error",ErrorMessage.NG )
					.addMessage( row+1,ErrorMessage.NG,"MATCHES"
						, "keys=[" + StringUtil.array2csv( keys ) + "]"
						, "vals=[" + StringUtil.array2csv( regex ) + "]"
						, StringUtil.array2csv( data )
					)
					.addMessage( ex );
			}
		}

		return table;
	}
}
