/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.system.HybsConst ;						// 6.1.0.0 (2014/12/26)
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.db.Transaction;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.ResourceManager;

import java.util.Locale;
import java.util.Map;
import java.util.LinkedHashMap ;
import java.util.concurrent.ConcurrentMap;							// 6.4.3.3 (2016/03/04)
import java.util.concurrent.ConcurrentHashMap;						// 6.4.3.1 (2016/02/12) refactoring
import java.util.Collections;										// 6.4.3.1 (2016/02/12) refactoring

/**
 * AbstractTableFilter は、TableUpda インターフェースを継承した、DBTableModel 処理用の
 * Abstract実装クラスです。
 *
 * @og.rev 5.5.2.6 (2012/05/25) protected変数をprivateに変更。インターフェースにメソッド追加
 * @og.rev 6.4.1.1 (2016/01/16) keysMap を、サブクラスから設定させるように変更。
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public abstract class AbstractTableFilter implements TableFilter {
	/** システムの改行コードを設定します。*/
	protected static final String CR		 = HybsConst.CR;			// 6.1.0.0 (2014/12/26) refactoring
	/** StringBilderなどの初期値を設定します。	{@value} */
	protected static final int BUFFER_MIDDLE = HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

	// 5.5.2.6 (2012/05/25) protected変数をprivateに変更。インターフェースにメソッド追加
	private DBTableModel	table		;
	private String			modifyType	;
	private int[]			rowNo		;
	private boolean			useDebug	;		// 6.0.2.5 (2014/10/31) refactoring メソッドと同じなので名称変更
	private Transaction		tran		;		// 5.1.9.0 (2010/08/01) 追加
	private String			sql			;		// 4.2.4.0 (2008/06/23)
	private String			dbid		;		// 4.2.4.0 (2008/06/23)
	private ResourceManager resource	;		// 4.3.7.4 (2009/07/01)

	private int 			errCode		= ErrorMessage.OK;
	private ErrorMessage	errMessage	;

	/** 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。  */
	private final ConcurrentMap<String,String> keyValMap = new ConcurrentHashMap<>();

	// 5.6.6.0 (2013/07/05) keys の整合性チェックを行います。
	// 6.4.1.1 (2016/01/16) keysMap を、サブクラスから設定させるように変更
	/** 6.4.3.1 (2016/02/12) Collections.synchronizedMap で同期処理を行います。  */
	private final Map<String,String> keysMap = Collections.synchronizedMap( new LinkedHashMap<>() ) ;

	/** 各種定数 */
	// 6.0.2.3 (2014/10/10) plugin.table.TableFilter_XXXX から移動 
	protected static final String XML_START_TAG	= "<?xml version='1.0' encoding='UTF-8'?>" + CR + "<ROWSET tableName='xxx'>";
	protected static final String XML_END_TAG	= "</ROWSET>";
	protected static final String EXEC_START_TAG= "<EXEC_SQL>";
	protected static final String EXEC_END_TAG	= "</EXEC_SQL>";

	/** XML形式かどうか  */
	// 6.0.2.3 (2014/10/10) isXml で、CR + EXEC_END_TAG のキャッシュを作成します。
	protected boolean	isXml		;		// 6.0.2.3 (2014/10/10)
	protected String 	execEndTag	;		// 6.0.2.3 (2014/10/10)

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	protected AbstractTableFilter() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * keys の整合性チェックを行うための初期設定を行います。
	 * サブクラスのコンストラクタ内で、設定するようにしてください。
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) keys の整合性チェック対応
	 * @og.rev 6.4.3.1 (2016/02/12) ConcurrentMap 系は、key,val ともに not null 制限です。
	 *
	 * @param	key  整合性チェックを行うための keysMap に設定するキー
	 * @param	cmnt 整合性チェックを行うための キー の説明
	 */
	protected void initSet( final String key , final String cmnt ) {
		if( key != null && cmnt != null ) {
			keysMap.put( key , cmnt );
		}
	}

	/**
	 * DBTableModel をセットします。
	 *
	 * @param	table DBTableModelオブジェクト
	 */
	public void setDBTableModel( final DBTableModel table ) {
		this.table = table;
	}

	/**
	 * DBTableModel を取得します。
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 *
	 * @return	内部のDBTableModel
	 */
	public DBTableModel getDBTableModel() {
		return table;
	}

	/**
	 * データ処理の方法(A:追加 C:更新 D:削除)を指定します。
	 *
	 * 通常は、DBTableModel に自動設定されている modifyType を元に、データ処理方法を
	 * 選別します。(A:追加 C:更新 D:削除)
	 * この場合、行単位で modifyType の値を取得して判別する必要がありますが、一般には
	 * 処理対象は、全件おなじ modifyType である可能性が高いです。
	 * また、selectedAll などで強制的に全件処理対象とする場合は、modifyType に値が
	 * 設定さていません。その様な場合に外部より modifyType を指定します。
	 * 初期値は、自動判定 です。
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) 廃止
	 *
	 * @param  type データ処理の方法(A:追加 C:更新 D:削除)
	 */
	public void setModifyType( final String type ) {
		modifyType = type;
	}

	/**
	 * データ処理の方法(A:追加 C:更新 D:削除)を取得します。
	 *
	 * 初期値は、自動判定 です。
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 *
	 * @return  データ処理の方法(A:追加 C:更新 D:削除)
	 */
	public String getModifyType() {
		return modifyType ;
	}

	/**
	 * キーと値のペアの変数配列を受け取ります。
	 *
	 * ここでは、この方式以外に、パラメーターMapを受け取る方法もあります。
	 * この受け取る時に、キーを大文字化します。TableFilter の keys は、
	 * 大文字のみで定義しておくことで、HTMLやWindows世代の曖昧な表記方法に
	 * 対応しています。(unixやxmlのような厳格な方が好きですけど)
	 *
	 * keys,vals とパラメーターMapを同時に指定した場合は、両方とも有効です。
	 * ただし、キーが重複した場合は、不定と考えてください。
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを行います。
	 *
	 * @param   keys キー配列
	 * @param   vals 値配列
	 * @see		#setParamMap( ConcurrentMap )
	 */
	public void setKeysVals( final String[] keys,final String[] vals ) {
		if( keys != null && vals != null ) {
			for( int i=0; i<keys.length; i++ ) {
				// 5.6.6.0 (2013/07/05) 共通のセッターメソッド経由で登録します。
				setKeyVal( keys[i],vals[i] );
			}
		}
	}

	/**
	 * キーと値のペアを受け取り、内部の keyValMap マップに追加します。
	 *
	 * キーか値のどちらかが null の場合は、何もしません。つまり、val に
	 * null をセットすることはできません。
	 *
	 * このメソッドは、setKeysVals( String[] ,String[] ) メソッドと、
	 * setParamMap( Map<String,String> ) メソッドの両方から、使用します。
	 * 処理を行うに当たり、下記の処理を行います。
	 * １．キーを大文字化します。
	 * ２．各クラスの keys と整合性チェックを行います。
	 *
	 * ただし、setKeysVals と setParamMap の登録順は、不定と考えてください。
	 * 両方に同じキーを指定すると、どちらの値がセットされたかは、不定です。
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを行います。
	 * @og.rev 6.4.3.4 (2016/03/12) Map#forEach で対応する。
	 * @og.rev 6.7.9.1 (2017/05/19) keysMap が、空の場合も、keyValMap に登録する。(initSet 未登録時)
	 * @og.rev 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
	 *
	 * @param   key キー文字列(null の場合は、処理しない)
	 * @param   val 値文字列(null の場合は、処理しない)
	 * @see		#setKeysVals( String[] ,String[] )
	 * @see		#setParamMap( ConcurrentMap )
	 */
	private void setKeyVal( final String key,final String val ) {
		// key か val かどちらかが null の場合は、処理を行わない。
		if( key == null || val == null ) { return; }

		final String upKey = key.toUpperCase(Locale.JAPAN);

		// 6.7.9.1 (2017/05/19) keysMap が、空の場合も、keyValMap に登録する。(initSet 未登録時)
		if(  keysMap.isEmpty() || keysMap.containsKey( upKey ) ) {		// keysMap は、各サブクラスで定義
			keyValMap.put( upKey,val );
		}
		else {
//			final String BR = "<br />" + CR ;
			final String BR = "<br>" + CR ;				// 7.0.1.0 (2018/10/15)
			final StringBuilder errMsg = new StringBuilder( BUFFER_MIDDLE );
			// 6.0.2.5 (2014/10/31) char を append する。
			errMsg.append( BR )
				  .append( "指定のキーは、この tableFilter では、使用できません。" ).append( BR )
				  .append( "  class=[" ).append( getClass().getName() ).append( ']' ).append( BR )
				  .append( "  key  =[" ).append( key				  ).append( ']' ).append( BR )
				  .append( "  ======== usage keys ======== " ).append( BR ) ;
			// 6.4.3.4 (2016/03/12) Map#forEach で対応する。
			keysMap.forEach( (k,v) -> { errMsg.append( ' ' ).append( k ).append( ':' ).append( v ).append( BR ); } );
			errMsg.append( "  ============================ " ).append( BR );

			throw new HybsSystemException( errMsg.toString() );
		}
	}

	/**
	 * 選択された行番号の配列をセットします。
	 *
	 * 表示データの HybsSystem.ROW_SELECTED_KEY を元に、選ばれた 行を
	 * 処理の対象とします。
	 *
	 * @param   rowNoTmp 行番号配列(可変長引数)
	 */
	public void setParameterRows( final int... rowNoTmp ) {
		if( rowNoTmp != null && rowNoTmp.length > 0 ) {		// 6.1.1.0 (2015/01/17) 可変長引数でもnullは来る。
			final int size = rowNoTmp.length ;
			rowNo = new int[size];
			System.arraycopy( rowNoTmp,0,rowNo,0,size );
		}
	}

	/**
	 * 選択された行番号の配列を取得します。
	 *
	 * 表示データの HybsSystem.ROW_SEL_KEY を元に、選ばれた 行を
	 * 処理の対象とします。
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 * @og.rev 6.0.2.5 (2014/10/31) null ではなく、サイズ０の配列を返すように変更。
	 *
	 * @return   行番号の配列(選ばれていない場合は、サイズ０の配列を返す)
	 * @og.rtnNotNull
	 */
	public int[] getParameterRows() {
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		return rowNo == null ? new int[0] : rowNo.clone() ;
	}

	/**
	 * アクセスログ取得の為,Transactionオブジェクトを設定します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応(新規追加)
	 *
	 * @param   tran Transactionオブジェクト
	 */
	public void setTransaction( final Transaction tran ) {
		this.tran = tran;
	}

	/**
	 * アクセスログ取得の為,Transactionオブジェクトを取得します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応(新規追加)
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 *
	 * @return   Transactionオブジェクト
	 */
	public Transaction getTransaction() {
		return tran;
	}

	/**
	 * DBIDを指定します。
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 *
	 * @param dbid 接続先ID
	 */
	public void setDbid( final String dbid ) {
		this.dbid = dbid;
	}

	/**
	 * DBIDを取得します。
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 *
	 * @return DBID(接続先情報)
	 */
	public String getDbid() {
		return dbid;
	}

	/**
	 * ボディー部分のSQLを指定します。
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 *
	 * @param sql ボディー部分のSQL
	 */
	public void setSql( final String sql ) {
		this.sql = sql;
	}

	/**
	 * ボディー部分のSQLを取得します。
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 *
	 * @return ボディー部分のSQL
	 */
	public String getSql() {
		return sql;
	}

	/**
	 * パラメーターMapを指定します。
	 *
	 * keys,vals と パラメーターMapを同時に指定した場合は、両方とも有効です。
	 * ただし、キーが重複した場合は、不定と考えてください。
	 *
	 * この受け取る時に、キーを大文字化します。TableFilter の keys は、
	 * 大文字のみで定義しておくことで、HTMLやWindows世代の曖昧な表記方法に
	 * 対応しています。(unixやxmlのような厳格な方が好きですけど)
	 *
	 * @og.rev 5.6.5.2 (2013/06/21) 新規追加
	 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを行います。
	 * @og.rev 6.4.3.3 (2016/03/04) ConcurrentHashMap を受け取ることを明確にするため、I/FをConcurrentMapに変更します。
	 *
	 * @param paramMap パラメーターMap
	 * @see		#setKeysVals( String[] ,String[] )
	 */
	public void setParamMap( final ConcurrentMap<String,String> paramMap ) {
		// 6.4.3.3 (2016/03/04) Map#forEach に変更
		if( paramMap != null ) {
			paramMap.forEach( (k,v) -> setKeyVal( k,v ) );
		}
	}

	/**
	 * リソースオブジェクトを指定します。
	 *
	 * @og.rev 4.3.7.4 (2009/07/01) 新規追加
	 *
	 * @param resource リソースオブジェクト
	 */
	public void setResource( final ResourceManager resource ) {
		this.resource = resource;
	}

	/**
	 * リソースオブジェクトを取得します。
	 *
	 * @og.rev 4.3.7.4 (2009/07/01) 新規追加
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 *
	 * @return リソースオブジェクト
	 */
	public ResourceManager getResource() {
		return resource;
	}

	/**
	 * デバッグ情報を出力するかどうか[true:する/false:しない]を指定します。
	 * true でデバッグ情報を表示します。
	 *
	 * @param   flag  デバッグ出力するか [true:する/false:しない]
	 */
	public void setDebug( final boolean flag ) {
		useDebug = flag;	// 6.0.2.5 (2014/10/31) refactoring メソッドと同じなので名称変更
	}

	/**
	 * デバッグ情報を出力するかどうか[true:する/false:しない]を取得します。
	 * true でデバッグ情報を表示します。
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 *
	 * @return  デバッグ出力 [true:する/false:しない]
	 */
	public boolean isDebug() {
		return useDebug ;	// 6.0.2.5 (2014/10/31) refactoring メソッドと同じなので名称変更
	}

	/**
	 * エラーコード を取得します。
	 * エラーコード は、ErrorMessage クラスで規定されているコードです。
	 *
	 * @return   エラーコード
	 */
	public int getErrorCode() {
		return errCode;
	}

	/**
	 * エラーメッセージオブジェクト を取得します。
	 *
	 * @return   エラーメッセージオブジェクト
	 */
	public ErrorMessage getErrorMessage() {
		return errMessage;
	}

	/**
	 * タイトルとエラーコードを指定して、エラーメッセージオブジェクト を作成します。
	 * すでに、作成済みの場合は、作成済みのオブジェクトを、まだ、未作成の場合は、
	 * 新規に作成します。
	 *
	 * @param	title	タイトル
	 * @param	code	エラーコード
	 *
	 * @return	エラーメッセージオブジェクト
	 */
	protected ErrorMessage makeErrorMessage( final String title,final int code ) {
		if( errMessage == null ) {
			errMessage = new ErrorMessage( title );
		}
		if( errCode < code ) { errCode = code; }
		return errMessage;
	}

	/**
	 *  カラム名配列(String[])より、対応するカラムNo配列(int[])を作成します。
	 *
	 * @param	nameArray カラム名配列
	 *
	 * @return	カラムNo配列(可変長引数)
	 */
	protected int[] getTableColumnNo( final String... nameArray ) {
		int[] clmNo = new int[nameArray.length];
		for( int i=0; i<clmNo.length; i++ ) {
			clmNo[i] = table.getColumnNo( nameArray[i] );
		}
		return clmNo;
	}

	/**
	 * 設定されたパラメータキーに対する値を取得します。
	 * 引数、および、パラメータが null の場合は、 null を返します。
	 *
	 * @og.rev 6.4.3.3 (2016/03/04) ConcurrentHashMap の not null制限のチェック追加
	 *
	 * @param	key	パラメータキー
	 *
	 * @return	パラメータ値
	 */
	protected String getValue( final String key ) {
		return key == null ? null : keyValMap.get( key );
	}

	/**
	 * keyValMapに持っているキーの配列を取得します。
	 * これは、サブクラスで、initSet(String,String) を行わない場合、keys には
	 * 値を自由に設定できます。
	 * その値を取り出すためです。
	 *
	 * @og.rev 6.7.9.1 (2017/05/19) 新規追加
	 *
	 * @return	キー値の配列(keyValMapに持っているキー)
	 */
	protected String[] getKeys() {
		return keyValMap.keySet().toArray( new String[keyValMap.size()] );
	}
}
