#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include "ppport.h"

#include "const-c.inc"

#define _UNICODE
#define UNICODE

#include <windows.h>
#include <tchar.h>
#include <time.h>

#include <dokan.h>

HANDLE PerlReadyEvent;
CRITICAL_SECTION ReadySection;
CRITICAL_SECTION ThreadCountSection;
int ThreadCount = 0;
BOOL Mounted = FALSE;

HANDLE PerlStartOKEvent;

typedef int (Func_t)(SV* self, void* pParam);

Func_t *pCurrentFunc;
void* pCurrentParam;
int* pReturn;
HANDLE PerlEndEvent;

static SV* create_dokan_options(DOKAN_OPTIONS *pDokanOptions)
{
    SV* options;
    int count;
    char drive[2];

    dSP ;

    drive[0] = (char)pDokanOptions->DriveLetter;
    drive[1] = 0;

    ENTER ;
    SAVETMPS ;

    PUSHMARK(SP) ;
    XPUSHs(sv_2mortal(newSVpv("Win32::Dokan::DokanOptions", 0)));
    PUTBACK ;
    count = call_method("new", G_SCALAR);
    SPAGAIN;

    if (count != 1) croak("create_file must return scalar");
    options = SvREFCNT_inc(POPs);
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(options);
    XPUSHs(sv_2mortal(newSVpv(drive, 1)));
    PUTBACK ;
    call_method("drive_letter", G_DISCARD);

    PUSHMARK(SP) ;
    XPUSHs(options);
    XPUSHs(sv_2mortal(newSVuv((unsigned)pDokanOptions->ThreadCount)));
    PUTBACK ;
    call_method("thread_count", G_DISCARD);

    PUSHMARK(SP) ;
    XPUSHs(options);
    XPUSHs(sv_2mortal(newSVuv((unsigned)pDokanOptions->DebugMode)));
    PUTBACK ;
    call_method("debug_mode", G_DISCARD);

    PUSHMARK(SP) ;
    XPUSHs(options);
    XPUSHs(sv_2mortal(newSVuv((unsigned)pDokanOptions->UseStdErr)));
    PUTBACK ;
    call_method("use_std_err", G_DISCARD);

    PUSHMARK(SP) ;
    XPUSHs(options);
    XPUSHs(sv_2mortal(newSVuv((unsigned)pDokanOptions->UseAltStream)));
    PUTBACK ;
    call_method("use_alt_stream", G_DISCARD);

    PUSHMARK(SP) ;
    XPUSHs(options);
    XPUSHs(sv_2mortal(newSVuv((unsigned)pDokanOptions->UseKeepAlive)));
    PUTBACK ;
    call_method("use_keep_alive", G_DISCARD);

    PUSHMARK(SP) ;
    XPUSHs(options);
    XPUSHs(sv_2mortal(newSVuv((unsigned)pDokanOptions->GlobalContext)));
    PUTBACK ;
    call_method("global_context", G_DISCARD);

    FREETMPS ;
    LEAVE ;

    return options;
}

static DOKAN_OPTIONS* restore_dokan_options(DOKAN_OPTIONS *pOptions,
					    SV *opt)
{
    int count;
    dSP ;

    memset(pOptions, 0, sizeof(*pOptions));

    ENTER ;
    SAVETMPS ;

    PUSHMARK(SP) ;
    XPUSHs(opt);
    PUTBACK ;
    count = call_method("drive_letter", G_SCALAR);
    SPAGAIN;

    if (count == 1) {
	SV* value = POPs;
        if (SvPOK(value)) {
	    int len;
	    const char* p = SvPV(value, len);
	    pOptions->DriveLetter = (_TCHAR)*p;
	}
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(opt);
    PUTBACK ;
    count = call_method("thread_count", G_SCALAR);
    SPAGAIN;
    if (count == 1) {
	SV* value = POPs;
	pOptions->ThreadCount = (USHORT)(SvIOK(value) ? SvIV(value) : 0);
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(opt);
    PUTBACK ;
    count = call_method("debug_mode", G_SCALAR);
    SPAGAIN;
    if (count == 1) {
	SV* value = POPs;
	pOptions->DebugMode = (UCHAR)(SvIOK(value) ? SvIV(value) : 0);
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(opt);
    PUTBACK ;
    count = call_method("use_std_err", G_SCALAR);
    SPAGAIN;
    if (count == 1) {
	SV* value = POPs;
	pOptions->UseStdErr = (UCHAR)(SvIOK(value) ? SvIV(value) : 0);
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(opt);
    PUTBACK ;
    count = call_method("use_alt_stream", G_SCALAR);
    SPAGAIN;
    if (count == 1) {
	SV* value = POPs;
	pOptions->UseAltStream = (UCHAR)(SvIOK(value) ? SvIV(value) : 0);
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(opt);
    PUTBACK ;
    count = call_method("use_keep_alive", G_SCALAR);
    SPAGAIN;
    if (count == 1) {
	SV* value = POPs;
	pOptions->UseKeepAlive = (UCHAR)(SvIOK(value) ? SvIV(value) : 0);
    }
    PUTBACK ;

    FREETMPS ;
    LEAVE ;

    pOptions->GlobalContext = 0;

    return pOptions;
}

static SV* create_dokan_file_info(DOKAN_FILE_INFO *pDFileInfo)
{
    int count;
    SV* dfileinfo;
    SV* options;

    dSP ;

    ENTER ;
    SAVETMPS ;

    options  = create_dokan_options(pDFileInfo->DokanOptions);

    PUSHMARK(SP) ;
    XPUSHs(sv_2mortal(newSVpv("Win32::Dokan::DokanFileInfo", 0)));
    PUTBACK ;
    count = call_method("new", G_SCALAR);
    SPAGAIN;

    if (count != 1) croak("create_file must return scalar");
    dfileinfo = SvREFCNT_inc(POPs);
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(dfileinfo);
    XPUSHs(sv_2mortal(newSVuv((unsigned)pDFileInfo->ProcessId)));
    PUTBACK ;
    call_method("process_id", G_DISCARD);

    PUSHMARK(SP) ;
    XPUSHs(dfileinfo);
    XPUSHs(sv_2mortal(newSVuv((unsigned)pDFileInfo->IsDirectory)));
    PUTBACK ;
    call_method("is_directory", G_DISCARD);

    PUSHMARK(SP) ;
    XPUSHs(dfileinfo);
    XPUSHs(sv_2mortal(newSVuv((unsigned)pDFileInfo->DeleteOnClose)));
    PUTBACK ;
    call_method("delete_on_close", G_DISCARD);

    PUSHMARK(SP) ;
    XPUSHs(dfileinfo);
    XPUSHs(sv_2mortal(options));
    PUTBACK ;
    call_method("dokan_options", G_DISCARD);

    FREETMPS ;
    LEAVE ;

    return dfileinfo;
}

static DOKAN_FILE_INFO* restore_dokan_file_info(DOKAN_FILE_INFO *pDFileInfo,
						SV *dfi)
{
    int count;

    dSP ;

    ENTER ;
    SAVETMPS ;

    PUSHMARK(SP) ;
    XPUSHs(dfi);
    PUTBACK ;
    count = call_method("context", G_SCALAR);
    SPAGAIN;

    if (count == 1) {
	SV* value = POPs;
        if (SvIOK(value)) pDFileInfo->Context = SvIV(value);
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(dfi);
    PUTBACK ;
    count = call_method("is_directory", G_SCALAR);
    SPAGAIN;

    if (count == 1) {
        int n = POPi;
	pDFileInfo->IsDirectory = n ? 1 : 0;
    }
    PUTBACK ;

    FREETMPS ;
    LEAVE ;

    return pDFileInfo;
}

static void set_file_time(LPFILETIME pFileTime, time_t t)
{
    struct tm tm;
    SYSTEMTIME st;
    __int64 ttmp = t;
    _gmtime64_s(&tm, &ttmp);

    st.wYear = tm.tm_year + 1900;
    st.wMonth = tm.tm_mon + 1;
    st.wDayOfWeek = tm.tm_wday;
    st.wDay = tm.tm_mday;
    st.wHour = tm.tm_hour;
    st.wMinute = tm.tm_min;
    st.wSecond = tm.tm_sec;
    st.wMilliseconds = 0;

    SystemTimeToFileTime(&st, pFileTime);
}

static time_t get_file_time(const FILETIME* pFileTime)
{
    struct tm tm;
    SYSTEMTIME st;
    __int64 ttmp, t1, t2, tzdiff;

    /* check TZ */
    ttmp = 123456;
    _localtime64_s(&tm, &ttmp);
    t1 = mktime(&tm);
    _gmtime64_s(&tm, &ttmp);
    t2 = mktime(&tm);
    tzdiff = t1 - t2;

    FileTimeToSystemTime(pFileTime, &st);
    tm.tm_year = st.wYear - 1900;
    tm.tm_mon = st.wMonth - 1;
    tm.tm_wday = st.wDayOfWeek;
    tm.tm_mday = st.wDay;
    tm.tm_hour = st.wHour;
    tm.tm_min = st.wMinute;
    tm.tm_sec = st.wSecond;
    
    return mktime(&tm) + tzdiff;
}

static SV* create_file_info()
{
    int count;
    SV* fileinfo;

    dSP ;

    ENTER ;
    SAVETMPS ;

    PUSHMARK(SP) ;
    XPUSHs(sv_2mortal(newSVpv("Win32::Dokan::FileInfo", 0)));
    PUTBACK ;
    count = call_method("new", G_SCALAR);
    SPAGAIN;

    if (count != 1) croak("FileInfo construction error");
    fileinfo = SvREFCNT_inc(POPs);
    PUTBACK ;

    FREETMPS ;
    LEAVE ;

    return fileinfo;
}

static LPBY_HANDLE_FILE_INFORMATION
restore_file_info(LPBY_HANDLE_FILE_INFORMATION pFileInfo, SV *dfi)
{
    int count;

    dSP ;

    memset(pFileInfo, 0, sizeof(*pFileInfo));

    ENTER ;
    SAVETMPS ;

    PUSHMARK(SP) ;
    XPUSHs(dfi);
    PUTBACK ;
    count = call_method("creation_time", G_SCALAR);
    SPAGAIN;

    if (count == 1) {
	SV* value = POPs;
        if (SvIOK(value)) set_file_time(&pFileInfo->ftCreationTime, SvIV(value));
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(dfi);
    PUTBACK ;
    count = call_method("last_access_time", G_SCALAR);
    SPAGAIN;

    if (count == 1) {
	SV* value = POPs;
        if (SvIOK(value))
	    set_file_time(&pFileInfo->ftLastAccessTime, SvIV(value));
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(dfi);
    PUTBACK ;
    count = call_method("last_write_time", G_SCALAR);
    SPAGAIN;

    if (count == 1) {
	SV* value = POPs;
        if (SvIOK(value))
	    set_file_time(&pFileInfo->ftLastWriteTime, SvIV(value));
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(dfi);
    PUTBACK ;
    count = call_method("volume_serial_number", G_SCALAR);
    SPAGAIN;

    if (count == 1) {
	SV* value = POPs;
        if (SvIOK(value)) pFileInfo->dwVolumeSerialNumber = SvIV(value);
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(dfi);
    PUTBACK ;
    count = call_method("file_size", G_SCALAR);
    SPAGAIN;

    pFileInfo->nFileSizeHigh = pFileInfo->nFileSizeLow = 0;
    if (count == 1) {
	SV* value = POPs;
        if (SvIOK(value)) pFileInfo->nFileSizeLow = SvIV(value);
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(dfi);
    PUTBACK ;
    count = call_method("file_attributes", G_SCALAR);
    SPAGAIN;

    pFileInfo->dwFileAttributes = 0;
    if (count == 1) {
	SV* value = POPs;
        if (SvIOK(value)) pFileInfo->dwFileAttributes = SvIV(value);
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(dfi);
    PUTBACK ;
    count = call_method("number_of_links", G_SCALAR);
    SPAGAIN;

    pFileInfo->nNumberOfLinks = 1;
    if (count == 1) {
	SV* value = POPs;
        if (SvIOK(value)) pFileInfo->nNumberOfLinks = SvIV(value);
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(dfi);
    PUTBACK ;
    count = call_method("file_index", G_SCALAR);
    SPAGAIN;

    pFileInfo->nFileIndexHigh = pFileInfo->nFileIndexLow = 0;
    if (count == 1) {
	SV* value = POPs;
        if (SvIOK(value)) pFileInfo->nFileIndexLow = SvIV(value);
    }
    PUTBACK ;

    FREETMPS ;
    LEAVE ;

    return pFileInfo;
}

static PWIN32_FIND_DATAW
restore_find_dataw(PWIN32_FIND_DATAW pFind_dataw, SV *fdw)
{
    int count;

    dSP ;

    memset(pFind_dataw, 0, sizeof(*pFind_dataw));

    ENTER ;
    SAVETMPS ;

    PUSHMARK(SP) ;
    XPUSHs(fdw);
    PUTBACK ;
    count = call_method("file_attributes", G_SCALAR);
    SPAGAIN;

    if (count == 1) {
	SV* value = POPs;
	if (SvIOK(value)) pFind_dataw->dwFileAttributes = SvIV(value);
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(fdw);
    PUTBACK ;
    count = call_method("creation_time", G_SCALAR);
    SPAGAIN;

    if (count == 1) {
	SV* value = POPs;
        if (SvIOK(value))
	    set_file_time(&pFind_dataw->ftCreationTime, SvIV(value));
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(fdw);
    PUTBACK ;
    count = call_method("last_access_time", G_SCALAR);
    SPAGAIN;

    if (count == 1) {
	SV* value = POPs;
        if (SvIOK(value))
	    set_file_time(&pFind_dataw->ftLastAccessTime, SvIV(value));
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(fdw);
    PUTBACK ;
    count = call_method("last_write_time", G_SCALAR);
    SPAGAIN;

    if (count == 1) {
	SV* value = POPs;
        if (SvIOK(value))
	    set_file_time(&pFind_dataw->ftLastWriteTime, SvIV(value));
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(fdw);
    PUTBACK ;
    count = call_method("file_size", G_SCALAR);
    SPAGAIN;

    pFind_dataw->nFileSizeHigh = pFind_dataw->nFileSizeLow = 0;
    if (count == 1) {
	SV* value = POPs;
        if (SvIOK(value)) pFind_dataw->nFileSizeLow = SvIV(value);
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(fdw);
    PUTBACK ;
    count = call_method("file_name", G_SCALAR);
    SPAGAIN;

    memset(pFind_dataw->cFileName, 0, sizeof(pFind_dataw->cFileName));
    if (count == 1) {
	SV* value = POPs;
        if (SvPOK(value)) {
	    int len;
	    const char* p = SvPV(value, len);
	    int maxlen = sizeof(pFind_dataw->cFileName) - sizeof(_TCHAR);
	    if (len > maxlen) len = maxlen;
	    memcpy(pFind_dataw->cFileName, p, len);
	}
    }
    PUTBACK ;

    PUSHMARK(SP) ;
    XPUSHs(fdw);
    PUTBACK ;
    count = call_method("alternative_file_name", G_SCALAR);
    SPAGAIN;

    memset(pFind_dataw->cAlternateFileName,
	   0,
	   sizeof(pFind_dataw->cAlternateFileName));

    if (count == 1) {
	SV* value = POPs;
        if (SvPOK(value)) {
	    int len;
	    const char* p = SvPV(value, len);
	    int maxlen =sizeof(pFind_dataw->cAlternateFileName)-sizeof(_TCHAR);
	    if (len > maxlen) len = maxlen;
	    memcpy(pFind_dataw->cAlternateFileName, p, len);
	}
    }
    PUTBACK ;

    FREETMPS ;
    LEAVE ;

    return pFind_dataw;
}

/* param for functions having "func(name, dokan_file_info)" style */
typedef struct {
    LPCWSTR fileName;
    PDOKAN_FILE_INFO pDFileInfo;
    const char* cbName;
} SimpleParam_t;

static int DP_Simple(void* self, void* pParam)
{
    SimpleParam_t *param = (SimpleParam_t*)pParam;
    int count, ret;
    SV* dfileinfo;

    dSP ;

    ENTER ;
    SAVETMPS ;

    dfileinfo = create_dokan_file_info(param->pDFileInfo);

    PUSHMARK(SP) ;
    XPUSHs(self);
    XPUSHs(sv_2mortal(newSVpv((const char*)param->fileName,
			      _tcslen(param->fileName)*sizeof(_TCHAR))));
    XPUSHs(sv_2mortal(dfileinfo));
    PUTBACK ;

    count = call_method(param->cbName, G_SCALAR);
    SPAGAIN;

    if (count != 1)
	croak("_cb_simple must return SCALAR");
    ret = POPi;

    PUTBACK;

    restore_dokan_file_info(param->pDFileInfo, dfileinfo);

    FREETMPS ;
    LEAVE ;

    return ret;
}


typedef struct {
    LPCWSTR fileName;
    DWORD desiredAccess;
    DWORD shareMode;
    DWORD createDisposition;
    DWORD flagsAndAttributes;
    PDOKAN_FILE_INFO pDFileInfo;
} CreateFileParam_t;

static int DP_CreateFile(void* self, void* pParam)
{
    CreateFileParam_t *param = (CreateFileParam_t*)pParam;
    int count, ret;
    SV* dfileinfo;

    dSP ;

    ENTER ;
    SAVETMPS ;

    dfileinfo = create_dokan_file_info(param->pDFileInfo);

    PUSHMARK(SP) ;
    XPUSHs(self);
    XPUSHs(sv_2mortal(newSVpv((const char*)param->fileName,
			      _tcslen(param->fileName)*sizeof(_TCHAR))));
    XPUSHs(sv_2mortal(newSVuv((unsigned)param->desiredAccess)));
    XPUSHs(sv_2mortal(newSVuv((unsigned)param->shareMode)));
    XPUSHs(sv_2mortal(newSVuv((unsigned)param->createDisposition)));
    XPUSHs(sv_2mortal(newSVuv((unsigned)param->flagsAndAttributes)));
    XPUSHs(sv_2mortal(dfileinfo));
    PUTBACK ;

    count = call_method("_cb_create_file", G_SCALAR);
    SPAGAIN;

    if (count != 1) croak("_cb_create_file must return SCALAR");
    ret = POPi;

    PUTBACK;

    restore_dokan_file_info(param->pDFileInfo, dfileinfo);

    FREETMPS ;
    LEAVE ;

    return ret;
}


typedef struct {
    LPCWSTR fileName;
    LPVOID buffer;
    DWORD numberOfBytesToRead;
    PDWORD pNumberOfBytesRead;
    LONGLONG offset;
    PDOKAN_FILE_INFO pDFileInfo;
} ReadFileParam_t;

static int DP_ReadFile(void* self, void* pParam)
{
    ReadFileParam_t *param = (ReadFileParam_t*)pParam;
    int count, ret;
    SV* data;
    SV* dfileinfo;

    dSP ;

    ENTER ;
    SAVETMPS ;

    dfileinfo = create_dokan_file_info(param->pDFileInfo);
    data = sv_newmortal();

    PUSHMARK(SP) ;
    XPUSHs(self);
    XPUSHs(sv_2mortal(newSVpv((const char*)param->fileName,
			      _tcslen(param->fileName)*sizeof(_TCHAR))));
    XPUSHs(data);
    XPUSHs(sv_2mortal(newSVuv((unsigned)param->offset)));
    XPUSHs(sv_2mortal(newSVuv((unsigned)param->numberOfBytesToRead)));
    XPUSHs(sv_2mortal(dfileinfo));
    PUTBACK ;

    count = call_method("_cb_read_file", G_SCALAR);
    SPAGAIN;

    if (count != 1) croak("_cb_read_file must return SCALAR");
    ret = POPi;

    PUTBACK;

    if (SvPOK(data)) {
	/* some data returned */
	int len;
	const char* p = SvPV(data, len);
	memcpy(param->buffer, p, len);
	*(param->pNumberOfBytesRead) = len;
    }

    restore_dokan_file_info(param->pDFileInfo, dfileinfo);

    FREETMPS ;
    LEAVE ;

    return ret;
}


typedef struct {
    LPCWSTR fileName;
    LPCVOID buffer;
    DWORD numberOfBytesToWrite;
    PDWORD pNumberOfBytesWritten;
    LONGLONG offset;
    PDOKAN_FILE_INFO pDFileInfo;
} WriteFileParam_t;

static int DP_WriteFile(void* self, void* pParam)
{
    WriteFileParam_t *param = (WriteFileParam_t*)pParam;
    int count, ret;
    SV* written;
    SV* dfileinfo;

    dSP ;

    ENTER ;
    SAVETMPS ;

    dfileinfo  = create_dokan_file_info(param->pDFileInfo);
    written = sv_newmortal();

    PUSHMARK(SP) ;
    XPUSHs(self);
    XPUSHs(sv_2mortal(newSVpv((const char*)param->fileName,
			      _tcslen(param->fileName)*sizeof(_TCHAR))));
    XPUSHs(sv_2mortal(newSVpv((const char*)param->buffer,
			      (int)(param->numberOfBytesToWrite))));
    XPUSHs(sv_2mortal(newSVuv((unsigned)param->offset)));
    XPUSHs(written);
    XPUSHs(sv_2mortal(dfileinfo));
    PUTBACK ;
    count = call_method("_cb_write_file", G_SCALAR);
    SPAGAIN;

    if (count != 1) croak("_cb_write_file must return SCALAR");
    ret = POPi;

    PUTBACK;
    if (SvIOK(written)) {
	*(param->pNumberOfBytesWritten) = SvIV(written);
    }

    restore_dokan_file_info(param->pDFileInfo, dfileinfo);

    FREETMPS ;
    LEAVE ;

    return ret;
}


typedef struct {
    LPCWSTR fileName;
    LPBY_HANDLE_FILE_INFORMATION pFileInfo;
    PDOKAN_FILE_INFO pDFileInfo;
} GetFileInformationParam_t;

static int DP_GetFileInformation(void* self, void* pParam)
{
    GetFileInformationParam_t *param = (GetFileInformationParam_t*)pParam;
    int count, ret;
    SV* fileinfo;
    SV* dfileinfo;

    dSP ;

    ENTER ;
    SAVETMPS ;

    fileinfo = create_file_info();
    dfileinfo = create_dokan_file_info(param->pDFileInfo);

    PUSHMARK(SP) ;
    XPUSHs(self);
    XPUSHs(sv_2mortal(newSVpv((const char*)param->fileName,
			      _tcslen(param->fileName)*sizeof(_TCHAR))));
    XPUSHs(sv_2mortal(fileinfo));
    XPUSHs(sv_2mortal(dfileinfo));
    PUTBACK ;

    count = call_method("_cb_get_file_information", G_SCALAR);
    SPAGAIN;

    if (count != 1) croak("_cb_get_file_inforation must return SCALAR");
    ret = POPi;
    PUTBACK;

    restore_file_info(param->pFileInfo, fileinfo);
    restore_dokan_file_info(param->pDFileInfo, dfileinfo);

    FREETMPS ;
    LEAVE ;

    return ret;
}


typedef struct {
    LPCWSTR pathName;
    PFillFindData pFindData;
    PDOKAN_FILE_INFO pDFileInfo;
} FindFilesParam_t;

static int DP_FindFiles(void* self, void* pParam)
{
    FindFilesParam_t *param = (FindFilesParam_t*)pParam;
    int i, count, ret;
    WIN32_FIND_DATAW fdw;
    SV* dfileinfo;

    dSP ;

    ENTER ;
    SAVETMPS ;

    dfileinfo = create_dokan_file_info(param->pDFileInfo);

    PUSHMARK(SP) ;
    XPUSHs(self);
    XPUSHs(sv_2mortal(newSVpv((const char*)param->pathName,
			      _tcslen(param->pathName)*sizeof(_TCHAR))));
    XPUSHs(sv_2mortal(dfileinfo));
    PUTBACK ;

    count = call_method("_cb_find_files", G_ARRAY);
    SPAGAIN;

    restore_dokan_file_info(param->pDFileInfo, dfileinfo);

    for(i=0; i<count; i++) {
	SV* value = POPs;
	if (SvROK(value)) {
	    param->pFindData(restore_find_dataw(&fdw, value),
			     param->pDFileInfo);
	}
	else {
	    ret = SvIV(value);
	}
    }
    PUTBACK;

    FREETMPS ;
    LEAVE ;

    return ret;
}


typedef struct {
    LPCWSTR fileName;
    DWORD attributes;
    PDOKAN_FILE_INFO pDFileInfo;
} SetFileAttributesParam_t;

static int DP_SetFileAttributes(void* self, void* pParam)
{
    SetFileAttributesParam_t *param = (SetFileAttributesParam_t*)pParam;
    int count, ret;
    SV* dfileinfo;

    dSP ;

    ENTER ;
    SAVETMPS ;

    dfileinfo = create_dokan_file_info(param->pDFileInfo);

    PUSHMARK(SP) ;
    XPUSHs(self);
    XPUSHs(sv_2mortal(newSVpv((const char*)param->fileName,
			      _tcslen(param->fileName)*sizeof(_TCHAR))));
    XPUSHs(sv_2mortal(newSVuv((unsigned)param->attributes)));
    XPUSHs(sv_2mortal(dfileinfo));
    PUTBACK ;

    count = call_method("_cb_set_file_attributes", G_SCALAR);
    SPAGAIN;

    if (count != 1)
	croak("_cb_set_file_attributes must return SCALAR");
    ret = POPi;

    PUTBACK;

    restore_dokan_file_info(param->pDFileInfo, dfileinfo);

    FREETMPS ;
    LEAVE ;

    return ret;
}


typedef struct {
    LPCWSTR fileName;
    const FILETIME* pCTime;
    const FILETIME* pATime;
    const FILETIME* pMTime;
    PDOKAN_FILE_INFO pDFileInfo;
} SetFileTimeParam_t;

static int DP_SetFileTime(void* self, void* pParam)
{
    SetFileTimeParam_t *param = (SetFileTimeParam_t*)pParam;
    int count, ret;
    SV* dfileinfo;

    dSP ;

    ENTER ;
    SAVETMPS ;

    dfileinfo = create_dokan_file_info(param->pDFileInfo);

    PUSHMARK(SP) ;
    XPUSHs(self);

    XPUSHs(sv_2mortal(newSVpv((const char*)param->fileName,
			      _tcslen(param->fileName)*sizeof(_TCHAR))));
    XPUSHs(sv_2mortal(newSVuv((unsigned)get_file_time(param->pCTime))));
    XPUSHs(sv_2mortal(newSVuv((unsigned)get_file_time(param->pATime))));
    XPUSHs(sv_2mortal(newSVuv((unsigned)get_file_time(param->pMTime))));
    XPUSHs(sv_2mortal(dfileinfo));

    PUTBACK ;

    count = call_method("_cb_set_file_time", G_SCALAR);
    SPAGAIN;

    if (count != 1)
	croak("_cb_set_file_time must return SCALAR");
    ret = POPi;

    PUTBACK;

    restore_dokan_file_info(param->pDFileInfo, dfileinfo);

    FREETMPS ;
    LEAVE ;

    return ret;
}


typedef struct {
    LPCWSTR existingName;
    LPCWSTR newFileName;
    BOOL replaceExisting;
    PDOKAN_FILE_INFO pDFileInfo;
} MoveFileParam_t;

static int DP_MoveFile(void* self, void* pParam)
{
    MoveFileParam_t *param = (MoveFileParam_t*)pParam;
    int count, ret;
    SV* dfileinfo;

    dSP ;

    ENTER ;
    SAVETMPS ;

    dfileinfo = create_dokan_file_info(param->pDFileInfo);

    PUSHMARK(SP) ;
    XPUSHs(self);
    XPUSHs(sv_2mortal(newSVpv((const char*)param->existingName,
			      _tcslen(param->existingName)*sizeof(_TCHAR))));
    XPUSHs(sv_2mortal(newSVpv((const char*)param->newFileName,
			      _tcslen(param->newFileName)*sizeof(_TCHAR))));
    XPUSHs(sv_2mortal(newSVuv((unsigned)param->replaceExisting)));
    XPUSHs(sv_2mortal(dfileinfo));
    PUTBACK ;

    count = call_method("_cb_move_file", G_SCALAR);
    SPAGAIN;

    restore_dokan_file_info(param->pDFileInfo, dfileinfo);

    if (count != 1)
	croak("_cb_move_file must return SCALAR");
    ret = POPi;

    PUTBACK;

    FREETMPS ;
    LEAVE ;

    return ret;
}


typedef struct {
    LPCWSTR fileName;
    LONGLONG length;
    PDOKAN_FILE_INFO pDFileInfo;
} SetEndOfFileParam_t;

static int DP_SetEndOfFile(void* self, void* pParam)
{
    SetEndOfFileParam_t *param = (SetEndOfFileParam_t*)pParam;
    int count, ret;
    SV* dfileinfo;

    dSP ;

    ENTER ;
    SAVETMPS ;

    dfileinfo = create_dokan_file_info(param->pDFileInfo);

    PUSHMARK(SP) ;
    XPUSHs(self);
    XPUSHs(sv_2mortal(newSVpv((const char*)param->fileName,
			      _tcslen(param->fileName)*sizeof(_TCHAR))));
    XPUSHs(sv_2mortal(newSVuv((unsigned)param->length)));
    XPUSHs(sv_2mortal(dfileinfo));
    PUTBACK ;

    count = call_method("_cb_set_end_of_file", G_SCALAR);
    SPAGAIN;

    restore_dokan_file_info(param->pDFileInfo, dfileinfo);

    if (count != 1)
	croak("_cb_move_file must return SCALAR");
    ret = POPi;

    PUTBACK;

    FREETMPS ;
    LEAVE ;

    return ret;
}


typedef struct {
    LPCWSTR fileName;
    LONGLONG offset;
    LONGLONG length;
    PDOKAN_FILE_INFO pDFileInfo;
    const char* callback;
} LockCtlParam_t;

static int DP_LockCtl(void* self, void* pParam)
{
    LockCtlParam_t *param = (LockCtlParam_t*)pParam;
    int count, ret;
    SV* dfileinfo;

    dSP ;

    ENTER ;
    SAVETMPS ;

    dfileinfo = create_dokan_file_info(param->pDFileInfo);

    PUSHMARK(SP) ;
    XPUSHs(self);
    XPUSHs(sv_2mortal(newSVpv((const char*)param->fileName,
			      _tcslen(param->fileName)*sizeof(_TCHAR))));
    XPUSHs(sv_2mortal(newSVuv((unsigned)param->offset)));
    XPUSHs(sv_2mortal(newSVuv((unsigned)param->length)));
    XPUSHs(sv_2mortal(dfileinfo));
    PUTBACK ;

    count = call_method(param->callback, G_SCALAR);
    SPAGAIN;

    restore_dokan_file_info(param->pDFileInfo, dfileinfo);

    if (count != 1)
	croak("_cb_lock_ctl must return SCALAR");
    ret = POPi;

    PUTBACK;

    FREETMPS ;
    LEAVE ;

    return ret;
}


typedef struct {
    PULONGLONG pAvailable;
    PULONGLONG pTotal;
    PULONGLONG pFree;

    PDOKAN_FILE_INFO pDFileInfo;
} GetDiskFreeSpaceParam_t;

static void set_space(PULONGLONG pDest, SV* value)
{
    if (SvIOK(value))
	*pDest = (ULONGLONG)SvIV(value);
    else if (SvNOK(value))
	*pDest = (ULONGLONG)SvNV(value);
    else
	*pDest = 0;
}

static int DP_GetDiskFreeSpace(void* self, void* pParam)
{
    GetDiskFreeSpaceParam_t *param = (GetDiskFreeSpaceParam_t*)pParam;
    int count, ret;
    SV* dfileinfo;

    dSP ;

    ENTER ;
    SAVETMPS ;

    dfileinfo = create_dokan_file_info(param->pDFileInfo);

    PUSHMARK(SP) ;
    XPUSHs(self);
    XPUSHs(sv_2mortal(dfileinfo));
    PUTBACK ;

    count = call_method("_cb_get_disk_free_space", G_ARRAY);
    SPAGAIN;

    restore_dokan_file_info(param->pDFileInfo, dfileinfo);

    if (count == 3) {
	set_space(param->pFree, POPs);
	set_space(param->pTotal, POPs);
	set_space(param->pAvailable, POPs);
	ret = 0;
    }
    else if (count == 1) {
        /* only error code */
        ret = POPi;
    }
    else {
	croak("_cb_get_disk_free_space must return SCALAR");
    }

    PUTBACK;

    FREETMPS ;
    LEAVE ;

    return ret;
}


typedef struct {
    LPWSTR nameBuffer;
    DWORD nameSize;
    LPDWORD pSerial;
    LPDWORD pMaxComponentLength;
    LPDWORD pFileSystemFlags;
    LPWSTR FileSystemNameBuffer;
    DWORD FileSystemNameSize;
    PDOKAN_FILE_INFO pDFileInfo;
} GetVolumeInformationParam_t;

static void set_strbuf(LPWSTR buf, DWORD size, SV* value)
{
    int destsize;

    if (buf == 0 || size <= 0) return;

    memset(buf, 0, size);
    destsize = size - sizeof(_TCHAR);
    if (destsize <= 0) return;

    if (SvPOK(value)) {
	int len;
	const char* p = SvPV(value, len);
	if (len < destsize) destsize = len;
	memcpy(buf, p, destsize);
	return;
    }
}

static void set_dword(PDWORD pdword, SV* value)
{
    if (SvIOK(value))
	*pdword = (DWORD)SvIV(value);
    else
	*pdword = 0;
}

static int DP_GetVolumeInformation(void* self, void* pParam)
{
    GetVolumeInformationParam_t *param = (GetVolumeInformationParam_t*)pParam;
    int count, ret;
    SV* dfileinfo;

    dSP ;

    ENTER ;
    SAVETMPS ;

    dfileinfo = create_dokan_file_info(param->pDFileInfo);

    PUSHMARK(SP) ;
    XPUSHs(self);
    XPUSHs(sv_2mortal(dfileinfo));
    PUTBACK ;

    count = call_method("_cb_get_volume_information", G_ARRAY);
    SPAGAIN;

    restore_dokan_file_info(param->pDFileInfo, dfileinfo);

    if (count == 5) {
	set_strbuf(param->FileSystemNameBuffer,
		   param->FileSystemNameSize, POPs);
	set_dword(param->pFileSystemFlags, POPs);
	set_dword(param->pMaxComponentLength, POPs);
	set_dword(param->pSerial, POPs);
	set_strbuf(param->nameBuffer, param->nameSize, POPs);

	ret = 0;
    }
    else if (count == 1) {
        /* only error code */
        ret = POPi;
    }
    else {
	croak("_cb_get_volume_information must return ARRAY(6)");
    }

    PUTBACK;

    FREETMPS ;
    LEAVE ;

    return ret;
}



typedef struct {
    PDOKAN_FILE_INFO pDFileInfo;
} UnmountParam_t;

static int DP_Unmount(void* self, void* pParam)
{
    UnmountParam_t *param = (UnmountParam_t*)pParam;
    int count, ret;
    SV* dfileinfo;

    dSP ;

    ENTER ;
    SAVETMPS ;

    dfileinfo = create_dokan_file_info(param->pDFileInfo);

    PUSHMARK(SP) ;
    XPUSHs(self);
    XPUSHs(sv_2mortal(dfileinfo));
    PUTBACK ;

    count = call_method("_cb_unmount", G_ARRAY);
    SPAGAIN;

    restore_dokan_file_info(param->pDFileInfo, dfileinfo);

    if (count != 1) croak("_cb_unmount must return SCALAR");
    ret = POPi;

    PUTBACK ;

    FREETMPS ;
    LEAVE ;

    return ret;
}


static int SyncExec(Func_t* pFunc, void* pParam)
{
    HANDLE hPerlEnd;
    int ret;

    WaitForSingleObject(PerlReadyEvent, INFINITE);

    pCurrentFunc = pFunc;
    pCurrentParam = pParam;
    pReturn = &ret;
    PerlEndEvent = hPerlEnd = CreateEvent(0, FALSE, FALSE, 0);

    SetEvent(PerlStartOKEvent);

    /* perl thread executed here */

    WaitForSingleObject(hPerlEnd, INFINITE);
    CloseHandle(hPerlEnd);

    return ret;
}

static int DOKAN_CALLBACK DCB_CreateFile(LPCWSTR fileName,
					 DWORD desiredAccess,
					 DWORD shareMode,
					 DWORD createDisposition,
					 DWORD flagsAndAttributes,
					 PDOKAN_FILE_INFO pDFileInfo)
{
    CreateFileParam_t param;

    param.fileName = fileName;
    param.desiredAccess = desiredAccess;
    param.shareMode = shareMode;
    param.createDisposition = createDisposition;
    param.pDFileInfo = pDFileInfo;

    return SyncExec(DP_CreateFile, &param);
}

static int DOKAN_CALLBACK DCB_OpenDirectory(LPCWSTR fileName,
					    PDOKAN_FILE_INFO pDFileInfo)
{
    SimpleParam_t param;

    param.fileName = fileName;
    param.pDFileInfo = pDFileInfo;
    param.cbName = "_cb_open_directory";

    return SyncExec(DP_Simple, &param);
}

static int DOKAN_CALLBACK DCB_CreateDirectory(LPCWSTR fileName,
					      PDOKAN_FILE_INFO pDFileInfo)
{
    SimpleParam_t param;

    param.fileName = fileName;
    param.pDFileInfo = pDFileInfo;
    param.cbName = "_cb_create_directory";

    return SyncExec(DP_Simple, &param);
}

static int DOKAN_CALLBACK DCB_Cleanup(LPCWSTR fileName,
				      PDOKAN_FILE_INFO pDFileInfo)
{
    SimpleParam_t param;

    param.fileName = fileName;
    param.pDFileInfo = pDFileInfo;
    param.cbName = "_cb_cleanup";

    return SyncExec(DP_Simple, &param);
}

static int DOKAN_CALLBACK DCB_CloseFile(LPCWSTR fileName,
					PDOKAN_FILE_INFO pDFileInfo)
{
    SimpleParam_t param;

    param.fileName = fileName;
    param.pDFileInfo = pDFileInfo;
    param.cbName = "_cb_close_file";

    return SyncExec(DP_Simple, &param);
}

static int DOKAN_CALLBACK DCB_ReadFile(LPCWSTR fileName,
				       LPVOID buffer,
				       DWORD numberOfBytesToRead,
				       LPDWORD pNumerOfBytesRead,
				       LONGLONG offset,
				       PDOKAN_FILE_INFO pDFileInfo)
{
    ReadFileParam_t param;

    param.fileName = fileName;
    param.buffer = buffer;
    param.numberOfBytesToRead = numberOfBytesToRead;
    param.pNumberOfBytesRead = pNumerOfBytesRead;
    param.offset = offset;
    param.pDFileInfo = pDFileInfo;

    return SyncExec(DP_ReadFile, &param);
}

static int DOKAN_CALLBACK DCB_WriteFile(LPCWSTR fileName,
					LPCVOID buffer,
					DWORD numberOfBytesToWrite,
					LPDWORD pNumberOfBytesWritten,
					LONGLONG offset,
					PDOKAN_FILE_INFO pDFileInfo)
{
    WriteFileParam_t param;

    param.fileName = fileName;
    param.buffer = buffer;
    param.numberOfBytesToWrite = numberOfBytesToWrite;
    param.pNumberOfBytesWritten = pNumberOfBytesWritten;
    param.offset = offset;
    param.pDFileInfo = pDFileInfo;

    return SyncExec(DP_WriteFile, &param);
}


static int DOKAN_CALLBACK DCB_FlushFileBuffers(LPCWSTR fileName,
					       PDOKAN_FILE_INFO pDFileInfo)
{
    SimpleParam_t param;

    param.fileName = fileName;
    param.pDFileInfo = pDFileInfo;
    param.cbName = "_cb_flush_file_buffers";

    return SyncExec(DP_Simple, &param);
}

static int DOKAN_CALLBACK DCB_GetFileInformation(LPCWSTR fileName,
						 LPBY_HANDLE_FILE_INFORMATION pFileInfo,
						 PDOKAN_FILE_INFO pDFileInfo)

{
    GetFileInformationParam_t param;

    param.fileName = fileName;
    param.pFileInfo = pFileInfo;
    param.pDFileInfo = pDFileInfo;

    return SyncExec(DP_GetFileInformation, &param);
}

static int DOKAN_CALLBACK DCB_FindFiles(LPCWSTR pathName,
					PFillFindData pFindData,
					PDOKAN_FILE_INFO pDFileInfo)
{
    FindFilesParam_t param;

    param.pathName = pathName;
    param.pFindData = pFindData;
    param.pDFileInfo = pDFileInfo;

    return SyncExec(DP_FindFiles, &param);
}

static int DOKAN_CALLBACK DCB_SetFileAttributes(LPCWSTR fileName,
						DWORD attributes,
						PDOKAN_FILE_INFO pDFileInfo)
{
    SetFileAttributesParam_t param;

    param.fileName = fileName;
    param.attributes = attributes;
    param.pDFileInfo = pDFileInfo;

    return SyncExec(DP_SetFileAttributes, &param);
}

static int DOKAN_CALLBACK DCB_SetFileTime(LPCWSTR fileName,
					  CONST FILETIME* pCTime,
					  CONST FILETIME* pATime,
					  CONST FILETIME* pMTime,
					  PDOKAN_FILE_INFO pDFileInfo)
{
    SetFileTimeParam_t param;

    param.fileName = fileName;
    param.pCTime = pCTime;
    param.pATime = pATime;
    param.pMTime = pMTime;
    param.pDFileInfo = pDFileInfo;

    return SyncExec(DP_SetFileTime, &param);
}

static int DOKAN_CALLBACK DCB_DeleteFile(LPCWSTR fileName,
					 PDOKAN_FILE_INFO pDFileInfo)
{
    SimpleParam_t param;

    param.fileName = fileName;
    param.pDFileInfo = pDFileInfo;
    param.cbName = "_cb_delete_file";

    return SyncExec(DP_Simple, &param);
}

static int DOKAN_CALLBACK DCB_DeleteDirectory(LPCWSTR fileName,
					      PDOKAN_FILE_INFO pDFileInfo)
{
    SimpleParam_t param;

    param.fileName = fileName;
    param.pDFileInfo = pDFileInfo;
    param.cbName = "_cb_delete_directory";

    return SyncExec(DP_Simple, &param);
}

static int DOKAN_CALLBACK DCB_MoveFile(LPCWSTR existingName,
				       LPCWSTR newFileName,
				       BOOL replaceExisting,
				       PDOKAN_FILE_INFO pDFileInfo)
{
    MoveFileParam_t param;

    param.existingName = existingName;
    param.newFileName = newFileName;
    param.replaceExisting = replaceExisting;
    param.pDFileInfo = pDFileInfo;

    return SyncExec(DP_MoveFile, &param);
}

static int DOKAN_CALLBACK DCB_SetEndOfFile(LPCWSTR fileName,
					   LONGLONG length,
					   PDOKAN_FILE_INFO pDFileInfo)
{
    SetEndOfFileParam_t param;

    param.fileName = fileName;
    param.length = length;
    param.pDFileInfo = pDFileInfo;

    return SyncExec(DP_SetEndOfFile, &param);
}

static int DOKAN_CALLBACK DCB_LockFile(LPCWSTR fileName,
				       LONGLONG offset,
				       LONGLONG length,
				       PDOKAN_FILE_INFO pDFileInfo)
{
    LockCtlParam_t param;

    param.fileName = fileName;
    param.offset = offset;
    param.length = length;
    param.pDFileInfo = pDFileInfo;
    param.callback = "_cb_lock_file";

    return SyncExec(DP_LockCtl, &param);
}

static int DOKAN_CALLBACK DCB_UnlockFile(LPCWSTR fileName,
					 LONGLONG offset,
					 LONGLONG length,
					 PDOKAN_FILE_INFO pDFileInfo)
{
    LockCtlParam_t param;

    param.fileName = fileName;
    param.offset = offset;
    param.length = length;
    param.pDFileInfo = pDFileInfo;
    param.callback = "_cb_unlock_file";

    return SyncExec(DP_LockCtl, &param);
}

static int DOKAN_CALLBACK DCB_GetDiskFreeSpace(PULONGLONG pAvailable,
					       PULONGLONG pTotal,
					       PULONGLONG pFree,
					       PDOKAN_FILE_INFO pDFileInfo)
{
    GetDiskFreeSpaceParam_t param;

    param.pAvailable = pAvailable;
    param.pTotal = pTotal;
    param.pFree = pFree;
    param.pDFileInfo = pDFileInfo;

    return SyncExec(DP_GetDiskFreeSpace, &param);
}

static int DOKAN_CALLBACK DCB_GetVolumeInformation(LPWSTR nameBuffer,
						   DWORD nameSize,
						   LPDWORD pSerial,
						   LPDWORD pMaxComponentLength,
						   LPDWORD pFileSystemFlags,
						   LPWSTR FileSystemNameBuffer,
						   DWORD FileSystemNameSize,
						   PDOKAN_FILE_INFO pDFileInfo)
{
    GetVolumeInformationParam_t param;

    param.nameBuffer = nameBuffer;
    param.nameSize = nameSize;
    param.pSerial = pSerial;
    param.pMaxComponentLength = pMaxComponentLength;
    param.pFileSystemFlags = pFileSystemFlags;
    param.FileSystemNameBuffer = FileSystemNameBuffer;
    param.FileSystemNameSize = FileSystemNameSize;
    param.pDFileInfo = pDFileInfo;

    return SyncExec(DP_GetVolumeInformation, &param);
}

static int DOKAN_CALLBACK DCB_Unmount(PDOKAN_FILE_INFO pDFileInfo)
{
    UnmountParam_t param;
    param.pDFileInfo = pDFileInfo;

    return SyncExec(DP_Unmount, &param);
}

static DOKAN_OPERATIONS Operations = {
    DCB_CreateFile,
    DCB_OpenDirectory,
    DCB_CreateDirectory,
    DCB_Cleanup,
    DCB_CloseFile,
    DCB_ReadFile,
    DCB_WriteFile,
    DCB_FlushFileBuffers,
    DCB_GetFileInformation,
    DCB_FindFiles,
    0,
    DCB_SetFileAttributes,
    DCB_SetFileTime,
    DCB_DeleteFile,
    DCB_DeleteDirectory,
    DCB_MoveFile,
    DCB_SetEndOfFile,
    DCB_LockFile,
    DCB_UnlockFile,
    DCB_GetDiskFreeSpace,
    DCB_GetVolumeInformation,
    DCB_Unmount
};

static int MyRelease(SV* self, void* pParam)
{
    return 0;
}

static void ThreadCleanup()
{
    DeleteCriticalSection(&ReadySection);
    DeleteCriticalSection(&ThreadCountSection);
    CloseHandle(PerlReadyEvent);
}

typedef int (CALLBACK* dokanMainProc_t)(PDOKAN_OPTIONS, PDOKAN_OPERATIONS);

typedef struct {
    dokanMainProc_t proc;
    DOKAN_OPTIONS options;
} DokanStartParam_t;

static void DokanStart(void* pParam)
{
    DokanStartParam_t *startParam = (DokanStartParam_t*)pParam;

    /* dokan main loop start */
    Mounted = TRUE;
    startParam->proc(&startParam->options, &Operations);

    /* dokan main loop end */
    Mounted = FALSE;

    /* release all threads */
    EnterCriticalSection(&ThreadCountSection);
    while(ThreadCount--) {
	SyncExec(MyRelease, 0);
    }
    LeaveCriticalSection(&ThreadCountSection);

    ThreadCleanup();
}

static int DokanThreadStart(HANDLE hModule, PDOKAN_OPTIONS pOptions)
{
    DokanStartParam_t startParam;
    uintptr_t mainThread;

    startParam.proc = (dokanMainProc_t)GetProcAddress(hModule, "DokanMain");
    startParam.options = *pOptions;

    if (!startParam.proc) return 0;

    InitializeCriticalSection(&ReadySection);
    InitializeCriticalSection(&ThreadCountSection);
    PerlReadyEvent = CreateEvent(0, FALSE, FALSE, 0);

    mainThread = _beginthread(DokanStart, 0, &startParam);
    sleep(3);

    if (!Mounted) {
	ThreadCleanup();
	return 0;
    }

    return (int)mainThread;
}

MODULE = Win32::Dokan		PACKAGE = Win32::Dokan		

INCLUDE: const-xs.inc

int
load_library(self, path)
    SV *self
    const char* path

    CODE:
	RETVAL = (int)LoadLibrary(path);
    OUTPUT:
	RETVAL

void
free_library(self, handle)
    SV *self
    int handle

    CODE:
	FreeLibrary((HANDLE)handle);

int
start_main_thread(self, handle, svopt)
    SV* self
    int handle
    SV* svopt

    INIT:
	DOKAN_OPTIONS options;

    CODE:
	dSP ;
	ENTER ;

	RETVAL = DokanThreadStart((HANDLE)handle,
				  restore_dokan_options(&options, svopt));
        LEAVE ;

    OUTPUT:
	RETVAL

SV*
convert_unicode_to_native(self, str)
    SV* self
    SV* str

    INIT:
	char* src;
	char* dest;
        unsigned int srclen;
	unsigned int destlen;
	int dummy;

    CODE:
	src = SvPV(str, srclen);
        destlen = WideCharToMultiByte(CP_ACP, 0,
				      (LPCWSTR)src, srclen/sizeof(WCHAR),
				      0, 0, 0, 0);
        RETVAL = newSVpv("", 0);
	SvGROW(RETVAL, destlen);
	dest = SvPV(RETVAL, dummy);
        WideCharToMultiByte(CP_ACP, 0,
			    (LPCWSTR)src, srclen/sizeof(WCHAR),
			    dest, destlen, 0, 0);
	SvCUR_set(RETVAL, destlen);

    OUTPUT:
        RETVAL

SV*
convert_native_to_unicode(self, str)
    SV* self
    SV* str

    INIT:
	char* src;
	char* dest;
        unsigned int srclen;
	unsigned int destlen;
	int dummy;

    CODE:
	src = SvPV(str, srclen);
        destlen = MultiByteToWideChar(CP_ACP, 0,
				      src, srclen, 0, 0);
	destlen *= sizeof(WCHAR);
        RETVAL = newSVpv("", 0);
	SvGROW(RETVAL, destlen);
	dest = SvPV(RETVAL, dummy);
        MultiByteToWideChar(CP_ACP, 0,
			    src, srclen, (LPWSTR)dest, destlen);
	SvCUR_set(RETVAL, destlen);

    OUTPUT:
        RETVAL


int
main(self)
    SV* self

    INIT:
	HANDLE hPerlStartOK = CreateEvent(0, FALSE, FALSE, 0);

	Func_t *pFunc;
	void *pParam;
	int *pRet;
	HANDLE hEnd;

    CODE:
	if (hPerlStartOK) {
	    EnterCriticalSection(&ThreadCountSection);
	    ThreadCount++;
	    LeaveCriticalSection(&ThreadCountSection);

	    while(Mounted) {
	        EnterCriticalSection(&ReadySection);
                PerlStartOKEvent = hPerlStartOK;

	        SetEvent(PerlReadyEvent);
	        WaitForSingleObject(hPerlStartOK, INFINITE);

	        pFunc = pCurrentFunc;
	        pParam = pCurrentParam;
		pRet = pReturn;
	        hEnd = PerlEndEvent;

                LeaveCriticalSection(&ReadySection);

	        if (pFunc) *pRet = pFunc(self, pParam);
	        SetEvent(hEnd);
	    }

	    CloseHandle(hPerlStartOK);
	    RETVAL = 1;
	}
	else {
	    RETVAL = 0;
	}

    OUTPUT:
	RETVAL
