#
# Sample class for Win32::Dokan
#
package Mirror;

use strict;

use base qw(Win32::Dokan::FS);

use Carp;
use Errno;

sub new {
    my $class = shift;
    my $dir = shift;

    bless {
	root_dir => $dir,
    }, $class;
}

sub encoding {
    return "X-MBCS";
    # return "CP932";
}

sub full_path {
    my $self = shift;
    my $path = shift;

    return $self->{root_dir} . "/" . $path;
}

sub open {
    my ($self, $path, $mode, $fileinfo) = @_;

    return -f $self->full_path($path);
}

sub create {
    my ($self, $path, $fileinfo) = @_;

    CORE::open(my $fh, "> " . $self->full_path($path)) or return undef;

    return 0;
}

sub truncate {
    my ($self, $path, $length, $fileinfo) = @_;

    CORE::open(my $fh, "<+ " . $self->full_path($path)) or return undef;
    binmode($fh);

    unless (CORE::truncate($fh, $length)) {
	return -1;
    }

    return 0;
}

sub opendir {
    my ($self, $path, $fileinfo) = @_;

    if (-d $self->full_path($path)) {
	return 0;
    }

    return -Errno::ENOENT();
}

sub mkdir {
    my ($self, $path, $fileinfo) = @_;

    if (CORE::mkdir($self->full_path($path))) {
	return 0;
    }
    return -1;
}

sub close {
    my ($self, $path, $fileinfo) = @_;

    return 0;
}

sub cleanup {
    my ($self, $path, $fileinfo) = @_;

    # OK: true
    # NG: false
    return 1;
}

sub read {
    my ($self, $path, $offset, $length, $fileinfo) = @_;

    CORE::open(my $fh, "< " . $self->full_path($path)) or return undef;
    binmode($fh);

    my $buf;
    unless (sysseek($fh, $offset, 0)) {
	return undef;
    }
    CORE::sysread($fh, $buf, $length) or return undef;

    return $buf;
}

sub write {
    my ($self, $path, $offset, $data, $fileinfo) = @_;

    CORE::open(my $fh, "+>> " . $self->full_path($path)) or return undef;
    binmode($fh);

    unless (sysseek($fh, $offset, 0)) {
	return undef;
    }
    syswrite($fh, $data) or return undef;

    return length($data);
}

sub flush {
    my ($self, $path, $fileinfo) = @_;
    # print STDERR "fs flush: $path\n";
    return 0;
}

sub stat {
    my ($self, $path, $fileinfo) = @_;

    my $attr = -d $self->full_path($path)
	? Win32::Dokan::FILE_ATTRIBUTE_DIRECTORY()
	: Win32::Dokan::FILE_ATTRIBUTE_NORMAL();

    my @s = stat($self->full_path($path));
    return undef unless (@s);

    my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
	$atime,$mtime,$ctime,$blksize,$blocks) = @s;
    
    return [$size, $attr, $ctime, $atime, $mtime];
}

sub readdir {
    my ($self, $path, $fileinfo) = @_;
    CORE::opendir(my $dh, $self->full_path($path)) or return -1;
    my @ret = readdir($dh);
    closedir($dh);

    return \@ret;
}

sub setattr {
    my ($self, $path, $attr, $fileinfo) = @_;

    # print STDERR "setattr for $path to $attr\n";

    return 0;
}

sub utime {
    my ($self, $path, $ctime, $atime, $mtime, $fileinfo) = @_;

    if (CORE::utime($atime, $mtime, $path)) {
	return 0;
    }

    return -1;
}

sub remove {
    my ($self, $path, $fileinfo) = @_;

    if (CORE::unlink($self->full_path($path))) {
	return 0;
    }
    return -1;
}

sub rename {
    my ($self, $path, $newpath, $fileinfo) = @_;

    if (CORE::rename($self->full_path($path), $self->full_path($newpath))) {
	return 0;
    }
    return -1;
}

sub rmdir {
    my ($self, $path, $fileinfo) = @_;

    if (CORE::rmdir($self->full_path($path))) {
	return 0;
    }
    return -1;
}

sub unmount {
    my ($self, $fileinfo) = @_;
    return;
}

1;
