#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# Overlay onto lib/libcommand_do.sh for portsreinstall-chroot
# - "do" command operation -
# Copyright (C) 2018 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Check the completion of build at the chroot environment =============
command_do_chk_build_completion ()
{
	[ -e "$opt_basedir/builder/${DBDIR_PROG}"/stage.complete/ALL_COMPLETE ]
}

# ============= Enter the guest =============
command_do_chroot_enter ()
{
	local shellcmd
	message_cat <<eof
=========================================================
You entered the builder chroot environment.
Complete the update of packages by portsreinstall(8),
Then exit to the target environment.
=========================================================

eof
	shellcmd=`id -P | sed -E 's|.*:([^:]*)$|\1|'`
	case `basename "$shellcmd"` in
	sh | ksh | zsh | '' )
		shellcmd="env PS1='\\\$@[builder]\h\\$ ' ${shellcmd:-/bin/sh}"
		;;
	bash )
		shellcmd="env PS1='\u@[builder]\h\\$ ' $shellcmd"
		;;
	esac
	chroot "$opt_basedir"/builder "$shellcmd"
}

# ============= Update of the files affecting package installation of the host environment  =============
command_do_update_host_files ()
{
	local dirpath filepath
	message_echo "Updating the files affecting package installation (packages, distfiles, port options, ports/packages management tools configurations) of the host environment..."
	for dirpath in "${DISTDIR}" "${PACKAGES}" "${PORT_DBDIR}"
	do
		(
			mkdir -p "$dirpath"
			cd $dirpath && find . -type f
		) | while read file
		do
			[ -e "$opt_basedir/builder$dirpath/$file" ] || rm "$dirpath/$file"
		done
		(
			cd "$opt_basedir"/builder$dirpath && find . -type f
		) | while read file
		do
			[ ! -e "$dirpath/$file" -o "$opt_basedir/builder$dirpath/$file" -nt "$dirpath/$file" ] || continue
			nodedir=`dirname "$dirpath/$file"`
			mkdir -p "$nodedir"
			cp -p "$opt_basedir/builder$dirpath/$file" "$dirpath/$file"
		done
		set +x
		find -d $dirpath -type d -empty -delete
	done
	for filepath in "${LOCALBASE}"/etc/portsreinstall.conf "${LOCALBASE}"/etc/pkgtools.conf "${LOCALBASE}"/etc/pkg.conf /etc/make.conf "${DBDIR}"
	do
		if [ -e "$opt_basedir"/builder$filepath ]
		then
			[ ! -e $filepath -o "$opt_basedir"/builder$filepath -nt $filepath ] && cp -p "$opt_basedir"/builder$filepath $filepath
		else
			rm -f $filepath
		fi
	done
	:
}

# ============= Ending process =============
command_do_ending_process ()
{
	temp_terminate_process () { :; }
	if [ $opt_no_opening_message = yes ]
	then
		message_echo "Done as ${APPNAME}"
		return
	fi
	message_section_title "COMPLETELY DONE"
	message_echo "- E N D -"
}

# ============= Initial clean up =============
command_do_chroot_cleanup  ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS=
	_program_exec_and_record_completion__operation ()
	{
		fs_destroy
	}
	program_exec_and_record_completion CLEANUP_GUEST
}

# ============= Enter the chroot environment =============
command_do_chroot_enter_if_incomplete ()
{
	local PROGRAM_DEPENDS
	command_do_chk_build_completion || program_deregister_stage_complete UPGRADE_GUEST
	PROGRAM_DEPENDS='CLEANUP_GUEST'
	_program_exec_and_record_completion__operation ()
	{
		message_echo
		if ! command_do_chroot_enter
		then
			message_echo "INFO: The last exit status in the builder chroot environment was non-zero." >&2
			exit 1
		fi
		if ! command_do_chk_build_completion
		then
			message_echo "ERROR: Exited from the builder chroot environment without completing the build." >&2
			exit 1
		fi
		message_echo "INFO: The package build completed."
		message_echo "Is it OK to apply the upgrade to the host environment? ([y]/n)"
		if ! message_query_yn_default_yes
		then
			message_echo "INFO: Terminated because the continuation is stopped by the user."
			exit 1
		fi
	}
	program_exec_and_record_completion UPGRADE_GUEST
}

# ============= Pack (re)installed packages at the chroot environment =============
command_do_chroot_pack_update ()
{
	local PROGRAM_DEPENDS
	if [ -e "$opt_basedir"/store/complete ]
	then
		if ! command_do_chk_build_completion
		then
			program_deregister_stage_complete PACK_PKGS
			rm -rf "$opt_basedir"/store/complete
		fi
	else
		program_deregister_stage_complete PACK_PKGS
	fi
	PROGRAM_DEPENDS='UPGRADE_GUEST'
	_program_exec_and_record_completion__operation ()
	{
		message_echo "Packing the upgrades..."
		[ $opt_batch_mode = yes ] && opt_quiet='-a '
		chroot "$opt_basedir"/builder /bin/csh -c "${PROGRAM} -S $opt_quiet packupgrade create && ${PROGRAM} -S $opt_quiet packupgrade crop /.${PROGRAM}"
	}
	program_exec_and_record_completion PACK_PKGS
}

# ============= Update of the files affecting package installation of the host environment  =============
command_do_chroot_update_host_files ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='PACK_PKGS'
	_program_exec_and_record_completion__operation ()
	{
		command_do_update_host_files
	}
	program_exec_and_record_completion UPDATE_HOST
}

# ============= Update of packages at the host environment =============
command_do_chroot_update_host_pkgs ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='UPDATE_HOST'
	_program_exec_and_record_completion__operation ()
	{
		message_echo "Extracting the upgrade archive..."
		(
			cd "$opt_basedir"/store
			rm -rf work
			mkdir work
			tar xzf portsreinstall-upgrade.tar.gz -C work
		)
	}
	program_exec_and_record_completion EXTRACT_SCRIPT_ARCHIVE
	PROGRAM_DEPENDS='EXTRACT_SCRIPT_ARCHIVE'
	_program_exec_and_record_completion__operation ()
	{
		message_echo "Updating the packages at the host environment..."
		(
			cd "$opt_basedir"/store
			opt_quiet=
			[ $opt_batch_mode = yes ] && opt_quiet='-a '
			work/portsreinstall-upgrade -S clean
			work/portsreinstall-upgrade -S $opt_quiet -P "${PACKAGES}/${PKGREPOSITORYSUBDIR}"
		)
	}
	program_exec_and_record_completion UPGRADE_HOST
}

# ============= Main operation of do/redo =============
command_do_main ()
{
	# Initial clean up
	command_do_chroot_cleanup
	
	# Build and mount of the chroot environment
	fs_build_chroot
	fs_terminate_if_mount_unavailable
	fs_mount
	
	# Enter the guest
	command_do_chroot_enter_if_incomplete
	
	# Pack (re)installed packages at the guest
	command_do_chroot_pack_update
	
	# Update of the files affecting package installation of the host environment 
	command_do_chroot_update_host_files
	
	# Update of packages at the host environment
	command_do_chroot_update_host_pkgs
}
