#!/bin/sh -e
# ==============================================================================
# portsreinstall-chroot library script for portsreinstall-chroot
# - File system operations -
# Copyright (C) 2018 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Check the safety of the base directory of builder chroot environment =============
fs_chk_safety_basedir ()
{
	local basedir
	basedir=$1
	[ -n "$basedir" ] || return
	case "$basedir" in
	/|/bin|/boot|/compat|/dev|/entropy|/etc|/home|/host|/lib|/libexec|/net|/proc|/rescue|/root|/sbin|/sys|/tmp|/usr|/var)
		return 1
		;;
	esac
	expr "$basedir" : '^/boot/.*' > /dev/null && return 1
	expr "$basedir" : '^/compat/.*' > /dev/null && return 1
	expr "$basedir" : '^/dev/.*' > /dev/null && return 1
	expr "$basedir" : '^/etc/.*' > /dev/null && return 1
	expr "$basedir" : '^/lib/.*' > /dev/null && return 1
	expr "$basedir" : '^/libexec/.*' > /dev/null && return 1
	expr "$basedir" : '^/proc/.*' > /dev/null && return 1
	expr "$basedir" : '^/sbin/.*' > /dev/null && return 1
	:
}

# ============= Safeguard of the base directory of builder chroot environment =============
fs_safeguard_basedir ()
{
	local basedir
	basedir=$1
	fs_chk_safety_basedir "$basedir" && return
	message_echo "ERROR: The base directory [$opt_basedir] is not safe." >&2
	exit 1
}

# ============= Build the file systems for the builder chroot environment =============
fs_build_chroot ()
{
	local systembase
	systembase=$1
	[ -e "${DBDIR}/mount_manifest" ] && return
	message_echo "Building the file systems for builder chroot environment (if not yet)."
	fs_safeguard_basedir "$opt_basedir"
	fs_unmount "$systembase"
	mkdir -p "$systembase/$opt_basedir"
	# Prescan the f file system of the original environment
	cp /dev/null "${TMPDIR}/fs_build_chroot:directories"
	(
		{
			echo bin compat etc lib libexec root sbin sys usr var | tr ' ' '\n'
			echo "$opt_extra_dirs" | tr "$opt_extra_dirs_delim" '\n'
		} | grep -v '^[[:space:]]*$' | sort -u > ${TMPDIR}/fs_build_chroot:sys_dirs
		sysdirs_ptn="^/*(`str_escape_regexp_filter < ${TMPDIR}/fs_build_chroot:sys_dirs | tr '\n' '|' | sed 's/\|$//'`)/+"
		while read directory
		do
			[ -e "$systembase"/$directory ] || continue
			if [ -L "$systembase"/$directory ]
			then
				src_mountpoint_real=`realpath "$systembase"/$directory`
					printf '%s\t%s\n' link $directory >> ${TMPDIR}/fs_build_chroot:directories
				if ! echo "$src_mountpoint_real/" | grep -qE "$sysdirs_ptn"
				then
					printf '%s\t%s\n' real "$src_mountpoint_real" >> ${TMPDIR}/fs_build_chroot:directories
					tmpdir_descendant=${TMPDIR}/fs_build_chroot:descendant/$src_mountpoint_real
					mkdir -p "$tmpdir_descendant"
					misc_get_descendant_mount_info "$systembase/$src_mountpoint_real" > $tmpdir_descendant/list
				fi
			elif [ -d "$systembase"/$directory ]
			then
				printf '%s\t%s\n' real $directory >> ${TMPDIR}/fs_build_chroot:directories
				tmpdir_descendant=${TMPDIR}/fs_build_chroot:descendant/$directory
				mkdir -p "$tmpdir_descendant"
				misc_get_descendant_mount_info "$systembase/$directory" > $tmpdir_descendant/list
			fi
		done < ${TMPDIR}/fs_build_chroot:sys_dirs
	)
	# Prepare the grand base of the chroot environment
	(
		cd "$systembase/$opt_basedir"
		for directory in builder mask store
		do
			[ -d $directory ] || mkdir $directory
		done
	)
	# Build target directories and the manifest for mounting
	cp /dev/null "${DBDIR}/mount_manifest.tmp"
	(
		cd "$systembase/$opt_basedir"/builder
		while read srcline
		do
			type=`echo "$srcline" | cut -f 1`
			directory=`echo "$srcline" | cut -f 2`
			case $type in
			link )
				[ -e "$directory" -o -L "$directory" ] || cp -RpP "$systembase/$directory" .
				;;
			real )
				mkdir -p "./$directory"
				masktarget=$systembase/$opt_basedir/mask/$directory
				mkdir -p "$masktarget"
				printf '%s\t%s\t%s\t%s\n' nullfs "$systembase"/$directory $directory ro >> ${DBDIR}/mount_manifest.tmp
				printf '%s\t%s\t%s\t%s\n' unionfs "$masktarget" $directory noatime >> ${DBDIR}/mount_manifest.tmp
				while read srcline
				do
					fs=`echo "$srcline" | cut -f 1`
					mp=`echo "$srcline" | cut -f 2`
					relative=`echo "$srcline" | cut -f 3`
					case $fs in
						normal )
							masktarget=$systembase/$opt_basedir/mask/$directory/$relative
							mkdir -p "$masktarget"
							printf '%s\t%s\t%s\t%s\n' nullfs "$mp" "$directory/$relative" ro >> ${DBDIR}/mount_manifest.tmp
							printf '%s\t%s\t%s\t%s\n' unionfs "$masktarget" "$directory/$relative" noatime >> ${DBDIR}/mount_manifest.tmp
							;;
						devfs )
							printf '%s\t%s\t%s\t%s\n' devfs devfs "$directory/$relative" '' >> ${DBDIR}/mount_manifest.tmp
							;;
						fdescfs )
							printf '%s\t%s\t%s\t%s\n' fdescfs fdesc "$directory/$relative" '' >> ${DBDIR}/mount_manifest.tmp
							;;
						procfs )
							printf '%s\t%s\t%s\t%s\n' procfs proc "$directory/$relative" '' >> ${DBDIR}/mount_manifest.tmp
							;;
						linprocfs )
							printf '%s\t%s\t%s\t%s\n' linprocfs linproc "$directory/$relative" '' >> ${DBDIR}/mount_manifest.tmp
							;;
						tmpfs )
							printf '%s\t%s\t%s\t%s\n' tmpfs tmpfs "$directory/$relative" mode=1777 >> ${DBDIR}/mount_manifest.tmp
							;;
					esac
				done < ${TMPDIR}/fs_build_chroot:descendant/$directory/list
				;;
			esac
		done < ${TMPDIR}/fs_build_chroot:directories
		for directory in dev proc tmp 
		do
			[ -e $directory ] || mkdir $directory
		done
		printf '%s\t%s\t%s\t%s\n' devfs devfs dev '' >> ${DBDIR}/mount_manifest.tmp
		printf '%s\t%s\t%s\t%s\n' fdescfs fdesc dev/fd '' >> ${DBDIR}/mount_manifest.tmp
		printf '%s\t%s\t%s\t%s\n' procfs proc proc '' >> ${DBDIR}/mount_manifest.tmp
		printf '%s\t%s\t%s\t%s\n' tmpfs tmpfs tmp mode=1777 >> ${DBDIR}/mount_manifest.tmp
		mkdir -p ."${PROGRAM}"
		cd "$systembase/$opt_basedir/mask"
		if [ ! -e root/.cshrc ]
		then
			tmp_cshrc=${TMPDIR}/fs_build_chroot:.cshrc
			[ -d root ] || mkdir root
			if [ -e "$systembase"/root/.cshrc ]
			then
				cp -p "$systembase"/root/.cshrc "$tmp_cshrc"
				cp -p "$systembase"/root/.cshrc "root/.cshrc.bak-${APPNAME}"
			elif [ -e "$systembase"/usr/share/skel/dot.cshrc ]
			then
				cp -p "$systembase"/usr/share/skel/dot.cshrc "$tmp_cshrc"
				touch "root/.cshrc.bak-${APPNAME}"
			else
				cp /dev/null "$tmp_cshrc"
			fi
			echo >> $tmp_cshrc
			echo 'set prompt="%N@[builder]%m:%~ %# "' >> $tmp_cshrc
			mv "$tmp_cshrc" root/.cshrc
		fi
		printf '%s\t%s\t%s\t%s\n' nullfs "$systembase/$opt_basedir"/store ".${PROGRAM}" '' >> ${DBDIR}/mount_manifest.tmp
	)
	mv "${DBDIR}/mount_manifest.tmp" "${DBDIR}/mount_manifest"
}

# ============= Check whether the file systems for the builder chroot environment are all mounted =============
fs_chk_mount ()
{
	local systembase
	systembase=$1
	[ -e "${DBDIR}/mount_manifest" ] || return
	rm -rf ${TMPDIR}/fs_chk_mount:remains
	while read srcline
	do
		type=`echo "$srcline" | cut -f 1`
		target=`echo "$srcline" | cut -f 2`
		[ "x$type" = xnullfs -o "x$type" = xunionfs ] && target=$systembase/$target
		directory=`echo "$srcline" | cut -f 3`
		opt=`echo "$srcline" | cut -f 4`
		if ! misc_chk_mounted "$type" "$target" "$systembase/$opt_basedir/builder/$directory"
		then
			touch "${TMPDIR}"/fs_chk_mount:remains
			break
		fi
	done < ${DBDIR}/mount_manifest
	[ ! -e "${TMPDIR}"/fs_chk_mount:remains ]
}

# ============= Terminate when the file systems for the builder chroot environment cannot be mounted =============
fs_terminate_if_mount_unavailable ()
{
	local systembase
	systembase=$1
	fs_chk_mount "$systembase" && return
	misc_chk_mount_privilege && return
	temp_terminate_process ()
	{
		local errno basedir
		errno=${1:-0}
		[ $opt_batch_mode = yes ] && return
		if [ $errno -ne 2 ]
		then
			message_echo "Aborted by unexpected error" >&2
			exit
		fi
		message_echo
		message_echo "INFO: Terminated for mounting file systems because this utility was executed at a virtual (chroot or jail) environment."
		message_echo "Execute"
		basedir=`misc_get_system_basedir`
		if [ -n "$basedir" ]
		then
			message_echo "  $basedir${SHAREDIR}/bin/portsreinstall-chroot-mount"
			message_echo "at the grand host environment."
		else
			message_echo "  \$BASEDIR${SHAREDIR}/bin/portsreinstall-chroot-mount"
			message_echo "at the grand host environment, where \$BASEDIR denotes the base directory of this virtual environment."
		fi
		message_echo "After its successful execution, rerun"
		if [ -n "$COMMAND_RESTART" ]
		then
			message_echo "  ${APPNAME} $COMMAND_RESTART"
		else
			message_echo "  ${APPNAME}"
		fi
	}
	exit 2
}

# ============= Mount the file systems for the builder chroot environment if not yet =============
fs_mount ()
{
	local systembase
	systembase=$1
	message_echo "Mounting the file systems for builder chroot environment."
	fs_safeguard_basedir "$opt_basedir"
	while read srcline
	do
		type=`echo "$srcline" | cut -f 1`
		target=`echo "$srcline" | cut -f 2`
		[ "x$type" = xnullfs -o "x$type" = xunionfs ] && target=$systembase/$target
		directory=`echo "$srcline" | cut -f 3`
		opt=`echo "$srcline" | cut -f 4`
		mp=$systembase/$opt_basedir/builder/$directory
		if ! misc_chk_mounted "$type" "$target" "$mp"
		then
			mount -t "$type" -o "$opt" "$target" "$mp"
		fi
	done < ${DBDIR}/mount_manifest
	if ! fs_chk_mount "$systembase"
	then
		message_echo "Error: Failed to mount the file systems. Some of them remain unmounted." >&2
		exit 1
	fi
	message_echo "Mounting done."
}

# ============= Check whether the file systems for the builder chroot environment are all unmounted or destroyed =============
fs_chk_unmount ()
{
	local systembase
	systembase=$1
	[ -e "${DBDIR}/mount_manifest" ] || return 0
	rm -rf ${TMPDIR}/fs_chk_unmount:remains
	tail -r "${DBDIR}/mount_manifest" | while read srcline
	do
		type=`echo "$srcline" | cut -f 1`
		target=`echo "$srcline" | cut -f 2`
		[ "x$type" = xnullfs -o "x$type" = xunionfs ] && target=$systembase/$target
		directory=`echo "$srcline" | cut -f 3`
		opt=`echo "$srcline" | cut -f 4`
		if misc_chk_mounted "$type" "$target" "$systembase/$opt_basedir/builder/$directory"
		then
			touch "${TMPDIR}"/fs_chk_unmount:remains
			break
		fi
	done
	[ ! -e "${TMPDIR}"/fs_chk_unmount:remains ]
}

# ============= Terminate when the file systems for the builder chroot environment cannot be unmounted =============
fs_terminate_if_unmount_unavailable ()
{
	local systembase
	systembase=$1
	fs_chk_unmount "$systembase" && return
	misc_chk_unmount_privilege && return
	temp_terminate_process ()
	{
		local errno basedir
		errno=${1:-0}
		[ $opt_batch_mode = yes ] && return
		if [ $errno -ne 3 ]
		then
			message_echo "Aborted by unexpected error" >&2
			exit
		fi
		message_echo
		message_echo "INFO: Terminated for unmounting file systems because this utility was executed at a virtual (chroot or jail) environment."
		message_echo "Execute"
		basedir=`misc_get_system_basedir`
		if [ -n "$basedir" ]
		then
			message_echo "  $basedir${SHAREDIR}/bin/portsreinstall-chroot-mount unmount"
			message_echo "at the grand host environment."
		else
			message_echo "  \$BASEDIR${SHAREDIR}/bin/portsreinstall-chroot-mount unmount"
			message_echo "at the grand host environment, where \$BASEDIR denotes the base directory of this virtual environment."
		fi
		message_echo "After its successful execution, rerun"
		if [ -n "$COMMAND_RESTART" ]
		then
			message_echo "  ${APPNAME} $COMMAND_RESTART"
		else
			message_echo "  ${APPNAME}"
		fi
	}
	exit 3
}

# ============= Unmount  file systems for the chroot environment =============
fs_unmount ()
{
	local systembase
	systembase=$1
	[ ! -d "$systembase/$opt_basedir"/builder ] && return
	[ -e "${DBDIR}/mount_manifest" ] || return 0
	message_echo "Unmounting the file systems for builder chroot."
	fs_safeguard_basedir "$opt_basedir"
	tail -r "${DBDIR}/mount_manifest" | while read srcline
	do
		type=`echo "$srcline" | cut -f 1`
		target=`echo "$srcline" | cut -f 2`
		[ "x$type" = xnullfs -o "x$type" = xunionfs ] && target=$systembase/$target
		directory=`echo "$srcline" | cut -f 3`
		opt=`echo "$srcline" | cut -f 4`
		mp=$systembase/$opt_basedir/builder/$directory
		if misc_chk_mounted "$type" "$target" "$mp"
		then
			umount -f "$mp"
		fi
	done
	if ! fs_chk_unmount "$systembase"
	then
		message_echo "Error: Failed to unmount the file systems. Some of them remain mounted." >&2
		exit 1
	fi
	message_echo "Unmounting done."
}

# ============= Destroy the chroot environment =============
fs_destroy ()
{
	local systembase
	systembase=$1
	fs_chk_safety_basedir "$opt_basedir" || return 0
	[ ! -d "$opt_basedir" ] && return
	fs_terminate_if_unmount_unavailable "$systembase"
	fs_unmount "$systembase"
	chflags -R noschg "$opt_basedir"
	rm -rf "$opt_basedir"
}
