#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Miscellaneous functions -
# Copyright (C) 2013-2018 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Variables =============
MISC_IS_READONLY_MODE=no
MISC_DEFAULT_CONSOLE_SIZE='25 80'

# ============= Check whether the current process is by the superuser privilege =============
misc_is_superuser_privilege ()
{
	[ `id -u` -eq 0 -a "$MISC_IS_READONLY_MODE" = no ]
}

# ============= Lock for duplicated executions =============
misc_lock_duplicated_executions ()
{
	local lockfile
	lockfile=$1
	MISC_IS_READONLY_MODE=no
	[ `id -u` -eq 0 ] || return 0
	if mktemp -q "$lockfile" > /dev/null
	then
		echo $$ > $lockfile
	elif [ x`cat "$lockfile" 2> /dev/null` != x$$ ]
	then
		message_echo "WARNING: Another ${APPNAME} process is running; commands requiring the superuser privilege is unavailable." >&2
		message_echo "If this report is incorrect, remove an obsolete lock file ($lockfile)." >&2
		message_echo >&2
		MISC_IS_READONLY_MODE=yes
	fi
	:
}

# ============= Abort if the execution is locked or the user does not have the superuser privilege =============
misc_chk_privilege ()
{
	misc_is_superuser_privilege && return
	[ "$MISC_IS_READONLY_MODE" = yes ] && message_echo "ERROR: Locked for this command." >&2
	[ `id -u` -ne 0 ] && message_echo "ERROR: The superuser privilege is required for this command." >&2
	exit 1
}

# ============= Get all shell variable definitions (without functions) =============
# Actually, this filtering is unnecessary for Bourne shell and only required for Bash.
# Since the overhead is not harmful, I will keep this for correspondence to possible future changes.
misc_get_all_vardefs ()
{
	set | sed -E '/^[^ ]+ \(\) *$/,/^\}$/d'
}

# ============= Initialize shell variable definitions =============
misc_init_vardefs ()
{
	eval `misc_get_all_vardefs | grep -E '^[a-z_][a-zA-Z0-9_]+=' | sed 's/=.*//;s/^/unset /'`
}

# ============= Get the size of the current console =============
misc_get_console_size ()
{
	local size
	size=`stty -f /dev/stdin size 2> /dev/null || :`
	if [ -n "$size" ]
	then
		echo "$size" > ${TMPDIR}/misc_get_console_size::size
	else
		if [ ! -e "${TMPDIR}/misc_get_console_size::size" ]
		then
			size=`stty -f /dev/console size 2> /dev/null || :`
		else
			size=`cat "${TMPDIR}/misc_get_console_size::size"`
		fi
	fi
	[ -n "$size" ] || size=$MISC_DEFAULT_CONSOLE_SIZE
	echo "$size"
}

# ============= Get the number of columns for the current console =============
misc_get_console_column_size ()
{
	misc_get_console_size | sed -E 's/^[^ ]+ +([^ ]+)/\1/'
}

# ============= Get the number of rows for the current console =============
misc_get_console_row_size ()
{
	misc_get_console_size | sed -E 's/^([^ ]+) +[^ ]+/\1/'
}

# ============= Get the global path of a possibly not yet created file/directory =============
misc_global_path ()
{
	local $path_src
	path_src=$1
	if [ -e "$path_src" ]
	then
		realpath "$path_src"
	else
		expr "$path_src" : '^/' > /dev/null || echo -n `realpath .`
		echo "$path_src"
	fi
}

# ============= Inspect the privilege of the current environment on file system operation =============
misc_inspect_fs_privilege ()
{
	local tgv mp mp_regexp mount_privilege basedir
	[ -d "${TMPDIR}"/fs_privilege ] && return
	tgv=${TMPDIR}/fs_privilege/test_tg
	mkdir -p "$tgv" "${TMPDIR}"/fs_privilege/test_mp
	mp=`realpath "${TMPDIR}"/fs_privilege/test_mp`
	mp_regexp=`str_escape_regexp "$mp"`
	echo yes > ${TMPDIR}/fs_privilege/fs_privilege
	cat > ${TMPDIR}/fs_privilege/fslist << eof
devfs	devfs
fdescfs	null
procfs	proc
tmpfs	tmpfs
nullfs	"$tgv"
unionfs	"$tgv"
eof
	while read fs tg
	do
		if mount -t $fs "$tgv" "$mp" 2> /dev/null && umount "$mp"
		then
			echo yes
		else
			echo no
			echo no > ${TMPDIR}/fs_privilege/fs_privilege
		fi > ${TMPDIR}/fs_privilege/fs_privilege:$fs 2> /dev/null
		umount -f "$mp" 2> /dev/null || :
		[ -e "${TMPDIR}"/fs_privilege/basedir ] && continue
		mount -t $fs "$tgv" "$mp" 2> /dev/null && \
			df "$mp" > ${TMPDIR}/fs_privilege/df:$fs && \
			umount "$mp"
		real_mp=`sed 1d "${TMPDIR}"/fs_privilege/df:$fs | tail -n 1 | \
			sed -E 's/^.*[[:space:]][0-9]+%[[:space:]]+//'`
		echo "$real_mp" | sed -E "s/$mp_regexp$//" > ${TMPDIR}/fs_privilege/basedir
	done < ${TMPDIR}/fs_privilege/fslist
	mount_privilege=`cat "${TMPDIR}"/fs_privilege/fs_privilege`
	if [ "x$mount_privilege" = xyes ]
	then
		mount -t nullfs /bin "$mp" 2> /dev/null
		if [ `ls "$mp" 2> /dev/null | wc -l` -gt 0 ]
		then
			echo yes
		else
			echo no
		fi > ${TMPDIR}/fs_privilege/nullfs_target_recognition
		umount -f "$mp" 2> /dev/null || :
		nullfs_target_recognition=`cat "${TMPDIR}"/fs_privilege/nullfs_target_recognition`
		if [ "x$nullfs_target_recognition" = xyes ]
		then
			message_echo "INFO: The current environment has the full required privilege of mounting/unmounting file systems."
		else
			message_echo "INFO: The current environment formally has the full required privilege of mounting/unmounting file systems but the recognition of nullfs/unionfs targets is incorrect."
		fi
	else
		echo no > ${TMPDIR}/fs_privilege/nullfs_target_recognition
		message_echo "INFO: The current environment does not have the privilege of mounting/unmounting for the following file system(s)."
		while read fs tg
		do
			 mount_privilege=`cat "${TMPDIR}"/fs_privilege/fs_privilege:$fs`
			 [ "x$mount_privilege" = xyes ] || echo '  '$fs
		done < ${TMPDIR}/fs_privilege/fslist | message_cat
	fi
	basedir=`cat "${TMPDIR}"/fs_privilege/basedir 2> /dev/null || :`
	if [ -n "$basedir" ]
	then
		message_echo "INFO: The current environment will be a chroot/jail guest whose base directory is \"$basedir\"."
	fi
	if [ "x$opt_invalidate_mount_privilege" = xyes ]
	then
		message_echo "INFO: The privilege of mounting/unmounting in this environment is forcibly invalidated."
	fi
}

# ============= Check whether mounting file systems are available at the current environment =============
misc_chk_mount_privilege ()
{
	local mount_privilege
	misc_inspect_fs_privilege
	[ "x$opt_invalidate_mount_privilege" = xno ] || return
	mount_privilege=`cat "${TMPDIR}"/fs_privilege/fs_privilege`
	nullfs_target_recognition=`cat "${TMPDIR}"/fs_privilege/nullfs_target_recognition`
	[ "x$nullfs_target_recognition" = xyes -a "x$mount_privilege" = xyes ]
}

# ============= Check whether mounting file systems are available at the current environment =============
misc_chk_unmount_privilege ()
{
	local mount_privilege
	misc_inspect_fs_privilege
	[ "x$opt_invalidate_mount_privilege" = xno ] || return
	mount_privilege=`cat "${TMPDIR}"/fs_privilege/fs_privilege`
	[ "x$mount_privilege" = xyes ]
}

# ============= Get the base directory the current environment (applicable in case of a chroot guest) =============
misc_get_system_basedir ()
{
	misc_inspect_fs_privilege
	cat "${TMPDIR}"/fs_privilege/basedir 2> /dev/null || :
}

# ============= Check whether the current environment is in a jail =============
misc_chk_in_jail ()
{
	local status
	status=`sysctl -n security.jail.jailed`
	[ x"$status" != x0 ]
}

# ============= Get the regular expression pattern of the actual mount point =============
misc_get_actual_mount_point_pattern ()
{
	local mountpoint basedir mountpoint_real
	mountpoint=$1
	[ -e  "$mountpoint" ] || return
	misc_inspect_fs_privilege
	basedir=`cat "${TMPDIR}"/fs_privilege/basedir 2> /dev/null || :`
	mountpoint_real=`realpath "$mountpoint"`
	mountpoint_real_full=`echo "$basedir$mountpoint_real" | sed 's|//*|/|'`
	str_escape_regexp "$mountpoint_real_full"
}

# ============= Get mount info at the descendant directory levels required for builder chroot environment =============
misc_get_descendant_mount_info ()
{
	local mountpoint mountpoint_real_regexp basedir
	mountpoint=$1
	mountpoint_real_regexp=`misc_get_actual_mount_point_pattern "$mountpoint"` || return
	basedir=`cat "${TMPDIR}"/fs_privilege/basedir 2> /dev/null || :`
	basedir_ptn=`str_escape_regexp "$basedir"`
	df | sed 1d | grep -E "%[[:space:]]+$mountpoint_real_regexp\/" | while read fs data
	do
		echo "$fs" | grep -q -e '^/' -e '^<above>:' && fs=normal
		mp_abs=`echo "$data" | sed -E  's|.*%[[:space:]]+(/.+)$|\1|'`
		mp=`echo "$mp_abs" | sed -E "s|^$basedir_ptn||"`
		relative=`echo "$mp_abs" | sed -E "s|^$mountpoint_real_regexp||"`
		printf '%s\t%s\t%s\n' "$fs" "$mp" "$relative"
	done
}

# ============= Check whether a directory is mounted properly =============
misc_chk_mounted ()
{
	local type target mountpoint target_ptn mountpoint_real_regexp tmpsrc
	type=$1
	target=$2
	mountpoint=$3
	target_ptn=`echo "$target" | sed 's|//*|/|g' | str_escape_regexp_filter`
	mountpoint_real_regexp=`misc_get_actual_mount_point_pattern "$mountpoint"` || return
	basedir=`cat "${TMPDIR}"/fs_privilege/basedir 2> /dev/null || :`
	basedir_target_ptn=`echo "$basedir/$target" | sed 's|//*|/|g' | str_escape_regexp_filter`
	tmpsrc=${TMPDIR}/misc_chk_mounted:src
	df | sed 1d | grep -E "%[[:space:]]+$mountpoint_real_regexp$" > $tmpsrc
	case $type in
		nullfs )
			grep -qE "^${target_ptn}[[:space:]]" "$tmpsrc" || grep -qE "^${basedir_target_ptn}[[:space:]]" "$tmpsrc"
			;;
		unionfs )
			grep -qE "^<above>:${target_ptn}[[:space:]]" "$tmpsrc" || grep -qE "^<above>:${basedir_target_ptn}[[:space:]]" "$tmpsrc"
			;;
		devfs | fdescfs | procfs | linprocfs | tmpfs )
			grep -q "^$type" "$tmpsrc"
			;;
		*)
			message_echo "ERROR: Unsupported fyle system [$type]" >&2
			exit 1
			;;
	esac
}

# ============= Selection of removing leaf ports =============
# Box options for dialog(1) are given via stdin.
misc_dialog_checklist ()
{
	local title desc dstfile itemlist cmdscript LINES COLUMNS hight width width_desc lines_desc hight_list width_list nlines iline len_tag tag len_tag_cur len_item_max len_item_trim
	title=$1
	desc=$2
	dstfile=$3
	itemlist=$4
	cmdscript=${TMPDIR}/misc_dialog_checklist::command
	if [ `wc -l < $itemlist` -eq 0 ]
	then
		cp /dev/null "$dstfile"
		return 0
	fi
	echo -n 'dialog --title "$title" --checklist "$desc" $hight $width $hight_list ' > $cmdscript
	COLUMNS=`misc_get_console_column_size`
	LINES=`misc_get_console_row_size`
	hight=$(($LINES-3))
	[ $hight -gt 0 ] || hight=$LINES
	width=$(($COLUMNS-5))
	[ $width -gt 0 ] || width=$COLUMNS
	width_desc=$(($width-4))
	[ $width_desc -gt 0 ] || width_desc=$COLUMNS
	lines_desc=`echo "$desc" | sed -E 's/\\n/\
/g' | fold -s -w $width_desc | wc -l`
	hight_list=$(($hight-$lines_desc-6))
	[ $hight_list -gt 0 ] || hight_list=$LINES
	width_list=$(($width-6))
	[ $width_list -gt 0 ] || width_list=$COLUMNS
	nlines=`wc -l < $itemlist`
	iline=1
	len_tag=0
	while [ $iline -le $nlines ]
	do
		tag=`sed -n "${iline}p" "$itemlist" | cut -f 1`
		iline=$(($iline+1))
		len_tag_cur=`echo -n "$tag" | wc -c`
		[ $len_tag_cur -gt $len_tag ] && len_tag=$len_tag_cur
	done
	iline=1
	len_item_max=$(($width_list-$len_tag-6))
	[ $len_item_max -gt 0 ] || len_item_max=$width_list
	len_item_trim=$(($len_item_max-3))
	[ $len_item_trim -gt 0 ] || len_item_trim=$len_item_max
	while [ $iline -le $nlines ]
	do
		tag=`sed -n "${iline}p" "$itemlist" | cut -f 1`
		item=`sed -n "${iline}p" "$itemlist" | cut -f 2`
		status=`sed -n "${iline}p" "$itemlist" | cut -f 3`
		iline=$(($iline+1))
		len_item=`echo -n "$item" | wc -c`
		if [ $len_item -gt $len_item_max ]
		then
			item=`echo -n "$item" | head -c $len_item_trim`...\"
		fi
		echo -n "$tag $item $status "
	done >> $cmdscript
	echo ' 2> ${TMPDIR}/misc_dialog_checklist::selected_items || :' >> $cmdscript
	. "$cmdscript"
	tr -d \" < ${TMPDIR}/misc_dialog_checklist::selected_items | tr ' ' '\n' > $dstfile
	echo >> $dstfile
}

