# session.rb
# $Id: session.rb,v 1.1.1.1.2.1 2003/12/29 09:50:35 komatsu Exp $
#
# Copyright (C) 2001 Satoru Takabayashi <satoru@namazu.org>
# Copyright (C) 2002 Hiroyuki Komatsu <komatsu@taiyaki.org>
#     All rights reserved.
#     This is free software with ABSOLUTELY NO WARRANTY.
#
# You can redistribute it and/or modify it under the terms of 
# the GNU General Public License version 2.
#

require 'socket'
require 'thread'

module PogemoServer
  class Command
    def initialize (name, args, description, min_nargs = nil)
      @name = name
      @args = args
      @nargs = args.length
      @min_nargs = (min_nargs or @nargs)
      @description = description
    end
    attr_reader :name
    attr_reader :args
    attr_reader :nargs
    attr_reader :min_nargs
    attr_reader :description
  end

  class SessionCore
    def initialize (pogemo, inport, outport)
      @in = inport
      @out = outport
      @pogemo  = pogemo
      @name    = "no name"
      @id      = "no ID"
      @debug   = false
      @command_table = Hash.new
      init_command_table
    end
    attr_accessor :name

    def set_debug (filename)
      @debug    = true
      @debug_io = File::open(filename, "w")
      @debug_io.chmod(0600) if @debug_io::stat.owned?
    end

    def init_command_table
      add_command(:close, [], "close the connection")
      add_command(:help, [], "print the help message")
      add_command(:version, [], "show the version number")
    end

    def add_command (name, args, description, min_nargs = nil)
      command = Command.new(name, args, description, min_nargs)
      @command_table[name] = command
    end

    def reply_successful
      @out.puts "ok"
      yield if block_given?
      @out.puts
    end

    def reply_unsuccessful
      @out.puts "error"
      yield if block_given?
      @out.puts "Try `help' for protocol information."
      @out.puts
    end

    def help
      commands = @command_table.values.sort {|a, b| 
	(a.nargs <=> b.nargs).nonzero? || a.name.to_s <=> b.name.to_s
      }
      reply_successful {
	commands.each {|c|
	  @out.printf "%s\t%s\t- %s\n", c.name, c.args.join(" "), c.description
	}
	@out.puts "Note: Use TAB for delimiters."
      }
    end

    def close
      @out.puts @pogemo.close
      @out.close
    end

    def version
      reply_successful {
	@out.puts "#{@name}\t#{@version}\t#{@id}"
      }
    end

    def execute (name, args)
      Mutex.new.synchronize {
	command = @command_table[name]
	if command.nil?
	  reply_unsuccessful {
	    @out.puts "Unknown command: #{name}"
	  }
	elsif command.min_nargs <= args.length and args.length <= command.nargs
	  send(command.name, *args)
	else
	  reply_unsuccessful {
	    @out.puts"Wrong number of arguments (expecting #{command.nargs})"
	  }
	end
      }
    end

    def start
      while line = @in.gets do
	logging(line)
	chunks = line.gsub(/[\r\n]/, "").split("\t")
	if chunks.length > 0 then
	  name = chunks.shift.intern
	  args = chunks
	  execute(name, args)
	  return if @in.closed? or @out.closed?
	end
      end
    end

    def logging(line)
      if @debug then
        line.chomp()
        @debug_io.puts(line)
        @debug_io.flush()
      end
    end
  end

  class Server
    def initialize (pogemo)
      @pogemo = pogemo
      @debug  = false
      @logfile = nil
    end

    public
    def accept
    end

    def set_debug (logfile)
      @debug   = true
      @logfile = logfile
    end
  end

  class UnixSocketServer < Server
    def initialize (pogemo, path)
      super(pogemo)
      @server = UNIXServer.new(path)
    end

    def accept
      while true
	socket = @server.accept
	t = Thread.new {
	  s = socket
	  session = Session.new(@pogemo, s, s)
          if @debug then
            session.set_debug(@logfile)
          end
	  session.start
	}
	t.abort_on_exception = true
      end
    end
  end

  class TaiyakiTCPServer < Server
    def initialize (pogemo, port)
      super(pogemo)
      @server = TCPServer.new(port)
    end

    def accept
      while true
	socket = @server.accept
	t = Thread.new {
	  s = socket
	  session = Session.new(@pogemo, s, s)
          if @debug then
            session.set_debug(@logfile)
          end
	  session.start
	}
	t.abort_on_exception = true
      end
    end
  end

  class StdioServer < Server
    def initialize (pogemo)
      super(pogemo)
      $stdout.sync = true
    end

    def accept
      session = Session.new(@pogemo, $stdin, $stdout)
      if @debug then
        session.set_debug(@logfile)
      end
      session.start
    end
  end
end
