;;*****************************************************************************
;;*****************************************************************************
;;  FILENAME: SDCard_SPI.asm
;;  Version: 1.2, Updated on 2009/5/7 at 21:11:30
;;  Generated by PSoC Designer 5.0.423.0
;;
;;  DESCRIPTION: SDCard User Module software implementation file
;;               for the low level SPI hardware.
;;
;;  NOTE: User Module APIs conform to the fastcall16 convention for marshalling
;;        arguments and observe the associated "Registers are volatile" policy.
;;        This means it is the caller's responsibility to preserve any values
;;        in the X and A registers that are still needed after the API functions
;;        returns. For Large Memory Model devices it is also the caller's 
;;        responsibility to perserve any value in the CUR_PP, IDX_PP, MVR_PP and 
;;        MVW_PP registers. Even though some of these registers may not be modified
;;        now, there is no guarantee that will remain the case in future releases.
;;-----------------------------------------------------------------------------
;;  Copyright (c) Cypress MicroSystems 2000-2006. All Rights Reserved.
;;*****************************************************************************
;;*****************************************************************************

include "m8c.inc"
include "memory.inc"
include "SDCard_SPI.inc"

;-----------------------------------------------
;  Global Symbols
;-----------------------------------------------
export   SDCard_InitHdwr
export  _SDCard_InitHdwr
export   SDCard_UnInitHdwr
export  _SDCard_UnInitHdwr
export   SDCard_Select 
export  _SDCard_Select
export   SDCard_SendTxData
export  _SDCard_SendTxData
export   SDCard_bReadRxData
export  _SDCard_bReadRxData
export   SDCard_bReadStatus
export  _SDCard_bReadStatus
export   SDCard_WriteBuff256
export  _SDCard_WriteBuff256
export   SDCard_ReadBuff256
export  _SDCard_ReadBuff256
export   SDCard_WriteBuff
export  _SDCard_WriteBuff
export   SDCard_ReadBuff
export  _SDCard_ReadBuff


;-----------------------------------------------
;  Constant Definitions
;-----------------------------------------------
bfCONTROL_REG_START_BIT:   equ   1     ; Control register start bit

;-----------------------------------------------
; Variable Allocation
;-----------------------------------------------
area SDCard_RAM (RAM, REL, CON)
    
 pMVI:   BLK  1


area UserModules (ROM, REL)


.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: SDCard_InitHdwr
;
;  DESCRIPTION:
;     Sets the start bit, SPI mode, and LSB/MSB first configuration of the SPIM
;     user module.
;
;     Transmission will begin transmitting when a byte is written into the TX buffer
;     using the SDCard_SendTxData function.
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;     BYTE bConfiguration - Consists of SPI Mode and LSB/MSB first bit.
;           Use defined masks - masks can be OR'd together.
;     PASSED in Accumulator.
;
;  RETURNS:  none
;
;  SIDE EFFECTS: 
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to perserve their values across calls to fastcall16 
;    functions.
;
;  THEORY of OPERATION or PROCEDURE:
;     1) Set all Slave Select outputs high
;     2) Set the specified SPI configuration bits in the Control register.
;
 SDCard_InitHdwr:
_SDCard_InitHdwr:
   RAM_PROLOGUE RAM_USE_CLASS_4
   RAM_SETPAGE_CUR >Port_2_Data_SHADE
   push A
   ; Drive the Card Select pin high (Deselected)
   or    [Port_2_Data_SHADE],SDCard_CS_Mask
   mov  A,[Port_2_Data_SHADE]
   mov  reg[SDCard_CS_Port],A 
   
   ; setup the SPIM configuration setting
   pop   A
   or    A, bfCONTROL_REG_START_BIT
   mov   REG[SDCard_CONTROL_REG], A

   mov  A, REG[SDCard_CONTROL_REG]  ; Flush Status
   mov  A, REG[SDCard_RX_BUFFER_REG]     ; Dummy read to clear all



IF ( SDCard_ENABLE_PRESENT)
   ; Drive the Card Present pin high for resistive pullup
   or    [@SD_CD_Port_Data_SHADE],SDCard_CD_Mask
   mov  A,[@SD_CD_Port_Data_SHADE]
   mov  reg[SDCard_CD_Port],A 
ENDIF
   
IF ( SDCard_ENABLE_WPROTECT)
   ; Drive the Write Protect pin high for resistive pullup
   or    [@SD_WP_Port_Data_SHADE],SDCard_WP_Mask
   mov  A,[@SD_WP_Port_Data_SHADE]
   mov  reg[SDCard_WP_Port],A 
ENDIF
   
   RAM_EPILOGUE RAM_USE_CLASS_4
   ret

.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: SDCard_Stop
;
;  DESCRIPTION:
;     Disables SPIM operation.
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:  none
;
;  RETURNS:  none
;
;  SIDE EFFECTS: 
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to perserve their values across calls to fastcall16 
;    functions.
;
;  THEORY of OPERATION or PROCEDURE:
;     Clear the start bit in the Control register.
;
 SDCard_UnInitHdwr:
_SDCard_UnInitHdwr:
   RAM_PROLOGUE RAM_USE_CLASS_1

   ; Drive the Card Select pin high (Deselected)

   mov  A,0      ; Deselect Card
   call SDCard_Select

    ; clear the SPIM stop bits
   and   REG[SDCard_CONTROL_REG], ~bfCONTROL_REG_START_BIT
   
   RAM_EPILOGUE RAM_USE_CLASS_1
   ret

.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: void SDCard_Select(BYTE bEnable)
;
;  DESCRIPTION:
;     Enable or disable card select signal.
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;     BYTE bEnable => 0 Disable Card
;                  => 1 Enable Card
;     PASSED in Accumulator.
;
;  RETURNS:  none
;
;  SIDE EFFECTS: 
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to perserve their values across calls to fastcall16 
;    functions.
;
;  THEORY of OPERATION or PROCEDURE:
;
;-----------------------------------------------------------------------------
 SDCard_Select:
_SDCard_Select:
   RAM_PROLOGUE RAM_USE_CLASS_4
   RAM_SETPAGE_CUR >Port_2_Data_SHADE
   ; setup the SPIM configuration setting
   and   A, 0xFF
   jz    .DeSelectCard

.SelectCard:
   and   [Port_2_Data_SHADE],~SDCard_CS_Mask
   mov  A,[Port_2_Data_SHADE]
   mov  reg[SDCard_CS_Port],A 
   jmp  .EndSelect

.DeSelectCard:
   or    [Port_2_Data_SHADE],SDCard_CS_Mask
   mov  A,[Port_2_Data_SHADE]
   mov  reg[SDCard_CS_Port],A 

.EndSelect:
   
   RAM_EPILOGUE RAM_USE_CLASS_4
   ret

.ENDSECTION


.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: SDCard_SendTxData
;
;  DESCRIPTION:
;     Initiates an SPI data transfer.
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;     BYTE  bTxData - data to transmit.
;        Passed in Accumulator.
;
;  RETURNS:  none
;
;  SIDE EFFECTS: 
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to perserve their values across calls to fastcall16 
;    functions.
;
;  THEORY of OPERATION or PROCEDURE:
;     Writes data to the TX buffer register.
;
 SDCard_SendTxData:
_SDCard_SendTxData:
   RAM_PROLOGUE RAM_USE_CLASS_1
   ; Transmit the data!
   mov REG[SDCard_TX_BUFFER_REG], A
   RAM_EPILOGUE RAM_USE_CLASS_1
   ret

.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: SDCard_WriteBuff256(char * sRamBuff)
;
;  DESCRIPTION:
;     Writes a 256 byte buffer to the SPI port
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;     A:X  Pointer to String
;          A contains MSB of string address
;          X contains LSB of string address
;
;
;  RETURNS:  none
;
;  SIDE EFFECTS: 
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to perserve their values across calls to fastcall16 
;    functions.
;
;
;  THEORY of OPERATION or PROCEDURE:
;     Writes data to the TX buffer register.
;
 SDCard_WriteBuff256:
_SDCard_WriteBuff256:
    RAM_PROLOGUE RAM_USE_CLASS_3
   ; Transmit the data!


   RAM_SETPAGE_IDX A
   mov  X,0                              ; Assume 256 byte buffer is full page
.DataLoop:
   mov  A,[X]                            ; Get data
   mov  REG[SDCard_TX_BUFFER_REG], A
.RdyLoop:
   tst  REG[SDCard_CONTROL_REG], SDCard_SPIM_SPI_COMPLETE
   jz   .RdyLoop
   inc  X
   jnz  .DataLoop

   RAM_EPILOGUE RAM_USE_CLASS_3
   ret

.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: SDCard_WriteBuff(BYTE * pBuff, BYTE bCnt)
;
;  DESCRIPTION:
;     Writes n bytes to the SPI port
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;  FASTCALL16 ARGUMENTS:
;
;   [SP-3] => pBuff LSB Address.
;   [SP-4] => pBuff MSB Address.
;   [SP-5] => Buffer length to write.
;
;  RETURNS:  none
;
;  SIDE EFFECTS: 
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to perserve their values across calls to fastcall16 
;    functions.
;
;
;  THEORY of OPERATION or PROCEDURE:
;     Writes n data byte to the TX buffer register.
;
; Stack offset constants
WBUFF_LSB:          equ -3
WBUFF_MSB:          equ -4
WBUFF_CNT:          equ -5


 SDCard_WriteBuff:
_SDCard_WriteBuff:
    RAM_PROLOGUE RAM_USE_CLASS_4
    RAM_PROLOGUE RAM_USE_CLASS_2

// (1) Set the MVW_PP to the value of the MSB of the buffer
// (2) Save the count variable in X
// (3) Set the Cur

   mov   X, SP

   mov  A,[X+WBUFF_MSB]                    ; Get buffer MSB
   RAM_SETPAGE_MVR A                       ; Set MVI Write page

   mov  A,>pMVI                            ; Get pMVI address MSB
   RAM_SETPAGE_CUR A                       ; Set current page pointer with pMVI address MSB
   
   mov  A,[X+WBUFF_LSB]                    ; Put LSB of buffer address in [pMVI]
   mov  [pMVI],A

   mov  X,[X+WBUFF_CNT]                    ; Put the buff counter in X

.DataLoop:
   dec  X                                  ; Decrement data counter.
   jc   .EndDataLoop
   mvi  A,[pMVI]                           ; Get value from buffer
   mov  REG[SDCard_TX_BUFFER_REG], A              ; Write data to buffer
.RdyLoop:                                  ; Check if buffer is empty
   tst  REG[SDCard_CONTROL_REG], SDCard_SPIM_SPI_COMPLETE
   jz   .RdyLoop
   jmp  .DataLoop

.EndDataLoop:
    RAM_EPILOGUE RAM_USE_CLASS_2
    RAM_EPILOGUE RAM_USE_CLASS_4
   ret

.ENDSECTION


.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: SDCard_ReadBuff(BYTE * pBuff, BYTE bCnt)
;
;  DESCRIPTION:
;     Reads n bytes from the SPI port
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;  FASTCALL16 ARGUMENTS:
;
;   [SP-3] => pBuff LSB Address.
;   [SP-4] => pBuff MSB Address.
;   [SP-5] => Buffer length to read (bCnt).
;
;
;  RETURNS:  none
;
;  SIDE EFFECTS: 
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to perserve their values across calls to fastcall16 
;    functions.
;
;
;  THEORY of OPERATION or PROCEDURE:
;     Writes data to the TX buffer register.
;
; Stack offset constants
RBUFF_LSB:          equ -3
RBUFF_MSB:          equ -4
RBUFF_CNT:          equ -5


 SDCard_ReadBuff:
_SDCard_ReadBuff:
    RAM_PROLOGUE RAM_USE_CLASS_4
    RAM_PROLOGUE RAM_USE_CLASS_2

// (1) Set the MVW_PP to the value of the MSB of the buffer
// (2) Save the count variable in X
// (3) Set the Cur

   mov   X, SP

   mov  A,[X+RBUFF_MSB]                    ; Get buffer MSB
   RAM_SETPAGE_MVW A                       ; Set MVI Write page

   mov  A,>pMVI                            ; Get pMVI address MSB
   RAM_SETPAGE_CUR A                       ; Set current page pointer with pMVI address MSB
   
   mov  A,[X+RBUFF_LSB]                    ; Put LSB of pMVI address in [pMVI]
   mov  [pMVI],A

   mov  X,[X+RBUFF_CNT]                    ; Put the buff counter in X


.DataLoop:
   dec  X                                  ; Decrement data counter.
   jc   .EndDataLoop
   mov  REG[SDCard_TX_BUFFER_REG], 0xFF           ; Write Dummy Byte
.RdyLoop:                                  ; Check if buffer is empty
   tst  REG[SDCard_CONTROL_REG], SDCard_SPIM_SPI_COMPLETE
   jz   .RdyLoop
   mov  A,REG[SDCard_RX_BUFFER_REG]               ; Read data from SPI data register
   mvi  [pMVI],A                           ; Write data to array
   jmp  .DataLoop


.EndDataLoop:
    RAM_EPILOGUE RAM_USE_CLASS_2
    RAM_EPILOGUE RAM_USE_CLASS_4
   ret

.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: SDCard_ReadBuff256(char * sRamBuff)
;
;  DESCRIPTION:
;     Reads 256 bytes into a buffer from the SPI port
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;     A:X  Pointer to String
;          A contains MSB of string address
;          X contains LSB of string address
;
;
;  RETURNS:  none
;
;  SIDE EFFECTS: 
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to perserve their values across calls to fastcall16 
;    functions.
;
;
;  THEORY of OPERATION or PROCEDURE:
;     Reads data from the TX buffer register.
;
 SDCard_ReadBuff256:
_SDCard_ReadBuff256:
    RAM_PROLOGUE RAM_USE_CLASS_3
   ; Transmit the data!


   RAM_SETPAGE_IDX A
   mov  X,0                               ; Assume 256 byte buffer is full page
.DataLoop:
   mov  REG[SDCard_TX_BUFFER_REG], 0xFF          ; Transmit dummy byte
.RdyLoop:
   tst  REG[SDCard_CONTROL_REG], SDCard_SPIM_SPI_COMPLETE
   jz   .RdyLoop
   mov  A, REG[SDCard_RX_BUFFER_REG]
   mov  [X],A                             ; Get data

   inc  X
   jnz  .DataLoop

   RAM_EPILOGUE RAM_USE_CLASS_3
   ret

.ENDSECTION





.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: SDCard_bReadRxData
;
;  DESCRIPTION:
;     Reads the RX buffer register.  Should check the status regiser to make
;     sure data is valid.
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:  none
;
;  RETURNS:
;     bRxData - returned in A.
;
;  SIDE EFFECTS: 
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to perserve their values across calls to fastcall16 
;    functions.
;
;  THEORY of OPERATION or PROCEDURE:
;
 SDCard_bReadRxData:
_SDCard_bReadRxData:
   RAM_PROLOGUE RAM_USE_CLASS_1
   mov A, REG[SDCard_RX_BUFFER_REG]
   RAM_EPILOGUE RAM_USE_CLASS_1   
   ret


.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: SDCard_ReadStatus
;
;  DESCRIPTION:
;     Reads the SPIM Status bits in the Control/Status register.
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:  none
;
;  RETURNS:
;     BYTE  bStatus - transmit status data.  Use the defined bit masks.
;        Returned in Accumulator.
;
;  SIDE EFFECTS: 
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to perserve their values across calls to fastcall16 
;    functions.
;
;  THEORY of OPERATION or PROCEDURE:
;     Read the status and control register.
;
 SDCard_bReadStatus:
_SDCard_bReadStatus:
   RAM_PROLOGUE RAM_USE_CLASS_1
   mov A,  REG[SDCard_CONTROL_REG]
   RAM_EPILOGUE RAM_USE_CLASS_1   
   ret

.ENDSECTION

; End of File SDCard.asm
