from twisted.internet import reactor, defer
import habu
import habu.log as log

import sys
import os, os.path

__version__ = "0.3.0"

def printUsage(name):
    print "Usage : %s [options] [config_file_path]" % name
    print "Valid Options:\n" \
        "  -h, --help: Show this message.\n" \
        "  -v, --version: Show Version\n" \
        "  -t, --target=TARGET_PIPELINE: Run specific pipeline.\n" \
        "  -a, --plugin-path=PLUGIN_PATH: Append plugin paht.\n" \
        "  --download-module=PLUGIN_NAME: download specified plugin.\n" \
        "  --download-url=DOWNLOAD_URL: download from specified urln.\n" \
        "  -e, --enable-netinstall: enable to install module from net.\n" \
        "  -d, --disable-server: disable server functions\n" \
        "  -l, --log-level=LOG_LEVEL: set log level.\n" \
        "  --log-file=LOG_FILE: set log files. if stdout, ouput to stdout.\n" \
        "  --proxy-host=PROXY_HOST_NAME: Proxy Host Name.\n" \
        "  --proxy-port=PROXY_HOST_NUMBER: Proxy Host Port Number.\n" \
        "  --pid-file=PID_FILE_PATH\n"
    sys.exit(0)

def printVersion():
    print "habu " + __version__
    sys.exit(0)

def parseArgs(args):
    from getopt import gnu_getopt
    import os.path
    
    try:
        options, arguments = gnu_getopt(args[1:], "hVt:a:del:",
                                        ["help", "version",
                                         "enable-netinstall", 
                                         "disable-server",
                                         "target=", "plugin-path=",
                                         "download-module=", "download-url=",
                                         "log-level=", "log-file=",
                                         "proxy-host=", "proxy-port=",
                                         "pid-file="])
    except:
        printUsage(args[0])
    
    cfgFile = "habu.cfg"
    if arguments:
        cfgFile = arguments[0]
    if not os.path.exists(cfgFile):
        printUsage(args[0])

    bag = {}
    for opt, arg in options:
        if opt in ("-h", "--help"):
            printUsage(args[0])
        elif opt in ("-V", "--version"):
            printVersion()
        elif opt in ("-t", "--target"):
            bag["target"] = arg
        elif opt in ("-a", "--plugin-path"):
            l = bag.setdefault("plugin-path", [])
            l.append(arg)
        elif opt in ("-e", "--enable-netinstall"):
            bag["enable-netinstall"] = True
        elif opt in ("-d", "--disable-server"):
            bag["enable-server"] = False
        elif opt == "--proxy-host":
            bag["proxy-host"] = arg
        elif opt == "--proxy-port":
            try:
                bag["proxy-port"] = int(arg)
            except:
                printUsage(arg[0])
        elif opt in ("-l", "--log-level"):
            if arg.isdigit():
                arg = int(arg)
            else:
                arg = arg.upper()
            bag["log-level"] = arg
        elif opt == "--log-file":
            bag["log-file"] = arg
        elif opt == "--download-module":
            bag["module"] = arg
        elif opt == "--download-url":
            bag["url"] = arg
        elif opt == "--pid-file":
            bag["pid-file"] = arg
    
    return cfgFile, bag

def prepare(cfgFile, bag):
    mgr = habu.Habu()
    mgr.load(open(cfgFile).read())
    mgr.addPluginPath(bag.get("plugin-path", None))
    mgr.setProxy(bag.get("proxy-host", None), bag.get("proxy-port", 0))
    if not mgr.getPipelines():
        print "No pipeline is defined"
        sys.exit(1)

    if bag.has_key("log-level"):
        mgr.setLogLevel(bag["log-level"])
    if bag.get("log-file", None):
        mgr.setLogFile(bag.get("log-file"))
    mgr.startLogging()


    target = bag.get("target", None)
    enableServer = bag.get("enable-server", True)
    def isDaemon():
        if enableServer and mgr.servers:
            return True
        if not mgr.scheduler or target or not enableServer:
            return False
        
        return True

    return mgr, target, isDaemon()

def loadModule(mgr, enableNetInstall):
    log.debug("loading modules...")
    try:
        # TODO: network module loader
        result = mgr.preloadModules(enableNetInstall)
        log.debug("done to preload module.")
    except Exception, e:
        log.error(str(e))
        sys.exit(1)

    return result


def runHabu(mgr, target, daemonMode, pidFile):
    if not daemonMode:
        if target != None:
            if target not in mgr.getPipelineNames():
                log.critical("No target : %s" % target)
                sys.exit(1)
            allManagers = [mgr.run(target)]
        else:
            allManagers = mgr.runAll()
        
        def eventFired(firedManager):
            if firedManager.executeContexts:
                num = len(firedManager.executeContexts)
                log.debug("remaining context num : %d" % num)
                return
            allManagers.remove(firedManager)
            num = len(allManagers)
            log.debug("remaining manager : %d" % num)
            if not allManagers:
                log.info("Stopping reactor")
                reactor.stop()

        for m in allManagers:
            m.addHook(m.HOOK_TYPE_SUCCESS, eventFired)
            m.addHook(m.HOOK_TYPE_GOT_ERROR, eventFired)
        mode = "Task Mode"

    else:
        mgr.startScheduler()
        if mgr.servers:
            mgr.startServer()
            mode = "Server Mode"
        else:
            mode = "Scheduler Mode"

        if pidFile:
            fp = open(pidFile, "w")
            fp.write(str(os.getpid()))
            fp.close()

    log.info("Started habu with " + mode)

def downloadModule(url, module, pluginDir, proxy_host, proxy_port):
    if not module:
        print "download module is not specified."
        sys.exit(1)
    if not url:
        url = "http://svn.sourceforge.jp/svnroot/pyhabu/trunk/habu"
    if not pluginDir:
        pluginDir = "."
    else:
        pluginDir = pluginDir[0]
    
    from habu.moduleloader import downloadModule
    def successDownload(result):
        log.info("success download module: %s %s" % (url, module))
        reactor.stop()
        return result

    def failedDownload(failure):
        log.error("failed download module: %s %s" % (url, module))
        reactor.stop()
        return failure
    
    deferred = downloadModule(url, module, pluginDir, proxy_host, proxy_port)
    deferred.addCallback(successDownload)
    deferred.addErrback(failedDownload)

def main(*args):
    cfgFile, bag = parseArgs(args)
    module = bag.get("module", None)
    url = bag.get("url", None)
    pid = bag.get("pid-file", None)
    if module or url:
        log.startLogging(sys.stdout, bag.get("log-level", "DEBUG"))
        downloadModule(url, module, bag.get("plugin-path", ["."]),
                       bag.get("proxy-host", None), bag.get("proxy-port", 0))
    else:
        mgr, target, daemonMode = prepare(cfgFile, bag)
        result = loadModule(mgr, bag.get("enable-netinstall", False))
        if result:
            def handleAllResult(results):
                errs = 0
                log.debug("got network module load result")
                log.debug(str(results))
                for success, result in results:
                    if success:
                        log.info("network module load success")
                    else:
                        log.error(result)
                        errs += 1
                if errs:
                    log.error("reactor stop because failed to load module")
                    reactor.stop()
                else:
                    log.info("runHabu after internet module load")
                    reactor.callLater(0, runHabu, mgr, target, 
                                      daemonMode, pid)
                    
            from twisted.internet import defer
            defer.DeferredList(result, consumeErrors = True).addCallback(
                handleAllResult)
        else:
            runHabu(mgr, target, daemonMode, pid)

    reactor.run()
    
    if pid and os.path.exists(pid):
        os.unlink(pid)

if __name__ == "__main__":
    main(*(sys.argv))
