#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of Pysilhouette.
#
# Copyright (c) 2009 HDE, Inc.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#

"""
@author: Kei Funagayama <kei@karesansui-project.info>
"""

import os
import sys
import time
import signal
import traceback
import logging

from pysilhouette.log import reload_conf
from pysilhouette.prep import readconf, getopts, chkopts
from pysilhouette.util import write_pidfile, create_fifo

def sigterm_handler(signum, frame):
    global logger
    logger.info('Stop the schedulerd with signal- pid=%s, signal=%s' % (os.getpid(), signum))

def scheduler():
    global opts
    global cf
    global logger

    if os.access(cf["observer.mkfifo.path"], os.F_OK|os.R_OK|os.W_OK) is False:
        try:
            os.unlink(cf["observer.mkfifo.path"])
            logger.info('Deleted filo file. - file=%s' % cf["observer.mkfifo.path"])
        except:
            pass # Not anything
        create_fifo(cf["observer.mkfifo.path"],
                    cf["observer.mkfifo.user.name"],
                    cf["observer.mkfifo.group.name"],
                    cf["observer.mkfifo.perms"],
                    )

        logger.info('The fifo file was created. - file=%s' % cf["observer.mkfifo.path"])

    if opts.daemon is True:
        pid = os.getpid()
        if write_pidfile(opts.pidfile, pid):
            logger.info('The process file was created. - file=%s' % opts.pidfile)
        else:
            logger.info('Could not create process file. - file=%s' % opts.pidfile)
            return 1

    logger.info('schedulerd started!!')
    
    while True:
        try:
            fp = open(cf["observer.mkfifo.path"], 'w')
            try:
                fp.write(cf['observer.mkfifo.start.code'])
                logger.info('Start code was written. - file=%s : code=%s'
                            % (cf["observer.mkfifo.path"], cf['observer.mkfifo.start.code']))
            finally:
                fp.close()

            logger.debug('interval start, interval=%s' % (cf['scheduler.interval']))
            time.sleep(int(cf['scheduler.interval']))
        except IOError, i:
            if i.errno == 4:
                return 0 # When ending with the signal

    # beyond expectation
    logger.error('file=%s - 2 error write FIFO, code=%s'
                     % (self.fifo, cf['observer.mkfifo.start.code']))
    return 1
    
def main(): 
    global opts
    global cf
    global logger

    (opts, args) = getopts()
    if chkopts(opts) is True:
        return 1

    cf = readconf(opts.config)
    if cf is None:
        print >>sys.stderr, 'Failed to load the config file "%s". (%s)' % (opts.config, sys.argv[0])
        return 1
    
    if reload_conf(cf["env.sys.log.conf.path"]):
        logger = logging.getLogger('pysilhouette.scheduler')
    else:
        print >>sys.stderr, 'Failed to load the log file. (%s)' % sys.argv[0]
        return 1
    
    try:
        try:
            signal.signal(signal.SIGTERM, sigterm_handler)
            ret = scheduler() # start!!
            return ret
        except KeyboardInterrupt, k:
            logger.critical('Keyboard interrupt occurred. - %s' % str(k.args))
            print >>sys.stderr, 'Keyboard interrupt occurred. - %s' % str(k.args)
        except Exception, e:
            logger.critical('A system error has occurred. - %s' % str(e.args))
            print >>sys.stderr, 'A system error has occurred. - %s' % str(e.args)
            print >>sys.stderr, traceback.format_exc()
            t_logger = logging.getLogger('pysilhouette_traceback')
            t_logger.critical(traceback.format_exc())
            
    finally:
        if opts.daemon is True and os.path.isfile(opts.pidfile):
            os.unlink(opts.pidfile)
            logger.info('Process file has been deleted.. - pidfile=%s' % opts.pidfile)

if __name__ == '__main__':
    sys.exit(main())
