/**
 * Copyright (C) 2009 root42 Inc. All rights reserved.
 */

/**
 * R42フレームワーク
 *
 * フォームの入力状態をリセットする。
 *
 * @param document 操作対象ドキュメント
 * @author root42 Inc.
 * @since 1.0
 * @version $Id: FormResetter.js 25 2009-11-29 06:10:25Z takahashikzn $
 */
function FormResetter(/* Document */ document) {

    /** {@link #this} */
    var self = this;

    /** {@link #document} */
    /* Document */ self.document = document;

    /** 各フォーム要素に対応するコマンドのセット */
    /* private Map<String, Map<String, Function>> */ var commandSets = {};

    /**
     * コンストラクタ
     */
    (function(){

        // ----------------------------------------------------------
        // inputタグ向けコマンドセット
        commandSets.input = {};
        /* RegExp*/ var IGNORE_TYPES_PTRN = /button|submit|hidden|file|reset|image/i;
        /* RegExp*/ var CHECKED_TYPES_PTRN = /checkbox|radio/i;
        commandSets.input[FormResetter.CMD_SAVE] = /* void */ function(/* Element */ elem) {

            // 対象外タイプの場合は無視
            if (elem.type.match(IGNORE_TYPES_PTRN)) {
                return;
            }

            // チェックボックス/ラジオボタンは他と少し違う扱いを要する
            else if (elem.type.match(CHECKED_TYPES_PTRN)) {
                elem.savedState = elem.checked;
            }

            else {
                elem.savedState = elem.value;
            }
        };
        commandSets.input[FormResetter.CMD_REVERT] = /* void */ function(/* Element */ elem) {

            // 対象外タイプの場合は無視
            if (elem.type.match(IGNORE_TYPES_PTRN)) {
                return;
            }

            // チェックボックス/ラジオボタンは他と少し違う扱いを要する
            else if (elem.type.match(CHECKED_TYPES_PTRN)) {
                elem.checked = !!elem.savedState;
            }

            else {
                if (!!elem.savedState) {
                    elem.value = elem.savedState;
                } else {
                    elem.value = "";
                }
            }
        };
        commandSets.input[FormResetter.CMD_CLEAR] = /* void */ function(/* Element */ elem) {

            // 対象外タイプの場合は無視
            if (elem.type.match(IGNORE_TYPES_PTRN)) {
                return;
            }

            // チェックボックス/ラジオボタンは他と少し違う扱いを要する
            else if (elem.type.match(CHECKED_TYPES_PTRN)) {
                elem.checked = false;
            }

            else {
                elem.value = "";
            }
        };

        // ----------------------------------------------------------
        // selectタグ用コマンドセット
        commandSets.select = {};
        commandSets.select[FormResetter.CMD_SAVE] = /* void */ function(/* Element */ elem) {
            elem.savedState = elem.selectedIndex;
        };
        commandSets.select[FormResetter.CMD_REVERT] = /* void */ function(/* Element */ elem) {
            if (!!elem.savedState) {
                elem.selectedIndex = elem.savedState;
            } else {
                elem.selectedIndex = 0;
            }

            // NarrowDownSelectionList用
            if (!!elem.onchange) {
                elem.onchange();
            }
        };
        commandSets.select[FormResetter.CMD_CLEAR] = /* void */ function(/* Element */ elem) {
            elem.selectedIndex = 0;

            // NarrowDownSelectionList用
            if (!!elem.onchange) {
                elem.onchange();
            }
        };

        // ----------------------------------------------------------
        // textareaタグ用コマンドセット
        commandSets.textarea = {};
        commandSets.textarea[FormResetter.CMD_SAVE] = /* void */ function(/* Element */ elem) {
            elem.savedState = elem.innerHTML;
        };
        commandSets.textarea[FormResetter.CMD_REVERT] = /* void */ function(/* Element */ elem) {
            if (!!elem.savedState) {
                elem.innerHTML = elem.savedState;
            } else {
                elem.innerHTML = "";
            }
        };
        commandSets.textarea[FormResetter.CMD_CLEAR] = /* void */ function(/* Element */ elem) {
            elem.innerHTML = "";
        };

    })();

    /**
     * 各ターゲット要素に対し、指定したコマンドを実行する。
     *
     * @param commandName 実行するコマンド(関数)名
     */
    /* private Function */ var doCommand = /* void */ function(/* String */ commandName) {

        for (/* String */ var targetTagName in commandSets) {

            /* Map<String, Function> */ var commandSet = commandSets[targetTagName];
            /* Array<Element> */ var targetElems = self.document.body.getElementsByTagName(targetTagName);

            for (/* int */ var i = 0; i < targetElems.length; i++) {

                /* Element */ var elem = targetElems[i];
                commandSet[commandName](elem);
            }
        }
    };

    /**
     * フォームの入力状態を保存する。
     */
    self.saveFormState = /* void */ function() {
        doCommand(FormResetter.CMD_SAVE);
    };

    /**
     * フォームの入力状態を復元する。
     */
    self.revertFormState = /* void */ function() {
        doCommand(FormResetter.CMD_REVERT);
    };

    /**
     * フォームをクリアする。
     */
    self.clearFormState = /* void */ function() {
        doCommand(FormResetter.CMD_CLEAR);
    };
}


/**
 * 定数宣言
 */
(function() {
    /** フォーム値をセーブ */
    FormResetter.CMD_SAVE = "save";
    /** フォーム値を復元 */
    FormResetter.CMD_REVERT = "revert";
    /** フォーム値をクリア */
    FormResetter.CMD_CLEAR = "clear";
})();
