#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_defense.h"

#include "opuci_move.h"
#include "opuci_neck.h"

#include <rcsc/action/neck_turn_to_point.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/player/say_message_builder.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>

#include <rcsc/geom/line_2d.h>
#include <rcsc/geom/circle_2d.h>
#include <rcsc/geom/angle_deg.h>
#include <rcsc/geom/sector_2d.h>

/*-------------------------------------------------------------------*/
/*!
  execute action
*/
bool
Bhv_defense::execute( rcsc::PlayerAgent * agent )
{
    return true;
}
/*-------------------------------------------------------------------*/
/*!
ダッシュパワーを決定
*/
double
Bhv_defense::getDashPower( const rcsc::PlayerAgent * agent,
                               const rcsc::Vector2D & target_point )
{
    static bool s_recover_mode = false;

    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();
    int ball_reach_step = 0;
    if( !wm.existKickableTeammate()
        && !wm.existKickableOpponent() )
    {
        ball_reach_step = std::min( mate_min, opp_min );
    }
    const rcsc::Vector2D base_pos = wm.ball().inertiaPoint( ball_reach_step );
    /*--------------------------------------------------------*/
    // check recover
    /*
    if (( wm.ball().pos().x < 5.0 && wm.existKickableOpponent() )
	|| wm.ball().pos().x < wm.self().pos().x + 5.0 )
    {
        s_recover_mode = false;
    }
    */
    if ( wm.self().stamina() < SP.staminaMax() * 0.4 )
    {
        s_recover_mode = true;
    }
    else if ( wm.self().stamina() > SP.staminaMax() * 0.6 )
    {
        s_recover_mode = false;
    }

    /*--------------------------------------------------------*/
    double dash_power = SP.maxDashPower();
    const double my_inc
        = wm.self().playerType().staminaIncMax()
        * wm.self().recovery();

    if ( ( ( base_pos.x < wm.defenseLineX() )
           || ( base_pos.x < wm.self().pos().x ) )
         && ! wm.existKickableTeammate() )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": correct DF line. keep max power" );
        // keep max power
        dash_power = SP.maxDashPower();
    }
    else if( self_min <= mate_min
             && self_min <= opp_min )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": my ball. keep max power" );
        dash_power = SP.maxDashPower();
    }
    else if ( base_pos.x < -30.0
              && opp_min <= mate_min
              && opp_min <= self_min )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": exist kickable opponent dash" );
        dash_power = SP.maxDashPower();
    }
    else if( target_point.x < wm.self().pos().x
             || target_point.dist( SP.ourTeamGoalPos() ) < wm.self().pos().dist( SP.ourTeamGoalPos() ) )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": return to goal dash" );
        dash_power = SP.maxDashPower();
    }
    else if ( wm.self().stamina() > SP.staminaMax() * 0.8 )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": full stamina dash" );
        dash_power = SP.maxDashPower();
    }
    else if ( s_recover_mode )
    {
        dash_power = my_inc - 25.0; // preffered recover value
        if ( dash_power < 0.0 ) dash_power = 0.0;

        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": recovering" );
    }
    else if ( wm.existKickableTeammate()
              && wm.ball().distFromSelf() > 15.0 )
    {
        dash_power = std::min( my_inc - 20.0,
                               SP.maxPower() * 0.5 );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: exist kickable teammate. dash_power= %f"
                            ,__FILE__, __LINE__,
                            dash_power );
    }
    else if ( wm.existKickableTeammate()
              && wm.ball().distFromSelf() < 20.0 )
    {
        dash_power = std::min( my_inc * 1.1,
                               SP.maxDashPower() );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": exist kickable teammate. dash_power=%.1f",
                            dash_power );
    }
    // in offside area
    else if ( wm.self().pos().x > wm.offsideLineX() )
    {
        dash_power = SP.maxDashPower();
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": in offside area. dash_power=%.1f",
                            dash_power );
    }
    else if( wm.self().pos().dist( target_point ) > 15.0
             && wm.self().stamina() >  SP.staminaMax() * 0.7 )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": target point far from here" );
        dash_power = SP.maxDashPower();
    }
    //nomal
    else
    {
        dash_power = std::min( my_inc * 1.3,
                               SP.maxPower() * 0.75 );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: normal mode dash_power= %f"
                            ,__FILE__, __LINE__,
                            dash_power );
    }

    //ボールが大外に出る場合は見送る
    if( ( base_pos.absX() > SP.pitchHalfLength() + 5.0
          && wm.ball().inertiaPoint( self_min ).absX() > SP.pitchHalfLength() + 5.0 )
        || ( base_pos.absY() > SP.pitchHalfWidth() + 5.0
             && wm.ball().inertiaPoint( self_min ).absY() > SP.pitchHalfWidth() + 5.0 ) )
    {
        dash_power = my_inc * 0.5;
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: ball go to outside= %f"
                            ,__FILE__, __LINE__,
                            dash_power );
    }
    return dash_power;
}
/*-------------------------------------------------------------------*/
/*!
敵のFWの背番号を返す
*/
std::vector<int>
Bhv_defense::oppForward( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    double opp_offenseLineX = oppFwLineX( agent );

    std::vector<int> forward_opp;
    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
         it != opps.end();
         ++it )
    {
        if( (*it)->pos().x > opp_offenseLineX - 1.0 )
        {
            bool opp_is_forward = true;
            rcsc::Vector2D sector_point( opp_offenseLineX - 0.1, (*it)->pos().y );
            rcsc::Sector2D sector( sector_point, 0.0, ( (*it)->pos() - sector_point ).r(),
                                   rcsc::AngleDeg( -90.0 ), rcsc::AngleDeg( 90.0 ) );
            //rcsc::Circle2D circle( circle_point, ( (*it)->pos() - circle_point ).r() );
            for( rcsc::PlayerPtrCont::const_iterator itr = opps.begin();
                 itr != opps.end();
                 ++itr )
            {
                if( it != itr
                    && sector.contains( (*itr)->pos() ) )
                {
                    opp_is_forward = false;
                    break;
                }
            }
            if( opp_is_forward )
            {
                forward_opp.push_back( (*it)->unum() );
            }
        }
    }

    std::vector<rcsc::Vector2D> opp_pos;
    std::vector<int> opp_unum;
    rcsc::Vector2D max_pos( 52.5, 34.0 );
    for( unsigned int i = 0; i < forward_opp.size(); i++ )
    {
        int unum = 0;
        rcsc::Vector2D min_pos( 52.5, -34.0 );
        for( std::vector<int>::iterator it = forward_opp.begin();
             it != forward_opp.end();
             ++it )
        {
            if( wm.opponent( (*it) ) )
            {
                if( max_pos.y > wm.opponent( (*it) )->pos().y
                    && min_pos.y < wm.opponent( (*it) )->pos().y )
                {
                    min_pos = wm.opponent( (*it) )->pos();
                    unum = (*it);
                }
            }
        }
        if( min_pos.y > -34.0 )
        {
            opp_pos.push_back( min_pos );
            opp_unum.push_back( unum );
        }
        max_pos = min_pos;
    }
    
    for( unsigned int i = 1; i < opp_pos.size(); i++ )
    {
        agent->debugClient().addLine( opp_pos.at( i - 1 ), opp_pos.at( i ) );
    }
    
    return opp_unum;
}
/*-------------------------------------------------------------------*/
/*!
敵のMFの背番号を返す
*/
std::vector<int>
Bhv_defense::oppSemiForward( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    std::vector<int> semi_fw;
    std::vector<int> opp_fw = oppForward( agent );
    const rcsc::PlayerPtrCont & opps = oppOffenseCluster( agent );
    for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
         it != opps.end();
         it++ )
    {
        bool is_mf = true;
        for( std::vector<int>::iterator itr = opp_fw.begin();
             itr != opp_fw.end();
             itr++ )
        {
            if( (*it)->unum() == (*itr) )
            {
                is_mf = false;
                break;
            }
        }
        if( is_mf )
        {
            semi_fw.push_back( (*it)->unum() );
        }
    }

    /*
    double opp_semi_offenseLineX = oppSemiFwLineX( agent );

    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
         it != opps.end();
         it++ )
    {
        if( (*it)->pos().x > opp_semi_offenseLineX - 0.1 )
        {
            bool opp_is_forward = true;
            rcsc::Vector2D sector_point( opp_semi_offenseLineX - 0.1, (*it)->pos().y );
            rcsc::Sector2D sector( sector_point, 0.0, ( (*it)->pos() - sector_point ).r(),
                                   rcsc::AngleDeg( -90.0 ), rcsc::AngleDeg( 90.0 ) );
            for( rcsc::PlayerPtrCont::const_iterator itr = opps.begin();
                 itr != opps.end();
                 ++itr )
            {
                if( it != itr
                    && sector.contains( (*itr)->pos() ) )
                {
                    opp_is_forward = false;
                    break;
                }
            }
            if( opp_is_forward )
            {
                semi_fw.push_back( (*it)->unum() );
            }
        }
    }
    */

    std::vector<rcsc::Vector2D> opp_pos;
    std::vector<int> opp_unum;
    rcsc::Vector2D max_pos( 52.5, 34.0 );
    for( unsigned int i = 0; i < semi_fw.size(); i++ )
    {
        int unum = 0;
        rcsc::Vector2D min_pos( 52.5, -34.0 );
        for( std::vector<int>::iterator it = semi_fw.begin();
             it != semi_fw.end();
             ++it )
        {
            if( wm.opponent( (*it) ) )
            {
                if( max_pos.y > wm.opponent( (*it) )->pos().y
                    && min_pos.y < wm.opponent( (*it) )->pos().y )
                {
                    min_pos = wm.opponent( (*it) )->pos();
                    unum = (*it);
                }
            }
        }
        if( min_pos.y > -34.0 )
        {
            opp_pos.push_back( min_pos );
            opp_unum.push_back( unum );
        }
        max_pos = min_pos;
    }

    for( unsigned int i = 1; i < opp_pos.size(); i++ )
    {
        agent->debugClient().addLine( opp_pos.at( i - 1 ), opp_pos.at( i ) );
    }

    return opp_unum;
}
/*-------------------------------------------------------------------*/
/*!
敵をマークするべき位置を返す
*/
rcsc::Vector2D
Bhv_defense::defendOpponent( rcsc::PlayerAgent * agent,
                             const int opp_unum,
                             const rcsc::Vector2D goal_pos,
                             const rcsc::Vector2D mark_pos,
                             int * reach_cycle,
                             const int cost_cycle )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();
    rcsc::Vector2D home_pos = mark_pos;
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();
    int ball_reach_step = 0;
    if( !wm.existKickableTeammate()
        && !wm.existKickableOpponent() )
    {
        ball_reach_step = std::min( mate_min, opp_min );
    }
    const rcsc::Vector2D base_pos = wm.ball().inertiaPoint( ball_reach_step );
    if( ! wm.opponent( opp_unum )
        || base_pos.absX() > sp.pitchHalfLength()
        || base_pos.absY() > sp.pitchHalfWidth() )
    {
        if( reach_cycle )
        {
            *reach_cycle = cost_cycle;
        }
        agent->debugClient().addMessage("defaultPos");
        return home_pos;
    }

    const rcsc::Vector2D opp_pos = wm.opponent( opp_unum )->pos();
    const rcsc::AngleDeg angle = ( opp_pos - goal_pos ).th();

    for( int i = 1; i < cost_cycle; i++ )
    {
        const double dash_r = wm.self().playerTypePtr()->dashDistanceTable().at( i );
        const double opp_dash_r = wm.opponent( opp_unum )->playerTypePtr()->playerSpeedMax() * (double)( i );
        const rcsc::Circle2D my_circle( wm.self().pos(), dash_r );
        home_pos.x = mark_pos.x - angle.cos() * opp_dash_r;
        home_pos.y = mark_pos.y - angle.sin() * opp_dash_r;
        if( my_circle.contains( home_pos ) )
        {
            if( reach_cycle )
            {
                *reach_cycle = i;
            }
            return home_pos;
        }
    }

    if( reach_cycle )
    {
        *reach_cycle = cost_cycle;
    }
    int dist_self_to_goal = wm.self().pos().dist( goal_pos ) / 5;
    int dist_opp_to_goal = opp_pos.dist( goal_pos ) / 5;
    if( dist_self_to_goal < dist_opp_to_goal )
    {
        return home_pos;
    }
    else
    {
        return goal_pos;
    }
    return home_pos;
}
/*-------------------------------------------------------------------*/
/*!
敵が既に他の味方にマークされていればtrueを返す
*/
bool
Bhv_defense::oppMarked( rcsc::PlayerAgent * agent,
                        const int opp_unum )
{
    const rcsc::WorldModel & wm = agent->world();
    if( !wm.opponent( opp_unum ) )
    {
        return false;
    }
    rcsc::Vector2D opp_pos = wm.opponent( opp_unum )->pos();

    rcsc::AngleDeg angle = ( rcsc::ServerParam::i().ourTeamGoalPos() - opp_pos ).th();
    rcsc::Sector2D sector( opp_pos, 0.0, 5.0, angle - 45.0, angle + 45.0 );

    const rcsc::PlayerPtrCont & mates = agent->world().teammatesFromSelf();
    for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
         it != mates.end();
         ++it )
    {
        if( sector.contains( (*it)->pos() ) )
        {
            return true;
        }
    }
    return false;
}
/*-------------------------------------------------------------------*/
/*!
敵FWの先頭のx座標を返す
*/
double
Bhv_defense::oppFwLineX( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    double opp_offenseLineX = 52.5;
    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
         it != opps.end();
         ++it )
    {
        if( (*it)->pos().x < opp_offenseLineX
            && wm.self().pos().x - (*it)->pos().x < 30.0 )
        {
            {
                opp_offenseLineX = (*it)->pos().x;
            }
        }
    }

    return opp_offenseLineX;
}
/*-------------------------------------------------------------------*/
/*!
敵MFの先頭のx座標を返す
*/
double
Bhv_defense::oppSemiFwLineX( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    std::vector<int> opp_fw = oppForward( agent );

    double opp_offenseLineX = 52.5;
    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
         it != opps.end();
         it++ )
    {
        if( (*it)->pos().x < opp_offenseLineX
            && wm.self().pos().x - (*it)->pos().x < 30.0 )
        {
            bool fw_opp = false;
            for( std::vector<int>::iterator fw = opp_fw.begin();
                 fw != opp_fw.end();
                 fw++ )
            {
                if( (*it)->unum() == (*fw) )
                {
                    fw_opp = true;
                    break;
                }
            }
            if( ! fw_opp )
            {
                opp_offenseLineX = (*it)->pos().x;
            }
        }
    }
    return opp_offenseLineX;
}
/*-------------------------------------------------------------------*/
/*!
攻撃に参加している敵を返す
*/
rcsc::PlayerPtrCont
Bhv_defense::oppOffenseCluster( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    rcsc::Vector2D centroid( 0.0, 0.0 );
    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
         it != opps.end();
         ++it )
    {
        centroid += (*it)->pos();
    }
    centroid /= opps.size();
    rcsc::Vector2D offense_point = centroid;
    offense_point.x -= 1.0;
    rcsc::Vector2D defense_point = centroid;
    defense_point.x += 1.0;

    rcsc::PlayerPtrCont opp_offense;
    for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
         it != opps.end();
         ++it )
    {
        if( (*it)->pos().dist( offense_point ) < (*it)->pos().dist( defense_point ) )
        {
            opp_offense.push_back( (*it) );
            agent->debugClient().addCircle( (*it)->pos(), 3.0 );
        }
    }
    return opp_offense;
}
/*-------------------------------------------------------------------*/
/*!
オフサイドになっている敵をマークから外す
*/
rcsc::Vector2D
Bhv_defense::offsideTrap( rcsc::PlayerAgent * agent,
                          rcsc::Vector2D default_pos )
{
    const rcsc::WorldModel & wm = agent->world();
    rcsc::Vector2D mark_home_pos = default_pos;

    double first_mate_x = 0.0;
    const rcsc::PlayerPtrCont & mates = wm.teammatesFromSelf();
    for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
         it != mates.end();
         ++it )
    {
        if( !(*it)->goalie() )
        {
            if( (*it)->pos().x < first_mate_x )
            {
                first_mate_x = (*it)->pos().x;
            }
        }
    }
    if( wm.ball().pos().x < first_mate_x )
    {
        return default_pos;
    }

    if( wm.self().pos().x < first_mate_x )
    {
        const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
        for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
             it != opps.end();
             ++it )
        {
            if( (*it)->pos().x < first_mate_x
                && wm.self().pos().x < (*it)->pos().x )
            {
                mark_home_pos.x = first_mate_x;
                mark_home_pos.y = wm.self().pos().y;
                agent->debugClient().addMessage("offsideTrap");
                break;
            }
        }
    }
    return mark_home_pos;
}
/*-------------------------------------------------------------------*/
/*!
他の味方にメッセージを送信する
*/
bool
Bhv_defense::defenseMessage( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    //int self_min = wm.interceptTable()->selfReachCycle();
    //int mate_min = wm.interceptTable()->teammateReachCycle();

    if( wm.ball().rposCount() < 1
        && wm.self().pos().dist( wm.ball().pos() ) < 20.0 )
    {
        //ボールのrposcountが0ならばボールの位置・速度，自分の位置，体の角度をメッセージで配信
        agent->addSayMessage( new rcsc::BallPlayerMessage( agent->effector().queuedNextBallPos(), 
                                                           agent->effector().queuedNextBallVel(),
                                                           wm.self().unum(),
                                                           agent->effector().queuedNextMyPos(),
                                                           agent->effector().queuedNextMyBody() ) );
    }
    else
    {
        //ボールのrposcountが1以上ならば自分の位置をメッセージで配信 #SelfMessageの方がいいかもしれない
        //agent->addSayMessage( new rcsc::OnePlayerMessage( wm.self().unum(), agent->effector().queuedNextMyPos() ) );
        //const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
        std::vector<int> opp_fw = oppForward( agent );
        rcsc::Vector2D first_pos = rcsc::ServerParam::i().theirTeamGoalPos();
        rcsc::Vector2D second_pos = rcsc::ServerParam::i().theirTeamGoalPos();
        int first_unum = 0;
        int second_unum = 0;
        int first_count = 5;
        int second_count = 5;
        for( std::vector<int>::iterator it = opp_fw.begin();
             it != opp_fw.end();
             ++it )
        {
            if( wm.opponent( (*it) )
                && wm.opponent( (*it) )->seenPosCount() < first_count )
            {
                second_count = first_count;
                second_unum = first_unum;
                second_pos = first_pos;

                first_count = wm.opponent( (*it) )->posCount();
                first_unum = (*it);
                first_pos = wm.opponent( (*it) )->pos();
            }
            else if( wm.opponent( (*it) )->posCount() < second_count )
            {
                second_count = wm.opponent( (*it) )->posCount();
                second_unum = (*it);
                second_pos = wm.opponent( (*it) )->pos();
            }
        }

        if( first_unum != 0
            && second_unum != 0 )
        {
            agent->addSayMessage( new rcsc::ThreePlayerMessage( first_unum + 11,
                                                                first_pos,
                                                                second_unum + 11,
                                                                second_pos,
                                                                wm.self().unum(),
                                                                agent->effector().queuedNextMyPos() ) );
        }
        else if( first_unum != 0 )
        {
            agent->addSayMessage( new rcsc::TwoPlayerMessage( first_unum + 11,
                                                              first_pos,
                                                              wm.self().unum(),
                                                              agent->effector().queuedNextMyPos() ) );
        }
        else
        {
            agent->addSayMessage( new rcsc::OnePlayerMessage( wm.self().unum(), agent->effector().queuedNextMyPos() ) );
        }
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!
アテンションを向ける味方を決定する
*/
bool
Bhv_defense::defenseAttention( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    const rcsc::Vector2D base_pos = ballInertiaPoint( agent );
    if( base_pos.x < -30.0
        || ( wm.self().seenPosCount() < 1
             && wm.ball().pos().dist( wm.self().pos() ) < 20.0 ) )
    {
        int pos_count = 0;
        int mate_unum = -2;
        for( int i = 1; i <= 6; i++ )
        {
            if( wm.teammate( i )
                && wm.teammate( i )->posCount() >= pos_count )
            { 
                pos_count = wm.teammate( i )->posCount();
                mate_unum = i;
            }
        }
        if( wm.teammate( mate_unum ) )
        {
            agent->doAttentionto( wm.ourSide(), mate_unum );
            agent->debugClient().addMessage( "AttTo%d", mate_unum );
            return true;
        }
    }
    else
    {
        if( wm.getTeammateNearestToBall( 100 ) )
        {
            //最も近そうな味方が他にいればアテンションを向ける
            int mate_unum = wm.getTeammateNearestToBall( 100 )->unum();
            agent->doAttentionto( wm.ourSide(), mate_unum );
            agent->debugClient().addMessage( "AttTo%d", mate_unum );
            return true;
        }
    }

    agent->doAttentiontoOff();
    agent->debugClient().addMessage("AttOff");
    return true;
}
/*-------------------------------------------------------------------*/
/*!

*/
rcsc::Vector2D
Bhv_defense::ballInertiaPoint( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    const int self_min = wm.interceptTable()->selfReachCycle();
    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();
    int ball_reach_step = 0;
    if( !wm.self().isKickable()
        && !wm.existKickableTeammate()
        && !wm.existKickableOpponent() )
    {
        ball_reach_step = std::min( self_min, std::min( mate_min, opp_min ) );
    }
    const rcsc::Vector2D inertia_pos = wm.ball().inertiaPoint( ball_reach_step );
    const rcsc::Line2D inertia_line( wm.ball().pos(), inertia_pos );
    rcsc::Vector2D base_pos = inertia_pos;

    if( inertia_pos.x > 52.5 )
    {
        base_pos.x = 52.5;
        base_pos.y = inertia_line.getY( 52.5 );
    }
    else if( inertia_pos.x < -52.5 )
    {
        base_pos.x = -52.5;
        base_pos.y = inertia_line.getY( -52.5 );
    }

    if( inertia_pos.y > 34.0 )
    {
        base_pos.x = inertia_line.getX( 34.0 );
        base_pos.y = 34.0;
    }
    else if( inertia_pos.y < -34.0 )
    {
        base_pos.x = inertia_line.getX( -34.0 );
        base_pos.y = -34.0;
    }
    return base_pos;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Bhv_defense::prepareTackle( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int ball_reach_step = 0;
    if( !wm.existKickableTeammate()
        && !wm.existKickableOpponent() )
    {
        ball_reach_step = std::min( self_min, mate_min );
    }
    const rcsc::Vector2D base_pos = wm.ball().inertiaPoint( ball_reach_step );
    const rcsc::Segment2D ball_line( wm.ball().pos(), base_pos );
    const rcsc::Segment2D our_goal( rcsc::Vector2D( -52.5, -9.0 ), rcsc::Vector2D( -52.5, 9.0 ) );

    if( !ball_line.existIntersection( our_goal ) )
    {
        return false;
    }
    const rcsc::Vector2D next_ball_pos = wm.ball().pos() + wm.ball().vel();
    const rcsc::Vector2D next_self_pos = wm.self().pos() + wm.self().vel();

    if( next_ball_pos.dist( next_self_pos ) < sp.tackleDist() )
    {
        rcsc::AngleDeg angle = ( next_ball_pos - next_self_pos ).th() - wm.self().body();
        agent->doTurn( angle );
        agent->debugClient().addMessage("prepareTackle");
        return true;
    }
    return false;
}

bool 
Bhv_defense::ourDefenseLine( rcsc::PlayerAgent * agent, std::vector< rcsc::PlayerObject* > & def_line )
{
    const rcsc::WorldModel & wm = agent->world();
    def_line.clear();
    double def_x = 0;
    const rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromSelf().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
         it != end;
         ++it )
    {
        if( !(*it)->goalie() 
            && (*it)->pos().x < def_x )
        {
            def_x = (*it)->pos().x;
        }
    }
    
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
         it != end;
         ++it )
    {
        if( (*it)->goalie() 
            || (*it)->isTackling() 
            || (*it)->unum() == rcsc::Unum_Unknown )
        {
            continue;
        }
        
        if( (*it)->pos().x >= def_x - 1 )
        {

            bool defender = true;
            rcsc::Vector2D center( def_x - 1.0, (*it)->pos().y );
            rcsc::Sector2D sector( center,
                                   0.0,
                                   center.dist( (*it)->pos() ),
                                   rcsc::AngleDeg( 0 ),
                                   rcsc::AngleDeg( 90 ) );
            for( rcsc::PlayerPtrCont::const_iterator it2 = wm.teammatesFromSelf().begin();
                 it2 != wm.teammatesFromSelf().end();
                 ++it2 )
            {
                if( it != it2
                    && sector.contains( (*it2)->pos() ) )
                {
                    defender = false;
                    break;
                }
            }
            if( defender )
            {
                def_line.push_back( (*it) );
            }
        }
    }

    if( def_line.empty() )
    {
        return false;
    }
    else
    {
        for( std::vector< rcsc::PlayerObject* >::iterator it = def_line.begin();
             it != def_line.end();
             ++it )
        {
            for( std::vector< rcsc::PlayerObject* >::iterator it2 = it;
                 it2 != def_line.end();
                 ++it2 )
            {
                if( (*it)->pos().y > (*it2)->pos().y )
                {
                    std::swap( *it, *it2 );
                }
            }
        }
        if( def_line.size() >= 2 )
        {
            for( int i = 1; i < (int)def_line.size(); i++ )
            {
                agent->debugClient().addLine( def_line[i - 1]->pos(),
                                              def_line[i]->pos() );
            }
        }
        return true;
    }
}

bool 
Bhv_defense::surroundOppNum( rcsc::PlayerAgent * agent, std::vector< rcsc::PlayerObject* > & def_line, std::vector< int > & opp_num )
{
    const rcsc::WorldModel & wm = agent->world();
    opp_num.clear();
    for( int i = 0; i < 11; i++ )
    {
        opp_num.push_back( 0 );
    }
    const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
         it != end;
         ++it )
    {
        double min = 50;
        int min_unum = 0;
        for( int i = 1; i <= 11; i++ )
        {
            if( wm.teammate( i ) )
            {
                double dist = wm.teammate( i )->pos().dist( (*it)->pos() );
                if( dist < min )
                {
                    min = dist;
                    min_unum = i;
                }
            }
        }
        if( min_unum >= 1 )
        {
            opp_num[min_unum - 1]++;
        }
    }
    if( def_line.empty() )
    {
        return false;
    }
    for( std::vector< rcsc::PlayerObject* >::iterator it = def_line.begin();
         it != def_line.end();
         ++it )
    {
        for( std::vector< rcsc::PlayerObject* >::iterator it2 = it;
             it2 != def_line.end();
             ++it2 )
        {
            if( (*it) == (*it2) )
            {
                continue;
            }
            int num1 = 5;
            int num2 = 5;
            if( wm.teammate( (*it)->unum() ) )
            {
                num1 = opp_num[(*it)->unum() - 1];
            }
            if( wm.teammate( (*it2)->unum() ) )
            {
                num2 = opp_num[(*it2)->unum() - 1];
            }
            if( num1 > num2 )
            {
                std::swap( *it, *it2 );
            }
        }
    }
    for( std::vector< rcsc::PlayerObject* >::iterator it = def_line.begin();
         it != def_line.end();
         ++it )
    {
        agent->debugClient().addMessage( "%d>", (*it)->unum() );
    }
    return true;
}
