#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_goalie.h"
#include "bhv_goalie_free_kick.h"
#include "bhv_defense.h"
#include "opuci_neck.h"
#include "opuci_move.h"
#include "bhv_set_play.h"
#include "strategy.h"
#include "bhv_basic_tackle.h"
#include "bhv_danger_area_tackle.h"
#include "opuci_intention_dash.h"

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/say_message_builder.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>

#include <rcsc/action/body_clear_ball2009.h>
#include <rcsc/action/body_smart_kick.h>
#include <rcsc/action/body_pass.h>
#include <rcsc/action/body_intercept.h>
#include <rcsc/action/body_turn_to_ball.h>
#include <rcsc/action/neck_turn_to_ball.h>
#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>
#include <rcsc/action/body_hold_ball.h>

#include <rcsc/geom/angle_deg.h>
#include <rcsc/geom/line_2d.h>
#include <rcsc/geom/segment_2d.h>
#include <rcsc/geom/triangle_2d.h>
#include <rcsc/geom/rect_2d.h>

int Bhv_Goalie::M_passMate = -1;
int Bhv_Goalie::M_neck_cnt = -1;

/*-------------------------------------------------------------------*/
/*!
  execute action
*/
bool
Bhv_Goalie::execute( rcsc::PlayerAgent * agent )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Bhv_Goalie"
                        ,__FILE__, __LINE__ );
    
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    static const rcsc::Rect2D our_penalty( rcsc::Vector2D( -sp.pitchHalfLength(),
                                                           -sp.penaltyAreaHalfWidth() + 1.0 ),
                                           rcsc::Size2D( sp.penaltyAreaLength() - 1.0,
                                                         sp.penaltyAreaWidth() - 2.0 ) );

    if( ( wm.gameMode().type() == rcsc::GameMode::FreeKick_
          || wm.gameMode().type() == rcsc::GameMode::GoalieCatch_ )
        && wm.gameMode().side() == wm.ourSide() )
    {
        Bhv_GoalieFreeKick().execute( agent );
        return true;
    }
    else if( ( wm.gameMode().type() == rcsc::GameMode::BackPass_
               || wm.gameMode().type() == rcsc::GameMode::IndFreeKick_ )
             && wm.gameMode().side() != wm.ourSide() )
    {
        rcsc::Vector2D home_pos = Strategy::i().getPosition( wm.self().unum() );
        Bhv_SetPlay( home_pos ).execute( agent );
        return true;
    }
    else if( wm.gameMode().type() == rcsc::GameMode::GoalKick_ )
    {
        doMove( agent );
        return true;
    }
    //////////////////////////////////////////////////////////////
    // play_on play
    static int ball_holder_unum = -2;
    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    const rcsc::PlayerPtrCont & mates = wm.teammatesFromSelf();
    for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
         it != opps.end();
         it++ )
    {
        if( (*it)->pos().dist( wm.ball().pos() ) < (*it)->playerTypePtr()->kickableArea() )
        {
            ball_holder_unum = (*it)->unum() + 11;
        }
    }
    for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
         it != mates.end();
         it++ )
    {
        if( (*it)->pos().dist( wm.ball().pos() ) < (*it)->playerTypePtr()->kickableArea() )
        {
            ball_holder_unum = (*it)->unum();
        }
    }
    if( rcsc::Segment2D( wm.ball().pos(), wm.ball().inertiaFinalPoint() ).existIntersection(
            rcsc::Segment2D( rcsc::Vector2D( -52.5, -9.0 ), rcsc::Vector2D( -52.5, 9.0 ) ) ) )
    {
        ball_holder_unum = 30;
    }
    agent->debugClient().addMessage( "ballHolder:%d", ball_holder_unum );

    // catchable
    if ( wm.time().cycle()
         > wm.self().catchTime().cycle() + sp.catchBanCycle()
         && wm.ball().distFromSelf() < wm.self().catchableArea() - 0.05
         && our_penalty.contains( wm.ball().pos() )
         && ball_holder_unum > 11 )
    {
        rcsc::dlog.addText( rcsc::Logger::ROLE,
                            __FILE__": catchable. ball dist=%.1f, my_catchable=%.1f",
                            wm.ball().distFromSelf(),
                            wm.self().catchableArea() );
        agent->doCatch();
        agent->setNeckAction( new rcsc::Neck_TurnToBall() );
    }
    else if ( wm.self().isKickable() )
    {
        doKick( agent );
    }
    else
    {
        doMove( agent );
        Bhv_defense().defenseMessage( agent );
    }
    Bhv_defense().defenseAttention( agent );

    return true;
}
/*-------------------------------------------------------------------*/
/*!
  kick action
*/
bool
Bhv_Goalie::doKick( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    rcsc::Vector2D target_point;
    double pass_speed = 0.0;
    int receiver = 0;
    
    std::vector< rcsc::PlayerObject* > def_line;
    std::vector< int > opp_num;
    Bhv_defense().ourDefenseLine( agent, def_line );
    Bhv_defense().surroundOppNum( agent, def_line, opp_num );

    //direct pass
    const rcsc::Rect2D penalty_area( rcsc::Vector2D( -sp.pitchHalfLength(), -sp.penaltyAreaHalfWidth() ),
                                     rcsc::Vector2D( -sp.pitchHalfLength() + sp.penaltyAreaLength(),
                                                     sp.penaltyAreaHalfWidth() ) );
    const double ball_decay = sp.ballDecay();
    const double ball_speed_max = sp.ballSpeedMax();
    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    const rcsc::PlayerPtrCont & mates = wm.teammatesFromSelf();
    const rcsc::PlayerPtrCont::const_iterator opps_end = opps.end();
    const rcsc::PlayerPtrCont::const_iterator mates_end = mates.end();
    for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
         it != mates_end;
         ++it )
    {
        bool target_is_free = true;
        const rcsc::Sector2D sector( wm.ball().pos(),
                                     1.0, (*it)->pos().dist( wm.ball().pos() ) + 3.0,
                                     ( (*it)->pos() - wm.ball().pos() ).th() - rcsc::AngleDeg( 30.0 ),
                                     ( (*it)->pos() - wm.ball().pos() ).th() + rcsc::AngleDeg( 30.0 ) );
        for( rcsc::PlayerPtrCont::const_iterator itr = opps.begin();
             itr != opps.end();
             ++itr )
        {
            if( sector.contains( (*itr)->pos() ) )
            {
                target_is_free = false;
            }
        }

        if( target_is_free
            && (*it)->posCount() < 3
            && (*it)->pos().x > -25.0
            && (*it)->pos().x > wm.self().pos().x - 5.0
            && (*it)->pos().dist( wm.self().pos() ) > 10.0
            && rcsc::AngleDeg( ( (*it)->pos() - wm.self().pos() ).th() ).abs() < 70.0 )
        {
            /*
            int reach_cycle = log( 1.0 - ( 1.0 - ball_decay ) * wm.self().pos().dist( (*it)->pos() ) 
                                   / ball_speed_max ) / log( ball_decay );
            */
            int reach_cycle = (int)rcsc::calc_length_geom_series( ball_speed_max, 
                                                                  wm.self().pos().dist( (*it)->pos() ),
                                                                  ball_decay );
            if( reach_cycle < 0
                || reach_cycle > 30 )
            {
                reach_cycle = 30;
            }
            double ball_speed = (*it)->playerTypePtr()->kickableArea() * 1.5 * pow( 1.0 / ball_decay, reach_cycle - 1 );
            double first_speed = std::min( ball_speed_max, ball_speed );
            if( penalty_area.contains( wm.ball().pos() ) )
            {
                if( (*it)->seenPosCount() == 0 )
                {
                    if( rcsc::Body_SmartKick( (*it)->pos(),
                                              first_speed,
                                              first_speed * 0.8,
                                              1 ).execute( agent ) )
                    {
                        agent->debugClient().addMessage( "DirectOneStepPass" );
                        agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(),
                                                                     (*it)->pos(),
                                                                     agent->effector().queuedNextBallPos(),
                                                                     agent->effector().queuedNextBallVel() ) );
                    }
                    if( M_passMate != -1 )
                    {
                        Opuci_Neck().lookAtPassReceiver( agent, M_passMate );
                        M_neck_cnt--;
                        if( M_neck_cnt <= 0 )
                            M_passMate = -1;
                    }
                    else
                        agent->setNeckAction( new Opuci_Neck() );
                    
                }
                else if( M_passMate != -1 )
                {
                    rcsc::dlog.addText( rcsc::Logger::TEAM,
                                        "%s:%d: Keep locking on %d at the next cycle."
                                        ,__FILE__, __LINE__, (*it)->unum() );
                    rcsc::Body_HoldBall().execute( agent );
                    Opuci_Neck().lookAtPassReceiver( agent, M_passMate );
                    M_neck_cnt--;
                    if( M_neck_cnt <= 0 )
                        M_passMate = -1;
                }
                else
                {
                    rcsc::dlog.addText( rcsc::Logger::TEAM,
                                        "%s:%d: seenCount = %d. Lock on %d at the next cycle."
                                        ,__FILE__, __LINE__, (*it)->seenPosCount(), (*it)->unum() );
                    rcsc::Body_HoldBall().execute( agent );
                    Opuci_Neck().lookAtPassReceiver( agent, (*it)->unum() );
                    M_passMate = (*it)->unum();
                    M_neck_cnt = 3;
                }
                return true;
            }
            else
            {
                if( (*it)->seenPosCount() == 0 )
                {
                    rcsc::Body_SmartKick( (*it)->pos(),
                                          first_speed,
                                          first_speed * 0.8,
                                          3 ).execute( agent );
                    agent->debugClient().addMessage( "DirectPass" );
                    agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(),
                                                                 (*it)->pos(),
                                                                 agent->effector().queuedNextBallPos(),
                                                                 agent->effector().queuedNextBallVel() ) );

                    if( M_passMate != -1 )
                    {
                        Opuci_Neck().lookAtPassReceiver( agent, M_passMate );
                        M_neck_cnt--;
                        if( M_neck_cnt <= 0 )
                            M_passMate = -1;
                    }
                    else
                        agent->setNeckAction( new Opuci_Neck() );
                }
                else if( M_passMate != -1 )
                {
                    rcsc::dlog.addText( rcsc::Logger::TEAM,
                                        "%s:%d: Keep locking on %d at the next cycle."
                                        ,__FILE__, __LINE__, (*it)->unum() );
                    rcsc::Body_HoldBall().execute( agent );
                    Opuci_Neck().lookAtPassReceiver( agent, M_passMate );
                    M_neck_cnt--;
                    if( M_neck_cnt <= 0 )
                        M_passMate = -1;
                }
                else
                {
                    rcsc::dlog.addText( rcsc::Logger::TEAM,
                                        "%s:%d: seenCount = %d. Lock on %d at the next cycle."
                                        ,__FILE__, __LINE__, (*it)->seenPosCount(), (*it)->unum() );
                    rcsc::Body_HoldBall().execute( agent );
                    M_passMate = (*it)->unum();
                    Opuci_Neck().lookAtPassReceiver( agent, M_passMate );
                    M_neck_cnt = 3;
                }
                return true;
            }
        }
    }
    /*
    if  ( rcsc::Body_Pass::get_best_pass( agent->world(), &target_point, &pass_speed, &receiver )
          && target_point.dist( rcsc::Vector2D( -50.0, 0.0 ) ) > 20.0 )
    {
        double opp_dist = 100.0;
        const rcsc::PlayerObject * opp
            = agent->world().getOpponentNearestTo( target_point, 20, &opp_dist );
        agent->debugClient().addMessage( "GKickOppDist%.1f", opp ? opp_dist : 1000.0 );
        if ( ! opp || opp_dist > 3.0 )
        {
            rcsc::Body_SmartKick( target_point,
                                  pass_speed,
                                  pass_speed * 0.96,
                                  3 ).execute( agent );
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                __FILE__": register goalie kick intention. to (%.1f, %.1f)",
                                target_point.x,
                                target_point.y );
            agent->setNeckAction( new rcsc::Neck_ScanField() );
            if ( agent->config().useCommunication()
                 && receiver != rcsc::Unum_Unknown )
            {
                rcsc::dlog.addText( rcsc::Logger::ACTION,
                                    "%s:%d: execute() set pass communication.freekick"
                                    ,__FILE__, __LINE__ );
                rcsc::Vector2D target_buf = target_point - agent->world().self().pos();
                target_buf.setLength( 0.3 );
                agent->addSayMessage( new rcsc::PassMessage( receiver,
                                                             target_point,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
                return true;
            }
        }
    }
    */
    double best_angle = 90.0 * wm.ball().pos().y / wm.ball().pos().absY();
    for( double angle = -90.0; angle <= 90.0; angle += 10.0 )
    {
        bool angle_safety = true;
        const rcsc::Sector2D sector( wm.ball().pos(),
                                     1.0, 100.0,
                                     rcsc::AngleDeg( angle ) - rcsc::AngleDeg( 20.0 ),
                                     rcsc::AngleDeg( angle ) + rcsc::AngleDeg( 20.0 ) );
        for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
             it != opps_end;
             ++it )
        {
            if( sector.contains( (*it)->pos() ) )
            {
                angle_safety = false;
            }
        }
        if( angle_safety
            && abs( angle ) < abs( best_angle ) )
        {
            best_angle = angle;
        }
    }
    target_point.x = wm.ball().pos().x + rcsc::AngleDeg( best_angle ).cos();
    target_point.y = wm.ball().pos().y + rcsc::AngleDeg( best_angle ).sin();
    if( rcsc::Body_SmartKick( target_point,
                              ball_speed_max,
                              ball_speed_max * 0.8,
                              1 ).execute( agent ) )
    {
        agent->debugClient().addMessage( "GoalieKickAdvance" );
        agent->setNeckAction( new Opuci_Neck() );
        return true;
    }
    rcsc::Body_ClearBall2009().execute( agent );
    agent->setNeckAction( new rcsc::Neck_ScanField() );
    return true;
}
/*-------------------------------------------------------------------*/
/*!
  move action
*/
bool
Bhv_Goalie::doMove( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    //ball lost
    if( wm.ball().state().ghost_count_ > 0 )
    {
        agent->doTurn( 180.0 );
        rcsc::Neck_ScanField().execute( agent );
        agent->debugClient().addMessage("ballLost");
        return true;
    }

    const int self_min = wm.interceptTable()->selfReachCycle();
    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();
    int ball_reach_step = 0;
    if( !wm.self().isKickable()
        && !wm.existKickableTeammate()
        && !wm.existKickableOpponent() )
    {
        ball_reach_step = std::min( mate_min, opp_min );
    }
    const rcsc::Vector2D base_pos = wm.ball().inertiaPoint( ball_reach_step );

    //corner kick situation
    if( wm.gameMode().side() != wm.ourSide()
        && wm.gameMode().type() == rcsc::GameMode::CornerKick_ )
    {
        const rcsc::Vector2D move_point( -51.0, 6.0 * ( wm.ball().pos().y / wm.ball().pos().absY() ) );
        const double dash_power = dashPower( agent, move_point );
        if( !Opuci_Move( move_point, dash_power ).execute( agent ) )
        {
            rcsc::Body_TurnToBall().execute( agent );
        }
        agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
        agent->debugClient().setTarget( move_point );
        return true;
    }
    //kick in situation
    //offside situation
    else if( wm.gameMode().type() == rcsc::GameMode::KickIn_
             || wm.gameMode().type() == rcsc::GameMode::OffSide_ )
    {
        basicMove( agent );
        return true;
    }
    //goal kick situation
    else if( wm.gameMode().side() == wm.ourSide()
             && wm.gameMode().type() == rcsc::GameMode::GoalKick_ )
    {
        const rcsc::Vector2D move_point( -50.0, 0.0 );
        const double dash_power = dashPower( agent, move_point );
        if( !Opuci_Move( move_point, dash_power ).execute( agent ) )
        {
            rcsc::Body_TurnToBall().execute( agent );
        }
        agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
        agent->debugClient().setTarget( move_point );
        return true;
    }
    //kick off situation
    else if( wm.gameMode().type() == rcsc::GameMode::KickOff_ )
    {
        const rcsc::Vector2D move_point( -40.0, 0.0 );
        const double dash_power = dashPower( agent, move_point );
        if( !Opuci_Move( move_point, dash_power ).execute( agent ) )
        {
            rcsc::Body_TurnToBall().execute( agent );
        }
        agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
        agent->debugClient().setTarget( move_point );
        return true;
    }

    const rcsc::Rect2D our_penalty( rcsc::Vector2D( -sp.pitchHalfLength(),
                                                    -sp.penaltyAreaHalfWidth() + 1.0 ),
                                    rcsc::Size2D( sp.penaltyAreaLength() - 1.0,
                                                  sp.penaltyAreaWidth() - 2.0 ) );
    //tackle
    if( our_penalty.contains( wm.ball().pos() )
        && !wm.existKickableTeammate()
        && Bhv_DangerAreaTackle( 0.5 ).execute( agent ) )
    {
        return true;
    }
    else if( Bhv_BasicTackle( 0.9, 80.0 ).execute( agent ) )
    {
        return true;
    }
    else if( wm.ball().pos().x < -35.0 && wm.ball().distFromSelf() < 3.0 )
    {
        if( Bhv_BasicTackle( 0.5, 80.0 ).execute( agent ) )
        {
            return true;
        }
    }
    else if( wm.ball().pos().x < -32.0 
             && ! wm.existKickableTeammate() 
             && std::abs( wm.ball().pos().y - wm.self().pos().y ) < 2.0 
             && wm.ball().distFromSelf() < 6.0 )
    {
        if( Bhv_BasicTackle( 0.5, 80.0 ).execute( agent ) )
        {
            return true;
        }
    }
    else if( Bhv_defense().prepareTackle( agent ) )
    {
        return true;
    }

    //shoot situation
    if( rcsc::Segment2D( wm.ball().pos(), wm.ball().inertiaFinalPoint() ).existIntersection( 
            rcsc::Segment2D( rcsc::Vector2D( -52.5, -9.0 ), rcsc::Vector2D( -52.5, 9.0 ) ) ) )
    {
        agent->debugClient().addMessage("shootM");
        if( wm.ball().inertiaPoint( self_min ).x > -52.0 )
        {
            chaseMove( agent );
        }
        else
        {
            rcsc::Vector2D move_point = rcsc::Line2D( wm.ball().pos(), wm.ball().inertiaFinalPoint() ).intersection(
                rcsc::Line2D( rcsc::Vector2D( -52.5, -9.0 ), rcsc::Vector2D( -52.5, 9.0 ) ) );
            int cycle = 1;
            for( int i = 1; i < 15; i++ )
            {
                if( wm.ball().inertiaPoint( i ).dist( wm.self().pos() ) < move_point.dist( wm.self().pos() ) )
                {
                    move_point = wm.ball().inertiaPoint( i );
                    cycle = i;
                }
                else
                {
                    break;
                }
            }
            agent->setIntention( new Opuci_IntentionDash( move_point, cycle, sp.maxDashPower(), 1.0 ) );
            Opuci_Move( move_point, sp.maxDashPower() ).execute( agent );
            agent->setNeckAction( new rcsc::Neck_TurnToBall() );
            agent->debugClient().setTarget( move_point );
        }
    }
    else if( dangerSituation( agent ) )
    {
        chaseMove( agent );
        agent->debugClient().addMessage("goalieCharge");
    }
    /*
    else if( self_min <= opp_min
             && self_min < mate_min
             && our_penalty.contains( wm.ball().inertiaPoint( self_min ) ) )
    {
        chaseMove( agent );
        agent->debugClient().addMessage( "selfMin:%d,oppMin:%d,mateMin:%d", self_min, opp_min, mate_min );
    }
    */
    else if( self_min <= opp_min
             && self_min < mate_min
             //&& opp_min < mate_min + 3
             && base_pos.x > -sp.pitchHalfLength()
             && base_pos.absY() < sp.pitchHalfWidth() )
    {
        chaseMove( agent );
        agent->debugClient().addMessage( "selfMin:%d,oppMin:%d", self_min, opp_min );
    }
    else
    {
        basicMove( agent );
    }
    return true;
}
/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_Goalie::chaseMove( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    const int self_min = wm.interceptTable()->selfReachCycle();
    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();
    int ball_reach_step = 0;
    if( !wm.self().isKickable()
        && !wm.existKickableTeammate()
        && !wm.existKickableOpponent() )
    {
        ball_reach_step = std::min( self_min, std::min( mate_min, opp_min ) );
    }
    const rcsc::Vector2D base_pos = wm.ball().inertiaPoint( ball_reach_step );

    if( self_min < mate_min
        && self_min <= opp_min )
    {
        rcsc::Body_Intercept().execute( agent );
    }
    else if( wm.existKickableOpponent()
             && self_min < 3 )
    {
        rcsc::Vector2D opp_pos = wm.interceptTable()->fastestOpponent()->pos();
        agent->setIntention( new Opuci_IntentionDash( opp_pos, 2, sp.maxDashPower(), 0.5 ) );
        Opuci_Move( opp_pos, sp.maxDashPower(), 0.5 );
    }
    else
    {
        rcsc::Vector2D goal_center = centroid( agent, NULL, NULL, NULL, NULL );
        rcsc::Line2D base_line( base_pos, goal_center );
        rcsc::Vector2D move_point = base_line.projection( wm.self().pos() );
        if( base_line.dist( wm.self().pos() ) < 1.0 )
        {
            rcsc::Body_Intercept().execute( agent );
        }
        else
        {
            if( move_point.x < -51.5 )
            {
                move_point.x = -51.5;
                move_point.y = base_line.getY( -51.5 );
            }
            else if( move_point.x > base_pos.x )
            {
                move_point = base_pos;
            }
            const double dash_power = dashPower( agent, move_point );
            double dist_thr = move_point.dist( wm.self().pos() ) * 0.3;
            if ( dist_thr < 0.5 ) dist_thr = 0.5;
            Opuci_Move( move_point, dash_power, dist_thr ).execute( agent );
        }
    }
    agent->setNeckAction( new rcsc::Neck_TurnToBall() );
    agent->debugClient().addMessage("chaseMove");

    return true;
}
/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_Goalie::basicMove( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    const int self_min = wm.interceptTable()->selfReachCycle();
    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();
    int ball_reach_step = 0;
    if( !wm.self().isKickable()
        && !wm.existKickableTeammate()
        && !wm.existKickableOpponent() )
    {
        ball_reach_step = std::min( self_min, std::min( mate_min, opp_min ) );
    }
    const rcsc::Vector2D base_pos = wm.ball().inertiaPoint( ball_reach_step );

    rcsc::Vector2D left_side;//( -sp.pitchHalfLength(), -7.0 );
    rcsc::Vector2D right_side;//( -sp.pitchHalfLength(), 7.0 );
    const rcsc::Vector2D center = centroid( agent, &left_side, &right_side, NULL, NULL );
    const rcsc::Vector2D static_point( -51.5, center.y );
    const rcsc::Line2D line_base_to_center( base_pos, center );
    const rcsc::AngleDeg angle_base_to_center = ( base_pos - center ).th();
    rcsc::Vector2D move_point = static_point;

    //最もx座標の小さい敵を基準にどのくらい前へ出るかを決定する
    const double min_opp_x = Bhv_defense().oppFwLineX( agent );
    double ball_factor = ( min_opp_x + sp.pitchHalfLength() ) * 0.5;
    if( base_pos.x < wm.defenseLineX() + 1.0
        || base_pos.x < -30.0 )
    {
        ball_factor = ( base_pos.x + sp.pitchHalfLength() ) * 0.3;
    }
    move_point.x = static_point.x + angle_base_to_center.cos() * ball_factor;
    move_point.y = static_point.y + angle_base_to_center.sin() * ball_factor;

    if( move_point.dist( static_point ) > 25.0 )
    {
        move_point.x = static_point.x + angle_base_to_center.cos() * 25.0;
        move_point.y = static_point.y + angle_base_to_center.sin() * 25.0;
    }

    const rcsc::Rect2D our_goal( rcsc::Vector2D( -sp.pitchHalfLength(),
                                                 -sp.goalAreaHalfWidth() ),
                                 rcsc::Size2D( sp.goalAreaLength(),
                                               sp.goalAreaWidth() ) );
    if( our_goal.contains( move_point ) )
    {
        const rcsc::AngleDeg left_angle = ( rcsc::Vector2D( -47.0, -9.0 ) - center ).th();
        const rcsc::AngleDeg right_angle = ( rcsc::Vector2D( -47.0, 9.0 ) - center ).th();
        if( angle_base_to_center.degree() > left_angle.degree()
            && angle_base_to_center.degree() < right_angle.degree() )
        {
            move_point.x = -47.0;
            move_point.y = line_base_to_center.getY( -47.0 );
        }
        else
        {
            move_point.x = std::max( line_base_to_center.getX( 9.0 * ( base_pos.y / base_pos.absY() ) ), -52.0 );
            move_point.y = 9.0 * ( base_pos.y / base_pos.absY() );
        }
    }
    if( wm.self().inertiaFinalPoint().dist( move_point ) > 0.5
        && move_point.dist( static_point ) < wm.self().pos().dist( static_point )
        && wm.self().pos().dist( move_point ) < 5.0 )
    {
        const rcsc::AngleDeg angle_to_point = ( wm.self().pos() - move_point ).th();
        const rcsc::AngleDeg angle = 180.0 - ( wm.self().body() - angle_to_point ).degree();
        agent->doDash( sp.maxDashPower(), angle );
    }
    else
    {
        const double dash_power = dashPower( agent, move_point );
        double dist_thr = move_point.dist( wm.self().pos() ) * 0.3;
        if ( dist_thr < 0.5 ) dist_thr = 0.5;
        if( !Opuci_Move( move_point, dash_power, dist_thr ).execute( agent ) )
        {
            rcsc::Body_TurnToBall().execute( agent );
        }
    }
    agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
    agent->debugClient().addMessage( "GoalieBasicMove" );
    agent->debugClient().setTarget( move_point );
    return true;
}
/*-------------------------------------------------------------------*/
/*!

 */
rcsc::Vector2D
Bhv_Goalie::centroid( rcsc::PlayerAgent * agent,
                      rcsc::Vector2D * left_goal_side,
                      rcsc::Vector2D * right_goal_side,
                      int * left_cycle,
                      int * right_cycle )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    const int self_min = wm.interceptTable()->selfReachCycle();
    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();
    int ball_reach_step = 0;
    if( !wm.self().isKickable()
        && !wm.existKickableTeammate()
        && !wm.existKickableOpponent() )
    {
        ball_reach_step = std::min( self_min, std::min( mate_min, opp_min ) );
    }
    const rcsc::Vector2D base_pos = wm.ball().inertiaPoint( ball_reach_step );
    rcsc::Vector2D left_side( -sp.pitchHalfLength(), -7.0 );
    rcsc::Vector2D right_side( -sp.pitchHalfLength(), 7.0 );

    //近くに味方がいる場合は，守備範囲を狭める
    rcsc::Triangle2D left_triangle( base_pos, left_side, sp.ourTeamGoalPos() );
    rcsc::Triangle2D right_triangle( base_pos, right_side, sp.ourTeamGoalPos() );
    const rcsc::PlayerPtrCont & mates = wm.teammatesFromSelf();
    for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
         it != mates.end();
         ++it )
    {
        if( left_triangle.contains( (*it)->pos() ) )
        {
            rcsc::Line2D left_line( base_pos, (*it)->pos() );
            left_side = rcsc::Vector2D( -sp.pitchHalfLength(), left_line.getY( -sp.pitchHalfLength() ) + 1.0 );
            left_triangle = rcsc::Triangle2D( base_pos, left_side, sp.ourTeamGoalPos() );
        }
        else if( right_triangle.contains( (*it)->pos() ) )
        {
            rcsc::Line2D right_line( base_pos, (*it)->pos() );
            right_side = rcsc::Vector2D( -sp.pitchHalfLength(), right_line.getY( -sp.pitchHalfLength() ) - 1.0 );
            right_triangle = rcsc::Triangle2D( base_pos, right_side, sp.ourTeamGoalPos() );
        }
    }
    if( left_goal_side )
        *left_goal_side = left_side;
    if( right_goal_side )
        *right_goal_side = right_side;

    rcsc::Vector2D centroid = sp.ourTeamGoalPos();

    double cycle_ball_to_left_side = (int)rcsc::calc_length_geom_series( sp.ballSpeedMax(),
                                                                         base_pos.dist( left_side ),
                                                                         sp.ballDecay() );
    if( cycle_ball_to_left_side < 1
        || cycle_ball_to_left_side > 30 )
        cycle_ball_to_left_side = 30;

    if( left_cycle )
        *left_cycle = cycle_ball_to_left_side;

    double cycle_ball_to_right_side = (int)rcsc::calc_length_geom_series( sp.ballSpeedMax(),
                                                                          base_pos.dist( right_side ),
                                                                          sp.ballDecay() );
    if( cycle_ball_to_right_side < 1
        || cycle_ball_to_right_side > 30 )
        cycle_ball_to_right_side = 30;

    if( right_cycle )
        *right_cycle = cycle_ball_to_right_side;

    centroid.y = sp.goalWidth() * 0.5 - ( sp.goalWidth() * ( cycle_ball_to_right_side ) 
                                          / ( cycle_ball_to_left_side + cycle_ball_to_right_side ) );

    return centroid;
}
/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_Goalie::dangerSituation( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();
    int ball_reach_step = 0;
    if( !wm.existKickableTeammate()
        && !wm.existKickableOpponent() )
    {
        ball_reach_step = std::min( mate_min, opp_min );
    }
    const rcsc::Vector2D base_pos = wm.ball().inertiaPoint( ball_reach_step );

    if( self_min < opp_min
        || mate_min < opp_min )
    {
        //opps don't control ball
        return false;
    }

    if( base_pos.absY() > 15.0 )
    {
        //safety angle
        return false;
    }
    const rcsc::Triangle2D goal_triangle( base_pos, rcsc::Vector2D( -52.0, -6.0 ), rcsc::Vector2D( -52.5, 6.0 ) );
    const rcsc::PlayerPtrCont & mates = wm.teammatesFromSelf();
    for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
         it != mates.end();
         it++ )
    {
        if( goal_triangle.contains( (*it)->pos() ) )
        {
            //opps far from our goal
            return false;
        }
    }
    const rcsc::PlayerObject * fastest_opp = wm.interceptTable()->fastestOpponent();
    const rcsc::PlayerObject * fastest_mate = wm.interceptTable()->fastestTeammate();
    if( mate_min < self_min )
    {
        //exit defense mate
        return false;
    }

    const rcsc::AngleDeg left_angle = ( rcsc::Vector2D( -52.0, -7.0 ) - base_pos ).th();
    const rcsc::AngleDeg right_angle = ( rcsc::Vector2D( -52.0, 7.0 ) - base_pos ).th();
    bool is_left_contained = false;
    bool is_right_contained = false;
    for( int i = 0; i < 30; i++ )
    {
        const double dash_r = wm.self().playerTypePtr()->dashDistanceTable().at( i );
        const double ball_r = rcsc::calc_sum_geom_series( sp.ballSpeedMax(), sp.ballDecay(), i + 1 );
        const rcsc::Vector2D left_ball_pos( base_pos.x + left_angle.cos() * ball_r,
                                            base_pos.y + left_angle.sin() * ball_r );
        const rcsc::Vector2D right_ball_pos( base_pos.x + right_angle.cos() * ball_r,
                                             base_pos.y + right_angle.cos() * ball_r );
        const rcsc::Circle2D my_circle( wm.self().pos(), dash_r );
        if( my_circle.contains( left_ball_pos ) )
        {
            is_left_contained = true;
        }
        if( my_circle.contains( right_ball_pos ) )
        {
            is_right_contained = true;
        }

        if( left_ball_pos.x < -52.5
            && !is_left_contained )
        {
            agent->debugClient().addMessage("danger");
            return true;
        }
        if( right_ball_pos.x < -52.5
            && !is_right_contained )
        {
            agent->debugClient().addMessage("danger");
            return true;
        }
    }
    return false;
}
/*-------------------------------------------------------------------*/
/*!

 */
double
Bhv_Goalie::dashPower( rcsc::PlayerAgent * agent,
                       const rcsc::Vector2D & target_point )
{    static bool s_recover_mode = false;

    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();
    int ball_reach_step = 0;
    if( !wm.existKickableTeammate()
        && !wm.existKickableOpponent() )
    {
        ball_reach_step = std::min( wm.interceptTable()->teammateReachCycle(),
                                    wm.interceptTable()->opponentReachCycle() );
    }
    const rcsc::Vector2D base_pos = wm.ball().inertiaPoint( ball_reach_step );
    /*--------------------------------------------------------*/
    // check recover
    /*
    if (( wm.ball().pos().x < 5.0 && wm.existKickableOpponent() )
	|| wm.ball().pos().x < wm.self().pos().x + 5.0 )
    {
        s_recover_mode = false;
    }
    */
    if ( wm.self().stamina() < SP.staminaMax() * 0.4 )
    {
        s_recover_mode = true;
    }
    else if ( wm.self().stamina() > SP.staminaMax() * 0.6 )
    {
        s_recover_mode = false;
    }

    /*--------------------------------------------------------*/
    double dash_power = SP.maxDashPower();
    const double my_inc
        = wm.self().playerType().staminaIncMax()
        * wm.self().recovery();

    if( wm.gameMode().type() != rcsc::GameMode::PlayOn )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": not play on mode" );

        dash_power = my_inc * 0.8;
    }
    else if ( ( ( base_pos.x < wm.defenseLineX() )
                || ( base_pos.x < wm.self().pos().x ) )
              && ! wm.existKickableTeammate() )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": correct DF line. keep max power" );
        // keep max power
        dash_power = SP.maxDashPower();
    }
    else if( self_min <= mate_min
             && self_min <= opp_min )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": my ball. keep max power" );
        dash_power = SP.maxDashPower();
    }
    else if ( base_pos.x < -30.0
              && opp_min <= mate_min
              && opp_min <= self_min )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": exist kickable opponent dash" );
        dash_power = rcsc::ServerParam::i().maxDashPower();
    }
    else if( ( target_point.x < wm.self().pos().x
               || target_point.dist( SP.ourTeamGoalPos() ) < wm.self().pos().dist( SP.ourTeamGoalPos() ) )
             && opp_min <= mate_min
             && opp_min <= self_min )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": return to goal dash" );
        dash_power = rcsc::ServerParam::i().maxDashPower();
    }
    else if ( s_recover_mode )
    {
        dash_power = my_inc - 25.0; // preffered recover value
        if ( dash_power < 0.0 ) dash_power = 0.0;

        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": recovering" );
    }
    else if ( wm.existKickableTeammate()
              && wm.ball().distFromSelf() > 15.0 )
    {
        dash_power = std::min( my_inc - 20.0,
                               rcsc::ServerParam::i().maxPower() * 0.5 );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: exist kickable teammate. dash_power= %f"
                            ,__FILE__, __LINE__,
                            dash_power );
    }
    else if ( wm.existKickableTeammate()
              && wm.ball().distFromSelf() < 20.0 )
    {
        dash_power = std::min( my_inc * 1.1,
                               rcsc::ServerParam::i().maxDashPower() );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": exist kickable teammate. dash_power=%.1f",
                            dash_power );
    }
    else if( wm.self().pos().dist( target_point ) > 15.0
             && wm.self().stamina() >  SP.staminaMax() * 0.7 )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": target point far from here" );
        dash_power = SP.maxDashPower();
    }
    //nomal
    else
    {
        dash_power = std::min( my_inc * 1.3,
                               rcsc::ServerParam::i().maxPower() * 0.75 );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: normal mode dash_power= %f"
                            ,__FILE__, __LINE__,
                            dash_power );
    }
    return dash_power;
}
