#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_offense.h"
#include "opuci_move.h"
#include "opuci_dribble.h"
#include "opuci_one_step_kick.h"
#include "opuci_shoot.h"
#include "opuci_search_ball.h"
#include "opuci_chase_ball.h"
#include "opuci_neck.h"
#include "opuci_intention_dribble.h"
#include "strategy.h"


//#include "experiment.h"
//#include "opuci_intercept.h"
#include <rcsc/common/audio_memory.h>

#include <rcsc/soccer_math.h>

#include <rcsc/player/intercept_table.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/say_message_builder.h>
#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/sector_2d.h>
#include <rcsc/geom/circle_2d.h>
#include <rcsc/geom/ray_2d.h>
#include <rcsc/action/body_kick_one_step.h>
/*-------------------------------------------------------------------*/
/*!
  execute action
*/
/*-------------------------------------------------------------------*/

bool
Bhv_Offense::execute( rcsc::PlayerAgent * agent )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Bhv_Offense"
                        ,__FILE__, __LINE__ );
    const rcsc::WorldModel & wm = agent->world();


    agent->debugClient().addMessage( "WN%d SELF%d", wm.interceptTable()->selfReachCycle(), selfReachCycle( agent ) );
    agent->debugClient().addMessage( "BallSpeed%.2f", wm.ball().vel().r() ); 
//    std::cout << "Offense, " << wm.time().cycle() << ", " << wm.self().unum() << ", "
//              << wm.interceptTable()->selfReachCycle() << ", "
//              << selfReachCycle( agent ) << std::endl;
    if( wm.self().isKickable() )
    {
        doKick( agent );
        neckToMate( agent );
        return true;
    }
    if( wm.ball().posCount() > 5 )
    {
        //ボールのposcountが大きければ，ボールを探す
         Opuci_SearchBall().execute( agent );
    }
    else
    {
        //キックできない場合はムーブ，首はdoMove関数内
        doMove( agent );

        if( wm.ball().rposCount() < 1 )
        {
            //ボールのrposcountが0ならばボールの位置・速度，自分の位置，体の角度をメッセージで配信
            agent->addSayMessage( new rcsc::BallPlayerMessage( agent->effector().queuedNextBallPos(), 
                                                               agent->effector().queuedNextBallVel(),
                                                               wm.self().unum(),
                                                               agent->effector().queuedNextMyPos(),
                                                               agent->effector().queuedNextMyBody() ) );
        }
        else
        {
            //ボールのrposcountが1以上ならば自分の位置をメッセージで配信 #SelfMessageの方がいいかもしれない
            agent->addSayMessage( new rcsc::OnePlayerMessage( wm.self().unum(), agent->effector().queuedNextMyPos() ) );
        }
    }
    
    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();

    if( self_min < mate_min )
    {
        //自分がボールに一番近そうならばアテンションを切る
        agent->doAttentiontoOff();
    }
    else
    {
        if( wm.getTeammateNearestToBall( 100 ) )
        {
            //最も近そうな味方が他にいればアテンションを向ける
            agent->doAttentionto( wm.self().side(),
                                  wm.getTeammateNearestToBall( 100 )->unum() );
        }
    }
    agent->debugClient().addMessage("posCnt:%d",wm.ball().posCount());

    return true;
}

/*-------------------------------------------------------------------*/
/*!
  移動を行う関数
*/
/*-------------------------------------------------------------------*/

bool
Bhv_Offense::doMove( rcsc::PlayerAgent * agent )
{
  rcsc::dlog.addText( rcsc::Logger::TEAM,
                      "%s:%d: Bhv_Offsence::doMove"
                      ,__FILE__, __LINE__ );
  
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
    int self_min = wm.interceptTable()->selfReachCycle(); //自分がボールに到達可能なサイクル数
    int mate_min = wm.interceptTable()->teammateReachCycle(); //味方が〃
    int opp_min = wm.interceptTable()->opponentReachCycle(); //敵が〃

    const int agent_reach_cycle = std::min( std::min( self_min, mate_min ), opp_min ); //敵，味方，自分の中で最短でボールに到達可能なサイクル数
    const rcsc::Vector2D base_pos = wm.ball().inertiaPoint( agent_reach_cycle ); //エージェントが到達可能なサイクル数後の，慣性によるボールの位置
    const rcsc::Segment2D base_line( base_pos, M_target_point ); //ボールの予定位置と最終目標位置とを結ぶ線分
    const rcsc::Vector2D home_pos = getPosition( wm, wm.self().unum() ); //ホームポジション
    rcsc::Vector2D target_pos( home_pos ); //目標位置：ホームポジション
    double dash_power = dashPower( agent );


    if( self_min <= mate_min + 1 )
   {
        Opuci_ChaseBall().execute( agent );
        rcsc::Vector2D next_ball_pos = agent->effector().queuedNextBallPos();
        rcsc::Vector2D next_self_pos = agent->effector().queuedNextSelfPos();
        rcsc::AngleDeg next_self_body = agent->effector().queuedNextSelfBody();
        rcsc::Vector2D rpos = next_ball_pos - next_self_pos;
        rcsc::AngleDeg angle = rpos.th() - next_self_body;
        if( (wm.ball().posCount() >= 1
	     || wm.ball().velCount() >= 1)
	    && fabs( angle.degree() ) < 115.0 )
        {
	  rcsc::dlog.addText( rcsc::Logger::TEAM,
                              "%s:%d: doturnneck( %f )"
                              ,__FILE__, __LINE__, angle.degree() );
	  agent->doTurnNeck( angle -wm.self().neck() );
	  agent->debugClient().addMessage( "LookBall" );
        }
        else
        {
            neckToMate( agent );
            agent->debugClient().addMessage( "LookMate" );
        }
        return true;
    }
    else if( opp_min + 3 < mate_min
             || wm.getTeammateNearestToBall( 100 )->unum() < 7 )
    {
        //敵の方が味方より３サイクル以上早くボールに到達可能な場合
        dash_power = dashPower( agent, SP.maxPower() * 0.7 );
        if( target_pos.x > wm.offsideLineX() )
        {
            //目標位置がオフサイドラインを越える場合は修正
            target_pos.x = wm.offsideLineX();
        }
        //目標位置（ホームポジション）へ移動
        if( Opuci_Move( target_pos, dash_power, 3.0 ).execute( agent ) )
        {
            agent->debugClient().addMessage("OpponentBall");
            agent->debugClient().setTarget( home_pos );
        }
        else
        {
            //目標位置に到達している場合はゴールの方を向く
            rcsc::Vector2D opp_goal_pos( SP.pitchHalfLength(), 0.0 );
            agent->doTurn( ( opp_goal_pos - wm.self().pos() ).th() - wm.self().body() );
            agent->debugClient().addMessage("TurnToOppGoal");
        }
        Opuci_Neck().execute( agent );
        return true;
    }

    //experiment
    /*
    if( Experiment().passedToMe( agent ) )
    {
      if( wm.getTeammateNearestToBall( 100 ) )
      {
        agent->setIntention( new Opuci_Intercept( 10, wm.getTeammateNearestToBall( 100 )->unum() ) );
        agent->debugClient().addMessage( "ExpInTrCpt" );
        return true;
      }
    }
    else
    {
        if( agent->world().audioMemory().passTime() == agent->world().time()
            && ! agent->world().audioMemory().pass().empty()
            && ( agent->world().audioMemory().pass().front().receiver_
                 != agent->world().self().unum() )
            && wm.audioMemory().pass().front().sender_ >= 7 )
        {
            //experiment
            const rcsc::WorldModel & wm = agent->world();
            const rcsc::Vector2D goal_pos( 51.5, 0.0 );
            const rcsc::AngleDeg angle_to_goal = ( goal_pos - ( wm.ball().pos() - wm.ball().vel() ) ).th();
            const rcsc::Line2D offense_line( ( wm.ball().pos() - wm.ball().vel() ), angle_to_goal + 90.0 );
            int front_mate_count = 0;
            const rcsc::PlayerPtrCont & mates = wm.teammatesFromBall();
            for( rcsc::PlayerPtrCont::const_iterator mate = mates.begin();
                 mate != mates.end();
                 ++mate )
            {
                if( ( offense_line.a() * (*mate)->pos().x + offense_line.b() * (*mate)->pos().y + offense_line.c() )
                    * ( offense_line.a() * goal_pos.x + offense_line.b() * goal_pos.y + offense_line.c() ) > 0.0
                    && (*mate) != mates.front() )
                {
                    front_mate_count++;
                }
            }
            if( ( offense_line.a() * wm.self().pos().x + offense_line.b() * wm.self().pos().y + offense_line.c() )
                * ( offense_line.a() * goal_pos.x + offense_line.b() * goal_pos.y + offense_line.c() ) > 0.0 )
            {
                front_mate_count++;
            }
            
            if( front_mate_count >= 3 )
            {
                Experiment().discrimination( 3 );
            }
        }
    }
    */
    std::vector< rcsc::Vector2D > candidates;
    setFormationPos( agent, candidates );
    switch( wm.self().unum() )
    {
    case 7:
        target_pos = candidates.at( 0 );
        break;
    case 8:
        target_pos = candidates.at( 1 );
        break;
    case 9:
        target_pos = candidates.at( 2 );
        break;
    case 10:
        target_pos = candidates.at( 3 );
        break;
    case 11:
        target_pos = candidates.at( 4 );
        break;
    default:
        break;
    }
    if( target_pos.x < wm.self().pos().x
	&& wm.self().pos().x < wm.offsideLineX() )
      target_pos.x = wm.self().pos().x;
    

    agent->debugClient().setTarget( target_pos );

    if( target_pos.dist( wm.self().pos() ) > wm.self().kickableArea() )
    {
        //目標位置と自分との距離がキッカブル距離より大きい場合
        dash_power = dashPower( agent, rcsc::ServerParam::i().maxPower() ); //最大ダッシュパワー
        if( base_pos.x < -15.0 )
        {
            //ボールの予定位置が−１５より小さい場合
            dash_power = dashPower( agent, rcsc::ServerParam::i().maxPower() * 0.7 ); //ダッシュパワーの7割
        }
        rcsc::AngleDeg margin_angle = 15.0;
        double margin = wm.self().pos().dist( target_pos ) * margin_angle.tan();//目標位置との距離の2割ほど？
        //目標位置へ移動
        agent->debugClient().addMessage( "Positioning");
        Opuci_Move( target_pos, dash_power, margin ).execute( agent );
    }
    else
    {
        rcsc::Vector2D opp_goal_pos( rcsc::ServerParam::i().pitchHalfLength(), 0.0 );
        agent->doTurn( ( opp_goal_pos - wm.self().pos() ).th() - wm.self().body() );
        agent->debugClient().addMessage( "WaitTurn" );
    }
    //味方を見る
    if( !neckToMate( agent ) )
      {
	Opuci_Neck().execute( agent );
      }

    return true;
}

/*-------------------------------------------------------------------*/
/*!
  ホームポジションを決める関数
*/
/*-------------------------------------------------------------------*/

rcsc::Vector2D
Bhv_Offense::getPosition( const rcsc::WorldModel & wm,
                          const int unum )
{
    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    int ball_reach_step = std::min( opp_min, std::min( self_min, mate_min ) );

    rcsc::Vector2D static_pos = Strategy::i().getBeforeKickOffPos( unum ); //試合開始時の自分の位置
    rcsc::Vector2D base_pos = wm.ball().inertiaPoint( ball_reach_step ); //ボールの予定位置
    rcsc::Vector2D home_pos = base_pos + static_pos;

    home_pos.x = base_pos.x + ( static_pos.x * 2.0 + 30.0 ) - 1.0 / 2.0 * base_pos.x;
    home_pos.y = static_pos.y;

    return home_pos;
}

/*-------------------------------------------------------------------*/
/*!
  レシーブ位置の候補を求める関数
*/
/*-------------------------------------------------------------------*/

bool
Bhv_Offense::getPassPoint( rcsc::PlayerAgent * agent,
                           std::vector<rcsc::Vector2D> * pass_point )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
    int self_min = wm.interceptTable()->selfReachCycle(); //自分がボールに到達可能なサイクル数
    int mate_min = wm.interceptTable()->teammateReachCycle(); //味方が〃
    int opp_min = wm.interceptTable()->opponentReachCycle(); //敵が〃
    const int agent_reach_cycle = std::min( std::min( self_min, mate_min ), opp_min ); //敵，味方，自分の中で最短でボールに到達可能なサイクル数

    const rcsc::Vector2D base_pos = wm.ball().inertiaPoint( agent_reach_cycle ); //エージェントが到達可能なサイクル数後の，慣性によるボールの位置
    const double pass_line_dist = 1.2 * ( 1.0 - pow( ( 1.0 / SP.ballDecay() ), 10 ) ) //ボール10サイクル距離
        / ( 1.0 - ( 1.0 / SP.ballDecay() ) );

    const rcsc::AngleDeg target_angle = ( M_target_point - base_pos ).th(); //最終目標位置とボールの予定位置との角度

    //first_receiver
    for( int i = 0; i < 5; i++ )
    {
        rcsc::AngleDeg start_angle = target_angle + 50.0 * i - 125.0; //目標位置への角度から-125〜+75
        rcsc::AngleDeg end_angle = target_angle + 50.0 * i - 75.0; //目標位置への角度から-75〜+125
        rcsc::Sector2D pass_sector( base_pos, 0.0, pass_line_dist,
                                    start_angle, end_angle ); //ボールの位置から長さがボール10サイクルの距離で幅が50度の扇形

        std::vector<rcsc::Vector2D> opp_pos;
        //扇形の左右の端になる位置を追加
        opp_pos.push_back( rcsc::Vector2D( base_pos.x + pass_line_dist * start_angle.cos(),
                                           base_pos.y + pass_line_dist * start_angle.sin() ) );
        opp_pos.push_back( rcsc::Vector2D( base_pos.x + pass_line_dist * end_angle.cos(),
                                           base_pos.y + pass_line_dist * end_angle.sin() ) );
        const rcsc::PlayerPtrCont & opps = wm.opponentsFromBall();
        for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
             it != opps.end();
             ++it )
        {
            if( pass_sector.contains( (*it)->pos() ) )
            {
                //扇形に含まれる敵の位置を追加
                opp_pos.push_back( (*it)->pos() );
            }
        }

        //back receiver
        if( start_angle.degree() > end_angle.degree() )
        {
            //扇形の開始角度が終了角度より大きい場合 #-180を挟んだ場合？
            for( std::vector<rcsc::Vector2D>::iterator it = opp_pos.begin();
                 it != opp_pos.end();
                 ++it )
            {
                for( std::vector<rcsc::Vector2D>::iterator itr = it;
                     itr != opp_pos.end();
                     ++itr )
                {
                    if( it != itr
                        && (*itr).y < (*it).y )
                    {
                        //y座標が小さい順にソート
                        swap_ranges( it, it + 1, itr );
                    }
                }
            }
        }
        else
        {
            for( std::vector<rcsc::Vector2D>::iterator it = opp_pos.begin();
                 it != opp_pos.end();
                 ++it )
            {
                for( std::vector<rcsc::Vector2D>::iterator itr = it;
                     itr != opp_pos.end();
                     ++itr )
                {
                    if( it != itr
                        && ( (*itr) - base_pos ).th().degree() < ( (*it) - base_pos ).th().degree() )
                    {
                        //ボールの予定位置との角度が小さい順にソート
                        swap_ranges( it, it + 1, itr );
                    }
                }
            }
        }

        int size = opp_pos.size();
        rcsc::AngleDeg max_angle = 0.0;
        rcsc::Vector2D pass_target_point = M_target_point;
        rcsc::AngleDeg pass_target_angle = ( M_target_point - base_pos ).th();
        for( int j = 0; j < size - 1; j++ )
        {
            rcsc::Vector2D last_point = opp_pos.back(); //ボールとの角度が1番大きい敵の位置
            opp_pos.pop_back();
            rcsc::Vector2D start_point = opp_pos.back(); //ボールとの角度が２番目に大きい敵の位置
            rcsc::AngleDeg dist_angle = ( ( last_point - base_pos ).th() 
                                          - ( start_point - base_pos ).th() ).abs(); //敵２人とボールの成す角度の絶対値
            if( dist_angle.degree() > max_angle.degree() )
            {
                //敵の間の角度が最も大きくなる位置の中間をレシーブ位置
                max_angle = dist_angle;
                pass_target_point = rcsc::Vector2D( 0.5 * ( start_point.x + last_point.x ),
                                                    0.5 * ( start_point.y + last_point.y ) );
                pass_target_angle = ( pass_target_point - base_pos ).th();
            }
        }

        if( pass_target_point.absX() > SP.pitchHalfLength() )
        {
            //目標位値が前後のフィールドラインから出る場合
            pass_target_point.x = SP.pitchHalfLength() * pass_target_point.x / pass_target_point.absX();
        }
        else if( pass_target_point.absY() > SP.pitchHalfWidth() )
        {
            //目標位値が左右のフィールドラインから出る場合
            pass_target_point.y = SP.pitchHalfWidth() * pass_target_point.y / pass_target_point.absY();
        }
        //レシーブ位置候補へ追加
        (*pass_point).push_back( pass_target_point );
    }//end for i in 0..5

    return true;
}

/*-------------------------------------------------------------------*/
/*!
  敵の間を通すパス探索関数
  レシーブ位置とレシーバの背番号
*/
/*-------------------------------------------------------------------*/

bool
Bhv_Offense::searchPass( rcsc::PlayerAgent * agent,
                         rcsc::Vector2D * pass_target,
                         int * target_unum )
{
    const rcsc::WorldModel & wm = agent->world();

    std::vector<rcsc::Vector2D> target;
    const double pass_line_dist = 1.2 * ( 1.0 - pow( ( 1.0 / rcsc::ServerParam::i().ballDecay() ), 10 ) ) //10サイクル後に速さが1.2になるボールが移動する距離
        / ( 1.0 - ( 1.0 / rcsc::ServerParam::i().ballDecay() ) );

    rcsc::AngleDeg angle_to_target_point = ( M_target_point - wm.ball().pos() ).th(); //最終目標位置（ゴール前）とボールの角度
    std::vector<rcsc::Vector2D> opp_pos;
    const rcsc::PlayerPtrCont & opps = wm.opponentsFromBall();
    for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
         it != opps.end();
         ++it )
    {
        //ボールからの距離が，ボールが10サイクルで移動する距離より近い敵に注目
        if( wm.ball().pos().dist( (*it)->pos() ) < pass_line_dist )
        {
            opp_pos.push_back( (*it)->pos() );
        }
    }

    for( std::vector<rcsc::Vector2D>::iterator it = opp_pos.begin();
         it != opp_pos.end();
         ++it )
    {
        for( std::vector<rcsc::Vector2D>::iterator itr = it;
             itr != opp_pos.end();
             ++itr )
        {
            if( it != itr
                && ( (*itr) - wm.ball().pos() ).th().degree() < ( (*it) - wm.ball().pos() ).th().degree() )
            {
                //ボールとの角度が小さい順にソート
                swap_ranges( it, it + 1, itr );
            }
        }
    }

    int size = opp_pos.size();
    if( size >= 2 )
    {
        //敵が2人以上のときに間を探索
        for( int i = 0; i < size - 1; i++ )
        {
            rcsc::Vector2D last_point = opp_pos.back(); //1番角度が大きい敵の位置
            opp_pos.pop_back();
            rcsc::Vector2D start_point = opp_pos.back(); //2番目に角度が大きい敵の位置
            rcsc::AngleDeg dist_angle = ( last_point - wm.ball().pos() ).th() - ( start_point - wm.ball().pos() ).th(); //敵2人とボールの成す角度
            rcsc::Vector2D target_point( 0.5 * ( last_point.x + start_point.x ), //敵2人の中間の位置
                                         0.5 * ( last_point.y + start_point.y ) );
            if( dist_angle.degree() > 45.0
                && wm.dirCount( ( target_point - wm.ball().pos() ).th() ) < 3 )
            {
                //敵の間が45度以上空いており，且つその方向の情報を過去2サイクル以内に得ていれば「間の位置」追加
                target.push_back( target_point );
            }
        }
        if( target.empty() )
        {
            //「間の位置」がなければ終了
            return false;
        }

        double dist_mate_to_target = 1000.0;
        double dist_ball_to_target = 0.0;
        for( std::vector<rcsc::Vector2D>::iterator it = target.begin();
             it != target.end();
             ++it )
        {
            agent->debugClient().addLine( wm.ball().pos(), (*it) );
            const rcsc::PlayerPtrCont & mates = wm.teammatesFromBall();
            for( rcsc::PlayerPtrCont::const_iterator itr = mates.begin();
                 itr != mates.end();
                 ++itr )
            {
                rcsc::AngleDeg pass_angle = ( (*it) - wm.ball().pos() ).th(); //「間の位置」とボールの角度
                const double pass_dist = ( (*itr)->pos() - wm.ball().pos() ).r(); //味方とボールの距離
                rcsc::Vector2D pass_point( wm.ball().pos().x + pass_dist * pass_angle.cos(),
                                           wm.ball().pos().y + pass_dist * pass_angle.sin() ); //ボールから「間の位置」の方向へ味方との距離だけ離れた位置
                rcsc::Segment2D pass_line( wm.ball().pos(), pass_point ); //ボールとパス位置を結ぶ線分
                bool exist_opp = false;
                for( rcsc::PlayerPtrCont::const_iterator it_opp = opps.begin();
                     it_opp != opps.end();
                     ++it_opp )
                {
                    //it=間の位置，itr=味方，it_opp=敵
                    const rcsc::Line2D suisen( (*it), pass_angle + 90.0 ); //敵の位置を通り，「間の位置」とボールを結ぶ線と垂直な線
                    const rcsc::Line2D suisen2( wm.ball().pos(), pass_angle + 90.0 ); //ボールの位置を通り，〃
                    if( pass_line.dist( (*it_opp)->pos() ) < pass_dist
                        && ( suisen.a() * (*itr)->pos().x + suisen.b() * (*itr)->pos().y + suisen.c() )
                        * ( suisen.a() * wm.ball().pos().x + suisen.b() * wm.ball().pos().y + suisen.c() ) > 0.0
                        && ( suisen2.a() * (*itr)->pos().x + suisen2.b() * (*itr)->pos().y + suisen2.c() )
                        * ( suisen2.a() * (*it).x + suisen2.b() * (*it).y + suisen2.c() ) > 0.0 )
                    {
                        //ボールとパス位置の線分から敵への距離が，味方とボールの位置との距離より近い　且つ
                        //ボールと味方が，敵の位置を通る垂線に対して同じ側に位置している　且つ
                        //味方と「間の位置」が，ボールの位置を通る垂線に対して同じ側に位置している場合
                        int opp_cycle = 10;
                        double dist_opp_to_line = pass_line.dist( (*it_opp)->pos() )
                            - (*it_opp)->playerTypePtr()->kickableArea();
                        if( dist_opp_to_line < 0.0 )
                        {
                            //ボールとパス位置を結ぶ線分と敵への距離が，敵のキッカブルエリアより小さければ失敗
                            exist_opp = true;
                            break;
                        }
                        for( int i = 0; i < 10; i++ )
                        {
                            if( dist_opp_to_line < (*it_opp)->playerTypePtr()->dashDistanceTable().at( i ) )
                            {
                                opp_cycle = i; //敵がダッシュでパス線分に到達するサイクル数 #ターンなし
                                break;
                            }
                        }
                        double pass_line_dist = pass_dist; //ボールと味方の距離
                        double ball_speed = 1.2;
                        for( int i = 0; i < 10; i++ )
                        {
                            pass_line_dist -= ball_speed;
                            ball_speed /= rcsc::ServerParam::i().ballDecay(); //ボールが目標位置に到達したときに早さが1.2となる初速
                            if( pass_line_dist < 0.0 )
                            {
                                break;
                            }
                        }
                        const rcsc::Vector2D projection = pass_line.projection( (*it_opp)->pos() ); //敵の位置のパス線分に対する（たぶん）対称点
                        double dist_to_projection = wm.ball().pos().dist( projection ); //敵の対称点とボールとの距離
                        int ball_cycle = 9;
                        for( int i = 0; i < 10; i++ )
                        {
                            dist_to_projection -= ball_speed;
                            ball_speed *= rcsc::ServerParam::i().ballDecay();
                            if( dist_to_projection < 0.0 )
                            {
                                ball_cycle = i; //ボールが敵の対称点へ到達するサイクル数  #projectionがinvalidの可能性は不明
                                break;
                            }
                        }

                        if( ball_cycle > opp_cycle
                            || (*it).dist( (*itr)->pos() ) > (*it).dist( (*it_opp)->pos() ) )
                        {
                            //ボールより敵の方が早い　または
                            //「間の位置」との距離が，味方の方が敵より遠ければ失敗
                            exist_opp = true;
                            break;
                        }
                    }//end if pass_line.dist etc
                }//end for opponentsFromBall
                if( ! exist_opp
                    && pass_point.dist( (*itr)->pos() ) < dist_mate_to_target )
                {
                    //邪魔する敵がいない　且つ
                    //「間の位置」と味方との距離が最小になる組み合わせ
                    dist_mate_to_target = (*it).dist( (*itr)->pos() ); //「間の位置」と味方の距離
                    dist_ball_to_target = pass_dist; //ボールと味方の距離
                    *pass_target = pass_point; //レシーブ位置
                    *target_unum = (*itr)->unum();
                }
            }//end for teammatesFromBall
        }//end for target

        int cycle_ball_to_target = 10;
        double ball_speed = 1.0;
        for( int i = 0; i < 10; i++ )
        {
            dist_ball_to_target -= ball_speed;
            if( dist_ball_to_target < 0.0 )
            {
                cycle_ball_to_target = i; //終速1.0のボールが味方の位置に到達するサイクル数
                break;
            }
            ball_speed /= rcsc::ServerParam::i().ballDecay();
        }
        int cycle_mate_to_target = (int)dist_mate_to_target + 2; //味方が「間の位置」に到達するサイクル数 #ターン２回，１サイクルに１進む，小数点無視
        if( cycle_ball_to_target <= cycle_mate_to_target )
            //|| dist_mate_to_target > 5.0 )
        {
            //ボールが味方の位置に到達する方が，味方が「間の位置」に到達するより早い場合
            if( (*pass_target).x > wm.offsideLineX() - 5.0 )
            {
                //パス位置がオフサイドライン-5より前になる場合
                for( int i = 0; i < 5; i++ )
                {
                    (*pass_target).x += 1.0; //パス位置を１ずつ前方へ
                    dist_ball_to_target = (*pass_target).dist( wm.ball().pos() );
                    ball_speed = 1.0;
                    for( int i = 0; i < 10; i++ )
                    {
                        dist_ball_to_target -= ball_speed;
                        if( dist_ball_to_target < 0.0 )
                        {
                            cycle_ball_to_target = i; //ボールが終速１でパス位置に到達するサイクル数
                            break;
                        }
                        ball_speed /= rcsc::ServerParam::i().ballDecay();
                    }
                    cycle_mate_to_target++;
                    if( cycle_ball_to_target > cycle_mate_to_target )
                    {
                        //5サイクル以内で味方の移動の方が早くなれば成功
                        return true;
                    }
                }
            }
            //パス位置がオフサイドラインから遠ければ失敗
            return false;
        }
        //ボールの方が味方より遅ければ成功
        return true;
    }//end if size >= 2
    //敵が1人以下の場合は失敗
    return false;
}

/*-------------------------------------------------------------------*/
/*!
  ダイレクトパス探索関数
  レシーブ位置とレシーバの背番号
*/
/*-------------------------------------------------------------------*/

bool
Bhv_Offense::searchDirectPass( rcsc::PlayerAgent * agent,
                               rcsc::Vector2D * pass_target,
                               int *target_unum )
{
    const rcsc::WorldModel & wm = agent->world();

    std::vector<rcsc::PlayerObject*> target;
    bool exist_opponent_to_pass = false;
    const double pass_line_dist = 1.2 * ( 1.0 - pow( ( 1.0 / rcsc::ServerParam::i().ballDecay() ), 10 ) ) //ボール10サイクル距離
        / ( 1.0 - ( 1.0 / rcsc::ServerParam::i().ballDecay() ) );
    const rcsc::PlayerPtrCont & mates = wm.teammatesFromBall();
    for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
         it != mates.end();
         ++it )
    {
        if( wm.dirCount( ( (*it)->pos() - wm.self().pos() ).th() ) < 3
            && (*it)->posCount() < 3
            && (*it)->pos().x < wm.offsideLineX()
            && (*it)->pos().dist( wm.self().pos() ) > 3.0
            && (*it)->pos().dist( wm.self().pos() ) < pass_line_dist )
        {
            //その方向から２サイクル以内に情報を得た　且つ
            //posCountが２以下　且つ
            //位置がオフサイドラインより手前　且つ
            //自分との距離が３より大きい 且つ
            //自分との距離がボール10サイクルの距離より近い味方

            exist_opponent_to_pass = false;
            const rcsc::AngleDeg angle_to_mate = ( (*it)->pos() - wm.ball().pos() ).th(); //味方とボールの角度
            const rcsc::Segment2D segment( wm.ball().pos(), (*it)->pos() ); //味方とボールを結ぶ線分
            const double dist = wm.self().pos().dist( (*it)->pos() ); //自分と味方の距離
            const rcsc::PlayerPtrCont & opps = wm.opponentsFromBall();
            for( rcsc::PlayerPtrCont::const_iterator itr = opps.begin();
                 itr != opps.end();
                 ++itr )
            {
                //it=味方，itr=敵
                const rcsc::Line2D suisen( (*it)->pos(), angle_to_mate + 90.0 ); //味方の位置を通り，味方とボールの線に垂直な線
                const rcsc::Line2D suisen2( wm.ball().pos(), angle_to_mate + 90.0 ); //ボールの位置を通り，〃
                if( segment.dist( (*itr)->pos() ) < dist
                    && ( suisen.a() * (*itr)->pos().x + suisen.b() * (*itr)->pos().y + suisen.c() )
                    * ( suisen.a() * wm.ball().pos().x + suisen.b() * wm.ball().pos().y + suisen.c() ) > 0.0
                    && ( suisen2.a() * (*itr)->pos().x + suisen2.b() * (*itr)->pos().y + suisen2.c() )
                    * ( suisen2.a() * (*it)->pos().x + suisen2.b() * (*it)->pos().y + suisen2.c() ) > 0.0 )
                {
                    //ボールと味方の線分からの敵への距離が，自分と味方の距離より近い　且つ
                    //ボールと敵が，味方の位置を通る垂線に対して同じ側に位置している　且つ
                    //敵と味方が，ボールの位置を通る垂線に対して同じ側に位置している場合
                    int opp_cycle = 10;
                    double dist_opp_to_line = segment.dist( (*itr)->pos() )
                        - (*itr)->playerTypePtr()->kickableArea();
                    if( dist_opp_to_line < 0.0 )
                    {
                        //ボールと味方の線分から敵への距離が，敵のキッカブルエリアより小さければ失敗
                        exist_opponent_to_pass = true;
                        break;
                    }
                    for( int i = 0; i < 10; i++ )
                    {
                        if( dist_opp_to_line <= (*itr)->playerTypePtr()->dashDistanceTable().at( i ) )
                        {
                            opp_cycle = i; //敵がダッシュでパス線分に到達するサイクル数 #ターンなし
                            agent->debugClient().addMessage("oppCyc%d:%d", (*itr)->unum(), i );
                            break;
                        }
                    }
                    double pass_line_dist = dist; //自分と味方の距離 #ボールとの距離ではない
                    double ball_speed = 1.2;
                    for( int i = 0; i < 10; i++ )
                    {
                        pass_line_dist -= ball_speed; 
                        ball_speed /= rcsc::ServerParam::i().ballDecay(); //到達速度1.2となるボールの初速
                        if( pass_line_dist < 0.0 )
                        {
                            break;
                        }
                    }
                    const rcsc::Vector2D projection = segment.projection( (*itr)->pos() ); //敵の位置の，味方とボールの線分に対する対称点
                    double dist_to_projection = wm.ball().pos().dist( projection ); //敵の対称点とボールとの距離
                    int ball_cycle = 9;
                    for( int i = 0; i < 10; i++ )
                    {
                        dist_to_projection -= ball_speed;
                        ball_speed *= rcsc::ServerParam::i().ballDecay();
                        if( dist_to_projection < 0.0 )
                        {
                            ball_cycle = i; //ボールが敵の対称点へ到達するサイクル数
                            agent->debugClient().addMessage("ballCyc:%d", i );
                            break;
                        }
                    }
                    if( ball_cycle >= opp_cycle
                        || (*it)->pos().dist( (*itr)->pos() ) < 3.0 )
                    {
                        //ボールより敵の方が早い または
                        //味方と敵の距離が３以下ならば失敗
                        exist_opponent_to_pass = true;
                        break;
                    }
                }
            }//end for opponentsFromBall
            
            if( ! exist_opponent_to_pass )
            {
                //邪魔する敵がいなければレシーバの候補に追加
                target.push_back( (*it) );
                agent->debugClient().addMessage("canDrctPs");
            }
        }//end if dirCount etc
    }//end for teammatesFromBall
    
    if( target.empty() )
    {
        //候補がなければ失敗
        return false;
    }

    for( std::vector<rcsc::PlayerObject*>::iterator it = target.begin();
         it != target.end(); 
         ++it )
    {
        if( M_target_point.dist( (*it)->pos() ) < M_target_point.dist( *pass_target ) )
        {
            //候補の中で最終目標位置に最も近い味方を選択
            *pass_target = rcsc::Vector2D( (*it)->pos() );
            *target_unum = (*it)->unum();
        }
    }   
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  ドリブル探索関数
  目標位置，ダッシュカウント（2に固定されている）
*/
/*-------------------------------------------------------------------*/

bool
Bhv_Offense::searchDribble( rcsc::PlayerAgent * agent,
                            rcsc::Vector2D * dribble_target,
                            int * dash_count )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();

    rcsc::AngleDeg angle_to_target = ( M_target_point - wm.self().pos() ).th(); //最終目標位置と自分の位置の角度
    if( wm.dirCount( angle_to_target ) > 3 )
    {
        //目標位置の方向を３サイクル以上見ていなければ失敗
        return false;
    }
    const double dash_power = wm.self().playerType().playerSpeedMax() * 2.0; //最大速度で２サイクルの間に移動できる距離
    const rcsc::PlayerPtrCont & opps = wm.opponentsFromBall();
    const double opp_dash_two = ( SP.defaultPlayerSpeedMax() * 2.0 + SP.defaultKickableArea() );//デフォルトプレイヤが最大速度で２サイクルの間に移動できる距離＋キッカブルエリア

    const rcsc::Sector2D front_sector( wm.ball().pos(), 0.0, 10.0,
                                       angle_to_target - 90.0, angle_to_target + 90.0 ); //ボールから目標位置方向±90度の扇形
    std::list<double> angle;
    angle.push_back( angle_to_target.degree() - 90.0 ); //目標方向-90度を追加
    angle.push_back( angle_to_target.degree() + 90.0 ); //目標方向+90度を追加
    for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
         it != opps.end();
         ++it )
    {
        if( front_sector.contains( (*it)->pos() )
            && ! (*it)->goalie() )
        {
            //扇形内にいる敵とボールとの角度を追加
            angle.push_back( ( (*it)->pos() - wm.ball().pos() ).th().degree() );
        }
    }
    angle.sort(); //角度を小さい順にソート
    int size = angle.size();
    rcsc::AngleDeg dist_angle = 0.0;
    rcsc::AngleDeg target_angle = 0.0;
    for( int i = 0; i < size - 1; i++ )
    {
        rcsc::AngleDeg start_angle = angle.front(); //１番小さい角度
        angle.pop_front();
        rcsc::AngleDeg last_angle = angle.front(); //2番目に小さい角度
        if( last_angle.degree() - start_angle.degree() > dist_angle.degree() )
        {
            //角度の差が最も大きくなる隣接する2つの角度
            //その中間を目標角度とする
            dist_angle = last_angle - start_angle;
            target_angle = ( last_angle.degree() + start_angle.degree() ) / 2.0;
        }
    }

    rcsc::Vector2D target_point = rcsc::Vector2D ( wm.ball().pos().x + dash_power * target_angle.cos(),
                                                   wm.ball().pos().y + dash_power * target_angle.sin() ); //ボールの位置から目標角度へ最大速度で2サイクル移動したときの位置
    rcsc::Circle2D target_circle( target_point, opp_dash_two ); //目標位置を中心として，半径が敵が2サイクルできる距離＋キッカブルエリアとなる円
    for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
         it != opps.end();
         ++it )
    {
        if( target_circle.contains( (*it)->pos() ) )
        {
            //敵が円内にいた場合は失敗
            return false;
        }
    }
    *dash_count = 2;
    *dribble_target = target_point;

    return true;
}

/*-------------------------------------------------------------------*/
/*!
  ダッシュパワーを決定する関数
*/
/*-------------------------------------------------------------------*/

double
Bhv_Offense::dashPower( rcsc::PlayerAgent * agent,
                        const double limit_power )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
    if( wm.self().stamina() < SP.staminaMax() * SP.effortDecThr() * 1.2 
        || wm.self().stamina() < SP.staminaMax() * SP.recoverDecThr() * 1.2 )
    {
        //スタミナ切れが近い場合は停止する
        return 0.0;
    }
    //制限パワー：デフォルトは最大パワー
    return limit_power;
}

/*-------------------------------------------------------------------*/
/*!
  味方がいると思われる方向へ首を向ける関数
*/
/*-------------------------------------------------------------------*/

bool
Bhv_Offense::neckToMate( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    const rcsc::Vector2D next_ball_pos = wm.ball().pos() + wm.ball().vel(); //慣性による次サイクルのボールの位置
    const rcsc::Vector2D next_self_pos = agent->effector().queuedNextSelfPos(); //次サイクルの自分の位置
    const rcsc::AngleDeg next_self_body = agent->effector().queuedNextSelfBody(); //次サイクルの自分の体の向き

    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: Bhv_Offsence::neckToMate"
                        ,__FILE__, __LINE__ );

    /*
    if( wm.ball().posCount() > 1 )
    {
        //ボールのposcountが２以上の場合はボールを見る
        agent->doTurnNeck( ( next_ball_pos - next_self_pos ).th() - ( wm.self().neck() + next_self_body ) );
        agent->debugClient().addMessage( "LkBl" );
	rcsc::dlog.addText( rcsc::Logger::TEAM,
			    "%s:%d: ball.posCount = %d"
			    ,__FILE__, __LINE__, wm.ball().posCount() );
    }
    else */ if( wm.dirCount( next_self_body ) > 1 )
    {
        agent->doTurnNeck( - wm.self().neck() );
        agent->debugClient().addMessage( "LkFront" );
    }
    else
    {
        std::vector<rcsc::Vector2D> pass_target;
        //レシーブ位置候補
//        getPassPoint( agent, &pass_target );
        setFormationPos( agent, pass_target );
        int max_count = -1;
	int max_unum = -1;
        rcsc::AngleDeg search_angle = next_self_body;//目標角度
        for( std::vector<rcsc::Vector2D>::iterator it = pass_target.begin();
             it != pass_target.end();
             ++it )
        {
            const rcsc::AngleDeg target_angle = ( (*it) - next_self_pos ).th(); //レシーブ位置候補と自分の体の向きとの角度
            if( wm.getPointCount( (*it), 15 ) > max_count
                && ( target_angle - next_self_body ).abs() < 115.0 )
            {
                //向けられる範囲でPointCountが最も大きいレシーブ位置の方向
	      max_count = wm.getPointCount( (*it), 15 );
	      search_angle = target_angle;
            }
        }
	rcsc::dlog.addText( rcsc::Logger::TEAM,
			    "%s:%d: max_PointCount: %d, search_angle: %f"
			    ,__FILE__, __LINE__, max_count, search_angle.degree() );
        for( int num = 7; num <= 11; num++ )
        {
            if( wm.teammate( num ) )
            {
                const rcsc::AngleDeg target_angle = ( wm.teammate( num )->pos() - next_self_pos ).th(); //味方の位置と自分の体の向きとの角度
                int tmp_count = wm.teammate( num )->posCount();
                if( num >= 9 )
                {
                    tmp_count *= 1.5;
                }
                if( tmp_count > max_count
                    && ( target_angle - next_self_body ).abs() < 115.0 )
                {
                    //向けられる範囲でposCountが最も大きい味方の位置の方向
                    max_count = tmp_count;
                    search_angle = target_angle;
		    max_unum = num;
                }
            }
        }
	rcsc::dlog.addText( rcsc::Logger::TEAM,
			    "%s:%d: max_PosCount: %d, mate_unum: %d"
			    ,__FILE__, __LINE__, max_count, max_unum );
	if( max_count != -1 )
	  agent->doTurnNeck( search_angle - ( wm.self().neck() + next_self_body ) );
	else
	  {
	    agent->debugClient().addMessage( "NoMate2Look" );
	    return false;
	  }
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  前衛のフォーメーションを決める関数
*/
/*-------------------------------------------------------------------*/

double
Bhv_Offense::setFormationPos( rcsc::PlayerAgent * agent, std::vector< rcsc::Vector2D > & candidates )
{
#if 0 // 新しいフォーメーションの決定法：11番から逐次的に決定
  rcsc::dlog.addText( rcsc::Logger::TEAM,
		      "%s:%d:Bhv_Offense::setFormationPos"
		      ,__FILE__, __LINE__ );

  const rcsc::WorldModel & wm = agent->world();
  const rcsc::ServerParam & sp = rcsc::ServerParam::i();
  int self_min = wm.interceptTable()->selfReachCycle();
  int mate_min = wm.interceptTable()->teammateReachCycle();
  int opp_min = wm.interceptTable()->opponentReachCycle();
  int agent_reach_cycle = std::min( std::min( self_min, mate_min ), opp_min );

  rcsc::Vector2D reach_ball_pos = wm.ball().inertiaPoint( agent_reach_cycle );
  agent->debugClient().addCircle( reach_ball_pos, 0.5 );

  double max_pass_dist = rcsc::calc_sum_geom_series( 1.2, 1.0 / sp.ballDecay(), 10);
  rcsc::Vector2D margin( max_pass_dist, 0 ); //長さ＝パス距離，角度＝０のベクトル

  std::vector< rcsc::Vector2D > oppline;
  getOppEndLine( agent, oppline );

  rcsc::Vector2D center_forward = reach_ball_pos + margin;
  if( reach_ball_pos.x > 0 )
    center_forward.x = wm.offsideLineX() - 0.5;
  // y移動範囲: -15から15まで
  if( center_forward.y < -15 )
      center_forward.y = -15;
  else if( center_forward.y > 15 )
      center_forward.y = 15;

  // right_forward
  rcsc::Vector2D right_forward( center_forward.x, center_forward.y + 15.0);
  if( right_forward.x > wm.offsideLineX() - 1.0 )
    {
      //オフサイドラインに近い場合は修正
      const rcsc::PlayerObject * nearest_opp =
	wm.getOpponentNearestTo( right_forward, 50, NULL );
      if( nearest_opp )
	{
	  if( nearest_opp->pos().x > wm.offsideLineX() )
	    {
	      right_forward.x = wm.offsideLineX() - 0.5;
	    }
	  else
	    {
	      right_forward.x = wm.offsideLineX() - 0.5;//nearest_opp->pos().x;
	      if( wm.ball().pos().y > nearest_opp->pos().y )
		right_forward.y = nearest_opp->pos().y - 1.5;
	      else
		right_forward.y = nearest_opp->pos().y + 1.5;
	    }
	  rcsc::dlog.addText( rcsc::Logger::TEAM,
			      "%s:%d: RF position set near the opponent GATE"
			      ,__FILE__, __LINE__);
	}
    }

  // left_forward
  rcsc::Vector2D left_forward( center_forward.x, center_forward.y - 15.0);

  if( left_forward.x > wm.offsideLineX() - 1.0 )
    {
      //オフサイドラインに近い場合は修正
      const rcsc::PlayerObject * nearest_opp =
	wm.getOpponentNearestTo( left_forward, 50, NULL );
      if( nearest_opp )
	{
	  if( nearest_opp->pos().x > wm.offsideLineX() )
	    {
	      left_forward.x = wm.offsideLineX() - 0.5;
	    }
	  else
	    {
	      left_forward.x = wm.offsideLineX() - 0.5;//nearest_opp->pos().x;
	      if( wm.ball().pos().y > nearest_opp->pos().y )
		left_forward.y = nearest_opp->pos().y - 1.5;
	      else
		left_forward.y = nearest_opp->pos().y + 1.5;
	    }
	  rcsc::dlog.addText( rcsc::Logger::TEAM,
			      "%s:%d: LF position set near the opponent GATE"
			      ,__FILE__, __LINE__);
	}
    }

  rcsc::Vector2D right_back( reach_ball_pos.x, reach_ball_pos.y + margin.r() / 2.0 );
  rcsc::Vector2D left_back( reach_ball_pos.x, reach_ball_pos.y - margin.r() / 2.0 );

  candidates.push_back( left_back );
  candidates.push_back( right_back );
  candidates.push_back( left_forward );
  candidates.push_back( right_forward );
  candidates.push_back( center_forward );

  for( unsigned int i = 0 ; i < candidates.size() ; i++ )
    agent->debugClient().addCircle( candidates.at( i ), 2.0 - (double)i * 0.2 );

  return max_pass_dist;
#endif

#if 1
  const rcsc::WorldModel & wm = agent->world();
  const rcsc::ServerParam & sp = rcsc::ServerParam::i();
  int self_min = wm.interceptTable()->selfReachCycle();
  int mate_min = wm.interceptTable()->teammateReachCycle();
  int opp_min = wm.interceptTable()->opponentReachCycle();
  int agent_reach_cycle = std::min( std::min( self_min, mate_min ), opp_min );

  rcsc::Vector2D inertial_ball_pos = wm.ball().inertiaPoint( agent_reach_cycle );
  double max_pass_dist = rcsc::calc_sum_geom_series( 1.2, 1.0 / sp.ballDecay(), 10);

  rcsc::Vector2D rotation( max_pass_dist, 0 ); //長さ＝パス距離，角度＝０のベクトル
  rcsc::Vector2D ball_to_target = M_target_point - inertial_ball_pos;



  rcsc::Vector2D base_pos = inertial_ball_pos;

  rcsc::Vector2D left_back = base_pos + rotation.rotatedVector( -90 ) * 0.35;
  rcsc::Vector2D right_back = base_pos + rotation.rotatedVector( 90 ) * 0.35;

  rotation.rotate( ball_to_target.th() ); //長さ＝パス距離，角度＝最終目標位置の方向 のベクトル

  rcsc::Vector2D left_forward = base_pos + rotation.rotatedVector( -45 );
  rcsc::Vector2D right_forward = base_pos + rotation.rotatedVector( 45 );
  rcsc::Vector2D center_forward = base_pos + rotation;
  left_forward.x = center_forward.x + 2.0;
  right_forward.x = center_forward.x + 2.0;


  candidates.push_back( left_back );
  candidates.push_back( right_back );
  candidates.push_back( left_forward );
  candidates.push_back( right_forward );
  candidates.push_back( center_forward );

  double minimum_dist = 100;
  rcsc::Vector2D shift( 0, 0 );
  int passer_unum = 0;
  for( int i = 7; i <= 11; i++ )
    {
      if( wm.teammate( i ) )
        {
	  double dist = inertial_ball_pos.dist( wm.teammate( i )->pos() );
	  if( dist < minimum_dist )
            {
	      minimum_dist = dist;
	      passer_unum = i;
            }
        }
    }
  switch( passer_unum )
    {
    case 7:
      shift = inertial_ball_pos - candidates.at( 0 );
      break;
    case 8:
      shift = inertial_ball_pos - candidates.at( 1 );
      break;
    case 9:
      shift = inertial_ball_pos - candidates.at( 2 );
      break;
    case 10:
      shift = inertial_ball_pos - candidates.at( 3 );
      break;
    case 11:
      shift = inertial_ball_pos - candidates.at( 4 );
      break;
    default:
      break;
    }
  for( int i = 0; i < (int)candidates.size(); i++ )
    {
      candidates.at( i ) += shift;
    }
    
  agent->debugClient().addCircle( inertial_ball_pos, 0.5 );
  agent->debugClient().addLine( inertial_ball_pos, M_target_point );

  double first_speed = rcsc::calc_first_term_geom_series_last( 1.2, max_pass_dist, sp.ballDecay() );

  std::vector< rcsc::Vector2D > oppline;
  getOppEndLine( agent, oppline );

  int cnddts_size = (int)candidates.size();
  bool flag_xmodify = false;
  for( int i = cnddts_size - 1 ; i >= 0; i-- )
    {
      if( i >= 2 && inertial_ball_pos.x > 0 )
        {
	  candidates.at( i ).x = wm.offsideLineX() - 0.5;
	  flag_xmodify = true;
        }

      if( i < 2 && flag_xmodify )
	{
	  candidates.at( i ).x = wm.offsideLineX() - 19.0;
	}

      if( candidates.at( i ).x > wm.offsideLineX() - 1.0 )
	{
	  //オフサイドラインに近い場合は修正
	  const rcsc::PlayerObject * nearest_opp =
	    wm.getOpponentNearestTo( candidates.at( i ), 50, NULL );
	  if( nearest_opp )
	    {
	      if( nearest_opp->pos().x > wm.offsideLineX() )
		{
		  candidates.at( i ).x = wm.offsideLineX() - 0.5;
		}
	      else
		{
		  candidates.at( i ).x = wm.offsideLineX() - 0.5;//nearest_opp->pos().x;
		  if( wm.ball().pos().y > nearest_opp->pos().y )
		    candidates.at( i ).y = nearest_opp->pos().y - 1.5;
		  else
		    candidates.at( i ).y = nearest_opp->pos().y + 1.5;
		  if( wm.self().unum() == 11 - 4 + i )
		    agent->debugClient().addMessage( "NearOpp" );
		}
	    }
	  else
	    {
	      candidates.at( i ).x = wm.offsideLineX() - 0.5;
	    }
	}

      //味方とかぶらないように．中央優先
      if( i != cnddts_size - 1 
	  && fabs( candidates.at( i ).y
		   - candidates.at( cnddts_size - 1 ).y ) < 5.0 )
	{
	  if( i == cnddts_size - 2 )
	    candidates.at( i ).y = candidates.at( cnddts_size - 1 ).y + 15.0;
	  else if( i == cnddts_size - 3 )
	    candidates.at( i ).y = candidates.at( cnddts_size - 1 ).y - 15.0;
	}

      //フィールドライン近くの修正
      if( candidates.at( i ).x > 50 )
	{
	  candidates.at( i ).x = 50;
	}
      if( candidates.at( i ).x < -50 )
	{
	  candidates.at( i ).x = -50;
	}
      if( candidates.at( i ).y > 31 )
	{
	  candidates.at( i ).y = 31;
	}
      if( candidates.at( i ).y < -31 )
	{
	  candidates.at( i ).y = -31;
	}

      agent->debugClient().addCircle( candidates.at( i ), 2 );
    }

  return max_pass_dist;
#endif
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

bool
Bhv_Offense::holdBall( rcsc::PlayerAgent * agent )
{
  holdBall( agent, M_target_point );
  return true;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

bool
Bhv_Offense::holdBall( rcsc::PlayerAgent * agent, rcsc::Vector2D target_pos )
{
  const rcsc::WorldModel & wm = agent->world();
  rcsc::Vector2D next_ball_pos = wm.ball().pos() + wm.ball().vel();
  rcsc::Vector2D next_self_pos = wm.self().pos() + wm.self().vel();
    
  rcsc::Circle2D next_self_area( next_self_pos, wm.self().kickableArea() * 0.9 );

  rcsc::Vector2D turn_pos = target_pos - next_self_pos;
  rcsc::AngleDeg angle_dif = turn_pos.th() - wm.self().body();
  if( next_self_area.contains( next_ball_pos ) && angle_dif.abs() > 10 )
  {
      //turn
      agent->doTurn( angle_dif );
      agent->debugClient().addMessage( "HoldTurn" );
  }
  else
  {
      //kick
      rcsc::Vector2D two_step_self_pos = wm.self().inertiaPoint( 2 );
      rcsc::Vector2D tmp_target = target_pos - two_step_self_pos;
      tmp_target.setLength( wm.self().kickableArea() * 0.5 );
      rcsc::Vector2D hold_pos = two_step_self_pos + tmp_target;
      
      // calculate the next velocity of the ball so that
      // the ball reaches hold_pos in two cycles
      // p + a + a*decay = hold_pos
      rcsc::Vector2D ball_vel = (hold_pos - wm.ball().pos()) / (1 + rcsc::ServerParam::i().ballDecay() );
      
      // calculate kick vector to be applied to the ball
      rcsc::Vector2D accel = ball_vel - wm.ball().vel();
	
      // calculate kick power
      double kick_power = accel.r() / wm.self().kickRate();
      agent->doKick( kick_power, accel.th() - wm.self().body() );
      agent->debugClient().addMessage( "HoldKick" );
      agent->debugClient().setTarget( hold_pos );
  }
  
  
    
  return true;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

bool
Bhv_Offense::doKick( rcsc::PlayerAgent * agent )
{

    const rcsc::WorldModel & wm = agent->world();
    rcsc::Vector2D next_self_pos = wm.self().pos() + wm.self().vel();
    rcsc::Vector2D next_bpos = wm.ball().inertiaPoint( 1 );
    rcsc::Vector2D tmp_target = M_target_point - next_self_pos;
    rcsc::AngleDeg angle_dif = tmp_target.th() - wm.self().body();

    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Bhv_Offense::doKick"
                        ,__FILE__, __LINE__ );

    std::vector< rcsc::Vector2D > candidates;
    setFormationPos( agent, candidates );
    std::vector< rcsc::Vector2D > oppline;
    getOppEndLine( agent, oppline );

    bool frontier = true;
    // check if I'm in the frontier
    const rcsc::PlayerPtrCont::const_iterator end_self = wm.teammatesFromSelf().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
         it != end_self;
         ++it )
    {
        if( (*it)->pos().x > wm.self().pos().x )
	{
            frontier = false;
            break;
	}
    }

    //前線である場合
    if( frontier || wm.self().pos().x > (wm.offsideLineX() - 8))
    {
        agent->debugClient().addMessage( "ImFront" );
        if( wm.self().body().abs() > 90 )
	{
            agent->debugClient().addMessage( "HoldForward" );
            holdBall( agent );
            agent->addSayMessage( new rcsc::BallPlayerMessage( agent->effector().queuedNextBallPos(), 
                                                               agent->effector().queuedNextBallVel(),
                                                               wm.self().unum(),
                                                               agent->effector().queuedNextMyPos(),
                                                               agent->effector().queuedNextMyBody() ) );
            return true;
	}

        // 前にドリブル
        // detect two nearest opponents in the defense line
        if( oppline.size() == 1 && fabs(next_bpos.y) < 20.0 ) //およそペナルティエリア
	{
            for( int dash_cycle = 7; dash_cycle >= 3 ; dash_cycle -= 2 )
	    {
                for( int i = 0; i <= 45; i += 2 )
		{
                    double degree = (double)i;
                    rcsc::AngleDeg right_angle( wm.self().body().degree() + degree );
                    rcsc::AngleDeg left_angle( wm.self().body().degree() - degree );
		  
                    if( doSelfPass( agent, right_angle, dash_cycle ) )
		    {
                        agent->debugClient().addMessage( "SelfPass%d", dash_cycle );
                        return true;
		    }
                    if( right_angle != left_angle )
		    {
                        if( doSelfPass( agent, left_angle, dash_cycle ) )
			{
                            agent->debugClient().addMessage( "SelfPass%d", dash_cycle );
                            return true;
			}
		    }
		}
	    }
	}
        else
	{
            // detect two nearest opponents in the defense line
            double nearest_disy = 1000.0;
            double nearest_disy2 = 1500.0;
            rcsc::Vector2D nearestOppInLine( 0.0, 0.0 );
            rcsc::Vector2D nearestOppInLine2( 0.0, 0.0 );
            for( std::vector< rcsc::Vector2D >::iterator it = oppline.begin();
                 it < oppline.end() ; it++ )
	    {
                if( fabs( (*it).y - wm.self().pos().y ) < nearest_disy )
		{
                    nearest_disy2 = nearest_disy;
                    nearest_disy = fabs( (*it).y - wm.self().pos().y );
                    nearestOppInLine2 = nearestOppInLine;
                    nearestOppInLine = (*it);
		}
                else if( fabs( (*it).y - wm.self().pos().y ) < nearest_disy2 )
		{
                    nearest_disy2 = fabs( (*it).y = wm.self().pos().y );
                    nearestOppInLine2 = (*it);
		}
	    }
	  
            if( (wm.self().pos().y - nearestOppInLine.y)
                * (wm.self().pos().y - nearestOppInLine2.y) < 0 )
	    {
                // I'm between the two nearest opponents in the frontier
                rcsc::Vector2D between = nearestOppInLine + nearestOppInLine2;
                for( int dash_cycle = 7; dash_cycle >= 5 ; dash_cycle -= 2 )
		{
                    rcsc::AngleDeg dash_angle( between.th().degree() );
		  
                    if( doSelfPass( agent, dash_angle, dash_cycle ) )
		    {
                        agent->debugClient().addMessage( "SelfPass%d", dash_cycle );
                        return true;
		    }
		}
	    }
            else if( wm.self().pos().y < nearestOppInLine.y )
	    {
                // I'm in the top side of the field

                // try to penetrate
                for( int dash_cycle = 7; dash_cycle >= 5 ; dash_cycle -= 2 )
		{
                    rcsc::AngleDeg dash_angle( 0.0 );
                    if( doSelfPass( agent, dash_angle, dash_cycle ) )
		    {
                        agent->debugClient().addMessage( "SelfPass%d", dash_cycle );
                        return true;
		    }
		}
	      
                // go to the edge
                rcsc::Vector2D dash_target( nearestOppInLine.x, -31.5 );
                for( int dash_cycle = 5; dash_cycle >= 3 ; dash_cycle -= 1 )
		{
                    rcsc::AngleDeg dash_angle( (dash_target - wm.self().pos() ).th().degree() );
		  
                    if( doSelfPass( agent, dash_angle, dash_cycle ) )
		    {
                        agent->debugClient().addMessage( "SelfPass%d", dash_cycle );
                        return true;
		    }
		}
	    }
            else if( wm.self().pos().y > nearestOppInLine.y )
	    {
                // I'm in the bottom side of the field

                // try to penetrate
                for( int dash_cycle = 7; dash_cycle >= 5 ; dash_cycle -= 2 )
		{
                    rcsc::AngleDeg dash_angle( 0.0 );
                    if( doSelfPass( agent, dash_angle, dash_cycle ) )
		    {
                        agent->debugClient().addMessage( "SelfPass%d", dash_cycle );
                        return true;
		    }
		}
	      
                // go to the edge
                rcsc::Vector2D dash_target( nearestOppInLine.x, 31.5 );
                for( int dash_cycle = 5; dash_cycle >= 3 ; dash_cycle -= 1 )
		{
                    rcsc::AngleDeg dash_angle( (dash_target - wm.self().pos() ).th().degree() );
		  
                    if( doSelfPass( agent, dash_angle, dash_cycle ) )
		    {
                        agent->debugClient().addMessage( "SelfPass%d", dash_cycle );
                        return true;
		    }
		}
	    }
	}
    } // end if frontier

    if( angle_dif.abs() > 90 )
    {
        agent->debugClient().addMessage( "HoldForward" );
        holdBall( agent );
        agent->addSayMessage( new rcsc::BallPlayerMessage( agent->effector().queuedNextBallPos(), 
                                                           agent->effector().queuedNextBallVel(),
                                                           wm.self().unum(),
                                                           agent->effector().queuedNextMyPos(),
                                                           agent->effector().queuedNextMyBody() ) );
        return true;
    }

    if( next_bpos.dist( next_self_pos ) >= wm.self().kickableArea() )
    {
        //ターンしている間にボールが逃げていきそうならばトラップ
        agent->debugClient().addMessage( "Trap" );
        if( wm.ball().pos().x < wm.self().pos().x )
	{
            rcsc::Vector2D hold_pos = next_self_pos;
            if( wm.ball().pos().y < hold_pos.y )
	    {
                hold_pos.y -= wm.self().kickableArea() * 0.5;
	    }
            else
	    {
                hold_pos.y += wm.self().kickableArea() * 0.5;
	    }
            holdBall( agent, hold_pos );
	}
        else
	{
            holdBall( agent );
	}
      
        agent->addSayMessage( new rcsc::BallPlayerMessage( agent->effector().queuedNextBallPos(), 
                                                           agent->effector().queuedNextBallVel(),
                                                           wm.self().unum(),
                                                           agent->effector().queuedNextMyPos(),
                                                           agent->effector().queuedNextMyBody() ) );
        return true;        
    }

    rcsc::Vector2D pass_target = next_self_pos;
    int mate_unum = -1;

    if( wm.ball().pos().x > 0.0 )
    {
        const rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromBall().end();
        for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromBall().begin();
             it != end;
             ++it )
	{    
            if( (*it)->unum() < 7 )
	    {
                continue;
	    }
            const rcsc::PlayerObject * receiver = (*it);//wm.getTeammateNearestTo( candidates.at( i ), 5, NULL  );
            if( receiver )
	    {
                if( receiver->posCount() > 1 )
		{
                    continue;
		}
                if( fabs( receiver->pos().y - wm.self().pos().y ) > 5 )
		{
                    double final_ball_speed = 1.2;
                    if( checkThroughPass( agent, receiver, pass_target, final_ball_speed ) )
		    {
                        agent->debugClient().addMessage( "ThroughPassTo%d", receiver->unum() );
                        Opuci_OneStepKick( pass_target, final_ball_speed ).execute( agent );
                        agent->addSayMessage( new rcsc::PassMessage( receiver->unum(), pass_target,
                                                                     agent->effector().queuedNextBallPos(),
                                                                     agent->effector().queuedNextBallVel() ) );
                        return true;
		    }
                    if( checkThroughPass2( agent, receiver, pass_target, final_ball_speed ) )
		    {
                        agent->debugClient().addMessage( "ThroughPass2To%d", receiver->unum() );
                        Opuci_OneStepKick( pass_target, final_ball_speed ).execute( agent );
                        agent->addSayMessage( new rcsc::PassMessage( receiver->unum(), pass_target,
                                                                     agent->effector().queuedNextBallPos(),
                                                                     agent->effector().queuedNextBallVel() ) );
                        return true;
		    }
		}
	    }
	}
    }
    
    if( testPass( agent, pass_target, mate_unum, false ) )
    {
        agent->debugClient().addMessage( "PassTo%d", mate_unum );
        Opuci_OneStepKick( pass_target, 1.2 ).execute( agent );
        agent->addSayMessage( new rcsc::PassMessage( mate_unum, pass_target,
                                                     agent->effector().queuedNextBallPos(),
                                                     agent->effector().queuedNextBallVel() ) );
        return true;
    }        
    
    rcsc::Vector2D self_to_target = M_target_point - wm.self().pos();
    rcsc::AngleDeg base_angle = self_to_target.th();
    for( int dash_cycle = 3; dash_cycle >= 1 ; dash_cycle-- )
    {
        for( int i = 0; i <= 50; i += 1 )
	{
            double degree = (double)i;
            rcsc::AngleDeg right_angle( base_angle.degree() + degree );
            rcsc::AngleDeg left_angle( base_angle.degree() - degree );
	    
	    
            if( doSelfPass( agent, right_angle, dash_cycle ) )
	    {
                agent->debugClient().addMessage( "SelfPass%d", dash_cycle );
                return true;
		
	    }
            if( right_angle != left_angle )
	    {
                if( doSelfPass( agent, left_angle, dash_cycle ) )
		{
                    agent->debugClient().addMessage( "SelfPass%d", dash_cycle );
                    return true;
		    
		}
	    }
	}
    }

    if( testPass( agent, pass_target, mate_unum, true ) )
    {
        agent->debugClient().addMessage( "PassTo%d", mate_unum );
        Opuci_OneStepKick( pass_target, 1.2 ).execute( agent );
        agent->addSayMessage( new rcsc::PassMessage( mate_unum, pass_target,
                                                     agent->effector().queuedNextBallPos(),
                                                     agent->effector().queuedNextBallVel() ) );
        return true;
    } 

    for( int dash_cycle = 3; dash_cycle >= 1 ; dash_cycle-- )
    {
        for( int i = 51; i <= 90; i += 1 )
	{
            double degree = (double)i;
            rcsc::AngleDeg right_angle( base_angle.degree() + degree );
            rcsc::AngleDeg left_angle( base_angle.degree() - degree );
	    
	    
            if( doSelfPass( agent, right_angle, dash_cycle ) )
	    {
                agent->debugClient().addMessage( "SelfPass%d", dash_cycle );
                return true;
		
	    }
            if( right_angle != left_angle )
	    {
                if( doSelfPass( agent, left_angle, dash_cycle ) )
		{
                    agent->debugClient().addMessage( "SelfPass%d", dash_cycle );
                    return true;
		    
		}
	    }
	}
    }

    // ホールド
    rcsc::Vector2D look = candidates.at( wm.time().cycle() % 5 );
    agent->debugClient().addMessage( "HoldPoint" );
    holdBall( agent, look );
    agent->addSayMessage( new rcsc::BallPlayerMessage( agent->effector().queuedNextBallPos(), 
                                                       agent->effector().queuedNextBallVel(),
                                                       wm.self().unum(),
                                                       agent->effector().queuedNextMyPos(),
                                                       agent->effector().queuedNextMyBody() ) );
    return true;

}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

bool 
Bhv_Offense::testPass( rcsc::PlayerAgent * agent, rcsc::Vector2D & target_pos, int & mate_unum, bool allow_back )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Bhv_Offense::testPass"
                        ,__FILE__, __LINE__ );

    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    std::vector< rcsc::Vector2D > candidates;
    setFormationPos( agent, candidates );
    
    int max_opp = 0;
    int unum = 0;
    rcsc::Vector2D max_target = wm.self().pos();

    const rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromBall().begin();
         it != end;
         ++it )
    {
        if( (*it)->unum() < 7 || (*it)->pos().x > wm.offsideLineX() )
	{
            continue;
	}

        rcsc::Vector2D tmp_target( 52, 0 );
        switch( (*it)->unum() )
	{
	case 7:
            tmp_target = candidates.at( 0 );
            break;
	case 8:
            tmp_target = candidates.at( 1 );
            break;
	case 9:
            tmp_target = candidates.at( 2 );
            break;
	case 10:
            tmp_target = candidates.at( 3 );
            break;
	case 11:
            tmp_target = candidates.at( 4 );
            break;
	}

        int opp_cycle = 0;
        if( allow_back
            || tmp_target.x > wm.self().pos().x )
	{
            if( checkPassSafe( agent, tmp_target, *it, opp_cycle ) )
	    {
                if( opp_cycle > max_opp )
		{
                    max_opp = opp_cycle;
                    max_target = tmp_target;
                    unum = (*it)->unum();
		}
	    }
	}

        rcsc::Vector2D middle_target = ( tmp_target + (*it)->pos() ) / 2.0;

        if( (allow_back
             || middle_target.x > wm.self().pos().x)
            && middle_target.x > (*it)->pos().x )
	{
            if( checkPassSafe( agent, middle_target, *it, opp_cycle ) )
	    {
                if( opp_cycle > max_opp )
		{
                    max_opp = opp_cycle;
                    max_target = middle_target;
                    unum = (*it)->unum();
		}
	    }
	}

        rcsc::Vector2D leader( 2.5, 0 );
        rcsc::Vector2D lead_target = (*it)->pos() + leader;

        if( allow_back
            || lead_target.x > wm.self().pos().x )
	{
            if( checkPassSafe( agent, lead_target, *it, opp_cycle ) )
	    {
                if( opp_cycle > max_opp )
		{
                    max_opp = opp_cycle;
                    max_target = lead_target;
                    unum = (*it)->unum();
		}
	    }
	}

        if( allow_back
            || (*it)->pos().x > wm.self().pos().x )
	{
            if( checkPassSafe( agent, (*it)->pos(), *it, opp_cycle ) )
	    {
                if( opp_cycle + 2 > max_opp )
		{
                    max_opp = opp_cycle + 2;
                    max_target = (*it)->pos();
                    unum = (*it)->unum();
		}
	    }
	}
        
    }

    if( max_opp != 0 )
    {
        target_pos = max_target;
        mate_unum = unum;
        return true;
    }
    
    return false;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

bool 
Bhv_Offense::checkPassSafe( rcsc::PlayerAgent * agent, rcsc::Vector2D target_pos, rcsc::PlayerObject * receiver, int & opp_cycle )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Bhv_Offense::checkPassSafe"
                        ,__FILE__, __LINE__ );
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Considered receive point: (%f, %f)"
                        ,__FILE__, __LINE__, target_pos.x, target_pos.y );

    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();
    int turn_cycle = 1;
    if( receiver->pos().x > target_pos.x )
    {
        turn_cycle++;
    }
    
    opp_cycle = 0;
    double dist = ( target_pos - receiver->pos() ).r();
    dist -= receiver->playerTypePtr()->kickableMargin();
    int dash_cycle = receiver->playerTypePtr()->cyclesToReachDistance( dist );
    
    int reach_cycle = turn_cycle + dash_cycle;

    if( reach_cycle >= 10 )
    {
        return false;
    }
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: Mate%d reach cycle: %d"
                        ,__FILE__, __LINE__, receiver->unum(), reach_cycle );
    //agent->debugClient().addMessage( "M%d:%d", receiver->unum(), reach_cycle );
    double tmp_dist = wm.ball().pos().dist( target_pos );
    double first_speed = rcsc::calc_first_term_geom_series_last( 1.2, tmp_dist, sp.ballDecay() );
    int max_ball_cycle = rcsc::calc_length_geom_series( first_speed, tmp_dist, sp.ballDecay() );
    if( max_ball_cycle <= 0 )
    {
        max_ball_cycle = 1;
    }
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Ball reach cycle: %d"
                        ,__FILE__, __LINE__, max_ball_cycle );
    if( max_ball_cycle < reach_cycle )
    {
        //agent->debugClient().addMessage( "b%d", max_ball_cycle );
        return false;
    }
    rcsc::Vector2D rvec = target_pos - wm.ball().pos();
    rcsc::Vector2D max_accel( wm.self().kickRate() * 100, 0 );
    max_accel.rotate( rvec.th() );
    rcsc::Vector2D acceled_ball = wm.ball().vel() + max_accel;
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:acceled_ball.r() = %f, first_speed * 0.9 = %f"
                        ,__FILE__, __LINE__, acceled_ball.r(), first_speed*0.9 );
    if( acceled_ball.r() < first_speed * 0.9 )
    {
        //agent->debugClient().addMessage( "BadAccel" );
        return false;
    }
    
    int opp_min = 50;
    const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
         it != end;
         ++it )
    {    
        rcsc::Segment2D ball_to_target( wm.ball().pos(), target_pos );
        rcsc::Vector2D next_opp_pos = (*it)->pos() + (*it)->vel();

        rcsc::Circle2D opp_area( next_opp_pos, (*it)->playerTypePtr()->kickableMargin() * 1.2 );
        if( opp_area.intersection( ball_to_target, NULL, NULL ) > 0 )
	{
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d:next_opp_pos(%f, %f) intersects with b2target line."
                                ,__FILE__, __LINE__, next_opp_pos.x, next_opp_pos.y );
            return false;
	}

        double opp_dist = ball_to_target.dist( next_opp_pos );
        opp_dist -= (*it)->playerTypePtr()->kickableMargin() * 1.2;
        int opp_dash = (*it)->playerTypePtr()->cyclesToReachDistance( opp_dist );
        int opp_turn = 1;
        if( std::fabs( next_opp_pos.x - target_pos.x ) < 2.5
            || std::fabs( next_opp_pos.y - target_pos.y ) < 2.5 )
	{
            opp_turn = 0;
	}

        int tmp_opp_cycle = opp_turn + opp_dash;
        
        rcsc::Vector2D opp_target =  ball_to_target.nearestPoint( next_opp_pos );
        double ball_dist = wm.ball().pos().dist( opp_target );
        
        int ball_cycle = rcsc::calc_length_geom_series( first_speed, ball_dist, sp.ballDecay() );
        if( ball_cycle <= 0 )
	{
            ball_cycle = 1;
	}
        if( tmp_opp_cycle <= ball_cycle )
	{
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d:ball_cycle %d is longer than tmp_opp_cycle %d"
                                ,__FILE__, __LINE__, ball_cycle, tmp_opp_cycle );
            return false;
	}

        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d:Opponent%d reach cycle: %d"
                            ,__FILE__, __LINE__, (*it)->unum(), tmp_opp_cycle );
        //        agent->debugClient().addMessage( "O%d,%d>%d", (*it)->unum(), tmp_opp_cycle, ball_cycle );

        /*
        if( max_ball_cycle >= tmp_opp_cycle )
	{
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d:max_ball_cycle(%d) >= tmp_opp_cycle(%d)"
                                ,__FILE__, __LINE__, max_ball_cycle, tmp_opp_cycle );
            return false;
	}
        */

        if( tmp_opp_cycle < opp_min )
	{
            opp_min = tmp_opp_cycle;
	}
    }
    if( opp_min < 50 )
    {
        opp_cycle = opp_min;
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d:Pass OK. opp_cycle = %d"
                            ,__FILE__, __LINE__, opp_cycle );
        return true;
    }

    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: No clear reason but not sure about successful pass"
                        ,__FILE__, __LINE__, opp_cycle );
    return false;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

bool
Bhv_Offense::checkDribbleSafe( rcsc::PlayerAgent * agent, rcsc::Vector2D target_pos, int dash_cycle )
{
  const rcsc::WorldModel & wm = agent->world();
  const rcsc::ServerParam & sp = rcsc::ServerParam::i();
  int kick_cycle = 1;
  int turn_cycle = 1;
  int dribble_cycle = kick_cycle + turn_cycle + dash_cycle;
   
  rcsc::Vector2D inertial_self_pos = wm.self().inertiaPoint( dribble_cycle );

  rcsc::Vector2D rpos = target_pos - inertial_self_pos;
    
  rcsc::Vector2D move_pos = inertial_self_pos;
  double self_speed = 0;
  double self_dist = 0;
  for( int j = 0; j < dash_cycle; j++ )
    {
      self_speed += wm.self().dashRate() * 10;
      if( self_speed >= wm.self().playerType().playerSpeedMax() )
	self_speed = wm.self().playerType().playerSpeedMax();
      self_dist += self_speed;
      self_speed *= wm.self().playerTypePtr()->playerDecay();
    }    
  rcsc::Vector2D dash_vector( self_dist, 0 );
  dash_vector.rotate( rpos.th() );
  move_pos += dash_vector;
  const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
  for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
       it != end;
       ++it )
    {
      rcsc::Vector2D inertial_opp_pos = (*it)->playerTypePtr()->inertiaPoint( (*it)->pos(),
									      (*it)->vel(),
									      dribble_cycle );
      double dist = inertial_opp_pos.dist( move_pos );
      dist -= sp.tackleDist();//wm.self().kickableArea();
      dist -= (*it)->playerTypePtr()->kickableMargin();
      if( dist <= 0 )
	{
	  return false;
	}
      if( (*it)->playerTypePtr()->cyclesToReachDistance( dist ) < dribble_cycle )
	{
	  return false;
	}
    }
  return true;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

bool
Bhv_Offense::doSelfPass( rcsc::PlayerAgent * agent, rcsc::AngleDeg angle, int dash_cycle )
{
  rcsc::dlog.addText( rcsc::Logger::TEAM,
		      "%s:%d:Bhv_Offense::doSelfPass. angle = %f"
		      ,__FILE__, __LINE__, angle.degree() );

  const rcsc::WorldModel & wm = agent->world();
  const rcsc::ServerParam & sp = rcsc::ServerParam::i();

  int turn_cycle = 1;
  int kick_cycle = 1;
    
  double degree = angle.degree();
  double moment = degree - wm.self().body().degree();
  if( fabs( moment ) < 10 )
    {
      turn_cycle = 0;
    }
  if( wm.self().canTurn( moment ) )
    {
      int total_cycle = kick_cycle + turn_cycle + dash_cycle;
      rcsc::Vector2D moved_pos = wm.self().pos();
      rcsc::Vector2D inertial_move = wm.self().vel();
      
      double stamina = wm.self().stamina();
      double effort = wm.self().effort();
      double recovery = wm.self().recovery();
      double power = 100;
      
      for( int i = 0; i < (kick_cycle + turn_cycle); i++ )
	{
	  moved_pos += inertial_move;
	  inertial_move *= wm.self().playerType().playerDecay();
	  stamina += recovery * wm.self().playerType().staminaIncMax();
	  if( stamina > sp.staminaMax() )
	    stamina = sp.staminaMax();
	}
      
      rcsc::Vector2D dash_move( 1, 0 );
      dash_move.rotate( degree );
      for( int i = 0; i < dash_cycle; i++ )
	{
	  if( i == 0 )
	    {
	      dash_move.setLength( effort * wm.self().playerType().dashPowerRate() * power );
	    }
	  else
	    {
	      dash_move.setLength( dash_move.r() + effort * wm.self().playerType().dashPowerRate() * power );
	    }
	  if( dash_move.r() > wm.self().playerType().playerSpeedMax() )
	    dash_move.setLength( wm.self().playerType().playerSpeedMax() );
	  
	  rcsc::Vector2D tmp_move = dash_move + inertial_move;
	  if( tmp_move.r() > wm.self().playerType().playerSpeedMax() )
	    tmp_move.setLength( wm.self().playerType().playerSpeedMax() );
	  
	  moved_pos += tmp_move;
          
	  stamina -= power;
	  if( stamina <= sp.recoverDecThr() * sp.staminaMax() )
	    {
	      if( recovery > sp.recoverMin() )
		recovery -= sp.recoverDec();
	      if( recovery < sp.recoverMin() )
		recovery = sp.recoverMin();
	    }
	  if( stamina <= sp.effortDecThr() * sp.staminaMax() )
	    {
	      if( effort > wm.self().playerType().effortMin() )
		effort -= sp.recoverDec();
	      if( effort < wm.self().playerType().effortMin() )
		effort = wm.self().playerType().effortMin();
	    }
	  if( stamina >= sp.effortIncThr() * sp.staminaMax() )
	    {
	      if( effort < wm.self().playerType().effortMax() )
		{
		  effort += sp.effortInc();
		  if( effort > wm.self().playerType().effortMax() )
		    effort = wm.self().playerType().effortMax();
		}
	    }
	  stamina += recovery * wm.self().playerType().staminaIncMax();
	  if( stamina > sp.staminaMax() )
	    stamina = sp.staminaMax();
	  
	  dash_move *= wm.self().playerType().playerDecay();
	  inertial_move *= wm.self().playerType().playerDecay();
	}
      
      rcsc::Vector2D ball_margin( wm.self().kickableArea() * 0.5, 0 );
      ball_margin.rotate( degree );
      
      rcsc::Vector2D ball_target = moved_pos + ball_margin;
      
      if( ball_target.absX() > 51
	  || ball_target.absY() > 33 )
	{
	  agent->debugClient().addMessage( "TargetOB" );
	  return false;
	}
      
      
      
      double ball_dist = wm.ball().pos().dist( ball_target );
      double ball_speed = rcsc::calc_first_term_geom_series( ball_dist, sp.ballDecay(), total_cycle );
      rcsc::Vector2D ball_vector( ball_speed, 0 );
      rcsc::Vector2D rpos = ball_target - wm.ball().pos();
      ball_vector.rotate( rpos.th() );
      rcsc::Vector2D accel = ball_vector - wm.ball().vel();
      double kick_power = accel.r() / wm.self().kickRate();
      
      if( kick_power > sp.maxPower() )
	{
	  rcsc::dlog.addText( rcsc::Logger::TEAM,
			      "%s:%d: Not enough power. Stop selfpass."
			      ,__FILE__, __LINE__ );
	  return false;
	}
      
      rcsc::Vector2D next_ball_pos = wm.ball().pos() + ball_vector;
      rcsc::Circle2D next_ball_area( next_ball_pos, sp.ballSize() );
      
      rcsc::Vector2D next_self_pos = wm.self().pos() + wm.self().vel();
      rcsc::Circle2D next_self_area( next_self_pos, wm.self().playerTypePtr()->playerSize() + 0.1 );
      
      if( next_self_area.intersection( next_ball_area, NULL, NULL ) > 0 
	  || next_self_area.contains( next_ball_pos ) )
	{
	  rcsc::dlog.addText( rcsc::Logger::TEAM,
			      "%s:%d: Ball will collide. Stop selfpass."
			      ,__FILE__, __LINE__ );
	  return false;
	}
      
      const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
      for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
	   it != end;
	   ++it )
	{
	  if( (*it)->isTackling() )
	    continue;
	  
	  const rcsc::Vector2D selfToIt = (*it)->pos() - wm.self().pos();
	  const rcsc::AngleDeg difAD( selfToIt.th().degree() - angle.degree() );
	  if( fabs( difAD.tan() ) > 6 )
	    continue;  // たぶん抜ける
	  
	  rcsc::Vector2D inertial_opp_pos = (*it)->playerTypePtr()->inertiaPoint( (*it)->pos(),
										  (*it)->vel(),
										  total_cycle );
	  double opp_dist = inertial_opp_pos.dist( ball_target );
	  opp_dist -= sp.tackleDist();//(*it)->playerTypePtr()->kickableMargin() * 1.5;
	  if( opp_dist <= 0 )
	    {
	      rcsc::dlog.addText( rcsc::Logger::TEAM,
				  "%s:%d: Ball will be tackled. Stop selfpass."
				  ,__FILE__, __LINE__ );
	      return false;
	    }
	  if( (*it)->playerTypePtr()->cyclesToReachDistance( opp_dist ) <= total_cycle )
	    {
	      rcsc::dlog.addText( rcsc::Logger::TEAM,
				  "%s:%d: Ball will be stolen by %d. Stop selfpass."
				  ,__FILE__, __LINE__, (*it)->unum() );
	      return false;
	    }
	}
      
      agent->doKick( kick_power, accel.th() - wm.self().body() );
      agent->debugClient().addCircle( ball_target, 0.1 );
      agent->debugClient().addCircle( moved_pos, 1.0 );
      
      agent->setIntention( new Opuci_IntentionDribble( ball_target, dash_cycle ) );
      
      
      return true;
      
    }//end if canturn

  return false;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

bool 
Bhv_Offense::checkThroughPass( rcsc::PlayerAgent * agent, const rcsc::PlayerObject * receiver, rcsc::Vector2D & target_pos, double & final_ball_speed )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Bhv_Offense::checkThroughPass"
                        ,__FILE__, __LINE__ );
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Receiver(%f,%f)"
                        ,__FILE__, __LINE__, receiver->pos().x, receiver->pos().y );

    if( receiver->body().abs() > 60 )
    {
        return false;
    }


    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    int inertial_cycle = 1;

    rcsc::Vector2D inertial_self_pos = wm.self().inertiaPoint( inertial_cycle );
    rcsc::Vector2D inertial_mate_pos = receiver->playerTypePtr()->inertiaPoint( receiver->pos(), receiver->vel(), inertial_cycle );
    

    std::vector< rcsc::PlayerObject* > opponents;
    const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
         it != end;
         ++it )
    {    
        if( (*it)->pos().x > wm.self().pos().x )
	{
            opponents.push_back( (*it) );
	}
    }


    if( opponents.empty() )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d:No opponents ahead of me. stop checking."
                            ,__FILE__, __LINE__ );
        return false;
    }

    for( std::vector< rcsc::PlayerObject* >::iterator it1 = opponents.begin();
         it1 != opponents.end();
         ++it1 )
    {
        for( std::vector< rcsc::PlayerObject* >::iterator it2 = it1;
             it2 != opponents.end();
             ++it2 )
	{
            if( (*it2)->pos().y < (*it1)->pos().y )
	    {
                swap_ranges( it1, it1 + 1, it2 );
	    }
	}
    }
    std::vector< rcsc::Vector2D > middle_pos;
    if( opponents.size() == 1 )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d:Only one opponent found at (%f, %f). stop checking."
                            ,__FILE__, __LINE__, opponents.at(0)->pos().x, opponents.at(0)->pos().y );
        return false;
    }
    else
    {
        for( int i = 0; i < (int)opponents.size() - 1; i++ )
	{
            rcsc::Vector2D mpos = ( opponents.at( i )->pos() + opponents.at( i + 1 )->pos() ) / 2.0;
            middle_pos.push_back( mpos );
	}
    }
    int cycle_margin = 0;
    rcsc::Vector2D target( 0, 0 );
    double speed = 0;
    
    for( int i = 0; i < (int)middle_pos.size(); i++ )
    {
        rcsc::Ray2D pass_course( wm.ball().pos(), middle_pos.at( i ) );
        rcsc::Ray2D receive_course( inertial_mate_pos, rcsc::AngleDeg( 0 ) );
        
        rcsc::Vector2D receive_pos = pass_course.intersection( receive_course );
        if( receive_pos.valid() )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: receive_pos(%f, %f)"
                                ,__FILE__, __LINE__, receive_pos.x, receive_pos.y);
        }
        else
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: There is not a vlid receive_pos"
                                ,__FILE__, __LINE__ );
        }
        
        if( !receive_pos.valid() || receive_pos.x < wm.self().pos().x )
	{
            continue;
	}
        agent->debugClient().addLine( wm.ball().pos(), receive_pos );
        double mate_dist = receive_pos.dist( inertial_mate_pos );
        mate_dist -= receiver->playerTypePtr()->kickableMargin() * 0.5;
        int mate_dash = receiver->playerTypePtr()->cyclesToReachDistance( mate_dist );
        int mate_turn = 1;
        int mate_cycle = mate_turn + mate_dash;
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: Cycles for the receiver to intercept: %d"
                            ,__FILE__, __LINE__, mate_cycle );
        
        double ball_dist = receive_pos.dist( wm.ball().pos() );
        double ball_speed = rcsc::calc_first_term_geom_series( ball_dist, sp.ballDecay(), mate_cycle + 1 );
        rcsc::Vector2D ball_vector( ball_speed, 0 );
        rcsc::Vector2D rpos = receive_pos - wm.ball().pos();
        ball_vector.rotate( rpos.th() );
        rcsc::Vector2D accel = ball_vector - wm.ball().vel();
        double kick_power = accel.r() / wm.self().kickRate();
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: Necessary kick power: %f"
                            ,__FILE__, __LINE__, kick_power );
        if( kick_power > sp.maxPower() )
	{
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: Over capacity. Continue."
                                ,__FILE__, __LINE__, mate_cycle );
            continue;
	}
        
        rcsc::Segment2D pass_seg( wm.ball().pos(), receive_pos );
        
        int opp_min = 100;
        bool success = true;
        for( rcsc::PlayerPtrCont::iterator it = opponents.begin();
             it != opponents.end();
             ++it )
	{    

            rcsc::Vector2D inertial_opp_pos = (*it)->playerTypePtr()->inertiaPoint( (*it)->pos(), (*it)->vel(), inertial_cycle );
            rcsc::Circle2D opp_area( inertial_opp_pos, (*it)->playerTypePtr()->kickableMargin() * 1.2 );
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: inertial_opp_pos for Opp%d: (%f, %f)"
                                ,__FILE__, __LINE__, (*it)->unum(), inertial_opp_pos.x, inertial_opp_pos.y );

            if( opp_area.intersection( pass_seg, NULL, NULL ) > 0 )
            {
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: Opp%d close to pass course. Continue"
                                    ,__FILE__, __LINE__, (*it)->unum() );
                success = false;
                continue; // used to be break;
            }
            double opp_dist = pass_seg.dist( inertial_opp_pos );
            opp_dist -= (*it)->playerTypePtr()->kickableMargin();
            if( opp_dist <= 0 )
            {
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: opp_dist = %f. continue."
                                    ,__FILE__, __LINE__, opp_dist );
                success = false;
                continue; // used to be break;
            }
            int opp_dash = (*it)->playerTypePtr()->cyclesToReachDistance( opp_dist );
            int opp_turn = 1;
            if( (*it)->pos().x < middle_pos.at( i ).x )
	    {
                opp_turn = 0;
	    }
            int opp_cycle = opp_turn + opp_dash;
            
            rcsc::Vector2D opp_target = pass_seg.nearestPoint( inertial_opp_pos );
            double tmp_ball_dist = wm.ball().pos().dist( opp_target );
            int tmp_ball_cycle = rcsc::calc_length_geom_series( ball_speed, tmp_ball_dist, sp.ballDecay() );
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: opp_cycle = %d, cycle to be intercept by Opp%d = %d"
                                ,__FILE__, __LINE__, opp_cycle, (*it)->unum(), tmp_ball_cycle );
            if( tmp_ball_cycle <= 0 )
	    {
                tmp_ball_cycle = 1;
	    }
            if( tmp_ball_cycle >= opp_cycle )
	    {
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: The ball will not pass the opponent gate. continue"
                                    ,__FILE__, __LINE__, opp_cycle, (*it)->unum(), tmp_ball_cycle );
                success = false;
                continue; // used to be break;
	    }
            else if( opp_cycle < opp_min )
	    {
                opp_min = opp_cycle;
	    }
	}
        if( success && opp_min < 100 )
	{
            if( opp_min > cycle_margin )
	    {
                cycle_margin = opp_min;
                target = receive_pos;
                speed = ball_speed * pow( sp.ballDecay(), mate_cycle );
	    }

	}

    }//end for middle_pos
    if( cycle_margin > 0 )
    {
        target_pos = target;
        final_ball_speed = speed;
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

bool 
Bhv_Offense::checkThroughPass2( rcsc::PlayerAgent * agent, const rcsc::PlayerObject * receiver, rcsc::Vector2D & target_pos, double & final_ball_speed )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    int inertial_cycle = 1;

    if( receiver->body().abs() > 60 )
    {
        return false;
    }


    rcsc::Vector2D inertial_self_pos = wm.self().inertiaPoint( inertial_cycle );
    rcsc::Vector2D inertial_mate_pos = receiver->playerTypePtr()->inertiaPoint( receiver->pos(), receiver->vel(), inertial_cycle );
    agent->debugClient().addMessage( "Mate%d", receiver->unum() );
    if( wm.self().pos().x > 30 )
    {
        agent->debugClient().addMessage( "BadX");
        return false;
    }

    std::vector< rcsc::PlayerObject* > opponents;
    const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
         it != end;
         ++it )
    {    
        if( (*it)->pos().x > wm.self().pos().x )
	{
            opponents.push_back( (*it) );
	}
    }
    
    rcsc::Vector2D receive_pos( 33, wm.ball().pos().y );

    agent->debugClient().addLine( wm.ball().pos(), receive_pos );
    double mate_dist = receive_pos.dist( inertial_mate_pos );
    mate_dist -= receiver->playerTypePtr()->kickableMargin() * 0.5;
    if( mate_dist <= 0 )
    {
        mate_dist = 0.5;
    }
    int mate_dash = receiver->playerTypePtr()->cyclesToReachDistance( mate_dist );
    int mate_turn = 1;
    int mate_cycle = mate_turn + mate_dash;
    agent->debugClient().addMessage( "Cycle%d", mate_cycle );

    double ball_dist = receive_pos.dist( wm.ball().pos() );
    double ball_speed = rcsc::calc_first_term_geom_series( ball_dist, sp.ballDecay(), mate_cycle + 1 );
    rcsc::Vector2D ball_vector( ball_speed, 0 );
    rcsc::Vector2D rpos = receive_pos - wm.ball().pos();
    ball_vector.rotate( rpos.th() );
    rcsc::Vector2D accel = ball_vector - wm.ball().vel();
    double kick_power = accel.r() / wm.self().kickRate();
    if( kick_power > sp.maxPower() )
    {
        agent->debugClient().addMessage( "CantEnoughKick" );
        return false;
    }

    rcsc::Segment2D pass_seg( wm.ball().pos(), receive_pos );

    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
         it != end;
         ++it )
    {   
        agent->debugClient().addMessage( "Opp%d", (*it)->unum() );
        rcsc::Vector2D inertial_opp_pos = (*it)->playerTypePtr()->inertiaPoint( (*it)->pos(), (*it)->vel(), inertial_cycle );
        rcsc::Circle2D opp_area( inertial_opp_pos, (*it)->playerTypePtr()->kickableMargin() * 1.2 );
        if( opp_area.intersection( pass_seg, NULL, NULL ) > 0 )
        {
            agent->debugClient().addMessage( "Intersection" );
            return false;
        }
        double opp_dist = pass_seg.dist( inertial_opp_pos );
        opp_dist -= (*it)->playerTypePtr()->kickableMargin();
        if( opp_dist <= 0 )
        {
            agent->debugClient().addMessage( "Dist0" );
            return false;
        }
        int opp_dash = (*it)->playerTypePtr()->cyclesToReachDistance( opp_dist );
        int opp_turn = 1;
        int opp_cycle = opp_turn + opp_dash;
        
            
        rcsc::Vector2D opp_target = pass_seg.nearestPoint( inertial_opp_pos );
        double tmp_ball_dist = wm.ball().pos().dist( opp_target );
        int tmp_ball_cycle = rcsc::calc_length_geom_series( ball_speed, tmp_ball_dist, sp.ballDecay() );
        if( tmp_ball_cycle <= 0 )
        {
            tmp_ball_cycle = 1;
        }
        if( tmp_ball_cycle >= opp_cycle )
        {
            agent->debugClient().addMessage( "Cycle%d", opp_cycle );
            return false;
        }

        opp_dist = receive_pos.dist( inertial_opp_pos );
        opp_dist -= (*it)->playerTypePtr()->kickableMargin();
        opp_dash = (*it)->playerTypePtr()->cyclesToReachDistance( opp_dist );
        int tmp_opp_cycle = opp_turn + opp_dash;

        if( tmp_opp_cycle <= mate_cycle )
        {
            agent->debugClient().addMessage( "Cycle%d", opp_cycle );
            return false;
        }
        agent->debugClient().addMessage( "OK" );
    }      

    target_pos = receive_pos;
    final_ball_speed = ball_speed * pow( sp.ballDecay(), mate_cycle );

    return true;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

void
Bhv_Offense::getOppEndLine( rcsc::PlayerAgent * agent, std::vector< rcsc::Vector2D > & oppline )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Bhv_Offense::getOppEndLine"
                        ,__FILE__, __LINE__ );


    const rcsc::WorldModel & wm = agent->world();

    double opp_defX = wm.offsideLineX() + 1.0;
    // detect candidate defense line
    for( rcsc::PlayerCont::const_iterator it = wm.opponents().begin();
         it != wm.opponents().end();
         ++it )
    {
        if( (*it).goalie() || (*it).isTackling() )
            continue;
 
        if( (*it).pos().x <= opp_defX )
        {
            bool defender = true;
            rcsc::Vector2D center( opp_defX + 1.0, (*it).pos().y );
            rcsc::Sector2D sector( center,
                                   0.0,
                                   center.dist( (*it).pos() ),
                                   rcsc::AngleDeg( 90 ),
                                   rcsc::AngleDeg( 270 ) );
            for( rcsc::PlayerCont::const_iterator it2 = wm.opponents().begin();
                 it2 != wm.opponents().end();
                 ++it2 )
            {
                if( it != it2
                    && sector.contains( (*it2).pos() ) )
                {
                    defender = false;
                    break;
                }
            }
            if( defender )
            {
                oppline.push_back( (*it).pos() );
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: oppline added (%f,%f)"
                                    ,__FILE__, __LINE__,
                                    (*it).pos().x, (*it).pos().y );
            }
       }
    }

    // sort by Y
    rcsc::Vector2D tmp_vec2d;
    for( std::vector<rcsc::Vector2D>::iterator it = oppline.begin();
         it != oppline.end();
         ++it )
    {
        for( std::vector<rcsc::Vector2D>::iterator it2 = it + 1;
             it2 != oppline.end();
             ++it2 )
        {
            if( (*it).y < (*it2).y )
            {
                tmp_vec2d = (*it);
                (*it) = (*it2);
                (*it2) = tmp_vec2d;
            }
        }
    }
    // visualize
    if( oppline.size() > 1 )
    {
        for( int i = 1 ; i < (int) oppline.size(); i++ )
        {
            agent->debugClient().addLine( oppline.at( i - 1 ),
                                          oppline.at( i ) );
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: oppline (%f,%f)-(%f,%f)"
                                ,__FILE__, __LINE__,
                                oppline.at(i-1).x, oppline.at(i-1).y,
                                oppline.at(i).x, oppline.at(i).y );
        }
    }

    return;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

bool
Bhv_Offense::getMateFrontLine( rcsc::PlayerAgent * agent, std::vector< rcsc::Vector2D > & matefront )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Bhv_Offense::getMateFrontLine"
                        ,__FILE__, __LINE__ );

    bool Imfront = false;

    const rcsc::WorldModel & wm = agent->world();
    double mate_maxX = -100;
    // Detect the frontier mate
    for( rcsc::AbstractPlayerCont::const_iterator it = wm.allTeammates().begin();
         it != wm.allTeammates().end();
         ++it )
    {
        if( (*it)->goalie() || wm.offsideLineX() < (*it)->pos().x )
            continue;
        if( (*it)->pos().x > mate_maxX )
            mate_maxX = (*it)->pos().x;
    } 
    mate_maxX += 1.0;


    // detect candidate offense line
    for( rcsc::AbstractPlayerCont::const_iterator it = wm.allTeammates().begin();
         it != wm.allTeammates().end();
         ++it )
    {
        if( (*it)->goalie() || (*it)->isTackling() || wm.offsideLineX() < (*it)->pos().x )
            continue;
 
        if( (*it)->pos().x <= mate_maxX )
        {
            bool frontline = true;
            rcsc::Vector2D center( mate_maxX + 1.0, (*it)->pos().y );
            rcsc::Sector2D sector( center,
                                   0.0,
                                   center.dist( (*it)->pos() ),
                                   rcsc::AngleDeg( 90 ),
                                   rcsc::AngleDeg( 270 ) );
            for( rcsc::AbstractPlayerCont::const_iterator it2 = wm.allTeammates().begin();
                 it2 != wm.allTeammates().end();
                 ++it2 )
            {
                if( it != it2
                    && sector.contains( (*it2)->pos() ) )
                {
                    frontline = false;
                    break;
                }
            }
            if( frontline )
            {
                matefront.push_back( (*it)->pos() );
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: mate frontline added (%f,%f)"
                                    ,__FILE__, __LINE__,
                                    (*it)->pos().x, (*it)->pos().y );
                if( (*it)->unum() == wm.self().unum() )
                {
                    rcsc::dlog.addText( rcsc::Logger::TEAM,
                                        "%s:%d: I am in the front line." ,__FILE__, __LINE__ );
                    Imfront = true;
                }
            }
        }
    }

    // sort by Y
    rcsc::Vector2D tmp_vec2d;
    for( std::vector<rcsc::Vector2D>::iterator it = matefront.begin();
         it != matefront.end();
         ++it )
    {
        for( std::vector<rcsc::Vector2D>::iterator it2 = it + 1;
             it2 != matefront.end();
             ++it2 )
        {
            if( (*it).y < (*it2).y )
                std::swap( *it, *it2 );
        }
    }
    // visualize
    if( matefront.size() > 1 )
    {
        for( int i = 1 ; i < (int) matefront.size(); i++ )
        {
            agent->debugClient().addLine( matefront.at( i - 1 ),
                                          matefront.at( i ) );
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: oppline (%f,%f)-(%f,%f)"
                                ,__FILE__, __LINE__,
                                matefront.at(i-1).x, matefront.at(i-1).y,
                                matefront.at(i).x, matefront.at(i).y );
        }
    }

    return Imfront;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

int
Bhv_Offense::selfReachCycle( rcsc::PlayerAgent * agent )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: Bhv_Offense::selfReachCycle"
                        ,__FILE__, __LINE__ );

    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();


    if( wm.self().isKickable() )
    {
        return 0;
    }


    rcsc::Ray2D self_ray( wm.self().pos(), wm.self().body() );
    rcsc::Ray2D ball_ray( wm.ball().pos(), wm.ball().vel().th() );
    rcsc::Vector2D next_ball_pos = wm.ball().pos() + wm.ball().vel();
    if( wm.ball().pos().equals( next_ball_pos ) )
    {
        double tmp_dist = wm.ball().pos().dist( wm.self().pos() );
        int tmp_dash_cycle = wm.self().playerTypePtr()->cyclesToReachDistance( tmp_dist );
        int tmp_turn_cycle = 0;
        rcsc::Vector2D tmp_self_pos = wm.self().inertiaPoint( tmp_dash_cycle );
        rcsc::Vector2D rpos = wm.ball().pos() - tmp_self_pos;
        rcsc::AngleDeg dif = rpos.th() - wm.self().body();
        if( dif.abs() > 5 )
        {
            tmp_turn_cycle++;
        }
        return tmp_turn_cycle + wm.self().playerTypePtr()->cyclesToReachDistance( rpos.r() );
    }
    rcsc::Line2D ball_line( wm.ball().pos(), next_ball_pos );

    agent->debugClient().addLine( rcsc::Vector2D( -52, ball_line.getY( -52 ) ),
                                  rcsc::Vector2D( 52, ball_line.getY( 52 ) ) );

    rcsc::Vector2D intersection = self_ray.intersection( ball_ray );
    int turn_cycle = 0;
    int dash_cycle = 0;

    rcsc::AngleDeg modified_self_body = wm.self().body();
    if( !intersection.valid() )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: no intersection between body angle and ball trajectory"
                            ,__FILE__, __LINE__ );

        //turn
        intersection = ball_line.projection( wm.self().pos() );
        turn_cycle++;
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: added turn: currently %d"
                            ,__FILE__, __LINE__, turn_cycle );
        rcsc::Vector2D rpos = intersection - wm.self().pos();
        rcsc::AngleDeg dif = rpos.th() - wm.self().body();
        if( !wm.self().canTurn( dif.degree() ) )
        {
            turn_cycle++;
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: another added turn: currently %d"
                                ,__FILE__, __LINE__, turn_cycle );
        }
        modified_self_body = rpos.th();
    }

    
    int cycle = cyclesViaPoint( agent, intersection );
    

    return cycle;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

int 
Bhv_Offense::cyclesViaPoint( rcsc::PlayerAgent * agent, rcsc::Vector2D way_pos )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();
    
    agent->debugClient().addLine( wm.self().pos(), way_pos );
    agent->debugClient().addCircle( way_pos, 5 );
    rcsc::Ray2D self_ray( wm.self().pos(), wm.self().body() );
    rcsc::Ray2D ball_ray( wm.ball().pos(), wm.ball().vel().th() );
    rcsc::Vector2D next_ball_pos = wm.ball().pos() + wm.ball().vel();
    rcsc::Line2D ball_line( wm.ball().pos(), next_ball_pos );

    
    int turn_cycle = 0;
    int dash_cycle = 0;
    int opp_min = wm.interceptTable()->opponentReachCycle();

    rcsc::Vector2D modified_self_vel = wm.self().vel();
    rcsc::Vector2D moved_self_pos = wm.self().pos();
    rcsc::Vector2D moved_ball_pos = wm.ball().pos();
    rcsc::Vector2D ball_vector = wm.ball().vel();
    rcsc::Circle2D self_area( moved_self_pos, wm.self().kickableArea() );
    rcsc::AngleDeg modified_self_body = wm.self().body();
    rcsc::Vector2D self_to_way = way_pos - wm.self().pos();
    rcsc::AngleDeg dif =  self_to_way.th() - wm.self().body();

    if( dif.abs() > 5 )
    {
        turn_cycle++;
        if( !wm.self().canTurn( dif.degree() ) )
        {
            turn_cycle++;
        }
        modified_self_body = self_to_way.th();
    }
    double simple_dash_dist = way_pos.dist( wm.self().pos() );
    if( ball_ray.inRightDir( wm.self().pos() )
        && ball_line.dist( wm.self().pos() ) < wm.self().kickableArea() )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: already on the ball line. dist set to 0.0."
                            ,__FILE__, __LINE__, turn_cycle );
        simple_dash_dist = 0.0;
    }
    int max_dash_cycle = wm.self().playerTypePtr()->cyclesToReachDistance( simple_dash_dist ) + 5;


    for( int i = 0; i < turn_cycle; i++ )
    {
        if( simple_dash_dist == 0.0 )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: turn not necessary"
                                ,__FILE__, __LINE__, turn_cycle );
            break;
        }
        
        //turn
        moved_self_pos += modified_self_vel;
        moved_ball_pos += ball_vector;
        
        agent->debugClient().addCircle( moved_self_pos, wm.self().kickableArea() );
        agent->debugClient().addCircle( moved_ball_pos, sp.ballSize() );

        modified_self_vel *= wm.self().playerTypePtr()->playerDecay();
        ball_vector *= sp.ballDecay();
    }


    rcsc::Vector2D dash_vector( 0, 0 );
    dash_vector.setPolar( wm.self().dashRate() * 100,
                          modified_self_body );    
    for( int i = 0; i < max_dash_cycle; i++ )
    {
        if( i == 0 )
        {
            if( self_area.intersection( ball_line, NULL, NULL ) > 0 ) 
            {
                break;
            }
        }
        dash_cycle++;
        modified_self_vel += dash_vector;

        moved_self_pos += modified_self_vel;
        moved_ball_pos += ball_vector;

        agent->debugClient().addCircle( moved_self_pos, wm.self().kickableArea() );
        agent->debugClient().addCircle( moved_ball_pos, sp.ballSize() );

        modified_self_vel *= wm.self().playerTypePtr()->playerDecay();
        ball_vector *= sp.ballDecay();
        
        self_area.assign( moved_self_pos, wm.self().kickableArea() );
        if( self_area.intersection( ball_line, NULL, NULL ) > 0 )
        {
            break;
        }
    }

    self_area.assign( moved_self_pos, wm.self().kickableArea() );
    if( self_area.contains( moved_ball_pos ) )
    {
        return turn_cycle + dash_cycle;
    }
    else
    {
        rcsc::Ray2D moved_ball_ray( moved_ball_pos, ball_vector.th() );
        if( self_area.intersection( moved_ball_ray, NULL, NULL ) > 0 )
        {
            //wait ball
            double remain_dist = moved_ball_pos.dist( moved_self_pos );
            int wait_cycle = rcsc::calc_length_geom_series( ball_vector.r(),
                                                      remain_dist,
                                                      sp.ballDecay() );
            agent->debugClient().addMessage( "WAIT" );

            if( wait_cycle < 0 || wait_cycle >= opp_min - turn_cycle - dash_cycle )
            {
                //need dash
                int add_dash = wm.self().playerTypePtr()->cyclesToReachDistance( remain_dist );
                int add_turn = 1;
                return turn_cycle + dash_cycle + add_dash + add_turn;
            }
            else
            {
                return turn_cycle + dash_cycle + wait_cycle;
            }
            
        }
        else
        {
            //chase ball 
            
            int add_turn = 1;
            for( int i = 0; i < add_turn; i++ )
            {
                //turn
//                moved_self_pos += modified_self_vel;
                moved_ball_pos += ball_vector;

                ball_vector *= sp.ballDecay();
//                modified_self_vel *= wm.self().playerTypePtr()->playerDecay();

                agent->debugClient().addCircle( moved_self_pos, wm.self().kickableArea() );
                agent->debugClient().addCircle( moved_ball_pos, sp.ballSize() );
            }
//            rcsc::Vector2D self_to_ball = moved_ball_pos - moved_self_pos;
//            dash_vector.setPolar( wm.self().dashRate() * 100,
//                                  self_to_ball.th() );    

            modified_self_vel.assign( 0, 0 );
            dash_vector.setPolar( wm.self().dashRate() * 100,
                                  wm.ball().vel().th() );
            agent->debugClient().addMessage( "CHASE" );
            for( int i = 0; i < 100; i++ )
            {

                modified_self_vel += dash_vector;
                moved_self_pos += modified_self_vel;
                moved_ball_pos += ball_vector;

                agent->debugClient().addCircle( moved_self_pos, wm.self().kickableArea() );
                agent->debugClient().addCircle( moved_ball_pos, sp.ballSize() );

                self_area.assign( moved_self_pos, wm.self().kickableArea() );
                if( self_area.contains( moved_ball_pos ) )
                {
                    return dash_cycle + i + add_turn;
                }
                modified_self_vel *= wm.self().playerTypePtr()->playerDecay();
                ball_vector *= sp.ballDecay();
            }
        }
    }
    
    return 1000;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

int 
Bhv_Offense::checkReceiver( rcsc::PlayerAgent * agent, rcsc::Vector2D receiver_pos )
{
    return checkReceiver( agent, receiver_pos, M_target_point );
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

int 
Bhv_Offense::checkReceiver( rcsc::PlayerAgent * agent, rcsc::Vector2D receiver_pos, rcsc::Vector2D target_pos )
{
    const rcsc::WorldModel & wm = agent->world();
    

    rcsc::Vector2D r2t = target_pos - receiver_pos;
    rcsc::Vector2D center = receiver_pos - r2t.setLengthVector( 7.0 );
/*
    rcsc::Line2D base_line( receiver->pos(), M_target_point );
    double x,y;
    if( fabs( base_line.a() ) < 0.01 )
    {
        x = receiver->pos().x - 7.0;
        y = receiver->pos().y;
    }
    else
    {
        y = base_line.getY( receiver->pos().x - 7.0 );
        x = base_line.getX( y );
    }
    rcsc::Vector2D center( x, y );
    rcsc::Vector2D c2t = M_target_point - center;
*/
    double min_r = center.dist( receiver_pos ) - 0.5;
    if( min_r < 0 )
    {
        min_r = 0;
    }
    double max_r = center.dist( receiver_pos ) + 8.5;

    
    rcsc::Sector2D sector( center, 
                           min_r,
                           max_r,
                           r2t.th() - 15,
                           r2t.th() + 15 );

    int opp_num = 0;
    const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
         it != end;
         ++it )
    {   
        if( sector.contains( (*it)->pos() ) )
        {
            opp_num++;
        }
    }
    
    
    return opp_num;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

int 
Bhv_Offense::checkPassReceive( rcsc::PlayerAgent * agent, rcsc::Vector2D passer_pos, rcsc::Vector2D receiver_pos )
{
    const rcsc::WorldModel & wm = agent->world();
    
    rcsc::Vector2D p2r = receiver_pos - passer_pos;
    
    rcsc::Vector2D center = passer_pos - p2r.setLengthVector( 7.0 );
    double min_r = center.dist( passer_pos ) - 0.5;
    if( min_r < 0 )
    {
        min_r = 0;
    }
    double max_r = center.dist( receiver_pos ) + 0.5;

    rcsc::Sector2D sector( center,
                           min_r,
                           max_r,
                           p2r.th() - 15,
                           p2r.th() + 15 );

    
    int opp_num = 0;

    const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
         it != end;
         ++it )
    {   
        if( sector.contains( (*it)->pos() ) )
        {
            opp_num++;
        }
    }

    return opp_num;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

void 
Bhv_Offense::ranking( rcsc::PlayerAgent * agent, std::vector< rcsc::AbstractPlayerObject* > & candidates )
{
    if( candidates.empty() )
    {
        return;
    }
    const rcsc::WorldModel & wm = agent->world();
    
    for( std::vector< rcsc::AbstractPlayerObject* >::iterator it = candidates.begin();
         it != candidates.end();
         ++it )
    {
        rcsc::Vector2D dribble_target = (*it)->pos() + rcsc::Vector2D( 10, 0 );
        if( wm.ball().pos().x > 25 )
        {
            dribble_target = rcsc::Vector2D( 52, 0 );
        }
        int front_opp = checkReceiver( agent, (*it)->pos(), dribble_target );
        int mid_opp = checkPassReceive( agent, wm.self().pos(), (*it)->pos() );
        const rcsc::PlayerPtrCont::const_iterator mate_end = wm.teammatesFromBall().end();
        for( rcsc::PlayerPtrCont::const_iterator mate = wm.teammatesFromBall().begin();
             mate != mate_end;
             ++mate )
        {
            
        }
    }

    
    return;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

int 
Bhv_Offense::countSectorOpp( rcsc::PlayerAgent * agent, rcsc::Vector2D receiver_pos, rcsc::AngleDeg angle, double length, rcsc::AngleDeg width, bool goalie )
{
    const rcsc::WorldModel & wm = agent->world();
    int opp_num = 0;
    
    rcsc::Vector2D back;
    back.setPolar( 1.0, angle );
    rcsc::Vector2D center = receiver_pos - back;
    double min_r = center.dist( receiver_pos ) - 0.5;
    if( min_r < 0 )
    {
        min_r = 0;
    }
    double max_r = center.dist( receiver_pos ) + length;
    rcsc::Sector2D sector( center,
                           min_r,
                           max_r,
                           angle - width,
                           angle + width );
    const rcsc::PlayerPtrCont::const_iterator opp_end = wm.opponentsFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
         it != opp_end;
         ++it )
    {
        if( !goalie && (*it)->goalie() )
            continue;

        if( sector.contains( (*it)->pos() ) )
        {
            opp_num++;
        }
    }


    return opp_num;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

rcsc::Sector2D 
Bhv_Offense::freeSector( rcsc::PlayerAgent * agent, rcsc::Vector2D receiver_pos, rcsc::AngleDeg & angle, double & length )
{
    rcsc::Sector2D free_sector( rcsc::Vector2D( 0, 0 ),
                                0,
                                10,
                                -30,
                                30 );
    angle = rcsc::AngleDeg( 0 );
    length = 0;
    rcsc::AngleDeg width( 30 );
    double r = 1.0;
    rcsc::Vector2D c;
    double max_length = 20.0;
    for( int i = 0; i < 60; i += 5 )
    {
        rcsc::AngleDeg tmp_angle( i );
        double tmp_length = 20;
        
        rcsc::Vector2D back;
        back.setPolar( 5.0, tmp_angle );
        
        rcsc::Vector2D center = receiver_pos - back;
        double min_r = center.dist( receiver_pos ) - 0.5;
        if( min_r < 0 )
        {
            min_r = 0;
        }
        rcsc::Sector2D tmp_sector = free_sector;
        while( tmp_length > 0.1 )
        {
            double max_r = center.dist( receiver_pos ) + tmp_length;
            
            rcsc::Sector2D sector( center,
                                   min_r,
                                   max_r,
                                   tmp_angle - width,
                                   tmp_angle + width );
            if( countSectorOpp( agent, receiver_pos, tmp_angle, tmp_length, width ) == 0 )
            {
                tmp_sector = sector;
                
                break;
            }
            tmp_length -= 1.0;
        }
        if( tmp_length > length )
        {
            length = tmp_length;
            angle = tmp_angle;
            free_sector = tmp_sector;
            r = center.dist( receiver_pos ) + tmp_length;
            c = center;
            if( length == max_length )
            {
                break;
            }
        }
        /////////////////////////////////
        if( i != 0 )
        {
            tmp_angle = rcsc::AngleDeg( -i );
            tmp_length = 20;
            back.setPolar( 5.0, tmp_angle );
            center = receiver_pos - back;
            min_r = center.dist( receiver_pos ) - 0.5;
            if( min_r < 0 )
            {
                min_r = 0;
            }
            while( tmp_length > 0.1 )
            {
                double max_r = center.dist( receiver_pos ) + tmp_length;
                
                rcsc::Sector2D sector( center,
                                       min_r,
                                       max_r,
                                       tmp_angle - width,
                                       tmp_angle + width );
                if( countSectorOpp( agent, receiver_pos, tmp_angle, tmp_length, width ) == 0 )
                {
                    tmp_sector = sector;
                    break;
                }
                tmp_length -= 1.0;
            }
            if( tmp_length > length )
            {
                length = tmp_length;
                angle = tmp_angle;
                free_sector = tmp_sector;
                r = center.dist( receiver_pos ) + tmp_length;
                c = center;
                if( length == max_length )
                {
                    break;
                }
            }
        }
        
    }
/*
    agent->debugClient().addCircle( c, r );
    rcsc::Vector2D dif;
    dif.setPolar( r, angle - width );
    rcsc::Vector2D left = c + dif;
    dif.setPolar( r, angle + width );
    rcsc::Vector2D right = c + dif;
    agent->debugClient().addLine( c, left );
    agent->debugClient().addLine( c, right );
*/  
    
    return free_sector;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

void
Bhv_Offense::findFrontFreeMate( std::vector< int > & unumffmate, rcsc::PlayerAgent *agent )
{
    const rcsc::WorldModel & wm = agent->world();
    rcsc::Vector2D front;

    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
         it != wm.teammatesFromSelf().end();
         ++it )
    {
        front.assign( (*it)->pos().x + 10.0 , (*it)->pos().y );

        if( Bhv_Offense( front ).checkReceiver( agent, (*it)->pos() ) == 0 )
        {
            unumffmate.push_back( (*it)->unum() );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

bool 
Bhv_Offense::getSubReceiver( rcsc::PlayerAgent * agent, const rcsc::PlayerObject * last_receiver, std::vector< rcsc::PlayerObject* > & subs )
{
    subs.clear();
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();
    const rcsc::PlayerPtrCont::const_iterator mate_end = wm.teammatesFromBall().end();
    const rcsc::PlayerPtrCont::const_iterator opp_end = wm.opponentsFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator mate = wm.teammatesFromBall().begin();
         mate != mate_end;
         ++mate )
    {
        if( last_receiver == (*mate) )
        {
            continue;
        }
        if( (*mate)->pos().dist( last_receiver->pos() ) > 20 )
        {
            continue;
        }
        if( (*mate)->pos().dist( wm.self().pos() ) > 20 )
        {
            continue;
        }
        bool success1 = true; //self to sub
        bool success2 = true; //sub to last
        rcsc::Segment2D self_to_sub( wm.self().pos(), (*mate)->pos() );
        rcsc::Segment2D sub_to_last( (*mate)->pos(), last_receiver->pos() );
        for( rcsc::PlayerPtrCont::const_iterator opp = wm.opponentsFromBall().begin();
             opp != opp_end;
             ++opp )
        {
            rcsc::Circle2D opp_area( (*opp)->pos(), (*opp)->playerTypePtr()->kickableMargin() );
            if( opp_area.intersection( self_to_sub, NULL, NULL ) > 0 )
            {
                success1 = false;
                break;
            }
            if( opp_area.intersection( sub_to_last, NULL, NULL ) > 0 )
            {
                success2 = false;
                break;
            }

            double opp_dist1 = self_to_sub.dist( (*opp)->pos() );
            double opp_dist2 = sub_to_last.dist( (*opp)->pos() );
            
            opp_dist1 -= (*opp)->playerTypePtr()->kickableMargin();
            opp_dist2 -= (*opp)->playerTypePtr()->kickableMargin();
            
            if( opp_dist1 <= 0 )
            {
                success1 = false;
                break;
            }
            if( opp_dist2 <= 0 )
            {
                success2 = false;
                break;
            }
            
            int opp_dash1 = (*opp)->playerTypePtr()->cyclesToReachDistance( opp_dist1 );
            int opp_dash2 = (*opp)->playerTypePtr()->cyclesToReachDistance( opp_dist2 );
            int opp_turn = 1;

            int opp_cycle1 = opp_dash1 + opp_turn;
            int opp_cycle2 = opp_dash2 + opp_turn;
            
            rcsc::Vector2D opp_target1 = self_to_sub.nearestPoint( (*opp)->pos() );
            rcsc::Vector2D opp_target2 = sub_to_last.nearestPoint( (*opp)->pos() );
            
            double ball_dist1 = wm.self().pos().dist( opp_target1 );
            double ball_dist2 = (*mate)->pos().dist( opp_target2 );

            int ball_cycle1 = rcsc::calc_length_geom_series( sp.ballSpeedMax(),
                                                             ball_dist1,
                                                             sp.ballDecay() );
            int ball_cycle2 = rcsc::calc_length_geom_series( sp.ballSpeedMax(),
                                                             ball_dist2,
                                                             sp.ballDecay() );
            if( ball_cycle1 < 0
                || ball_cycle1 >= opp_cycle1 )
            {
                success1 = false;
                break;
            }
            if( ball_cycle2 < 0
                || ball_cycle2 >= opp_cycle2 )
            {
                success2 = false;
                break;
            }
        }
        if( success1 && success2 )
        {
            subs.push_back( (*mate) );
        }
    }
    if( !subs.empty() )
    {
        return true;
    }
    return false;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

void 
Bhv_Offense::surroundOppNum( rcsc::PlayerAgent * agent, std::vector< int > & opp_num )
{
    const rcsc::WorldModel & wm = agent->world();
    opp_num.clear();
    for( int i = 0; i < 5; i++ )
    {
        opp_num.push_back( 0 );
    }
    const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
         it != end;
         ++it )
    {   
        double min = 50;
        int min_unum = 0;
        for( int i = 1; i <= 11; i++ )
        {
            if( wm.teammate( i ) )
            {
                double dist = wm.teammate( i )->pos().dist( (*it)->pos() );
                if( dist < min )
                {
                    min = dist;
                    min_unum = i;
                }
            }
        }
        switch( min_unum )
        {
        case 7:
            opp_num[0]++;
            break;
        case 8:
            opp_num[1]++;
            break;
        case 9:
            opp_num[2]++;
            break;
        case 10:
            opp_num[3]++;
            break;
        case 11:
            opp_num[4]++;
            break;
        default:
            break;
        }

    }
    agent->debugClient().addMessage( "7:%d,8:%d,9:%d,10:%d,11:%d", opp_num[0], opp_num[1], opp_num[2], opp_num[3], opp_num[4] );
    return;
}

/*-------------------------------------------------------------------*/
/*!


*/
bool 
Bhv_Offense::pointingPass( rcsc::PlayerAgent * agent, rcsc::PlayerObject * receiver, rcsc::Vector2D & pass_target, double & first_speed )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();
    const rcsc::PlayerPtrCont::const_iterator opp_end = wm.opponentsFromBall().end();
    const rcsc::PlayerPtrCont::const_iterator end_self = wm.teammatesFromSelf().end();
    const rcsc::PlayerObject * goalie = wm.getOpponentGoalie();
    
    double min_speed = 2.5;
    if( receiver->pos().dist( wm.ball().pos() ) < 10 )
    {
        min_speed = 2.0;
    }
    else if( receiver->pos().dist( wm.ball().pos() ) < 15 )
    {
        min_speed = 2.25;
    }
    if( receiver->pos().x > wm.offsideLineX() 
        || receiver->posCount() > 3 )
    {
        return false;
    }
    if( !goalie )
    {
        return false;
    }

    std::vector< rcsc::Vector2D > mate_pos;
    rcsc::Vector2D dash;
    rcsc::Vector2D current = receiver->pos();
    rcsc::Vector2D vel( 0, 0 );
    bool finish = false;
    if( receiver->pointtoAngle().abs() >= 90 )
    {
        return false;
    }
    dash.setPolar( receiver->playerTypePtr()->dashRate( receiver->playerTypePtr()->effortMax() ) * 100, receiver->pointtoAngle() );
    agent->debugClient().addMessage( "pointAngle%.1f", receiver->pointtoAngle().degree() );
    mate_pos.push_back( current );
    int count = 0;
    while( !finish )
    {
        vel += dash;
        current += vel;
        
        double move_dist = current.dist( mate_pos[0] );
        double goalie_dist = current.dist( goalie->pos() );
        
        if( current.x > 50 
            || current.absY() > 32 )
        {
            finish = true;
            break;
        }
        if( goalie_dist < move_dist )
        {
            finish = true;
            break;
        }
        
        mate_pos.push_back( current );
        agent->debugClient().addCircle( current, 1 );
        count++;
        vel *= receiver->playerTypePtr()->playerDecay();
        if( count > 30 )
        {
            finish = true;
            break;
        }
    }
    if( mate_pos.size() < 1 )
    {
        return false;
    }
    for( int i = (int)mate_pos.size() - 1; i >= 0; i-- )
    {
        rcsc::Vector2D b2m = mate_pos[i] - wm.ball().pos();
        rcsc::Vector2D ball_vel = rcsc::Body_KickOneStep::get_max_possible_vel( b2m.th(), wm.self().kickRate(), wm.ball().vel() );
        double ball_speed = ball_vel.r();
        rcsc::Vector2D ball_pos = wm.ball().pos();
        int max_cycle = rcsc::calc_length_geom_series( ball_vel.r(), b2m.r(), sp.ballDecay() );
        if( max_cycle < 0
            || max_cycle > 40 )
        {
            double move_dist = rcsc::calc_sum_geom_series( ball_vel.r(), sp.ballDecay(), 40 );
            double receive_dist = mate_pos[i].dist( wm.ball().pos() );
            receive_dist -= receiver->playerTypePtr()->kickableArea();
            receive_dist -= move_dist;
            if( receive_dist > 0 )
            {
                continue;
            }
        }
        if( max_cycle < i )
        {
            double dist = b2m.r() - receiver->playerTypePtr()->kickableMargin();
            if( dist <= 1 )
            {
                continue;
            }
            double tmp_speed = rcsc::calc_first_term_geom_series( b2m.r(), sp.ballDecay(), i );
            if( tmp_speed < ball_speed )
            {
                if( tmp_speed < min_speed )
                {
                    tmp_speed = min_speed;
                }
                ball_speed = tmp_speed;
                ball_vel.setPolar( ball_speed, b2m.th() );
            }
        }
        bool success = true;
        for( int j = 1; j <= max_cycle; j++ )
        {
            ball_pos += ball_vel;
            for( rcsc::PlayerPtrCont::const_iterator opp = wm.opponentsFromBall().begin();
                 opp != opp_end;
                 ++opp )
            {
                if( (*opp)->pos().x < wm.self().pos().x )
                {
                    continue;
                }
                double opp_dist = (*opp)->pos().dist( ball_pos );
                if( j < 3 )
                {
                    opp_dist -= (*opp)->playerTypePtr()->kickableArea();
                }
                else if( j >= 3 )
                {
                    opp_dist -= (*opp)->playerTypePtr()->dashDistanceTable()[j-2];
                }
                if( opp_dist <= 0 )
                {
                    success = false;
                    break;
                }
/*
                int opp_cycle = (*opp)->playerTypePtr()->cyclesToReachDistance( opp_dist );
                if( opp_cycle + 3 < j )
                {
                    success = false;
                    break;
                }
*/
            }
            if( !success )
            {
                break;
            }
            ball_vel *= sp.ballDecay();
        }
        if( success )
        {
            pass_target = mate_pos[i];
            if( pass_target.x < wm.offsideLineX() )
            {
                return false;
            }

            first_speed = ball_speed;
            rcsc::Segment2D segment( wm.ball().pos(), pass_target );
            for( rcsc::PlayerPtrCont::const_iterator opp = wm.opponentsFromBall().begin();
                 opp != opp_end;
                 ++opp )
            {
                if( (*opp)->pos().x < wm.self().pos().x )
                {
                    continue;
                }
                rcsc::Line2D straight( (*opp)->pos(), rcsc::AngleDeg( 0 ) );
                rcsc::Vector2D opp_target = segment.intersection( straight );
                if( opp_target.valid() )
                {
                    rcsc::Vector2D b2ot = opp_target - wm.ball().pos();
                    int ball_cycle = rcsc::calc_length_geom_series( first_speed, b2ot.r(), sp.ballDecay() );
                    rcsc::Vector2D o2ot = opp_target - (*opp)->pos();
                    int opp_cycle = (*opp)->playerTypePtr()->cyclesToReachDistance( o2ot.r() );
                    if( ball_cycle > opp_cycle + 2 )
                    {
                        return false;
                    }
                }
                else
                {
                    opp_target = pass_target;
                    rcsc::Vector2D o2ot = opp_target - (*opp)->pos();
                    int opp_cycle = (*opp)->playerTypePtr()->cyclesToReachDistance( o2ot.r() );
                    if( i > opp_cycle + 2 )
                    {
                        return false;
                    }
                }
            }            

            agent->debugClient().addMessage( "Speed%.1f", first_speed ); 
            return true;
        }
    } 
    return false;
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/

bool
Bhv_Offense::decideTurnAngleWhenImThere( rcsc::PlayerAgent * agent, rcsc::Vector2D & t_receive )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: Bhv_Offense::decideTurnAngleWhenImThere"
                        ,__FILE__, __LINE__ );

    const rcsc::WorldModel & wm = agent->world();

    if( wm.getOpponentGoalie() )
    {
        t_receive.x = (wm.self().pos().x + wm.getOpponentGoalie()->pos().x) / 2.0;
    }
    else
    {
        t_receive.x = (wm.self().pos().x + 40.0) / 2.0;
    }

    std::vector<rcsc::Vector2D> matefront;
    bool front =  getMateFrontLine( agent, matefront );
    if( !front )
    {
        t_receive.y = wm.self().pos().y;
        return true;
    }
    else if( std::fabs( wm.self().pos().y - wm.ball().pos().y ) > 10.0 )
    {
        t_receive.y = wm.self().pos().y;
        return true;
    }
    else
    {
        for( std::vector<rcsc::Vector2D>::iterator it = matefront.begin();
             it != matefront.end(); ++it )
        {
            if( (*it).y == wm.self().pos().y || (*it).y == wm.ball().pos().y )
                continue;
            
            if( wm.self().pos().y < (*it).y && (*it).y < wm.ball().pos().y )
            {
                t_receive.y = wm.self().pos().y;
                return true;
            }
            else if( wm.ball().pos().y < (*it).y && (*it).y < wm.self().pos().y )
            {
                t_receive.y = wm.self().pos().y;
                return true;
            }
        }
    }

    std::vector<rcsc::Vector2D> oppline;
    getOppEndLine( agent, oppline );

    if( oppline.empty() || oppline.size() == 1 )
    {
        t_receive.y = wm.self().pos().y;
        return true;
    }

    if( wm.ball().pos().y > oppline.front().y )
    {
        if( wm.time().cycle() % 2 )
        {
            // 敵前方を目標
            t_receive.y = oppline.front().y;
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: Receive t_receive(%f, %f)"
                                ,__FILE__, __LINE__, t_receive.x, t_receive.y );
            return true;
        }
        else
        {
            // キッカー前方
            t_receive.y = wm.ball().pos().y;
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: Receive t_receive(%f, %f)"
                                ,__FILE__, __LINE__, t_receive.x, t_receive.y );
            return true;
        }
    }
    else if( wm.ball().pos().y < oppline.back().y )
    {
        if( wm.time().cycle() % 2 )
        {
            // 敵前方を目標
            t_receive.y = oppline.back().y;
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: Receive t_receive(%f, %f)"
                                ,__FILE__, __LINE__, t_receive.x, t_receive.y );
            return true;
        }
        else
        {
            // キッカー前方
            t_receive.y = wm.ball().pos().y;
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: Receive t_receive(%f, %f)"
                                ,__FILE__, __LINE__, t_receive.x, t_receive.y );
            return true;
        }
    }
    else
    {
        for( unsigned int i = 0 ; i < oppline.size() - 2 ; i++ )
        {
            if( oppline[i].y > wm.ball().pos().y && wm.ball().pos().y > oppline[i+1].y )
            {
                if( oppline[i].y < wm.self().pos().y && wm.self().pos().y - oppline[i].y < 3.0 )
                {
                    // oppline[i] 前方
                    t_receive.y = oppline[i].y;
                    rcsc::dlog.addText( rcsc::Logger::TEAM,
                                        "%s:%d: Receive t_receive(%f, %f)"
                                        ,__FILE__, __LINE__, t_receive.x, t_receive.y );
                    return true;
                }
                else if( oppline[i+1].y > wm.self().pos().y && oppline[i+1].y - wm.self().pos().y < 3.0 )
                {
                    // oppline[i+1] 前方
                    t_receive.y = oppline[i+1].y;
                    rcsc::dlog.addText( rcsc::Logger::TEAM,
                                        "%s:%d: Receive t_receive(%f, %f)"
                                        ,__FILE__, __LINE__, t_receive.x, t_receive.y );
                    return true;
                }
                else
                {

                    // キッカ前方
                    t_receive.y = wm.ball().pos().y;
                    rcsc::dlog.addText( rcsc::Logger::TEAM,
                                        "%s:%d: Receive t_receive(%f, %f)"
                                        ,__FILE__, __LINE__, t_receive.x, t_receive.y );
                    return true;
                }
            }
        }

        // どれにもひっかからなかったらレシーバ前方
        t_receive.y = wm.self().pos().y;
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: Receive t_receive(%f, %f)"
                            ,__FILE__, __LINE__, t_receive.x, t_receive.y );
        return true;
    }
}

/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/


/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/
/*!
*/
/*-------------------------------------------------------------------*/
