// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_opuci_set_play.h"
#include "opuci_move.h"
#include "opuci_neck.h"
#include "opuci_kick.h"
#include "opuci_search_ball.h"
#include "opuci_chase_ball.h"

#include "strategy.h"

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/bhv_before_kick_off.h>
#include <rcsc/action/bhv_scan_field.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/body_kick_one_step.h>
#include <rcsc/action/body_turn_to_point.h>
#include <rcsc/action/body_pass.h>
#include <rcsc/action/body_clear_ball.h>
#include <rcsc/action/body_advance_ball.h>
#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>
#include <rcsc/player/say_message_builder.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/action/neck_turn_to_low_conf_teammate.h>
#include <rcsc/player/intercept_table.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/line_2d.h>


/*-------------------------------------------------------------------*/
/*!

*/
bool
Bhv_OpuciSetPlay::execute( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: Bhv_OpuciSetPlay"
                        ,__FILE__, __LINE__ );
    
    if( wm.self().goalie() )
    {
        return false;
    }
    agent->debugClient().addMessage( "RposCount%d", wm.ball().rposCount() );
    if( wm.ball().posCount() >= 10 )
    {
        Opuci_SearchBall().execute( agent );
        return true;
    }
    Opuci_Neck().execute( agent );
    if( isKicker( agent ) )
    {
        agent->debugClient().addMessage( "Kicker" );
        doKickerAction( agent );
    }
    else
    {
        agent->debugClient().addMessage( "NotKicker" );
        if( wm.gameMode().type() == rcsc::GameMode::GoalKick_ 
            &&  wm.gameMode().side() == wm.ourSide() )
        {
            if( wm.self().isKickable() )
            {
                rcsc::Vector2D target_pos( 52, 0 );
                const rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromBall().end();
                for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromBall().begin();
                     it != end;
                     ++it )
                {
                    if( !(*it)->goalie()
                        && (*it)->pos().x < wm.self().pos().x )
                    {
                        target_pos = (*it)->pos();
                    }
                }
                agent->debugClient().setTarget( target_pos );
                agent->debugClient().addMessage( "GoalKick" );
                if( Opuci_Kick( target_pos ).execute( agent ) )
                {
                    return true;
                }
                else
                {
                    agent->debugClient().addMessage( "Miss" );
                    return false;
                }
            }
            else
            {
                int self_min = wm.interceptTable()->selfReachCycle();
                int mate_min = wm.interceptTable()->teammateReachCycle();
                int opp_min = wm.interceptTable()->opponentReachCycle();
                if( !wm.existKickableTeammate()
                    && !wm.existKickableOpponent()
                    && self_min <= mate_min )
                {
                    int self_cycle = Opuci_ChaseBall().selfCycle( agent );
                    rcsc::Vector2D target_pos = wm.ball().inertiaPoint( self_cycle );
                    if( Opuci_Move( target_pos, 100, 0.5 ).execute( agent ) )
                    {
                        agent->debugClient().addMessage( "GoToBall" );
                        return true;
                    }
                }
            }

        }

        doPositioning( agent );

    }


    return true;
}


bool
Bhv_OpuciSetPlay::isKicker( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    if( wm.gameMode().side() == wm.theirSide()
        || wm.gameMode().type() == rcsc::GameMode::GoalieCatch_ )
    {
        return false;
    }
    if( wm.teammatesFromBall().empty() )
    {
        return true;
    }
    if( wm.gameMode().type() == rcsc::GameMode::GoalKick_ 
        && ( wm.ball().pos().x > -45 
             || wm.ball().pos().absY() > 11 
             || wm.ball().pos().absY() < 7 ) )
    {
        return false;
    }
    const rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromBall().begin();
         it != end;
         ++it )
    {
        if( !(*it)->goalie()
            && (*it)->distFromBall() < wm.ball().distFromSelf() )
        {
            return false;
        }
    }
    return true;
}

bool
Bhv_OpuciSetPlay::doKickerAction( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
//    Opuci_Neck().execute( agent );
    if( goToBall( agent ) )
    {
        agent->debugClient().addMessage( "MoveToBall" );
        return true;
    }

    const rcsc::PlayerObject * nearest_mate = wm.getTeammateNearestToSelf( 10, false );
    if( nearest_mate )
    {
        rcsc::Vector2D target_pos = nearest_mate->pos();
        const rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromSelf().end();
        for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
             it != end;
             ++it )
        {
            if( !(*it)->goalie()
                && (*it)->pos().x > wm.ball().pos().x )
            {
                target_pos = (*it)->pos();
                break;
            }
        }
        if( Opuci_Kick( target_pos ).execute( agent ) )
        {
            agent->debugClient().addMessage( "KickTo%d", nearest_mate->unum() );
        }
        else
        {
            agent->debugClient().addMessage( "MissKick" );
        }
    }
    else
    {
        agent->debugClient().addMessage( "KickToFace");
        agent->doKick( 100, 0 );
    }

    return true;
}

bool
Bhv_OpuciSetPlay::goToBall( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
    rcsc::Vector2D target_pos = wm.ball().pos();
    rcsc::Vector2D sub_target = target_pos;
    static bool reached_sub = false;
    rcsc::Vector2D vector_dif = wm.ball().pos() - wm.self().pos();
    rcsc::AngleDeg angle_dif = vector_dif.th() - wm.self().body();
    double dist_margin = 0.5;
    double angle_margin = 15;
    double dash_power = 50;
    if( wm.self().stamina() < SP.staminaMax() * SP.effortDecThr() * 1.2
        || wm.self().stamina() < SP.staminaMax() * SP.recoverDecThr() * 1.2 )
    {
        dash_power = 0.0;
    }
    else if( wm.self().stamina() < SP.staminaMax() * 0.8 )
    {
        dash_power = 30;
    }
    agent->debugClient().addMessage( "dist%f", vector_dif.r() );
    agent->debugClient().addMessage( "angle%f", angle_dif.degree() );
    if( vector_dif.r() <= dist_margin
        && angle_dif.abs() <= angle_margin
        && wm.self().isKickable() 
        && reached_sub )
    {
        agent->debugClient().addMessage( "Reached%f", angle_dif.abs() );
        reached_sub = false;
        return false;
    }
    if( wm.lastSetPlayStartTime() == wm.time() )
    {
        agent->debugClient().addMessage( "Reset" );
        reached_sub = false;
    }
    switch( wm.gameMode().type() )
    {
    case rcsc::GameMode::KickIn_:
    case rcsc::GameMode::CornerKick_:
        if( wm.ball().pos().y > 0 )
        {
            sub_target.y += 2.0;
        }
        else
        {
            sub_target.y -= 2.0;
        }
        break;
    case rcsc::GameMode::KickOff_:
    case rcsc::GameMode::GoalKick_:
    case rcsc::GameMode::BackPass_:
    case rcsc::GameMode::IndFreeKick_:
    case rcsc::GameMode::FreeKick_:
    case rcsc::GameMode::OffSide_:
    default:
        sub_target.x -= 2.0;
        break;
    }
    if( !reached_sub )
    {
        if( wm.self().pos().dist( sub_target ) < dist_margin )
        {
            reached_sub = true;
        }
        else
        {
            if( Opuci_Move( sub_target, dash_power, dist_margin ).execute( agent ) )
            {
                agent->debugClient().addMessage( "MoveToSub" );
                return true;
            }
            else
            {
                //can not reach
                reached_sub = true;
            }
        }
    }
    if( reached_sub )
    {
        if( vector_dif.r() > dist_margin )
        {
            if( Opuci_Move( target_pos, dash_power, dist_margin ).execute( agent ) )
            {
                agent->debugClient().addMessage( "DashToBall" );
                return true;
            }
            else if( angle_dif.abs() > angle_margin )
            {
                agent->debugClient().addMessage( "TurnToBall" );
                agent->doTurn( angle_dif );
                return true;
            }
            else
            {
                if( wm.self().isKickable() )
                {
                    agent->debugClient().addMessage( "Finish" );
                    reached_sub = false;
                    return false;
                }
                else
                {
                    agent->debugClient().addMessage( "NotReached" );
//                    agent->doTurn( 0 );
                    return false;
                }
            }
        }
        else if( angle_dif.abs() > angle_margin )
        {
            agent->debugClient().addMessage( "TurnToBall" );
            agent->doTurn( angle_dif );
            return true;
        }
        else
        {
            agent->debugClient().addMessage( "NG1" );
            return false;
        }
    }
    agent->debugClient().addMessage( "NG2" );
    return false;
}

bool
Bhv_OpuciSetPlay::doPositioning( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
//    std::vector< rcsc::Vector2D > candidates;
//    setMovePos( agent, candidates );


    
    rcsc::Vector2D target_pos = getPosition( agent );
    double dash_power = 50;
    if( wm.self().stamina() < SP.staminaMax() * SP.effortDecThr() * 1.2
        || wm.self().stamina() < SP.staminaMax() * SP.recoverDecThr() * 1.2 )
    {
        dash_power = 0.0;
    }
    else if( wm.self().stamina() < SP.staminaMax() * 0.8 )
    {
        dash_power = 30;
    }
//    agent->debugClient().setTarget( target_pos );
    if( Opuci_Move( target_pos, dash_power ).execute( agent ) )
    {
        agent->debugClient().addMessage( "MoveToPosition1" );
        return true;
    }
    else
    {
        agent->debugClient().addMessage( "Reached" );
        rcsc::AngleDeg angle = wm.ball().rpos().th() - wm.self().body();
        agent->doTurn( angle );
        return true;
    }
    
    return true;
}

bool 
Bhv_OpuciSetPlay::setMovePos( rcsc::PlayerAgent * agent, std::vector< rcsc::Vector2D > & candidates )
{
    const rcsc::WorldModel & wm = agent->world();
    rcsc::Vector2D base_pos = wm.ball().pos();
    int reverse = 1;

    if( wm.ball().pos().y < 0 )
    {
        reverse = -1;
        base_pos.y *= reverse;
    }

    rcsc::Vector2D pos1 = base_pos;
    rcsc::Vector2D pos2 = base_pos;
    rcsc::Vector2D pos3 = base_pos;
    rcsc::Vector2D pos4 = base_pos;
    rcsc::Vector2D pos5 = base_pos;
    rcsc::Vector2D pos6 = base_pos;
    rcsc::Vector2D pos7 = base_pos;
    rcsc::Vector2D pos8 = base_pos;
    rcsc::Vector2D pos9 = base_pos;
    rcsc::Vector2D pos10 = base_pos;

    double field_length = 105.0;
//    double field_width = 68.0;

    if( wm.gameMode().side() == wm.ourSide() )
    {
        switch( wm.gameMode().type() )
        {
        case rcsc::GameMode::KickIn_:
            if( base_pos.x < -field_length / 4.0 )
            {
                pos1.x = base_pos.x;
                pos1.y = base_pos.y - 8;

                pos2.x = base_pos.x + 10;
                pos2.y = base_pos.y - 8;

                pos3.x = base_pos.x + 20;
                pos3.y = base_pos.y - 8;

                pos4.x = base_pos.x;
                pos4.y = base_pos.y - 16;
                
                pos5.x = base_pos.x + 10;
                pos5.y = base_pos.y - 16;
                
                pos6.x = base_pos.x + 20;
                pos6.y = base_pos.y - 16;

                pos7.x = base_pos.x;
                pos7.y = base_pos.y - 32;

                pos8.x = base_pos.x + 10;
                pos8.y = base_pos.y - 32;

                pos9.x = base_pos.x + 20;
                pos9.y = base_pos.y - 32;
                
            }
            else if( base_pos.x < field_length / 4.0 )
            {
                
            }
            else
            {
                
            }
            break;
        case rcsc::GameMode::CornerKick_:
            break;
        case rcsc::GameMode::KickOff_:
            break;
        case rcsc::GameMode::BackPass_:
        case rcsc::GameMode::IndFreeKick_:
            break;
        case rcsc::GameMode::GoalKick_:
        case rcsc::GameMode::GoalieCatch_:
            break;
        case rcsc::GameMode::FreeKick_:
        case rcsc::GameMode::OffSide_:
        default:
            break;
        }

        candidates.push_back( pos1 );
        candidates.push_back( pos2 );
        candidates.push_back( pos3 );
        candidates.push_back( pos4 );
        candidates.push_back( pos5 );
        candidates.push_back( pos6 );
        candidates.push_back( pos7 );
        candidates.push_back( pos8 );
        candidates.push_back( pos9 );
        for( int i = 0; i < (int)candidates.size(); i++ )
        {
            candidates.at( i ).y *= reverse;
        }
        agent->debugClient().addCircle( pos1, 1 );
        agent->debugClient().addCircle( pos2, 1 );
        agent->debugClient().addCircle( pos3, 1 );
        agent->debugClient().addCircle( pos4, 1 );
        agent->debugClient().addCircle( pos5, 1 );
        agent->debugClient().addCircle( pos6, 1 );
        agent->debugClient().addCircle( pos7, 1 );
        agent->debugClient().addCircle( pos8, 1 );
        agent->debugClient().addCircle( pos9, 1 );
    }
    else
    {
        switch( wm.gameMode().type() )
        {
        case rcsc::GameMode::KickIn_:
            break;
        case rcsc::GameMode::CornerKick_:
            break;
        case rcsc::GameMode::KickOff_:
            break;
        case rcsc::GameMode::BackPass_:
        case rcsc::GameMode::IndFreeKick_:
            break;
        case rcsc::GameMode::GoalKick_:
        case rcsc::GameMode::GoalieCatch_:
            break;
        case rcsc::GameMode::FreeKick_:
        case rcsc::GameMode::OffSide_:
        default:
            break;
        }    
    }
    return true;
}


rcsc::Vector2D
Bhv_OpuciSetPlay::getPosition( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();


    const rcsc::Vector2D static_pos = Strategy::i().getBeforeKickOffPos( wm.self().unum() );
    rcsc::Vector2D ball_pos = wm.ball().pos();
    rcsc::Vector2D home_pos = static_pos;

    home_pos.x = ball_pos.x / 2.0 + static_pos.x * 2.0 + 30.0;
    home_pos.y = ball_pos.y / 3.0 + static_pos.y * 1.5;

    if( wm.gameMode().type() == rcsc::GameMode::KickOff_ )
    {
        home_pos = static_pos;
    }
    if( wm.gameMode().type() == rcsc::GameMode::IndFreeKick_
        && wm.gameMode().side() == wm.theirSide() )
    {
        if( wm.self().unum() < 6 )
        {
            home_pos.x = -50;
            home_pos.y = static_pos.y / 2.0;
        }
    }
    return home_pos;
}
