// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_set_play.h"

#include "opuci_direct_pass.h"
#include "opuci_direct_pass.h"
#include <rcsc/action/basic_actions.h>
#include <rcsc/action/bhv_before_kick_off.h>
#include <rcsc/action/bhv_scan_field.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/body_kick_one_step.h>
#include <rcsc/action/body_turn_to_point.h>
#include <rcsc/action/body_pass.h>
#include <rcsc/action/body_clear_ball.h>
#include <rcsc/action/body_advance_ball.h>
#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>
#include <rcsc/action/body_smart_kick.h>
#include <rcsc/player/say_message_builder.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/intercept_table.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/line_2d.h>

#include <rcsc/geom/polygon_2d.h>

#include "strategy.h"
#include "soccer_role.h"

bool Bhv_SetPlay::S_kicker_canceled = false;

/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::execute( rcsc::PlayerAgent * agent )
{

    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: Bhv_SetPlay"
                        ,__FILE__, __LINE__ );

    if ( ! agent->world().ball().posValid() )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: invalid ball pos"
                            ,__FILE__, __LINE__ );
        return rcsc::Bhv_ScanField().execute( agent );
    }
    if( agent->world().gameMode().side() != agent->world().ourSide()
        && agent->world().gameMode().type() != rcsc::GameMode::IndFreeKick_ )
    {
        rcsc::Vector2D target_point = M_home_pos;
        if ( agent->world().gameMode().type() == rcsc::GameMode::KickOff_ 
             && target_point.x > -0.5 )
        {
            target_point.x = -0.5;
        }
        doBasicTheirSetPlayMove( agent, target_point );
    }
    else
    {
        switch ( agent->world().gameMode().type() )
        {
        case rcsc::GameMode::KickOff_:
            if( isKicker( agent ) )
            {
                agent->debugClient().addMessage("KickOff");
                return doKickOff( agent );
            }
            else
            {
                agent->debugClient().addMessage("KickOffMove");
                return doKickOffMove( agent );
            }
            break;
        case rcsc::GameMode::KickIn_:
        case rcsc::GameMode::CornerKick_:
            if( isKicker( agent ) )
            {
                agent->debugClient().addMessage("KickIn");
                return doKickIn( agent );
            }
            else
            {
                agent->debugClient().addMessage("KickInMove");
                return doKickInMove( agent );
            }
            break;
        case rcsc::GameMode::GoalKick_:
            if( isKicker( agent ) )
            {
                agent->debugClient().addMessage("GoalKick");
                return doGoalKick( agent );
            }
            else
            {
                agent->debugClient().addMessage("GoalKickMove");
                return doGoalKickMove( agent );
            }
            break;
        case rcsc::GameMode::BackPass_:
        case rcsc::GameMode::IndFreeKick_:
            if( isKicker( agent ) )
            {
                agent->debugClient().addMessage("IndirectFreeKick");
                return doIndirectFreeKick( agent );
            }
            else
            {
                agent->debugClient().addMessage("IndirectFreeKickMove");
                return doIndirectFreeKickMove( agent );
            }
            break;
        case rcsc::GameMode::FreeKick_:
        case rcsc::GameMode::GoalieCatch_: // after catch
        case rcsc::GameMode::OffSide_:
        case rcsc::GameMode::FreeKickFault_:
        case rcsc::GameMode::CatchFault_:
        default:
            if( isKicker( agent ) )
            {
                agent->debugClient().addMessage("FreeKick");
                return doFreeKick( agent );
            }
            else
            {
                agent->debugClient().addMessage("FreeKickMove");
                return doFreeKickMove( agent );
            }
            break;
        }
    }


    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
double
Bhv_SetPlay::get_set_play_dash_power( const rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    if ( wm.gameMode().type() == rcsc::GameMode::PenaltySetup_ )
    {
        return wm.self().getSafetyDashPower( rcsc::ServerParam::i().maxDashPower() );
    }

    double rate;
    if ( wm.self().stamina() > rcsc::ServerParam::i().staminaMax() * 0.8 )
    {
        rate = 1.5
            * wm.self().stamina()
            / rcsc::ServerParam::i().staminaMax();
    }
    else
    {
        rate = 0.9
            * ( wm.self().stamina()
                - rcsc::ServerParam::i().recoverDecThrValue() )
            / rcsc::ServerParam::i().staminaMax();
        rate = std::max( 0.0, rate );
    }

    return ( wm.self().playerType().staminaIncMax()
             * wm.self().recovery()
             * rate );
}

/*-----------------------------------------------------------------------------*
 * recursive function
 *
 *-----------------------------------------------------------------------------*/
namespace {

rcsc::Vector2D
get_avoid_circle_point( const rcsc::WorldModel & world,
                        const rcsc::Vector2D & point,
                        int depth )
{
    if ( depth > 5 )
    {
        return point;
    }
    if( world.gameMode().type() == rcsc::GameMode::BackPass_
        || world.gameMode().type() == rcsc::GameMode::IndFreeKick_ )
    {
        const double circle_r = world.gameMode().type() == rcsc::GameMode::BackPass_
            ? rcsc::ServerParam::i().goalAreaLength() + 0.5
            : rcsc::ServerParam::i().centerCircleR() + 0.5;        
        if ( ( world.self().pos().x > -rcsc::ServerParam::i().pitchHalfLength()
               || world.self().pos().absY() > rcsc::ServerParam::i().goalHalfWidth()
               || world.ball().distFromSelf() < world.self().pos().dist( point ) )
             && ( rcsc::Line2D( world.self().pos(), point ).dist2( world.ball().pos() )
                  < circle_r * circle_r )
             && ( ( world.ball().pos() - point ).th() - ( world.self().pos() - point ).th() ).abs()
             < 90.0
             && ( world.ball().angleFromSelf() - ( point - world.self().pos() ).th() ).abs()
             < 90.0
            )
        {
            rcsc::Vector2D new_point = world.ball().pos();
            rcsc::AngleDeg self2target = ( point - world.self().pos() ).th();
            if ( world.ball().angleFromSelf().isLeftOf( self2target ) )
            {
                new_point += rcsc::Vector2D::polar2vector( 10.5,
                                                           self2target + 90.0 );
            }
            else
            {
                new_point += rcsc::Vector2D::polar2vector( 10.5,
                                                           self2target - 90.0 );
            }
            
            // recursive
            return get_avoid_circle_point( world, new_point, depth + 1 );
        }
        
    }
    else if ( world.ball().distFromSelf() < world.self().pos().dist( point )
         && ( ( world.ball().pos() - point ).th()
              - ( world.self().pos() - point ).th() ).abs() < 90.0
         && ( world.ball().angleFromSelf()
              - ( point - world.self().pos() ).th() ).abs() < 90.0
         && ( rcsc::Line2D( world.self().pos(), point).dist2( world.ball().pos() )
              < 10.0 * 10.0 )
         )
    {
        rcsc::Vector2D new_point = world.ball().pos();
        rcsc::AngleDeg self2target = ( point - world.self().pos() ).th();
        if ( world.ball().angleFromSelf().isLeftOf( self2target ) )
        {
            new_point += rcsc::Vector2D::polar2vector( 11.5,
                                                       self2target + 90.0 );
        }
        else
        {
            new_point += rcsc::Vector2D::polar2vector( 11.5,
                                                       self2target - 90.0 );
        }
        // recursive
        return get_avoid_circle_point( world, new_point, depth + 1 );
    }

    return point;
}

} // end noname namespace

/*-------------------------------------------------------------------*/
/*!
  execute action
*/
void
Bhv_SetPlay::doBasicTheirSetPlayMove( rcsc::PlayerAgent * agent,
                                      const rcsc::Vector2D & target_point )
{
    double dash_power = get_set_play_dash_power( agent );
    rcsc::Vector2D new_target = target_point;

    rcsc::Vector2D ball_to_target = target_point - agent->world().ball().pos();
    double target2ball_dist = ball_to_target.r();
    if ( target2ball_dist < 11.0 )
    {
        new_target.x = agent->world().ball().pos().x
            - std::sqrt( 11.0 * 11.0
                         - ball_to_target.y * ball_to_target.y );
        if ( new_target.x < -45.0 )
        {
            new_target = agent->world().ball().pos();
            new_target += ball_to_target.setLengthVector( 11.0 );
            target2ball_dist = 11.0;
        }
    }

    /*敵をマークする仕組み*/
    const rcsc::WorldModel & wm = agent->world();
    if( wm.self().unum() < 9 )
    {
        if( wm.gameMode().type() != rcsc::GameMode::KickOff_ )
        {
            rcsc::Vector2D opp_pos( 52.5, 0.0 );
            int opp_unum = -1;
            const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
            for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
                 it != opps.end();
                 ++it )
            {
                if( (*it)->pos().dist( M_home_pos ) < opp_pos.dist( M_home_pos ) )

                {
                    opp_pos = (*it)->pos();
                    opp_unum = (*it)->unum();
                }
            }
            //自分に最も近い敵に，自分が最も近い場合，その敵をマーク
            for( int i = 2; i < 9; i++ )
            {
                if( wm.teammate( i )
                    && i != wm.self().unum() )
                {
                    rcsc::Vector2D mate_home_pos = Strategy::i().getPosition( i );
                    if( M_home_pos.dist( opp_pos ) > mate_home_pos.dist( opp_pos )
                        && M_home_pos.dist( wm.ball().pos() ) > mate_home_pos.dist( wm.ball().pos() ) )
                    {
                        opp_unum = -1;
                        break;
                    }
                }
            }
            if( wm.opponent( opp_unum ) )
            {
                rcsc::AngleDeg angle = ( opp_pos - wm.ball().pos() ).th();
                rcsc::Vector2D mark_pos;
                mark_pos.x = opp_pos.x - angle.cos() * 3.0;
                mark_pos.y = opp_pos.y - angle.sin() * 3.0;
                new_target = mark_pos;
                agent->debugClient().addMessage("mark_opp");
            }
        }
    }

    if ( agent->world().self().pos().absY() > rcsc::ServerParam::i().pitchHalfWidth()
         && agent->world().self().pos().x < agent->world().ball().pos().x + 11.0
         && agent->world().ball().pos().x < agent->world().self().pos().x )
    {
        // subtarget may be out of area.
        // at first, player should back to safety area
        new_target = agent->world().ball().pos();
        new_target.x += 12.0;
        new_target.y *= 0.9;
    }
    else
    {
        // recursive search
        new_target = get_avoid_circle_point( agent->world(), new_target, 0 );
    }
    agent->debugClient().setTarget( new_target );

    double dist_thr = agent->world().ball().distFromSelf() * 0.1;
    if ( dist_thr < 0.7 ) dist_thr = 0.7;

    if ( ! rcsc::Body_GoToPoint( new_target,
                                 dist_thr,
                                 dash_power
                                 ).execute( agent ) )
    {
        // already there
        /*
        rcsc::AngleDeg body_angle = agent->world().ball().angleFromSelf();
        if ( body_angle.degree() < 0.0 ) body_angle -= 90.0;
        else body_angle += 90.0;

        rcsc::Vector2D body_point = agent->world().self().pos();
        body_point += rcsc::Vector2D::polar2vector( 10.0, body_angle );

        rcsc::Body_TurnToPoint( body_point ).execute( agent );
        */
        agent->doTurn( 60.0 );
    }
    //agent->setNeckAction( new rcsc::Neck_TurnToBall() );
    agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
    if( agent->world().getTeammateNearestToBall( 100 ) )
    {
        agent->doAttentionto( agent->world().self().side(), 
                              agent->world().getTeammateNearestToBall( 100 )->unum() );
    }
}


/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::isKicker( const rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::PlayerObject * nearest_mate
        = static_cast< rcsc::PlayerObject * >( 0 );
    long max_wait1 = 30;
    long max_wait2 = 50;

/*
    if( wm.gameMode().type() == rcsc::GameMode::CornerKick_ 
        && wm.gameMode().side() == wm.ourSide() )
    {
        if( wm.ball().pos().y > 0 
            && wm.self().unum() == 8 )
        {
            return true;
        }
        else if( wm.ball().pos().y < 0
                 && wm.self().unum() == 7 )
        {
            return true;
        }
        else
        {
            return false;
        }
    } 
*/
    if( wm.gameMode().type() ==  rcsc::GameMode::KickOff_ )
    {
        if( !wm.teammatesFromBall().empty()
            && ( wm.teammatesFromBall().front()->distFromBall()
                 < wm.ball().distFromSelf() ) )
        {
            return false;
        }
        return true;
    }
    else if( wm.gameMode().type() == rcsc::GameMode::KickIn_ )
    {
        if ( wm.teammatesFromBall().empty() )
        {
            return true;
        }

        
        nearest_mate = wm.teammatesFromBall().front();
        
        if ( wm.setplayCount() < max_wait1
             || ( wm.setplayCount() < max_wait2
                  && wm.self().pos().dist( M_home_pos ) > 20.0 )
             || ( nearest_mate
                  && nearest_mate->distFromBall() < wm.ball().distFromSelf() * 0.9 )
            )
        {
            return false;
        }
        return true;
    }
    else if( wm.gameMode().type() == rcsc::GameMode::GoalKick_ )
    {
        if ( wm.setplayCount() < 30 )
        {
            return false;
        }
        if ( wm.teammatesFromBall().empty() )
        {
            return true;
        }
        const rcsc::PlayerPtrCont::const_iterator end
            = wm.teammatesFromBall().end();
        for ( rcsc::PlayerPtrCont::const_iterator it
                  = wm.teammatesFromBall().begin();
              it != end;
              ++it )
        {
            if ( ! (*it)->goalie()
                 && (*it)->distFromBall() < wm.ball().distFromSelf() )
            {
                // exist other kicker
                return false;
            }
        }
        return true;
    }
    else if( wm.gameMode().type() == rcsc::GameMode::BackPass_
             || wm.gameMode().type() == rcsc::GameMode::IndFreeKick_ )
    {

    if ( ( wm.gameMode().type() == rcsc::GameMode::BackPass_
           && wm.gameMode().side() == wm.ourSide() )
         || ( wm.gameMode().type() == rcsc::GameMode::IndFreeKick_
              && wm.gameMode().side() == wm.theirSide() )
         )
    {
        S_kicker_canceled = true;
        return false;
    }

    if ( wm.setplayCount() < 5 )
    {
        if ( wm.time().cycle() < wm.lastSetPlayStartTime().cycle() + 5 )
        {
            S_kicker_canceled = false;
        }
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": isKicker() wait period" );
        return false;
    }

    if ( S_kicker_canceled )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__":  isKicker() kicker canceled" );
        return false;
    }

    nearest_mate = ( wm.teammatesFromBall().empty()
                     ? static_cast< rcsc::PlayerObject * >( 0 )
                     : wm.teammatesFromBall().front() );

    if ( ! nearest_mate )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__":  no nearest teammate" );
        return true;
    }

    if ( nearest_mate->distFromBall() < wm.ball().distFromSelf() * 0.85 )
    {
        return false;
    }

    if ( nearest_mate->distFromBall() < 3.0
         && wm.ball().distFromSelf() < 3.0
         && nearest_mate->distFromBall() < wm.ball().distFromSelf() )
    {
        S_kicker_canceled = true;

        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__":  isKicker() kicker canceled" );
        return false;
    }

    return true;        
    }
    else
    {
        // goalie catch mode && I'm not a goalie
        if ( wm.gameMode().type() == rcsc::GameMode::GoalieCatch_
             && wm.gameMode().side() == wm.ourSide()
             && ! wm.self().goalie() )
        {
            return false;
        }
        
        nearest_mate = wm.teammatesFromBall().front();
        
        if ( wm.setplayCount() < max_wait1
             || ( wm.setplayCount() < max_wait2
                  && wm.self().pos().dist( M_home_pos ) > 20.0 )
             || ( nearest_mate
                  && nearest_mate->distFromBall() < wm.ball().distFromSelf() * 0.9 )
            )
        {
            return false;
        }
        
        return true;
    }

    return false;
}


/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::doKickOffMove( rcsc::PlayerAgent * agent )
{
    rcsc::Vector2D target_pos = M_home_pos;
    target_pos.x = std::min( -0.5, target_pos.x );

    double dash_power
        = get_set_play_dash_power( agent );
    double dist_thr = agent->world().ball().distFromSelf();// * 0.07;
    if ( dist_thr < 1.0 ) dist_thr = 1.0;

    if ( ! rcsc::Body_GoToPoint( target_pos,
                                 dist_thr,
                                 dash_power
                                 ).execute( agent ) )
    {
        // already there
        rcsc::Body_TurnToBall().execute( agent );
    }
    agent->setNeckAction( new rcsc::Neck_ScanField() );    
    return true;
}


/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::doKickOff( rcsc::PlayerAgent * agent )
{
    static int S_scan_count = -5;
    const rcsc::WorldModel & wm = agent->world();
    // go to ball
    agent->debugClient().addMessage( "ssc%d", S_scan_count );
    if( goToStaticBall( agent, 0.0 ) )
    {
        return false;
    }



    // already kick point
    if ( wm.self().body().abs() < 175.0 )
    {
        rcsc::Vector2D face_point( -rcsc::ServerParam::i().pitchHalfLength(),
                                   0.0 );
        rcsc::Body_TurnToPoint( face_point ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return false;
    }

    if ( S_scan_count < 0 )
    {
        int dif = 0;
        if( wm.ourSide() == rcsc::LEFT )
        {
            dif = wm.gameMode().scoreLeft() - wm.gameMode().scoreRight();
        }
        else
        {
            dif = wm.gameMode().scoreRight() - wm.gameMode().scoreLeft();
        }
        
        if( dif == 1 
            && wm.time().cycle() >= 4500 )
        {
            if( wm.time().cycle() % 20 == 0 )
            {
                S_scan_count++;
            }
        }
        else if( dif >= 2
                 && wm.time().cycle() >= 3000 )
        {
            if( wm.time().cycle() % 20 == 0 )
            {
                S_scan_count++;
            }
        }
        else
        {
            S_scan_count++;
        }

        rcsc::Body_TurnToAngle( 180.0 ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return false;
    }

    if ( S_scan_count < 10
         && wm.teammatesFromSelf().empty() )
    {
        S_scan_count++;
        rcsc::Body_TurnToAngle( 180.0 ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return false;
    }

    S_scan_count = -5;

    const double max_kick_speed = wm.self().kickRate() * rcsc::ServerParam::i().maxPower();

    rcsc::Vector2D target_point;
    double ball_speed = 0.0;

    // teammate not found
    if ( wm.teammatesFromSelf().empty() )
    {
        target_point.assign( rcsc::ServerParam::i().pitchHalfLength(),
                             static_cast< double >
                             ( -1 + 2 * wm.time().cycle() % 2 )
                             * 0.8 * rcsc::ServerParam::i().goalHalfWidth() );
        ball_speed = max_kick_speed;
    }
    else
    {
        double dist = wm.teammatesFromSelf().front()->distFromSelf();
        // too far
        if ( dist > 35.0 )
        {
            target_point.assign( rcsc::ServerParam::i().pitchHalfLength(),
                                 static_cast< double >
                                 ( -1 + 2 * wm.time().cycle() % 2 )
                                 * 0.8 * rcsc::ServerParam::i().goalHalfWidth() );
            ball_speed = max_kick_speed;
        }
        else
        {
            target_point = wm.teammatesFromSelf().front()->pos();
            ball_speed = rcsc::calc_first_term_geom_series_last( 1.4,
                                                                 dist,
                                                                 rcsc::ServerParam::i().ballDecay() );
            ball_speed = std::max( ball_speed,
                                   wm.self().playerType().playerSize() + wm.self().playerType().kickableArea() );
            ball_speed = std::min( ball_speed, max_kick_speed );
        }
    }

    rcsc::Body_KickOneStep( target_point,
                            ball_speed
                            ).execute( agent );
    agent->setNeckAction( new rcsc::Neck_ScanField() );
    agent->addSayMessage( new rcsc::BallMessage( agent->effector().queuedNextBallPos(),
                                                 agent->effector().queuedNextBallVel() ) );
    return true;
}


/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::doGoalKickMove( rcsc::PlayerAgent * agent )
{
    if( agent->world().ball().vel().r() < 0.1 )
    {
        double dash_power
            = get_set_play_dash_power( agent );
        double dist_thr = agent->world().ball().distFromSelf() * 0.07;
        if ( dist_thr < 1.0 ) dist_thr = 1.0;

        rcsc::Vector2D move_point = M_home_pos;
        move_point.y += agent->world().ball().pos().y * 0.5;

        if ( ! rcsc::Body_GoToPoint( move_point,
                                     dist_thr,
                                     dash_power
             ).execute( agent ) )
        {
            // already there
            /*
            if( agent->world().ball().posCount() < 3 ){
                rcsc::AngleDeg t_angle( 0.0 );
                rcsc::Body_TurnToAngle( t_angle ).execute( agent );
            }
            else
                rcsc::Body_TurnToBall().execute( agent );
            */
            agent->doTurn( 60.0 );
        }
        //agent->setNeckAction( new rcsc::Neck_ScanField() );
        agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
        if( agent->world().getTeammateNearestToBall( 100 ) )
        {
            agent->doAttentionto( agent->world().self().side(), 
                                  agent->world().getTeammateNearestToBall( 100 )->unum() );
        }
    }
    else
    {
        SoccerRole::Ptr role_ptr = Strategy::i().createRole( agent->world().self().unum(), agent->world() );
        role_ptr->execute( agent );
    }
    return true;
}


/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::doGoalKick( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    static int S_scan_count = -5;


    // go to point to kick the ball
    if ( goToStaticBall( agent, 0.0 ) )
    {
        return false;
    }

    // already kick point
    if ( ( agent->world().ball().angleFromSelf()
           - agent->world().self().body() ).abs() > 3.0 )
    {
        rcsc::Body_TurnToBall().execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return false;
    }

    // wait1
    if ( S_scan_count < 0 )
    {
        int dif = 0;
        if( wm.ourSide() == rcsc::LEFT )
        {
            dif = wm.gameMode().scoreLeft() - wm.gameMode().scoreRight();
        }
        else
        {
            dif = wm.gameMode().scoreRight() - wm.gameMode().scoreLeft();
        }
        
        if( dif == 1 
            && wm.time().cycle() >= 4500 )
        {
            if( wm.time().cycle() % 20 == 0 )
            {
                S_scan_count++;
            }
        }
        else if( dif >= 2
                 && wm.time().cycle() >= 3000 )
        {
            if( wm.time().cycle() % 20 == 0 )
            {
                S_scan_count++;
            }
        }
        else
        {
            S_scan_count++;
        }        
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return false;
    }

    // wait2
    if ( S_scan_count < 40
         && agent->world().teammatesFromSelf().empty() )
    {
        S_scan_count++;
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return false;
    }

    S_scan_count = -5;

    // kick to teammate
    rcsc::Vector2D target_point;

    //check if there is a good segment between opps
    
    // preperation
    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    const rcsc::PlayerPtrCont & mates = wm.teammatesFromSelf();
    const rcsc::PlayerPtrCont::const_iterator mates_end = mates.end();
    const rcsc::PlayerPtrCont::const_iterator opps_end = opps.end();

    // pick up six nearest opponents from you
    std::vector< rcsc::Vector2D > six_opps;
    for( int i = 0 ; i < 6 ; i++ )
    {
        if( (int)opps.size() <= i )
        {
            six_opps.push_back( rcsc::Vector2D( 0, 0 ) );
        }
        else
        {
            six_opps.push_back( opps[i]->pos() );
        }
    }
    // sort six opponents along y-axis in descending order
    rcsc::Vector2D tmp_2D;
    for( int i = 0 ; i < 6 ; i++ )
    {
        for( int j = i ; j < 6 ; j++ )
        {
            if( six_opps[i].y < six_opps[j].y )
            {
                tmp_2D = six_opps[i];
                six_opps[i] = six_opps[j];
                six_opps[j] = tmp_2D;
            }
        }
    }
    for( int i = 0 ; i < 6 ; i++ )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: six_opps[%d](%f, %f)"
                            ,__FILE__, __LINE__,
                            i, six_opps[i].x, six_opps[i].y );
    }
    // find the segment that the projection of you directly hits
    for( int i = 0 ; i < 5 ; i++ )
    {
        rcsc::Segment2D seg( six_opps[i], six_opps[i+1] );
        rcsc::Line2D line = seg.line();
        if( line.dist( six_opps[i] ) == seg.dist( six_opps[i] ) )
        {
            // found the segment. kick the ball to it.
            rcsc::Vector2D target_pos = (seg.origin() + seg.terminal())/2.0;
            // search for the nearest mate to target_pos
            double min_dist = rcsc::ServerParam::i().pitchLength() * 2.0; // very large
            int unum_receiver = -1;
            for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
                 it != mates_end;
                 ++it )
            {
                if( min_dist > target_pos.dist( (*it)->pos() ) )
                {
                    min_dist = target_pos.dist( (*it)->pos() );
                    unum_receiver = (*it)->unum();
                }
            }
            
            // kick action
            double first_speed = std::min( wm.self().kickRate() * rcsc::ServerParam::i().maxPower(),
                                           rcsc::ServerParam::i().ballSpeedMax() );
            double simed_speed = first_speed;
            double travel_dist = 0.0;
            int elapsed_time = 0;
            while( travel_dist < target_pos.dist( wm.ball().pos() ) )
            {
                elapsed_time++;
                travel_dist += simed_speed;
                simed_speed *= rcsc::ServerParam::i().ballDecay();
                if( elapsed_time > 20 )
                    break;
            }
            travel_dist -= simed_speed / rcsc::ServerParam::i().ballDecay();
            if( travel_dist > 10 )
                continue;

            rcsc::Vector2D self2tgt = target_pos - wm.ball().pos();
            self2tgt.setLength( travel_dist );
            target_pos = wm.ball().pos() + self2tgt;
            rcsc::Body_KickOneStep( target_pos, first_speed ).execute( agent );

            if ( agent->config().useCommunication() )
            {
                rcsc::dlog.addText( rcsc::Logger::ACTION,
                              __FILE__": goal kick pass communication to %d", unum_receiver );
                agent->addSayMessage( new rcsc::PassMessage( unum_receiver,
                                                             target_pos,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
            }
            return true;
        }
    }
    

    //direct pass
    for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
         it != mates_end;
         ++it )
    {
        bool target_is_free = true;
        const rcsc::Sector2D sector( wm.ball().pos(),
                                     0.0, (*it)->pos().dist( wm.ball().pos() ) + 5.0,
                                     ( (*it)->pos() - wm.ball().pos() ).th() - rcsc::AngleDeg( 20.0 ),
                                     ( (*it)->pos() - wm.ball().pos() ).th() + rcsc::AngleDeg( 20.0 ) );
        for( rcsc::PlayerPtrCont::const_iterator itr = opps.begin();
             itr != opps.end();
             ++itr )
        {
            if( sector.contains( (*itr)->pos() ) )
            {
                target_is_free = false;
            }
        }

        if( target_is_free
            && (*it)->posCount() < 3
            && (*it)->pos().x > wm.self().pos().x
            && (*it)->pos().dist( wm.self().pos() ) > 10.0 )
        {
            const double ball_decay = rcsc::ServerParam::i().ballDecay();
            const double ball_speed_max = rcsc::ServerParam::i().ballSpeedMax();
            int reach_cycle = (int)(log( 1.0 - ( 1.0 - ball_decay ) * wm.self().pos().dist( (*it)->pos() ) / ball_speed_max ) / log( ball_decay ));
            double ball_speed = (*it)->playerTypePtr()->kickableArea() * 1.5 * pow( 1.0 / ball_decay, reach_cycle - 1 );
            double first_speed = std::min( ball_speed_max, ball_speed );
            target_point = wm.ball().pos() * 0.2 + (*it)->pos() * 0.8;
            if( target_point.x > -35.0
                || target_point.absY() > 20.0 )
            {
                if( rcsc::Body_KickOneStep( target_point,
                                            first_speed ).execute( agent ) )
                {
                    agent->debugClient().addMessage( "DirectPass" );
                    agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(),
                                                                 target_point,
                                                                 agent->effector().queuedNextBallPos(),
                                                                 agent->effector().queuedNextBallVel() ) );
                    agent->setNeckAction( new rcsc::Neck_ScanField() );
                    return true;
                }
            }
        }
    }

    if  ( rcsc::Body_Pass::get_best_pass( agent->world(),
                                          & target_point,
                                          NULL,
                                          NULL )
          && target_point.dist( rcsc::Vector2D(-50.0, 0.0) ) > 20.0
          )
    {
        double opp_dist = 100.0;
        const rcsc::PlayerObject * opp
            = agent->world().getOpponentNearestTo( target_point,
                                                   10,
                                                   & opp_dist );
        if ( ! opp
             || opp_dist > 5.0 )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: pass to (%f, %f)"
                                ,__FILE__, __LINE__,
                                target_point.x, target_point.y );
            rcsc::Body_Pass().execute( agent );
            agent->setNeckAction( new rcsc::Neck_ScanField() );
            return true;
        }
    }

    //advance kick
    double ball_speed_max = rcsc::ServerParam::i().ballSpeedMax();
    double best_angle = 90.0 * wm.ball().pos().y / wm.ball().pos().absY();
    for( double angle = -80.0; angle <= 80.0; angle += 10.0 )
    {
        bool angle_safety = true;
        const rcsc::Sector2D sector( wm.ball().pos(),
                                     0.0, 20.0,
                                     angle - 15.0,
                                     angle + 15.0 );
        for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
             it != opps_end;
             ++it )
        {
            if( sector.contains( (*it)->pos() ) )
            {
                angle_safety = false;
            }
        }
        if( angle_safety
            && fabs( angle ) < fabs( best_angle ) )
        {
            best_angle = angle;
        }
    }
    target_point.x = wm.ball().pos().x + rcsc::AngleDeg( best_angle ).cos();
    target_point.y = wm.ball().pos().y + rcsc::AngleDeg( best_angle ).sin();
    target_point = wm.ball().pos() * 0.2 + target_point * 0.8;
    if( target_point.x > -35.0 )
    {
        if( rcsc::Body_KickOneStep( target_point,
                                    ball_speed_max ).execute( agent ) )
        {
            agent->debugClient().addMessage( "DefKickAdvance:%lf",best_angle );
            agent->setNeckAction( new rcsc::Neck_ScanField() );
            agent->addSayMessage( new rcsc::BallMessage( agent->effector().queuedNextBallPos(),
                                                         agent->effector().queuedNextBallVel() ) );
            return true;
        }
    }

    // clear
    rcsc::Body_ClearBall().execute( agent );
    agent->setNeckAction( new rcsc::Neck_ScanField() );    
    agent->addSayMessage( new rcsc::BallMessage( agent->effector().queuedNextBallPos(),
                                                 agent->effector().queuedNextBallVel() ) );
    return true;
}


/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::doKickInMove( rcsc::PlayerAgent * agent )
{
    rcsc::Vector2D target_point = M_home_pos;
    target_point = getMiddleMovePos( agent );
    double dash_power
        = get_set_play_dash_power( agent );
    double dist_thr = agent->world().ball().distFromSelf() * 0.07;
    if ( dist_thr < 1.0 ) dist_thr = 1.0;

    agent->debugClient().setTarget( target_point );

    if ( ! rcsc::Body_GoToPoint( target_point,
                                 dist_thr,
                                 dash_power
                                 ).execute( agent ) )
    {
        //rcsc::Body_TurnToBall().execute( agent );
        agent->doTurn( 60.0 );
    }
    //agent->setNeckAction( new rcsc::Neck_ScanField() );
    agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
    if( agent->world().getTeammateNearestToBall( 100 ) )
    {
        agent->doAttentionto( agent->world().self().side(), 
                              agent->world().getTeammateNearestToBall( 100 )->unum() );
    }
    return true;
}


/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::doKickIn( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    rcsc::AngleDeg ball_place_angle = ( wm.ball().pos().y > 0.0
                                        ? -90.0
                                        : 90.0 );
    int wait_cycle = 10;

    int dif = 0;
    if( wm.ourSide() == rcsc::LEFT )
    {
        dif = wm.gameMode().scoreLeft() - wm.gameMode().scoreRight();
    }
    else
    {
        dif = wm.gameMode().scoreRight() - wm.gameMode().scoreLeft();
    }
    if( dif == 1 
        && wm.time().cycle() >= 4500 )
    {
        wait_cycle = 100;
    }
    else if( dif >= 2
             && wm.time().cycle() >= 3000 )
    {
        wait_cycle = 100;
    }
    
    if ( prepareSetPlayKick( agent, ball_place_angle, wait_cycle ) )
    {
        // go to kick point
        return false;
    }


    rcsc::Vector2D target_point;
    double ball_speed;
    int receiver = -1;
    target_point = getMiddleKickPos( agent, receiver );
    if ( receiver >= 2 )
    {
        double dist = target_point.dist( wm.ball().pos() );
        ball_speed = getBestSpeed( dist );
        rcsc::Body_KickOneStep( target_point, ball_speed ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        agent->debugClient().setTarget( target_point );
        agent->debugClient().addMessage( "OneStepKick:%d", receiver );
    }
    else if  ( rcsc::Body_Pass::get_best_pass( wm,
                                          & target_point,
                                          & ball_speed,
                                          & receiver )
          && target_point.x > -25.0
          && target_point.x < 48.0 )
    {
        // enforce one step kick
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: pass to (%f, %f)"
                            ,__FILE__, __LINE__,
                            target_point.x, target_point.y );
        rcsc::Body_KickOneStep( target_point,
                                ball_speed
                                ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
    }
    /*
    else if( wm.getTeammateNearestToBall( 5, false ) )
    {
        receiver == wm.getTeammateNearestToBall( 5, false )->unum();
        const rcsc::Circle2D circle( wm.ball().pos(), wm.ball().pos().dist( wm.teammate( receiver )->pos() ) );
        std::vector<rcsc::Vector2D> opp_pos_in_circle;
        const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
        for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
             it != opps.end();
             it++ )
        {
            if( circle.contains( (*it)->pos() ) )
            {
                opp_pos_in_circle.push_back( (*it)->pos() );
            }
        }
        if( opp_pos_in_circle.empty() )
        {
            target_point = wm.teammate( receiver )->pos();
        }
        else
        {
            double dist_opp_nearest_to_mate = wm.ball().pos().dist( wm.teammate( receiver )->pos() );
            rcsc::Vector2D nearest_opp_pos = wm.ball().pos();
            for( std::vector<rcsc::Vector2D>::iterator it = opp_pos_in_circle.begin();
                 it != opp_pos_in_circle.end();
                 it++ )
            {
                if( wm.teammate( receiver )->pos().dist( (*it) ) < dist_opp_nearest_to_mate )
                {
                    nearest_opp_pos = (*it);
                    dist_opp_nearest_to_mate = wm.teammate(receiver )->pos().dist( (*it) );
                }
            }
            const rcsc::AngleDeg angle_to_opp = ( nearest_opp_pos - wm.ball().pos() ).th();
            rcsc::AngleDeg left_angle( angle_to_opp.degree() - 60.0 );
            rcsc::AngleDeg right_angle( angle_to_opp.degree() + 60.0 );
            rcsc::Vector2D left_opp_pos( wm.ball().pos().x + left_angle.cos() * dist_opp_nearest_to_mate,
                                         wm.ball().pos().y + left_angle.sin() * dist_opp_nearest_to_mate );
            rcsc::Vector2D right_opp_pos( wm.ball().pos().x + right_angle.cos() * dist_opp_nearest_to_mate,
                                          wm.ball().pos().y + right_angle.sin() * dist_opp_nearest_to_mate );
            for( std::vector<rcsc::Vector2D>::iterator it = opp_pos_in_circle.begin();
                 it != opp_pos_in_circle.end();
                 it++ )
            {
                rcsc::AngleDeg angle = ( (*it) - wm.ball().pos() ).th();
                if( angle.degree() < angle_to_opp.degree()
                    && angle.degree() > left_angle.degree() )
                {
                    left_angle = angle;
                    left_opp_pos = (*it);
                }
                else if( angle.degree() > angle_to_opp.degree()
                         && angle.degree() < right_angle.degree() )
                {
                    right_angle = angle;
                    right_opp_pos = (*it);
                }
            }
            rcsc::AngleDeg angle_to_target = ( wm.teammate( receiver )->pos() - wm.ball().pos() ).th();
            if( ( angle_to_opp - left_angle ).degree() > ( right_angle - angle_to_opp ).degree() )
            {
                angle_to_target = ( angle_to_opp + left_angle ).degree() / 2.0;
            }
            else
            {
                angle_to_target = ( angle_to_opp + right_angle ).degree() / 2.0;
            }
            rcsc::Line2D target_line( wm.ball().pos(), angle_to_target );
            target_point = target_line.projection( wm.teammate( receiver )->pos() );
            if( target_point.absY() > 33.0 )
            {
                target_point.y = 33.0 * ( wm.ball().pos().y / wm.ball().pos().absY() );
            }
        }
        ball_speed = getBestSpeed( wm.ball().pos().dist( target_point ) );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: pass to (%f, %f)"
                            ,__FILE__, __LINE__,
                            target_point.x, target_point.y );
        rcsc::Body_KickOneStep( target_point,
                                ball_speed
                                ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        agent->debugClient().addMessage("NarimotoKickIn");
        agent->debugClient().setTarget( target_point );
    }
    */
    else if ( wm.self().pos().x < 20.0 )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: advance"
                            ,__FILE__, __LINE__);
        rcsc::Body_AdvanceBall().execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
    }
    else
    {
        rcsc::Vector2D target_point( rcsc::ServerParam::i().pitchHalfLength() - 2.0,
                                     ( rcsc::ServerParam::i().pitchHalfWidth() - 5.0 )
                                     * ( 1.0 - ( wm.self().pos().x
                                                 / rcsc::ServerParam::i().pitchHalfLength() ) ) );
        if ( wm.self().pos().y < 0.0 )
        {
            target_point.y *= -1.0;
        }
        // enforce one step kick
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: advance 2 to (%f, %f)"
                            ,__FILE__, __LINE__,
                            target_point.x, target_point.y );
        rcsc::Body_KickOneStep( target_point,
                                rcsc::ServerParam::i().ballSpeedMax()
                                ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
    }

    if ( agent->config().useCommunication()
         && receiver != rcsc::Unum_Unknown )
    {
       // cout << agent->world().self().unum() << "say" << receiver <<"("<< target_point.x
       //     << ","<< target_point.y << ")" << endl;
       rcsc::dlog.addText( rcsc::Logger::ACTION,
			     "%s:%d: execute() set pass communication.freekick"
       ,__FILE__, __LINE__ );
       rcsc::Vector2D target_buf = target_point - agent->world().self().pos();
       target_buf.setLength( 0.3 );
//       std::cout << wm.time().cycle() << " " << "pass to "
//		 << target_point.x << "," << target_point.y << std::endl;

       agent->addSayMessage( new rcsc::PassMessage( receiver,
                                                    target_point,
                                                    agent->effector().queuedNextBallPos(),
                                                    agent->effector().queuedNextBallVel() ) );
    }
    
    return true;
}


/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::doFreeKickMove( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    rcsc::Vector2D target_point = M_home_pos;
    /*
      if ( wm.gameMode().type() == rcsc::GameMode::GoalieCatch_
      && wm.gameMode().side() == wm.ourSide()
      && wm.self().unum() <= 8 )
      {
      target_point.x += 15.0;
      }
    */

    double dash_power
        = get_set_play_dash_power( agent );
    double dist_thr = wm.ball().distFromSelf() * 0.07;
    if ( dist_thr < 1.0 ) dist_thr = 1.0;
    if ( ! rcsc::Body_GoToPoint( target_point,
                                 dist_thr,
                                 dash_power
                                 ).execute( agent ) )
    {
       // already there
        /*
 	if( agent->world().ball().posCount() < 3 ){
	    rcsc::AngleDeg t_angle( wm.self().body().degree() + rand() );
	    rcsc::Body_TurnToAngle( t_angle ).execute( agent );
	}
	else
	    rcsc::Body_TurnToBall().execute( agent );
        */
        agent->doTurn( 60.0 );
    }
    agent->debugClient().setTarget( target_point );
    //agent->setNeckAction( new rcsc::Neck_ScanField() );
    agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
    if( agent->world().getTeammateNearestToBall( 100 ) )
    {
        agent->doAttentionto( agent->world().self().side(), 
                              agent->world().getTeammateNearestToBall( 100 )->unum() );
    }
    return true;
}


/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::doFreeKick( rcsc::PlayerAgent * agent )
{
    // I am kickaer
    const rcsc::WorldModel & wm = agent->world();
    static int S_scan_count = -5;

    // go to ball
    if ( goToStaticBall( agent, 0.0 ) )
    {
        return false;
    }
    

    // already ball point

    const rcsc::Vector2D face_point( 40.0, 0.0 );
    const rcsc::AngleDeg face_angle
        = ( face_point - wm.self().pos() ).th();

    if ( ( face_angle - wm.self().body() ).abs() > 5.0 )
    {
        rcsc::Body_TurnToPoint( face_point ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return false;
    }

    if ( S_scan_count < 0 )
    {
        int dif = 0;
        if( wm.ourSide() == rcsc::LEFT )
        {
            dif = wm.gameMode().scoreLeft() - wm.gameMode().scoreRight();
        }
        else
        {
            dif = wm.gameMode().scoreRight() - wm.gameMode().scoreLeft();
        }
        
        if( dif == 1 
            && wm.time().cycle() >= 4500 )
        {
            if( wm.time().cycle() % 20 == 0 )
            {
                S_scan_count++;
            }
        }
        else if( dif >= 2
                 && wm.time().cycle() >= 3000 )
        {
            if( wm.time().cycle() % 20 == 0 )
            {
                S_scan_count++;
            }
        }
        else
        {
            S_scan_count++;
        }
        rcsc::Body_TurnToPoint( face_point ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return false;
    }

    if ( S_scan_count < 10
         && wm.teammatesFromSelf().empty() )
    {
        S_scan_count++;
        rcsc::Body_TurnToPoint( face_point ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return false;
    }

    if ( wm.time().stopped() != 0 )
    {
        rcsc::Body_TurnToPoint( face_point ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return false;
    }

    S_scan_count = -5;

    const double max_kick_speed = wm.self().kickRate() * rcsc::ServerParam::i().maxPower();

    rcsc::Vector2D target_point;
    double ball_speed = 0.0;

    std::vector< rcsc::PlayerObject* > cand_mates;
    rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromSelf().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
         it != end;
         ++it )
    {
        if( (*it)->pos().x >= wm.self().pos().x - 5
            && (*it)->pos().dist( wm.self().pos() ) >= 5 
            && (*it)->posCount() <= 3 )
        {
            cand_mates.push_back( (*it) );
        }
    }
    if( !cand_mates.empty() )
    {
        for( std::vector< rcsc::PlayerObject* >::iterator it1 = cand_mates.begin();
             it1 != cand_mates.end();
             ++it1 )
        {
            for( std::vector< rcsc::PlayerObject* >::iterator it2 = it1;
                 it2 != cand_mates.end();
                 ++it2 )
            {
                if( (*it2)->pos().x > (*it1)->pos().x )
                {
                    swap_ranges( it1, it1 + 1, it2 );
                }
            }
        }
        rcsc::Vector2D pass_target;
        double power;
        rcsc::AngleDeg angle;
        for( int i = 0; i < (int)cand_mates.size(); i++ )
        {
            const rcsc::PlayerObject * receiver = cand_mates.at( i );
            if( Opuci_DirectPass().search( agent, receiver, pass_target, power, angle ) )
            {
                agent->debugClient().addMessage( "Target%dPower%dAngle%d", receiver->unum(), (int)power, (int)angle.degree() );
                agent->debugClient().addMessage( "Body%.0f", wm.self().body().degree() );
                agent->debugClient().setTarget( pass_target );
                agent->doKick( power, angle );
                agent->addSayMessage( new rcsc::PassMessage( receiver->unum(), pass_target,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
                agent->setNeckAction( new rcsc::Neck_ScanField() );
                return true;
            }
            
        }
    }    
    

    if ( rcsc::Body_Pass::get_best_pass( wm,
                                         &target_point,
                                         &ball_speed,
                                         NULL )
         && target_point.x > -25.0 )
    {

    }
    else if ( wm.teammatesFromSelf().empty()
              || wm.teammatesFromSelf().front()->distFromSelf() > 35.0
              || wm.teammatesFromSelf().front()->pos().x < -30.0 )
    {
        target_point
            = rcsc::Vector2D( rcsc::ServerParam::i().pitchHalfLength(),
                              static_cast< double >( -1 + 2 * wm.time().cycle() % 2 )
                              * 0.8 * rcsc::ServerParam::i().goalHalfWidth() );
        ball_speed = max_kick_speed;
    }
    else
    {
        double dist = wm.teammatesFromSelf().front()->distFromSelf();
        target_point = wm.teammatesFromSelf().front()->pos();
        ball_speed = rcsc::calc_first_term_geom_series_last( 1.4,
                                                             dist,
                                                             rcsc::ServerParam::i().ballDecay() );
        ball_speed = std::max( ball_speed,
                               wm.self().playerType().playerSize() + wm.self().playerType().kickableArea() );
        ball_speed = std::min( ball_speed, max_kick_speed );
    }

    rcsc::Body_KickOneStep( target_point,
                            ball_speed
                            ).execute( agent );
    agent->setNeckAction( new rcsc::Neck_ScanField() );
    agent->addSayMessage( new rcsc::BallMessage( agent->effector().queuedNextBallPos(),
                                                 agent->effector().queuedNextBallVel() ) );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::doIndirectFreeKickMove( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    const bool our_kick = ( ( wm.gameMode().type() == rcsc::GameMode::BackPass_
                              && wm.gameMode().side() == wm.theirSide() )
                            || ( wm.gameMode().type() == rcsc::GameMode::IndFreeKick_
                                 && wm.gameMode().side() == wm.ourSide() )
                            );

    rcsc::Vector2D target_point = M_home_pos;
    if ( our_kick )
    {
        target_point.x = std::min( wm.offsideLineX() - 1.0, target_point.x );

        double nearest_dist = 1000.0;
        const rcsc::PlayerObject * teammate = wm.getTeammateNearestTo( target_point, 10, &nearest_dist );
        if ( nearest_dist < 2.5 )
        {
            target_point += ( target_point - teammate->pos() ).setLengthVector( 2.5 );
            target_point.x = std::min( wm.offsideLineX() - 1.0, target_point.x );
        }
    }
    else
    {
        target_point = get_avoid_circle_point( wm, M_home_pos, 0 );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__":  their kick adjust target to (%.1f %.1f)->(%.1f %.1f) ",
                            M_home_pos.x, M_home_pos.y,
                            target_point.x, target_point.y );
    }

    double dash_power = wm.self().getSafetyDashPower( rcsc::ServerParam::i().maxDashPower() );

    double dist_thr = wm.ball().distFromSelf() * 0.07;
    if ( dist_thr < 0.5 ) dist_thr = 0.5;

    agent->debugClient().setTarget( target_point );
    agent->debugClient().addCircle( target_point, dist_thr );

    if ( ! rcsc::Body_GoToPoint( target_point,
                                 dist_thr,
                                 dash_power
                                 ).execute( agent ) )
    {
        // already there
        if ( our_kick )
        {
            rcsc::Vector2D turn_point( rcsc::ServerParam::i().pitchHalfLength(), 0.0 );
            rcsc::AngleDeg angle = ( turn_point - wm.self().pos() ).th();
            if ( angle.abs() > 100.0 )
            {
                turn_point = rcsc::Vector2D::polar2vector( 10.0, angle + 100.0 );
                if ( turn_point.x < 0.0 )
                {
                    turn_point.rotate( 180.0 );
                }
                turn_point += wm.self().pos();
            }

            rcsc::Body_TurnToPoint( turn_point ).execute( agent );
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                __FILE__":  our kick. turn to (%.1f %.1f)",
                                turn_point.x, turn_point.y );
        }
        else
        {
            rcsc::Vector2D turn_point = rcsc::Vector2D::polar2vector( 10.0, wm.ball().angleFromSelf() + 90.0 );
            if ( turn_point.x < 0.0 )
            {
                turn_point.rotate( 180.0 );
            }
            turn_point += wm.self().pos();
            rcsc::Body_TurnToPoint( turn_point ).execute( agent );
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                __FILE__":  their kick. turn to (%.1f %.1f)",
                                turn_point.x, turn_point.y );
        }
        S_kicker_canceled = false;
    }

    if ( our_kick
         && ! S_kicker_canceled
         && ( wm.self().pos().dist( target_point )
              > std::max( wm.ball().pos().dist( target_point ) * 0.2, dist_thr ) + 6.0 )
         )
    {
        agent->debugClient().addMessage( "Sayw" );
        agent->addSayMessage( new rcsc::WaitRequestMessage() );
    }

    if ( our_kick )
    {
        agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
    }
    else
    {
        agent->setNeckAction( new rcsc::Neck_TurnToBall() );
    }    
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::doIndirectFreeKick( rcsc::PlayerAgent * agent )
{
    static int S_scan_count = -5;
    const rcsc::WorldModel & wm = agent->world();

    // go to ball
    if ( goToStaticBall( agent, 0.0 ) )
    {
        return false;
    }

    // already ball point

    const rcsc::Vector2D face_point( 50.0, 0.0 );
    const rcsc::AngleDeg face_angle = ( face_point - wm.self().pos() ).th();

    if ( wm.time().stopped() > 0 )
    {
        rcsc::Body_TurnToPoint( face_point ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return false;
    }

    if ( ( face_angle - wm.self().body() ).abs() > 5.0 )
    {
        rcsc::Body_TurnToPoint( face_point ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return false;
    }

    if ( S_scan_count < 0 )
    {
        int dif = 0;
        if( wm.ourSide() == rcsc::LEFT )
        {
            dif = wm.gameMode().scoreLeft() - wm.gameMode().scoreRight();
        }
        else
        {
            dif = wm.gameMode().scoreRight() - wm.gameMode().scoreLeft();
        }
        
        if( dif == 1 
            && wm.time().cycle() >= 4500 )
        {
            if( wm.time().cycle() % 20 == 0 )
            {
                S_scan_count++;
            }
        }
        else if( dif >= 2
                 && wm.time().cycle() >= 3000 )
        {
            if( wm.time().cycle() % 20 == 0 )
            {
                S_scan_count++;
            }
        }
        else
        {
            S_scan_count++;
        }
        rcsc::Body_TurnToPoint( face_point ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return false;
    }

    if ( S_scan_count < 10
         && wm.teammatesFromSelf().empty() )
    {
        ++S_scan_count;
        rcsc::Body_TurnToPoint( face_point ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return false;
    }

    if ( wm.time().stopped() != 0 )
    {
        rcsc::Body_TurnToPoint( face_point ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        return false;
    }

    S_scan_count = -5;


    const double max_kick_speed = wm.self().kickRate() * rcsc::ServerParam::i().maxPower();

    rcsc::Vector2D target_point;
    double ball_speed = 0.0;

    rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromSelf().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
         it != end;
         ++it )
    {
        if( (*it)->pos().x > wm.self().pos().x - 5 )
        {
            if( Opuci_DirectPass().search( agent, (*it), target_point, ball_speed ) )
            {
                agent->debugClient().addMessage( "Target%d", (*it)->unum() );
                agent->debugClient().setTarget( target_point );
                if( rcsc::Body_SmartKick( target_point, ball_speed, ball_speed * 0.8, 1 ).execute( agent ) )
                {
                    agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(), target_point,
                                                                 agent->effector().queuedNextBallPos(),
                                                                 agent->effector().queuedNextBallVel() ) );
                }
                agent->setNeckAction( new rcsc::Neck_ScanField() );
                return true;
            }    
        }
    }

    
    if  ( rcsc::Body_Pass::get_best_pass( wm,
                                          &target_point,
                                          &ball_speed,
                                          NULL )
          && target_point.x > 35.0
          && target_point.x > wm.self().pos().x - 1.0 )
    {
        ball_speed = std::min( ball_speed, max_kick_speed );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__":  pass to (%.1f %.1f) speed=%.2f",
                            target_point.x, target_point.y,
                            ball_speed );
    }
    else if ( wm.teammatesFromSelf().empty()
              || wm.teammatesFromSelf().front()->distFromSelf() > 35.0
              || wm.teammatesFromSelf().front()->pos().x < -30.0 )
    {
        target_point
            = rcsc::Vector2D( rcsc::ServerParam::i().pitchHalfLength(),
                              static_cast< double >( -1 + 2 * wm.time().cycle() % 2 )
                              * ( rcsc::ServerParam::i().goalHalfWidth() - 0.8 ) );
        ball_speed = max_kick_speed;
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__":  kick to goal (%.1f %.1f) speed=%.2f",
                            target_point.x, target_point.y,
                            ball_speed );
    }
    else
    {
        const rcsc::Vector2D goal( rcsc::ServerParam::i().pitchHalfLength(),
                                   wm.self().pos().y * 0.8 );

        double min_dist = 100000.0;
        const rcsc::PlayerObject * receiver = static_cast< const rcsc::PlayerObject * >( 0 );

        const rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromBall().end();
        for ( rcsc::PlayerPtrCont::const_iterator t = wm.teammatesFromBall().begin();
              t != end;
              ++t )
        {
            if ( (*t)->posCount() > 5 ) continue;
            if ( (*t)->distFromBall() < 1.5 ) continue;
            if ( (*t)->distFromBall() > 20.0 ) continue;
            if ( (*t)->pos().x > wm.offsideLineX() ) continue;

            double dist = (*t)->pos().dist( goal ) + (*t)->distFromBall();
            if ( dist < min_dist )
            {
                min_dist = dist;
                receiver = (*t);
            }
        }

        double target_dist = 10.0;
        if ( ! receiver )
        {
            target_dist = wm.teammatesFromSelf().front()->distFromSelf();
            target_point = wm.teammatesFromSelf().front()->pos();
        }
        else
        {
            target_dist = receiver->distFromSelf();
            target_point = receiver->pos();
            target_point.x += 0.6;
        }
        ball_speed = rcsc::calc_first_term_geom_series_last( 1.8, // end speed
                                                             target_dist,
                                                             rcsc::ServerParam::i().ballDecay() );
        ball_speed = std::min( ball_speed, max_kick_speed );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__":  pass to nearest teammate (%.1f %.1f) speed=%.2f",
                            target_point.x, target_point.y,
                            ball_speed );
    }

    agent->debugClient().setTarget( target_point );

    rcsc::Body_KickOneStep( target_point, ball_speed ).execute( agent );

    agent->setNeckAction( new rcsc::Neck_ScanField() );
    agent->addSayMessage( new rcsc::BallMessage( agent->effector().queuedNextBallPos(),
                                                 agent->effector().queuedNextBallVel() ) );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::goToStaticBall( rcsc::PlayerAgent * agent, rcsc::AngleDeg ball_place_angle )
{
    const rcsc::WorldModel & wm = agent->world();
    const double dir_margin = 15.0;
    rcsc::AngleDeg angle_diff = ( wm.ball().angleFromSelf()
                                  - ball_place_angle );
    if( angle_diff.abs() < dir_margin
        && ( wm.ball().distFromSelf()
             < ( wm.self().playerType().playerSize()
                 + rcsc::ServerParam::i().ballSize()
                 + 0.08 ) )
        )
    {
        //already reach
        return false;
    }

    //decide sub-target point
    rcsc::Vector2D sub_target
        = wm.ball().pos()
        + rcsc::Vector2D::polar2vector( 2.3, ball_place_angle + 180.0 );
    
    double dash_power = 20.0;
    if( wm.ball().distFromSelf() > 2.0 )
    {
        dash_power = get_set_play_dash_power( agent );
    }
    
    //it is necessary to go to sub target point
    if( angle_diff.abs() > dir_margin )
    {
        rcsc::Body_GoToPoint( sub_target,
                              0.1,
                              dash_power,
                              3
            ).execute( agent );
    }
    //dir diff is small. go to ball
    else
    {
        //body dir is not right
        if( ( wm.ball().angleFromSelf() - wm.self().body() ).abs() > 5.0 )
        {
            rcsc::Body_TurnToBall().execute( agent );
        }
        //dash to ball
        else
        {
            agent->doDash( dash_power );
        }
    }
    agent->setNeckAction( new rcsc::Neck_ScanField() );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::prepareSetPlayKick( rcsc::PlayerAgent * agent, rcsc::AngleDeg ball_place_angle, int wait_cycle )
{
    static int S_rest_wait_cycle = -1;

    // not reach the ball side
    if ( goToStaticBall( agent, ball_place_angle ) )
    {
        return true;
    }
    agent->debugClient().addMessage( "ws%d", S_rest_wait_cycle );
    // reach to ball side

    if ( S_rest_wait_cycle < 0 )
    {
        S_rest_wait_cycle = wait_cycle;
    }

    if ( S_rest_wait_cycle == 0 )
    {
        if ( agent->world().self().stamina() < rcsc::ServerParam::i().staminaMax() * 0.9
             || agent->world().seeTime() != agent->world().time() )
        {
            S_rest_wait_cycle = 1;
        }
    }

    if ( S_rest_wait_cycle > 0 )
    {
        if ( agent->world().gameMode().type() == rcsc::GameMode::KickOff_ )
        {
            rcsc::AngleDeg moment( rcsc::ServerParam::i().visibleAngle() );
            agent->doTurn( moment );
        }
        else
        {
            rcsc::Body_TurnToBall().execute( agent );
        }
        agent->setNeckAction( new rcsc::Neck_ScanField() );
        S_rest_wait_cycle--;
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: wait. rest cycles=%d"
                            ,__FILE__, __LINE__,
                            S_rest_wait_cycle );
        return true;
    }

    S_rest_wait_cycle = -1;

    return false;    
}

/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::doCornerKickMove( rcsc::PlayerAgent * agent )
{
    rcsc::Vector2D target_point = M_home_pos;
    const rcsc::WorldModel & wm = agent->world();


    double thrX = 35;

    if( wm.ball().pos().x >= thrX 
        && wm.self().unum() >= 7 
        && wm.gameMode().side() == wm.ourSide() )
    {
       target_point = getMovePos( agent );
    }    


    double dash_power = get_set_play_dash_power( agent );
    double dist_thr = agent->world().ball().distFromSelf() * 0.07;
    if( dist_thr < 0.5 )
    {
        dist_thr = 0.5;
    }

    agent->debugClient().addMessage( "KickInMove" );
    agent->debugClient().setTarget( target_point );
    
    if ( ! rcsc::Body_GoToPoint( target_point,
                                 dist_thr,
                                 dash_power
                                 ).execute( agent ) )
    {
        agent->debugClient().addMessage( "KickInTurn" );
        if( wm.self().unum() >= 7 
            && wm.ball().pos().x >= thrX )
        {
            agent->doTurn( 10 );
            agent->setNeckAction( new rcsc::Neck_ScanField() );
            return true;
        }
        else
        {
            if( wm.ball().pos().x > wm.self().pos().x )
            {
                rcsc::Vector2D front( wm.self().pos().x + 3, wm.self().pos().y );
                rcsc::Body_TurnToPoint( front ).execute( agent );
                agent->setNeckAction( new rcsc::Neck_ScanField() );
                return true;
            }
            rcsc::Body_TurnToBall().execute( agent );
            agent->setNeckAction( new rcsc::Neck_ScanField() );
            return true;
        }

//       rcsc::Body_TurnToAngle( wm.self().body() + 30 );

    }

    agent->setNeckAction( new rcsc::Neck_ScanField() );    
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

bool
Bhv_SetPlay::doCornerKick( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    rcsc::AngleDeg ball_place_angle = ( wm.ball().pos().y > 0.0
                                        ? -90.0
                                        : 90.0 );
    double thrX = 35;
    if ( prepareSetPlayKick( agent, ball_place_angle, 20 ) )
    {
        agent->debugClient().addMessage( "MoveToBall" );
//        reachedBase = false;
        // go to kick point
        return true;
    }


    rcsc::Vector2D target_point;
    double ball_speed;
    int receiver = rcsc::Unum_Unknown;

    if( wm.ball().pos().x >= thrX 
        && wm.self().unum() >= 7 )
    {
       target_point = getKickPos( agent, receiver );
       double dist = target_point.dist( wm.ball().pos() );
       ball_speed = getBestSpeed( dist );
       rcsc::Body_KickOneStep( target_point, ball_speed ).execute( agent );
       agent->setNeckAction( new rcsc::Neck_ScanField() );

    }
    if ( agent->config().useCommunication()
         && receiver != rcsc::Unum_Unknown )
    {
       // cout << agent->world().self().unum() << "say" << receiver <<"("<< target_point.x
       //     << ","<< target_point.y << ")" << endl;
       rcsc::dlog.addText( rcsc::Logger::ACTION,
			     "%s:%d: execute() set pass communication.freekick"
       ,__FILE__, __LINE__ );
       rcsc::Vector2D target_buf = target_point - agent->world().self().pos();
       target_buf.setLength( 0.3 );
//       std::cout << wm.time().cycle() << " " << "pass to "
//		 << target_point.x << "," << target_point.y << std::endl;

       agent->addSayMessage( new rcsc::PassMessage( receiver,
                                                    target_point,
                                                    agent->effector().queuedNextBallPos(),
                                                    agent->effector().queuedNextBallVel() ) );
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

rcsc::Vector2D
Bhv_SetPlay::getMovePos( rcsc::PlayerAgent * agent )
{
   const rcsc::WorldModel & wm = agent->world();
   rcsc::Vector2D target_pos;
   rcsc::Vector2D base_pos = M_home_pos;
   double thrX = 35;
   static bool reachedBase = false;
   static int start = 0;
   if( wm.lastSetPlayStartTime().cycle() > start )
   {
      start = wm.lastSetPlayStartTime().cycle();
      reachedBase = false;
   }
   int reverse = 1;
   if( wm.ball().pos().y < 0 )
   {
      reverse = -1;
   }
   double length = 10.0;
   base_pos.x = wm.ball().pos().x;
   base_pos.x += length;
   if( base_pos.x > 52 )
   {
      base_pos.x = 51;
   }
   switch( wm.self().unum() )
   {
      case 7:
	 base_pos.y = 0;
	 base_pos.x -= 7.0;
	 break;
      case 8:
	 base_pos.y = 0;
	 base_pos.x -= 7.0;
	 break;
      case 9:
	 if( reverse < 0 )
	 {
	    base_pos.y = 20;
	 }
	 else
	 {
	    base_pos.y = 7;
	 }
	 base_pos.x -= 3.0;
	 break;
      case 10:
	 if( reverse < 0 )
	 {
	    base_pos.y = 7;
	 }
	 else
	 {
	    base_pos.y = 20;
	 }
	 base_pos.x -= 3.0;
	 break;
      case 11:
	 base_pos.y = 10;
	 base_pos.x -= 2.0;
	 break;
   }
   base_pos.y *= reverse;
   double yoko = 9;
   double tate = 9;
   if( wm.self().unum() == 10
       && reverse > 0 )
   {
      tate = 15;
   }
   if( wm.self().unum() == 9
       && reverse < 0 )
   {
      tate = 15;
   }
   rcsc::Vector2D top_left( base_pos.x - yoko, base_pos.y - tate );
   rcsc::Vector2D bottom_right( base_pos.x + yoko, base_pos.y + tate );
   rcsc::Rect2D self_range( top_left, bottom_right );

   target_pos = base_pos;

   if( wm.self().pos().dist( base_pos ) < 1.5 )
   {
      reachedBase = true;
   }


///////////////////////////////////////////////
   const rcsc::PlayerObject * passer = NULL;
   
   if( !passer )
   {
      const rcsc::PlayerPtrCont::const_iterator end2 = wm.teammatesFromSelf().end();
      for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
	   it != end2;
	   ++it )
      {
	 if( (*it)->pos().absY() > wm.self().pos().absY()
	 && (*it)->pos().x >= thrX
	 && (*it)->unum() >= 7 )
	 {
	    if( (*it)->pos().absY() > base_pos.absY() )
	    {
	       passer = (*it);
	       break;
	    }
	    else if( wm.ball().pos().absY() <= base_pos.absY() )
	    {
	       passer = (*it);
	       break;
	    }
	 }
      }
   }
///////////////////////////////////////////////////////////////
   if( reachedBase )
   {
      const rcsc::PlayerObject * target_opp1 = NULL;
      const rcsc::PlayerObject * target_opp2 = NULL;
      const rcsc::PlayerObject * target_opp3 = NULL;

      const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromSelf().end();
      for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromSelf().begin();
	   it != end;
	   ++it )
      {
	 if( !target_opp1 )
	 {
	    target_opp1 = (*it);
	 }
	 else if( !target_opp2 )
	 {
	    target_opp2 = (*it);
	 }
	 else if( !target_opp3 )
	 {
	    target_opp3 = (*it);
	    break;
	 }
      }

      rcsc::Vector2D opponent_pos = base_pos;

      if( target_opp1 )
      {
	 double length = 5.0;
	 opponent_pos = target_opp1->pos();
	 if( self_range.contains( opponent_pos ) )
	 {
	    if( opponent_pos.x >= target_pos.x - length )
	    {
	       target_pos.x = opponent_pos.x + length;
	    }
	    if( opponent_pos.absY() >= target_pos.absY() - 3 )
	    {
	       target_pos.y = opponent_pos.y + 3.0 * reverse;
	    }
	 }
	 
	 if( target_opp2 )
	 {

	    opponent_pos = target_opp2->pos();
	    if( self_range.contains( opponent_pos ) )
	    {
	       if( opponent_pos.absY() > target_pos.absY() - 3  )
	       {
		  target_pos.y = opponent_pos.y + 3.0 * reverse;
		  if( opponent_pos.x >= target_pos.x - length )
		  {
		     target_pos.x = opponent_pos.x + length;
		  }
	       }
	    }
	    if( target_opp3 )
	    {

	       opponent_pos = target_opp2->pos();
	       if( self_range.contains( opponent_pos ) )
	       {
		  if( opponent_pos.absY() > target_pos.absY() - 3 )
		  {
		     target_pos.y = opponent_pos.y + 3.0 * reverse;
		     if( opponent_pos.x >= target_pos.x - length )
		     {
			target_pos.x = opponent_pos.x + length;
		     }
		  }
	       }
	    }//end if target_opp3
	 }//end if target_opp2
      }//end if target_opp1
   }//end if reachedBase





/*
   if( target_pos.x > wm.offsideLineX() )
   {
	 target_pos.x = wm.offsideLineX() - 0.5;
   }
*/
   


   double defLine = 0;
   const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromSelf().end();
   for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromSelf().begin();
	it != end;
	++it )
   {
      if( !(*it)->goalie() && (*it)->pos().x > defLine )
      {
	 defLine = (*it)->pos().x;
      }
   }
   if( passer )
   {
      if( reachedBase )
      {
	 if( passer->pos().absY() <= 32 
	 && passer->pos().x > defLine )
	 {
	    if( passer->pos().x < target_pos.x + 1.5 )
	    {
	       target_pos.x = passer->pos().x - 1.5;
	    }
	 }
	 if( passer->pos().dist( target_pos ) < 7 )
	 {
	    target_pos.y = passer->pos().y - 7 * reverse;
	 }

	 if( reverse < 0 )
	 {
	    if( wm.self().unum() == 10 )
	    {
	       target_pos.x = defLine - 5;
	    }
	    else if( wm.self().unum() == 8 )
	    {
	       if( passer->pos().x > defLine )
	       {
		  target_pos.x = passer->pos().x - 1;
	       }
	       else
	       {
		  target_pos.x = defLine - 1;
	       }
	    }
	 }
	 else
	 {
	    if( wm.self().unum() == 9 )
	    {
	       target_pos.x = defLine - 5;
	    }
	    else if( wm.self().unum() == 7 )
	    {
	       if( passer->pos().x > defLine )
	       {
		  target_pos.x = passer->pos().x - 1;
	       }
	       else
	       {
		  target_pos.x = defLine - 1;
	       }
	    }
	 }
      }//end if reachedBase
   }//end if passer

   if( target_pos.x > 50.5 )
   {
      target_pos.x = 50.5;
   }
   if( target_pos.y > 32 )
   {
      target_pos.y = 32;
   }
   if( target_pos.y < -32 )
   {
      target_pos.y = -32;
   }
   return target_pos;
}

/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

rcsc::Vector2D
Bhv_SetPlay::getKickPos( const rcsc::PlayerAgent * agent, int & receiver )
{
   const rcsc::WorldModel & wm = agent->world();
   rcsc::Vector2D target_pos;
   double thrX = 35;
   const rcsc::PlayerObject * target_mate = NULL;
   const rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromBall().end();
   for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromBall().begin();
	it != end;
	++it )
   {
      if( (*it)->pos().absY() < wm.self().pos().absY()
          && (*it)->pos().x >= thrX
          && (*it)->distFromSelf() < 20 )
      {
	 target_mate = (*it);
	 break;
      }
   }
   if( !target_mate )
   {
      target_mate = wm.getTeammateNearestToSelf( 5, false );
   }

   if( target_mate )
   {
      receiver = target_mate->unum();
      rcsc::Vector2D tmp_target = target_mate->pos();
      bool found = false;
      std::vector< rcsc::Vector2D > candidates;
      std::vector< bool > eval;
      setCandidates( tmp_target, candidates );
      
      for( unsigned int i = 0; i < candidates.size(); i++ )
      {
	 std::vector< rcsc::Vector2D > vertices;
	 double tmpY = 0;
	 if( wm.self().pos().y > 0 )
	 {
	    tmpY = candidates.at( i ).y;
	 }
	 else
	 {
	    tmpY = candidates.at( i ).y;
	 }
	 vertices.push_back( rcsc::Vector2D( candidates.at( i ).x + 1, tmpY ) );
	 vertices.push_back( rcsc::Vector2D( candidates.at( i ).x - 1, tmpY ) );
	 vertices.push_back( wm.self().pos() );
	 rcsc::Polygon2D polygon( vertices );
	 if( !wm.existOpponentIn( polygon, 3, true )
	     && candidates.at( i ).x < 51 )
	 {
	    found = true;
	    eval.push_back( true );
	 }
	 else
	 {
	    eval.push_back( false );
	 }
      }
      if( !found )
      {
	 target_pos = tmp_target;
      }
      else
      {
	 double dist = 0;
	 double max = 0;
	 int best = 0;
	 for( unsigned int i = 0; i < eval.size(); i++ )
	 {
	    if( eval.at( i ) )
	    {

	       const rcsc::PlayerObject * target_opp = 
		  wm.getOpponentNearestTo( candidates.at( i ), 3, & dist );
	       if( target_opp )
	       {
		  if( max < dist )
		  {
		     max = dist;
		     best = i;
		  }
	       }
	       else
	       {
		  best = i;
		  dist = 100;
	       }
	    }
	 }
	 target_pos = candidates.at( best );
      }
   }
   else
   {
      target_pos = rcsc::Vector2D( 50, 0 );
   }
   return target_pos;
}

/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

double
Bhv_SetPlay::getBestSpeed( double dist )
{
   double speed = rcsc::ServerParam::i().ballSpeedMax();
   int cycle = 0;
   while( cycle < 10 )
   {
      cycle++;
      double sum = rcsc::calc_sum_geom_series( speed, rcsc::ServerParam::i().ballDecay(), cycle );
      if( sum >= dist )
      {
	 speed = rcsc::calc_first_term_geom_series( dist, rcsc::ServerParam::i().ballDecay(), cycle + 1 );
	 break;
      }
   }
   
   return speed;
}

/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

void
Bhv_SetPlay::setCandidates( rcsc::Vector2D base_pos, std::vector< rcsc::Vector2D > & candidates )
{
   candidates.clear();
   candidates.push_back( base_pos );
   
   rcsc::Vector2D cand1 = base_pos;
   rcsc::Vector2D cand2 = base_pos;
   rcsc::Vector2D cand3 = base_pos;
   rcsc::Vector2D cand4 = base_pos;
   rcsc::Vector2D cand5 = base_pos;
   rcsc::Vector2D cand6 = base_pos;
   rcsc::Vector2D cand7 = base_pos;
   rcsc::Vector2D cand8 = base_pos;

   double radius = 1.5;
   
   cand1.x += radius;

   cand2.x += radius * cos( M_PI_4 );
   cand2.y += radius * sin( M_PI_4 );

   cand3.y += radius;

   cand4.x += radius * cos( M_PI_4 * 3 );
   cand4.y += radius * sin( M_PI_4 * 3 );

   cand5.x -= radius;

   cand6.x += radius * cos( M_PI_4 * 5 );
   cand6.y += radius * sin( M_PI_4 * 5 );

   cand7.y -= radius;

   cand8.x += radius * cos( M_PI_4 * 7 );
   cand8.y += radius * sin( M_PI_4 * 7 );

   candidates.push_back( cand1 );
   candidates.push_back( cand2 );
   candidates.push_back( cand3 );
   candidates.push_back( cand4 );
   candidates.push_back( cand5 );
   candidates.push_back( cand6 );
   candidates.push_back( cand7 );
   candidates.push_back( cand8 );
   

   for( unsigned int i = 0; i < candidates.size(); i++ )
   {

      if( candidates.at( i ).x > 50.2 )
      {
	 candidates.at( i ).x = 50.5;
      }
      if( candidates.at( i ).y > 32 )
      {
	 candidates.at( i ).y = 32;
      }
      if( candidates.at( i ).y < -32 )
      {
	 candidates.at( i ).y = -32;
      }
   }
   return;
}

/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

rcsc::Vector2D
Bhv_SetPlay::getMiddleKickPos( const rcsc::PlayerAgent * agent, int & receiver )
{
   rcsc::Vector2D target_pos( 0, 0 );
   const rcsc::WorldModel & wm = agent->world();
   const rcsc::PlayerObject * target_mate1 = NULL;
   const rcsc::PlayerObject * target_mate2 = NULL;
   const rcsc::PlayerObject * target_mate3 = NULL;
   const rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromSelf().end();
   for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
	it != end;
	++it )
   {
      if( !target_mate1
	  && (*it)->pos().absY() < wm.self().pos().absY()
          && (*it)->distFromSelf() < 20 )
      {
	 target_mate1 = (*it);
	 continue;
      }
      else if( !target_mate2
               && (*it)->pos().absY() < wm.self().pos().absY()
               && (*it)->distFromSelf() < 20 )
      {
	 target_mate2 = (*it);
	 continue;
      }
      else if( !target_mate3
               && (*it)->pos().absY() < wm.self().pos().absY()
               && (*it)->distFromSelf() < 20 )
      {
	 target_mate3 = (*it);
	 break;
      }
   }
   if( !target_mate1 )
   {
      target_mate1 = wm.getTeammateNearestToSelf( 5, false );
   }
   if( target_mate1 )
   {
      rcsc::Vector2D tmp_target = target_mate1->pos();
      std::vector< rcsc::Vector2D > candidates1;
      std::vector< bool > eval1;
      setCandidates( tmp_target, candidates1 );
      for( unsigned int i = 0; i < candidates1.size(); i++ )
      {
	 std::vector< rcsc::Vector2D > vertices;
	 double tmpY = 0;
	 if( wm.self().pos().y > 0 )
	 {
	    tmpY = candidates1.at( i ).y;
	 }
	 else
	 {
	    tmpY = candidates1.at( i ).y;
	 }
	 vertices.push_back( rcsc::Vector2D( candidates1.at( i ).x + 1, tmpY ) );
	 vertices.push_back( rcsc::Vector2D( candidates1.at( i ).x - 1, tmpY ) );
	 vertices.push_back( wm.self().pos() );
	 rcsc::Polygon2D polygon( vertices );
	 if( !wm.existOpponentIn( polygon, 3, true )
	     && candidates1.at( i ).x < 51 )
	 {
	    eval1.push_back( true );
	 }
	 else
	 {
	    eval1.push_back( false );
	 }
      }
      double dist = 0;
      double max = 0;
      int best = 0;
      double x_max = -50;
      for( unsigned int i = 0; i < eval1.size(); i++ )
      {
	 if( eval1.at( i ) )
	 {
	    const rcsc::PlayerObject * target_opp = 
	       wm.getOpponentNearestTo( candidates1.at( i ), 3, & dist );
	    if( target_opp )
	    {
	       if( max < dist )
	       {
		  max = dist;
		  best = i;
	       }
	    }
	    else
	    {
	       best = i;
	       dist = 100;
	    }
	 }
      }
      target_pos = candidates1.at( best );
      x_max = candidates1.at( best ).x;
      receiver = target_mate1->unum();

/////////////////////////////////////
      if( target_mate2 )
      {
	 tmp_target = target_mate2->pos();
	 std::vector< rcsc::Vector2D > candidates2;
	 std::vector< bool > eval2;
	 setCandidates( tmp_target, candidates2 );
	 for( unsigned int i = 0; i < candidates2.size(); i++ )
	 {
	    std::vector< rcsc::Vector2D > vertices;
	    double tmpY = 0;
	    if( wm.self().pos().y > 0 )
	    {
	       tmpY = candidates2.at( i ).y;
	    }
	    else
	    {
	       tmpY = candidates2.at( i ).y;
	    }
	    vertices.push_back( rcsc::Vector2D( candidates2.at( i ).x + 1, tmpY ) );
	    vertices.push_back( rcsc::Vector2D( candidates2.at( i ).x - 1, tmpY ) );
	    vertices.push_back( wm.self().pos() );
	    rcsc::Polygon2D polygon( vertices );
	    if( !wm.existOpponentIn( polygon, 3, true )
	        && candidates2.at( i ).x < 51 )
	    {
	       eval2.push_back( true );
	    }
	    else
	    {
	       eval2.push_back( false );
	    }
	 }
	 double dist2 = 0;
	 double max2 = 0;
	 double best2 = 0;
	 for( unsigned int i = 0; i < eval2.size(); i++ )
	 {
	    if( eval2.at( i ) )
	    {
	       const rcsc::PlayerObject * target_opp = 
		  wm.getOpponentNearestTo( candidates2.at( i ), 3, & dist2 );
	       if( target_opp )
	       {
		  if( max2 < dist2 )
		  {
		     max2 = dist2;
		     best2 = i;
		  }
	       }
	       else
	       {
		  best2 = i;
		  dist2 = 100;
	       }
	    }
	 }
	 if( eval2.at( best2 ) && candidates2.at( best2 ).x > x_max )
	 {
	    target_pos = candidates2.at( best2 );
	    x_max = candidates2.at( best2 ).x;
	    receiver = target_mate2->unum();
	 }
	 ///////////////////////////////////
	 if( target_mate3 )
	 {
	    tmp_target = target_mate3->pos();
	    std::vector< rcsc::Vector2D > candidates3;
	    std::vector< bool > eval3;
	    setCandidates( tmp_target, candidates3 );
	    for( unsigned int i = 0; i < candidates3.size(); i++ )
	    {
	       std::vector< rcsc::Vector2D > vertices;
	       double tmpY = 0;
	       if( wm.self().pos().y > 0 )
	       {
		  tmpY = candidates3.at( i ).y;
	       }
	       else
	       {
		  tmpY = candidates3.at( i ).y;
	       }
	       vertices.push_back( rcsc::Vector2D( candidates3.at( i ).x + 1, tmpY ) );
	       vertices.push_back( rcsc::Vector2D( candidates3.at( i ).x - 1, tmpY ) );
	       vertices.push_back( wm.self().pos() );
	       rcsc::Polygon2D polygon( vertices );
	       if( !wm.existOpponentIn( polygon, 3, true )
	           && candidates3.at( i ).x < 51 )
	       {
		  eval3.push_back( true );
	       }
	       else
	       {
		  eval3.push_back( false );
	       }
	    }
	    double dist3 = 0;
	    double max3 = 0;
	    double best3 = 0;
	    for( unsigned int i = 0; i < eval3.size(); i++ )
	    {
	       if( eval3.at( i ) )
	       {
		  const rcsc::PlayerObject * target_opp = 
		     wm.getOpponentNearestTo( candidates3.at( i ), 3, & dist3 );
		  if( target_opp )
		  {
		     if( max3 < dist3 )
		     {
			max3 = dist3;
			best3 = i;
		     }
		  }
		  else
		  {
		     best3 = i;
		     dist3 = 100;
		  }
	       }
	    }
	    if( eval3.at( best3 ) && candidates3.at( best3 ).x > x_max )
	    {
	       target_pos = candidates3.at( best3 );
	       receiver = target_mate3->unum();
	    }
	 }//end if target_mate3 
      }//end if target_mate2
   }//end if target_mate1

   return target_pos;
}

/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/

rcsc::Vector2D
Bhv_SetPlay::getMiddleMovePos( const rcsc::PlayerAgent * agent )
{
   rcsc::Vector2D target_pos = M_home_pos;
   const rcsc::WorldModel & wm = agent->world();

   static bool reachedBase = false;
   static int start = 0;
   if( start < wm.lastSetPlayStartTime().cycle() )
   {
      start = wm.lastSetPlayStartTime().cycle();
      reachedBase = false;
   }
   if( wm.self().pos().dist( M_home_pos ) < 2.5 )
   {
      reachedBase = true;
      
   }
   bool receiver = false;

   const rcsc::PlayerObject * fastest = wm.interceptTable()->fastestTeammate();
   const rcsc::PlayerObject * second = wm.interceptTable()->secondTeammate();
   if( fastest && second )
   {
       double dist1 = fastest->pos().dist( wm.ball().pos() );
       double dist2 = second->pos().dist( wm.ball().pos() );
       double sdist = wm.self().pos().dist( wm.ball().pos() );
       if( second->unum() == wm.self().unum() 
           || ( dist1 < sdist
                && sdist < dist2 ) )
       {
           receiver = true;
       }
   }
   if( reachedBase && receiver )
   {
      target_pos = wm.self().pos();
      if( wm.ball().distFromSelf() < 20 )
      {
	 rcsc::Vector2D top_left( 0, 0 );
	 rcsc::Vector2D bottom_right( 0, 0 );
	 double buf = 3.0;
	 if( wm.ball().pos().x <= wm.self().pos().x )
	 {
	    top_left.x = wm.ball().pos().x - buf;
	    bottom_right.x = wm.self().pos().x + buf;
	 }
	 else
	 {
	    top_left.x = wm.self().pos().x - buf;
	    bottom_right.x = wm.ball().pos().x + buf;
	 }
	 if( wm.ball().pos().y <= wm.self().pos().y )
	 {
	    top_left.y = wm.ball().pos().y;
	    bottom_right.y = wm.self().pos().y + buf;
	 }
	 else
	 {
	    top_left.y = wm.self().pos().y - buf;
	    bottom_right.y = wm.ball().pos().y;
	 }
	 rcsc::Rect2D self_range( top_left, bottom_right );
	 
	 const rcsc::PlayerObject * target_opp = NULL;
	 const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromSelf().end();
	 for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromSelf().begin();
	      it != end;
	      ++it )
	 {
	    if( self_range.contains( (*it)->pos() ) )
	    {
	       target_opp = (*it);
	       break;
	    }
	 }
	 
	 if( target_opp )
	 {
	    if( wm.ball().pos().x <= wm.self().pos().x )
	    {
	       target_pos.x = target_opp->pos().x - 3;
	    }
	    else
	    {
	       target_pos.x = target_opp->pos().x + 3;
	    }
	    if( wm.ball().pos().y <= wm.self().pos().y )
	    {
	       target_pos.y = target_opp->pos().y - 3;
	    }
	    else
	    {
	       target_pos.y = target_opp->pos().y + 3;
	    }
	 }
	 else
	 {

	    target_opp = wm.getOpponentNearestToSelf( 5, false );
	    if( target_opp )
	    {
                bool checkOpponent = false;
                if( target_opp->distFromSelf() < 3 )
                {
                    if( wm.ball().pos().x <= wm.self().pos().x )
                    {
                        if( target_opp->pos().x > wm.ball().pos().x 
                            && target_opp->pos().x < wm.self().pos().x )
                        {
                            target_pos.x = target_opp->pos().x - 3;
                            checkOpponent = true;
                        }
                    }
                    else
                    {
                        if( target_opp->pos().x < wm.ball().pos().x 
                            && target_opp->pos().x > wm.self().pos().x )
                        {
                            target_pos.x = target_opp->pos().x + 3;
                            checkOpponent = true;
                        }
                    }
                    if( wm.ball().pos().y <= wm.self().pos().y )
                    {
                        if( target_opp->pos().y > wm.ball().pos().y
                            && target_opp->pos().y < wm.self().pos().y )
                        {
                            target_pos.y = target_opp->pos().y - 3;
                            checkOpponent = true;
                        }
                    }
                    else
                    {
                        if( target_opp->pos().y < wm.ball().pos().y
                            && target_opp->pos().y > wm.self().pos().y )
                        {
                            target_pos.y = target_opp->pos().y + 3;
                            checkOpponent = true;
                        }
                    }
                }
                if( !checkOpponent 
                    && wm.ball().distFromSelf() < 10 
                    && target_pos.absY() > 30 )
                {
                    if( wm.ball().pos().y > 0 )
                    {
                        target_pos.y -= 8;
                    }
                    else
                    {
                        target_pos.y += 8;
                    }
                }
            }
	    else if( wm.ball().distFromSelf() < 10
	             && target_pos.absY() > 30 )
	    {
                if( wm.ball().pos().y > 0 )
                {
                    target_pos.y -= 8;
                }
                else
                {
                    target_pos.y += 8;
                }
	    }
            
	 }
      }//end if distFromSelf
   }//end if reachedBase
   if( target_pos.x > 50.5 )
   {
      target_pos.x = 50.5;
   }
   if( target_pos.y > 32 )
   {
      target_pos.y = 32;
   }
   if( target_pos.y < -32 )
   {
      target_pos.y = -32;
   }
   return target_pos;
}

/*-------------------------------------------------------------------*/
/*!

*/
/*-------------------------------------------------------------------*/
