#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_shootchance.h"
#include "bhv_offense.h"
#include "opuci_move.h"
#include "opuci_dribble.h"
#include "opuci_one_step_kick.h"
#include "opuci_shoot.h"
#include "opuci_search_ball.h"
#include "opuci_chase_ball.h"
#include "opuci_neck.h"
#include "opuci_intention_dribble.h"
#include "opuci_other_pass.h"
#include "opuci_direct_pass.h"
#include "opuci_message.h"
#include "opuci_self_pass.h"
#include "opuci_neck.h"
#include "body_dribble_opuci.h"
#include "strategy.h"


#include <rcsc/soccer_math.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/say_message_builder.h>
#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/sector_2d.h>
#include <rcsc/geom/circle_2d.h>
#include <rcsc/geom/ray_2d.h>
#include <rcsc/action/neck_turn_to_goalie_or_scan.h>
#include <rcsc/action/neck_scan_field.h>

#include <rcsc/action/body_dribble2008.h>
#include <rcsc/action/body_hold_ball.h>
#include <rcsc/action/body_intercept2009.h>
#include <rcsc/action/body_smart_kick.h>
#include <rcsc/action/body_tackle_to_point.h>
#include <rcsc/action/body_kick_one_step.h>

int Bhv_Shootchance::M_passMate = -1;
int Bhv_Shootchance::M_neck_cnt = -1;

/*-------------------------------------------------------------------*/
/*!
  execute action
*/
bool
Bhv_Shootchance::execute( rcsc::PlayerAgent * agent )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Bhv_Shootchance"
                        ,__FILE__, __LINE__ );
    const rcsc::WorldModel & wm = agent->world();

    rcsc::Vector2D goal_center( 52.5, 0.0 );

    if( wm.self().isKickable() )
    {
        doKick_inProgress( agent );

        if( M_passMate != -1 )
        {
            Opuci_Neck().lookAtPassReceiver( agent, M_passMate );
            M_neck_cnt--;
            if( M_neck_cnt <= 0 )
                M_passMate = -1;
        }
        else
        {
            const rcsc::PlayerObject * goalie = wm.getOpponentGoalie();
            if( goalie )
            {
                if( goalie->posCount() > 1 )
                {
                    agent->setNeckAction( new rcsc::Neck_TurnToGoalieOrScan() );
                }
                else
                {
                    agent->setNeckAction( new rcsc::Neck_ScanField() );
                }
            }
            else
            {
                if( wm.time().cycle() % 2 )
                    agent->setNeckAction( new rcsc::Neck_TurnToGoalieOrScan() );
                else
                    neckToMate( agent );
            }
        }
    }
    else if( wm.self().tackleProbability() >= 0.65 && wm.ball().vel().r() > 1.0
             && wm.getOpponentGoalie() && wm.getOpponentGoalie()->pos().dist( wm.self().pos() ) < 15.0
             && wm.ball().inertiaPoint( 1 ).dist( wm.self().pos() + wm.self().vel() ) > wm.self().kickableArea() )
    {
        if( wm.getOpponentGoalie()->pos().y > 0 )
        {
            if( rcsc::Body_TackleToPoint( rcsc::Vector2D( 52.5, -6.0 ) ).execute( agent ) )
            {
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: Shoot tackle to (52,5, -6.0)"
                                    ,__FILE__, __LINE__ );
            }
            else
                doMove( agent );
        }
        else if( rcsc::Body_TackleToPoint( rcsc::Vector2D( 52.5, 6.0 ) ).execute( agent ) )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: Shoot tackle to (52,5, 6.0)"
                                ,__FILE__, __LINE__ );
        }
        else
            doMove( agent );
    }
    else
    {
        doMove( agent );
    }
    
    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();



    if( wm.self().isKickable() ||
//        (self_min > mate_min && wm.ball().seenPosCount() <= 1) )
        wm.ball().pos().dist( wm.self().pos() ) > 20 )
    {
        int max_pc = -10000;
        int max_unum;
        for( int i = 11 ; i > 6 ; i-- )
        {
            if( !wm.teammate( i ) )
                continue;

            if( wm.teammate( i )->posCount() > max_pc )
            {
                max_unum = wm.teammate(i)->unum();
                max_pc = wm.teammate(i)->posCount();
            }
        }

        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: Attention. max_unum = %d, max_pc = %d"
                            ,__FILE__, __LINE__, max_unum, max_pc );

        if( max_pc >= 0 )
            agent->doAttentionto( wm.self().side(), max_unum );
        else
            agent->doAttentiontoOff();
    }
    else
    {
        if( wm.getTeammateNearestToBall( 100 ) )
        {
            agent->doAttentionto( wm.self().side(),
                                  wm.getTeammateNearestToBall( 100 )->unum() );
        }
        else
        {
            agent->doAttentionto( wm.self().side(),
                                  wm.time().cycle()%5 + 7 );
        }
    }
    agent->debugClient().addMessage("posCount:%d",wm.ball().posCount());

    return true;
}


/*-------------------------------------------------------------------*/
/*!
  移動を行う関数
*/
bool
Bhv_Shootchance::doMove( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
    int self_min = wm.interceptTable()->selfReachCycle(); //自分がボールに到達可能なサイクル数
    int mate_min = wm.interceptTable()->teammateReachCycle(); //味方が〃
    int opp_min = wm.interceptTable()->opponentReachCycle(); //敵が〃

    const int agent_reach_cycle = std::min( std::min( self_min, mate_min ), opp_min ); //敵，味方，自分の中で最短でボールに到達可能なサイクル数
    const rcsc::Vector2D base_pos = wm.ball().inertiaPoint( agent_reach_cycle ); //エージェントが到達可能なサイクル数後の，慣性によるボールの位置
    const rcsc::Segment2D base_line( base_pos, M_target_point ); //ボールの予定位置と最終目標位置とを結ぶ線分
    const rcsc::Vector2D home_pos = getPosition( wm, wm.self().unum() ); //ホームポジション
    rcsc::Vector2D target_pos( home_pos ); //目標位置：ホームポジション
    double dash_power = dashPower( agent );

    if( self_min <= mate_min + 1 && !wm.existKickableTeammate() )
    {
        rcsc::Line2D lbody( wm.self().pos(), wm.self().body() );
        rcsc::Vector2D target( 52.5, lbody.getY(52.5) );
        if( std::fabs( lbody.getY(52.5) ) < 15.0 )
            rcsc::Body_Intercept2009( false, target ).execute( agent );
        else
            rcsc::Body_Intercept2009( false ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_TurnToGoalieOrScan() );        
/*
        if( wm.ball().posCount() >= 1
            || wm.ball().velCount() >= 1 )
        {
            rcsc::Vector2D next_ball_pos = agent->effector().queuedNextBallPos();
            rcsc::Vector2D next_self_pos = agent->effector().queuedNextSelfPos();
            rcsc::AngleDeg next_self_body = agent->effector().queuedNextSelfBody();
            rcsc::Vector2D rpos = next_ball_pos - next_self_pos;
            rcsc::AngleDeg angle = rpos.th() - ( next_self_body + wm.self().neck() );
            agent->doTurnNeck( angle );
            agent->debugClient().addMessage( "LookBall" );
        }
        else
        {
            neckToMate( agent );
            agent->debugClient().addMessage( "LookMate" );
        }
*/
    }
    else if( wm.existKickableOpponent() || opp_min + 3 < mate_min )
    {
        //敵の方が味方より３サイクル以上早くボールに到達可能な場合
        dash_power = dashPower( agent, SP.maxPower() * 0.7 );
        if( target_pos.x > wm.offsideLineX() )
        {
            //目標位置がオフサイドラインを越える場合は修正
            target_pos.x = wm.offsideLineX();
        }
        //目標位置（ホームポジション）へ移動
        if( Opuci_Move( target_pos, dash_power, 3.0 ).execute( agent ) )
        {
            agent->debugClient().addMessage("OpponentBall");
            agent->debugClient().setTarget( home_pos );
        }
        else
        {
            //目標位置に到達している場合はゴールの方を向く
            rcsc::Vector2D opp_goal_pos( SP.pitchHalfLength(), 0.0 );
            agent->doTurn( ( opp_goal_pos - wm.self().pos() ).th() - wm.self().body() );
            agent->debugClient().addMessage("TurnToOppGoal");
        }
        Opuci_Neck().execute( agent );
    }
    else
    {


        std::vector< rcsc::Vector2D > candidates;
        setFormationPos( agent, candidates );
        switch( wm.self().unum() )
        {
        case 7:
            target_pos = candidates.at( 0 );
            break;
        case 8:
            target_pos = candidates.at( 1 );
            break;
        case 9:
            target_pos = candidates.at( 2 );
            break;
        case 10:
            target_pos = candidates.at( 3 );
            break;
        case 11:
            target_pos = candidates.at( 4 );
            break;
        default:
            break;
        }
    

        if( target_pos.x > wm.offsideLineX() - 0.5 )
        {
            target_pos.x = wm.offsideLineX() - 0.5;
        }
        //フィールドライン近くの修正
        if( target_pos.x > 50 )
        {
            target_pos.x = 50;
        }
        if( target_pos.x < -50 )
        {
            target_pos.x = -50;
        }
        if( target_pos.y > 31 )
        {
            target_pos.y = 31;
        }
        if( target_pos.y < -31 )
        {
            target_pos.y = -31;
        }





        agent->debugClient().setTarget( target_pos );

        if( target_pos.dist( wm.self().pos() ) > wm.self().kickableArea() )
        {
            //目標位置と自分との距離がキッカブル距離より大きい場合
            dash_power = dashPower( agent, rcsc::ServerParam::i().maxPower() ); //最大ダッシュパワー
            rcsc::AngleDeg margin_angle = 15.0; //目標位置との距離の2割ほど？
            double margin = wm.self().pos().dist( target_pos );
            //目標位置へ移動
            if( margin > 2.0 )
            {
                agent->debugClient().addMessage( "Positioning");
                if( wm.self().pos().x < target_pos.x - 5 )
                {
                    agent->debugClient().addMessage( "SelfY" );
                    target_pos.y = wm.self().pos().y;
                }
            }
            else
            {
                agent->debugClient().addMessage( "App2Ball" );
                target_pos = wm.ball().pos();
            }
            margin = wm.self().pos().dist( target_pos );
            Opuci_Move( target_pos, dash_power, margin * margin_angle.tan() ).execute( agent );
            //味方を見る
            neckToMate( agent );
        }
        else
        {
            rcsc::Vector2D opp_goal_pos( rcsc::ServerParam::i().pitchHalfLength(), 0.0 );
            agent->doTurn( ( opp_goal_pos - wm.self().pos() ).th() - wm.self().body() );
            //味方を見る
            neckToMate( agent );

            agent->debugClient().addMessage( "WaitTurn" );
        }
    }

/*
  if( wm.ball().rposCount() < 1 )
  {
  //ボールのrposcountが0ならばボールの位置・速度，自分の位置，体の角度をメッセージで配信
  agent->addSayMessage( new rcsc::BallPlayerMessage( agent->effector().queuedNextBallPos(), 
  agent->effector().queuedNextBallVel(),
  wm.self().unum(),
  agent->effector().queuedNextMyPos(),
  agent->effector().queuedNextMyBody() ) );
  }
*/
    Opuci_Message().shootChanceSayMessage( agent, 5 );



    return true;
}

/*-------------------------------------------------------------------*/
/*!
  ダッシュパワーを決定する関数
*/
double
Bhv_Shootchance::dashPower( rcsc::PlayerAgent * agent,
                            const double limit_power )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
    if( wm.self().stamina() < SP.staminaMax() * SP.effortDecThr() * 1.2 
        || wm.self().stamina() < SP.staminaMax() * SP.recoverDecThr() * 1.2 )
    {
        //スタミナ切れが近い場合は停止する
        return 0.0;
    }
    //制限パワー：デフォルトは最大パワー
    return limit_power;
}

/*-------------------------------------------------------------------*/
/*!
  味方がいると思われる方向へ首を向ける関数
*/
bool
Bhv_Shootchance::neckToMate( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    const rcsc::Vector2D next_ball_pos = wm.ball().pos() + wm.ball().vel(); //慣性による次サイクルのボールの位置
    const rcsc::Vector2D next_self_pos = agent->effector().queuedNextSelfPos(); //次サイクルの自分の位置
    const rcsc::AngleDeg next_self_body = agent->effector().queuedNextSelfBody(); //次サイクルの自分の体の向き
    if( wm.ball().posCount() > 1 )
    {
        //ボールのposcountが２以上の場合はボールを見る
        agent->doTurnNeck( ( next_ball_pos - next_self_pos ).th() - ( wm.self().neck() + next_self_body ) );
    }
    else
    {
        std::vector<rcsc::Vector2D> pass_target;
        //レシーブ位置候補
//        getPassPoint( agent, &pass_target );
        setFormationPos( agent, pass_target );
        int max_count = 0;
        rcsc::AngleDeg search_angle = next_self_body;//目標角度
        for( std::vector<rcsc::Vector2D>::iterator it = pass_target.begin();
             it != pass_target.end();
             ++it )
        {
            const rcsc::AngleDeg target_angle = ( (*it) - next_self_pos ).th(); //レシーブ位置候補と自分の体の向きとの角度
            if( wm.dirCount( target_angle ) > max_count
                && ( target_angle - next_self_body ).abs() < 115.0 )
            {
                //向けられる範囲でdirCountが最も大きいレシーブ位置の方向
                max_count = wm.dirCount( target_angle );
                search_angle = target_angle;
            }
        }
        for( int num = 7; num <= 11; num++ )
        {
            if( wm.teammate( num ) )
            {
                const rcsc::AngleDeg target_angle = ( wm.teammate( num )->pos() - next_self_pos ).th(); //味方の位置と自分の体の向きとの角度
                int tmp_count = wm.teammate( num )->posCount();
                if( num >= 9 )
                {
                    tmp_count *= 1.5;
                }
                if( tmp_count > max_count
                    && ( target_angle - next_self_body ).abs() < 115.0 )
                {
                    //向けられる範囲でposCountが最も大きい味方の位置の方向
                    max_count = tmp_count;
                    search_angle = target_angle;
                }
            }
        }
        agent->doTurnNeck( search_angle - ( wm.self().neck() + next_self_body ) );
    }
    return true;
}


/*-------------------------------------------------------------------*/
/*!
  前衛のフォーメーションを決める関数
*/
double
Bhv_Shootchance::setFormationPos( rcsc::PlayerAgent * agent, std::vector< rcsc::Vector2D > & candidates )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();
    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();
    int agent_reach_cycle = std::min( std::min( self_min, mate_min ), opp_min );

    rcsc::Vector2D inertial_ball_pos = wm.ball().inertiaPoint( agent_reach_cycle );
    double max_pass_dist = rcsc::calc_sum_geom_series( 1.2, 1.0 / sp.ballDecay(), 10);

    rcsc::Vector2D rotation( max_pass_dist, 0 ); //長さ＝パス距離，角度＝０のベクトル
    rcsc::Vector2D ball_to_target = M_target_point - inertial_ball_pos;

#if 0
    rcsc::Vector2D base_pos = inertial_ball_pos;

    rcsc::Vector2D left_back = base_pos + rotation.rotatedVector( -90 );
    rcsc::Vector2D right_back = base_pos + rotation.rotatedVector( 90 );

    rotation.rotate( ball_to_target.th() ); //長さ＝パス距離，角度＝最終目標位置の方向 のベクトル

    rcsc::Vector2D left_forward = base_pos + rotation.rotatedVector( -45 );
    rcsc::Vector2D right_forward = base_pos + rotation.rotatedVector( 45 );
    rcsc::Vector2D center_forward = base_pos + rotation;

    
    candidates.push_back( left_back );
    candidates.push_back( right_back );
    candidates.push_back( left_forward );
    candidates.push_back( right_forward );
    candidates.push_back( center_forward );

    double minimum_dist = 100;
    rcsc::Vector2D shift( 0, 0 );
    int passer_unum = 0;
    for( int i = 7; i <= 11; i++ )
    {
        if( wm.teammate( i ) )
        {
            double dist = inertial_ball_pos.dist( wm.teammate( i )->pos() );
            if( dist < minimum_dist )
            {
                minimum_dist = dist;
                passer_unum = i;
            }
        }
    }
    switch( passer_unum )
    {
    case 7:
        shift = inertial_ball_pos - candidates.at( 0 );
        break;
    case 8:
        shift = inertial_ball_pos - candidates.at( 1 );
        break;
    case 9:
        shift = inertial_ball_pos - candidates.at( 2 );
        break;
    case 10:
        shift = inertial_ball_pos - candidates.at( 3 );
        break;
    case 11:
        shift = inertial_ball_pos - candidates.at( 4 );
        break;
    default:
        break;
    }
    for( int i = 0; i < (int)candidates.size(); i++ )
    {
        candidates.at( i ) += shift;
    }
    
    agent->debugClient().addCircle( inertial_ball_pos, 0.5 );
    agent->debugClient().addLine( inertial_ball_pos, M_target_point );

    double first_speed = rcsc::calc_first_term_geom_series_last( 1.2, max_pass_dist, sp.ballDecay() );

    for( int i = 0; i < (int)candidates.size(); i++ )
    {
        double dist_to_cand = inertial_ball_pos.dist( candidates.at( i ) );

        if( dist_to_cand > first_speed )
        {
            rcsc::Vector2D ball_to_cand = candidates.at( i ) - inertial_ball_pos;
            rcsc::Segment2D route( inertial_ball_pos, candidates.at( i ) );
            rcsc::Sector2D check_area( inertial_ball_pos,
                                       rcsc::calc_sum_geom_series( first_speed, sp.ballDecay(), 3 ),
                                       max_pass_dist,
                                       ball_to_cand.th() - 10,
                                       ball_to_cand.th() + 10 );
            double min_dist = 50;
            const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
            for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
                 it != end;
             ++it )
            {
                if( check_area.contains( (*it)->pos() ) )
                {
                    rcsc::Vector2D tmp_target = route.nearestPoint( (*it)->pos() );
                    double dist = tmp_target.dist( inertial_ball_pos );
                    if( dist < min_dist )
                    {
                        min_dist = dist;
                        candidates.at( i ) = tmp_target;
                    }
                }
            }            
        }
        agent->debugClient().addCircle( candidates.at( i ), 2 );
    }
#endif

#if 1
    double min_x = 52;
    double max_x = 0;
    double min_y = 15;
    double max_y = -15;
    const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
         it != end;
         ++it )
    {
        if( (*it)->pos().x > 35 
            && (*it)->pos().absY() < 25
            && !(*it)->goalie() )
        {
            if( (*it)->pos().x < min_x )
            {
                min_x = (*it)->pos().x;
            }
/*
            if( (*it)->pos().x > max_x )
            {
                max_x = (*it)->pos().x;
            }
*/
            if( (*it)->pos().y < min_y )
            {
                min_y = (*it)->pos().y;
            }
            if( (*it)->pos().y > max_y )
            {
                max_y = (*it)->pos().y;
            }
        }
    }
    if( min_x < 35 )
    {
        min_x = 35;
    }
    else if( min_x > 47 )
    {
        min_x = 47;
    }
/*
    if( max_x < min_x )
    {
        max_x = min_x + 5;
    }
*/
    max_x = 52;
    if( min_y > 0 )
    {
        min_y = 0;
    }
    if( max_y < 0 )
    {
        max_y = 0;
    }


    rcsc::Vector2D top_left( min_x - 3.0, min_y - 3.0 );
    rcsc::Vector2D bottom_right( max_x, max_y + 3.0 );
    rcsc::Rect2D rect( top_left, bottom_right );
    agent->debugClient().addRectangle( rect );
    rcsc::Vector2D left_back, center_back, right_back, center_forward, other_forward;
    left_back.x = rect.minX();
    center_back.x = rect.minX();
    right_back.x = rect.minX();
    center_forward.y = rect.center().y;
    other_forward = rect.center();


    bool back = false;

    if( wm.time().cycle() % 20 >= 10 )
    {
        back = true;
    }

    if( back )
    {
        center_forward.x = rect.minX();
    }
    else
    {
        center_forward.x = wm.offsideLineX() - 0.5;
    }
    if( wm.ball().pos().y < 0 )
    {
        left_back.y = rect.minY();
        center_back.y = left_back.y + 12.0;
        right_back.y = center_back.y + 12.0;
        if( back )
        {
            const rcsc::PlayerObject * nearest_center = wm.getOpponentNearestTo( right_back, 20, NULL );
            if( nearest_center )
            {
                rcsc::Vector2D o2b = wm.ball().pos() - nearest_center->pos();
                center_back = nearest_center->pos() + o2b.setLengthVector( 2.5 );
                right_back.x = rect.center().x;
            }
        }
        if( right_back.y > max_y )
        {
            right_back.y = max_y;
        }
        other_forward.y = max_y / 2.0;
        other_forward.x = min_x;
    }
    else
    {
        right_back.y = rect.maxY();
        center_back.y = right_back.y - 12.0;
        left_back.y = center_back.y - 12.0;
        if( back )
        {
            const rcsc::PlayerObject * nearest_center = wm.getOpponentNearestTo( center_back, 20, NULL );
            if( nearest_center )
            {
                rcsc::Vector2D o2b = wm.ball().pos() - nearest_center->pos();
                center_back = nearest_center->pos() + o2b.setLengthVector( 2.5 );
                left_back.x = rect.center().x;
            }
        }
        if( left_back.y < min_y )
        {
            left_back.y = min_y;
        }
        other_forward.y = min_y / 2.0;
        other_forward.x = min_x;
    }








    
    rcsc::Vector2D nana, hachi, ku, ju, juichi;
    if( wm.ball().pos().y < 0 )
    {
        nana = left_back;
        hachi = center_back;
        ku = other_forward;
        ju = right_back;
        juichi = center_forward;
    }
    else
    {
        nana = center_back;
        hachi = right_back;
        ku = left_back;
        ju = other_forward;
        juichi = center_forward;
    }


/*
    const rcsc::PlayerObject * holder = wm.getTeammateNearestToBall( 30, true );
    int holder_unum = -1;
    if( holder )
    {
        holder_unum = holder->unum();
    }

    switch( holder_unum )
    {
    case 7:
        nana = wm.ball().pos();
        hachi = center_back;
        ku = left_back;
        ju = right_back;
        juichi = center_forward;
        break;
    case 8:
        nana = center_back;
        hachi = wm.ball().pos();
        ku = left_back;
        ju = right_back;
        juichi = center_forward;
        break;
    case 9:
        nana = left_back;
        hachi = center_back;
        ku = wm.ball().pos();
        ju = right_back;
        juichi = center_forward;
        break;
    case 10:
        nana = center_back;
        hachi = right_back;
        ku = left_back;
        ju = wm.ball().pos();
        juichi = center_forward;
        break;
    case 11:
        if( wm.ball().pos().y < 0 )
        {
            nana = center_back;
            hachi = right_back;
            ku = left_back;
            ju = center_forward;
        }
        else
        {
            nana = left_back;
            hachi = center_back;
            ku = center_forward;
            ju = right_back;
        }
        juichi = wm.ball().pos();
        break;
    default:
        nana = left_back;
        hachi = right_back;
        ku.x =  ( rect.maxX() + rect.minX() ) / 2.0;
        ku.y = rect.minY();
        ju.x = ( rect.maxX() + rect.minX() ) / 2.0;
        ju.y = rect.maxY();
        juichi = center_forward;
        break;
    }
*/  

    candidates.push_back( nana );
    candidates.push_back( hachi );
    candidates.push_back( ku );
    candidates.push_back( ju );
    candidates.push_back( juichi );
/*
    rcsc::Vector2D left_back = inertial_ball_pos + rotation.rotatedVector( -135 ) * 0.7;
    rcsc::Vector2D right_back = inertial_ball_pos + rotation.rotatedVector( 135 ) * 0.7;
//    rotation.rotate( ball_to_target.th() ); //長さ＝パス距離，角度＝最終目標位置の方向 のベクトル

    rcsc::Vector2D left_forward = inertial_ball_pos + rotation.rotatedVector( -90 );
    rcsc::Vector2D right_forward = inertial_ball_pos + rotation.rotatedVector( 90 );
    rcsc::Vector2D center_forward = inertial_ball_pos - rotation.rotatedVector( 180 ) * 0.5;


    left_back.y = -10;
    right_back.y = 10;
    left_forward.y = -7;
    right_forward.y = 7);
    center_forward.y = 0;


    candidates.push_back( left_back );
    candidates.push_back( right_back );
    candidates.push_back( left_forward );
    candidates.push_back( right_forward );
    candidates.push_back( center_forward );
*/    
    agent->debugClient().addCircle( inertial_ball_pos, 2 );
    agent->debugClient().addLine( inertial_ball_pos, M_target_point );

    double first_speed = rcsc::calc_first_term_geom_series_last( 1.2, max_pass_dist, sp.ballDecay() );
    for( int i = 0; i < (int)candidates.size(); i++ )
    {
        if( candidates.at( i ).x > 50 )
        {
            candidates.at( i ).x = 50;
        }
        if( candidates.at( i ).y > 30 )
        {
            candidates.at( i ).y = 30;
        }
        if( candidates.at( i ).y < -30 )
        {
            candidates.at( i ).y = -30;
        }
/*
        rcsc::Vector2D ball_to_cand = candidates.at( i ) - inertial_ball_pos;
        rcsc::Sector2D check_area( inertial_ball_pos,
                                   rcsc::calc_sum_geom_series( first_speed, sp.ballDecay(), 2 ),
                                   max_pass_dist,
                                   ball_to_cand.th() - 45,
                                   ball_to_cand.th() + 45 );
        rcsc::Segment2D route( inertial_ball_pos, candidates.at( i ) );
        double min_dist = 50;
        const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
        for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
             it != end;
             ++it )
        {
            if( check_area.contains( (*it)->pos() ) )
            {
                rcsc::Vector2D tmp_target = route.nearestPoint( (*it)->pos() );
                double dist = tmp_target.dist( inertial_ball_pos );
                if( dist < min_dist )
                {
                    min_dist = dist;
                    candidates.at( i ) = tmp_target;
                }
            }
        }
move*/
        agent->debugClient().addCircle( candidates.at( i ), 2 );
    }
#endif
    return max_pass_dist;
}

/*-------------------------------------------------------------------*/
/*!
 */
bool
Bhv_Shootchance::holdBall( rcsc::PlayerAgent * agent )
{
    holdBall( agent, M_target_point );
    return true;
}



/*-------------------------------------------------------------------*/
/*!
 */
bool
Bhv_Shootchance::holdBall( rcsc::PlayerAgent * agent, rcsc::Vector2D target_pos )
{
    const rcsc::WorldModel & wm = agent->world();
    rcsc::Vector2D next_ball_pos = wm.ball().pos() + wm.ball().vel();
    rcsc::Vector2D next_self_pos = wm.self().pos() + wm.self().vel();
    
    rcsc::Circle2D next_self_area( next_self_pos, wm.self().kickableArea() * 0.9 );

    rcsc::Vector2D turn_pos = target_pos - next_self_pos;
    rcsc::AngleDeg angle_dif = turn_pos.th() - wm.self().body();
    if( next_self_area.contains( next_ball_pos ) && angle_dif.abs() > 10 )
    {
        //turn
        agent->doTurn( angle_dif );
        agent->debugClient().addMessage( "HoldTurn" );
    }
    else
    {
        //kick
        rcsc::Vector2D two_step_self_pos = wm.self().inertiaPoint( 2 );
        rcsc::Vector2D tmp_target = target_pos - two_step_self_pos;
        tmp_target.setLength( wm.self().kickableArea() * 0.5 );
        rcsc::Vector2D hold_pos = two_step_self_pos + tmp_target;

	// calculate the next velocity of the ball so that
	// the ball reaches hold_pos in two cycles
	// p + a + a*decay = hold_pos
	rcsc::Vector2D ball_vel = (hold_pos - wm.ball().pos()) / (1 + rcsc::ServerParam::i().ballDecay() );

	// calculate kick vector to be applied to the ball
        rcsc::Vector2D accel = ball_vel - wm.ball().vel();
	
	// calculate kick power
        double kick_power = accel.r() / wm.self().kickRate();
        agent->doKick( kick_power, accel.th() - wm.self().body() );
        agent->debugClient().addMessage( "HoldKick" );
        agent->debugClient().setTarget( hold_pos );
    }


    
    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
/*-------------------------------------------------------------------*/

void
Bhv_Shootchance::doKick_inProgress( rcsc::PlayerAgent *agent )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Bhv_Shootchance::dokick"
                        ,__FILE__, __LINE__ );

    kickaction( agent );

    agent->addSayMessage( new rcsc::BallPlayerMessage( agent->effector().queuedNextBallPos(), 
                                                       agent->effector().queuedNextBallVel(),
                                                       agent->world().self().unum(),
                                                       agent->effector().queuedNextMyPos(),
                                                       agent->effector().queuedNextMyBody() ) );

    return;
}

/*-------------------------------------------------------------------*/
/*!

 */
/*-------------------------------------------------------------------*/

bool
Bhv_Shootchance::canDribble2GoalieWhenFrontier( rcsc::PlayerAgent *agent )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Bhv_Shootchance::canDribble2GoalieWhenFrontier"
                        ,__FILE__, __LINE__ );

    const rcsc::WorldModel & wm = agent->world();
    const rcsc::PlayerObject * opp_nearest = wm.getOpponentNearestToSelf( 100, false );
    const rcsc::Vector2D goal_center( 52.5, 0.0 );

    if( !opp_nearest )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: No nearest opponent. Can dribble to Goalie."
                            ,__FILE__, __LINE__ );
        return true;
    }
    
    // ここ以降は最近傍の敵がいる
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: Found nearest opponent."
                        ,__FILE__, __LINE__ );

    if( std::fabs( opp_nearest->pos().y - wm.self().pos().y ) > 0.0 )
    {
        if( Bhv_Offense().countSectorOpp( agent, wm.ball().pos(), (goal_center - wm.ball().pos()).th(),
                            10.0, rcsc::AngleDeg( 30.0 ) ) == 0 )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: There is a space. Can dribble to Goalie."
                                ,__FILE__, __LINE__ );
            return true;
        }
    }
        
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: Better go straight to the goal line."
                        ,__FILE__, __LINE__ );
    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
/*-------------------------------------------------------------------*/

void
Bhv_Shootchance::kickaction( rcsc::PlayerAgent *agent )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Bhv_Shootchance::kickaction"
                        ,__FILE__, __LINE__ );

    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();
    rcsc::Vector2D goal_center( 52, 0 );
    int opp_min = wm.interceptTable()->opponentReachCycle();
    agent->debugClient().addMessage( "OppMin%d", opp_min );
    int count_thr = 2;
    if( opp_min <= 2 )
    {
        count_thr = 4;
    }


    bool exist_tri = existTri( agent, wm.ball().pos() );
    bool exist_front = false;

/*    
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
         it != wm.teammatesFromSelf().end();
         ++it )
    {
        rcsc::
    }
*/
    rcsc::Vector2D goal_left( 52, -7 );
    rcsc::Vector2D goal_right( 52, 7 );
    const rcsc::PlayerObject * fastest_opp = wm.interceptTable()->fastestOpponent();
    if( fastest_opp 
        && fastest_opp->pos().x > wm.self().pos().x )
    {
        exist_front = true;
    }
    if( !exist_tri 
        && wm.self().pos().absY() < 20 
        && !( opp_min <= 3 && exist_front ) )
    {
            
        rcsc::Vector2D b2g = goal_center - wm.ball().pos();
        rcsc::Line2D goal_line( goal_left, goal_right );
        rcsc::Ray2D ray( wm.self().pos(), wm.self().body() );
        rcsc::AngleDeg angle = b2g.th();
        rcsc::Vector2D intersection = ray.intersection( goal_line );
        bool use_self = false;
        if( wm.self().body().abs() < 90 
            && intersection.valid()
            &&  intersection.absY() <= 7  
            && wm.self().pos().absY() <= 10 )
        {
            angle = wm.self().body();
//            use_self = true;
        }

        int dash_cycle = 4;
/*
        const rcsc::PlayerObject * goalie = wm.getOpponentGoalie();
        if( goalie )
        {
            dash_cycle = Opuci_SelfPass( angle ).safeCycle( agent, goalie );
            if( dash_cycle < 4 )
            {
                dash_cycle = 4;
            }
        }
*/
        if( use_self 
            && wm.self().pos().x < 45 
            && Opuci_SelfPass( angle ).forceDash( agent, dash_cycle ) )
        {
            agent->debugClient().addMessage( "Force%d", dash_cycle );
            return;
        }
        else if( Body_DribbleOpuci( goal_center, 1.0, sp.maxPower(), 2, false ).execute( agent ) )
        {
            agent->debugClient().addMessage( "Dribble%d", 2 );
            return;
        }
    }


    std::vector< rcsc::PlayerObject* > cand_mates;
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
         it != wm.teammatesFromSelf().end();
         ++it )
    {
        if( (*it)->pos().x > 30
            && (*it)->seenPosCount() <= count_thr
            && (*it)->pos().x < wm.offsideLineX() )
        {
            cand_mates.push_back( (*it) );
        }
    }

    for( std::vector< rcsc::PlayerObject* >::iterator it = cand_mates.begin();
         it != cand_mates.end();
         ++it )
    {
        for( std::vector< rcsc::PlayerObject* >::iterator it2 = it;
             it2 != cand_mates.end();
             ++it2 )
        {
            if( (*it)->pos().dist( goal_center ) > (*it2)->pos().dist( goal_center ) )
                std::swap( *it, *it2 );
        }
    }

    for( std::vector< rcsc::PlayerObject* >::iterator it = cand_mates.begin();
         it != cand_mates.end();
         ++it )
    {
        agent->debugClient().addMessage( "%d>", (*it)->unum() );
    }


    std::vector< int > opp_num;
    Bhv_Offense().surroundOppNum( agent, opp_num );    

    rcsc::Vector2D s2g = goal_center - wm.self().pos();
    int self_opp = Bhv_Offense().countSectorOpp( agent, wm.self().pos(), s2g.th(), 15, 30 );
    int self_opp2 = 3;
    switch( wm.self().unum() )
    {
    case 7:
        self_opp2 = opp_num[0];
        break;
    case 8:
        self_opp2 = opp_num[1];
        break;
    case 9:
        self_opp2 = opp_num[2];
        break;
    case 10:
        self_opp2 = opp_num[3];
        break;
    case 11:
        self_opp2 = opp_num[4];
        break;
    default:
        break;
    }

    for( std::vector< rcsc::PlayerObject* >::iterator it = cand_mates.begin();
         it != cand_mates.end();
         ++it )
    {
        //other pass loop
        int rnum = 0;
        for( rcsc::PlayerPtrCont::const_iterator tm = wm.teammatesFromSelf().begin();
             tm != wm.teammatesFromSelf().end();
             ++tm )
        {
            if( (*it) == (*tm)
                || (*tm)->pos().x < 30 )
            {
                continue;
            }
            if( Bhv_Offense().checkPassReceive( agent, (*it)->pos(), (*tm)->pos() ) == 0)
            {
                rnum++;
            }
        }
        rcsc::Vector2D m2g = goal_center - (*it)->pos();
        int mate_opp = Bhv_Offense().countSectorOpp( agent, (*it)->pos(), m2g.th(), 15, 30 );
        int mate_opp2 = 5;
        switch( (*it)->unum() )
        {
        case 7:
            mate_opp2 = opp_num[0];
            break;
        case 8:
            mate_opp2 = opp_num[1];
            break;
        case 9:
            mate_opp2 = opp_num[2];
            break;
        case 10:
            mate_opp2 = opp_num[3];
            break;
        case 11:
            mate_opp2 = opp_num[4];
            break;
        default:
            break;
        }
        bool free_triangle = !existTri( agent, (*it)->pos() );
        if( mate_opp <= self_opp 
            || mate_opp2 <= self_opp2        
            || m2g.r() <= s2g.r() 
            || free_triangle 
            || rnum > 0 )
        {
            rcsc::PlayerObject * receiver = (*it);
            rcsc::Vector2D pass_target;
            double first_speed;
            if( Opuci_OtherPass().search( agent, receiver, pass_target, first_speed, false ) )
            {
                if( receiver->seenPosCount() == 0 )
                {
                    agent->debugClient().addMessage( "Target%d", receiver->unum() );
                    agent->debugClient().setTarget( pass_target );
                    if( rcsc::Body_SmartKick( pass_target, first_speed, first_speed * 0.9, 2 ).execute( agent ) )
                    {
                        agent->addSayMessage( new rcsc::PassMessage( receiver->unum(), pass_target,
                                                                     agent->effector().queuedNextBallPos(),
                                                                     agent->effector().queuedNextBallVel() ) );
                    }
                }
                else
                {
                    rcsc::dlog.addText( rcsc::Logger::TEAM,
                                        "%s:%d: seenCount = %d. Lock on %d at the next cycle."
                                        ,__FILE__, __LINE__, receiver->seenPosCount(), receiver->unum() );
                    M_passMate = receiver->unum();
                    rcsc::Body_HoldBall().execute( agent );
                    M_neck_cnt = 3;
                }
                return;
            }
        }
    }
    for( std::vector< rcsc::PlayerObject* >::iterator it = cand_mates.begin();
         it != cand_mates.end();
         ++it )
    {
        //direct pass loop
        int rnum = 0;
        for( rcsc::PlayerPtrCont::const_iterator tm = wm.teammatesFromSelf().begin();
             tm != wm.teammatesFromSelf().end();
             ++tm )
        {
            if( (*it) == (*tm)
                || (*tm)->pos().x < 30 )
            {
                continue;
            }
            if( Bhv_Offense().checkPassReceive( agent, (*it)->pos(), (*tm)->pos() ) == 0)
            {
                rnum++;
            }
        }
        
        rcsc::Vector2D m2g = goal_center - (*it)->pos();
        int mate_opp = Bhv_Offense().countSectorOpp( agent, (*it)->pos(), m2g.th(), 15, 30 );
        int mate_opp2 = 5;
        switch( (*it)->unum() )
        {
        case 7:
            mate_opp2 = opp_num[0];
            break;
        case 8:
            mate_opp2 = opp_num[1];
            break;
        case 9:
            mate_opp2 = opp_num[2];
            break;
        case 10:
            mate_opp2 = opp_num[3];
            break;
        case 11:
            mate_opp2 = opp_num[4];
            break;
        default:
            break;
        }
        bool free_triangle = !existTri( agent, (*it)->pos() );
        if( mate_opp < self_opp 
            || mate_opp2 < self_opp2        
            || m2g.r() < s2g.r() 
            || free_triangle 
            || rnum > 0 )
        {
            rcsc::PlayerObject * receiver = (*it);
            rcsc::Vector2D pass_target;
            double first_speed;
            if( Opuci_DirectPass().search( agent, receiver, pass_target, first_speed ) )
            {
                if( receiver->seenPosCount() == 0 )
                {
                    agent->debugClient().addMessage( "Target%d", receiver->unum() );
                    agent->debugClient().setTarget( pass_target );
                    if( rcsc::Body_SmartKick( pass_target, first_speed, first_speed * 0.8, 3 ).execute( agent ) )
                    {
                        agent->addSayMessage( new rcsc::PassMessage( receiver->unum(), pass_target,
                                                                     agent->effector().queuedNextBallPos(),
                                                                     agent->effector().queuedNextBallVel() ) );
                    }
                }
                else
                {
                    rcsc::dlog.addText( rcsc::Logger::TEAM,
                                        "%s:%d: seenCount = %d. Lock on %d at the next cycle."
                                        ,__FILE__, __LINE__, receiver->seenPosCount(), receiver->unum() );
                    M_passMate = receiver->unum();
                    rcsc::Body_HoldBall().execute( agent );
                    M_neck_cnt = 3;
                }

                return;
            }
        }
    }



    // 最前線にいる場合、ドリブルで前進
    std::vector< rcsc::Vector2D > matefront;
    bool frontier = Bhv_Offense( goal_center ).getMateFrontLine( agent, matefront );
    bool veryfront = true;
    for( std::vector<rcsc::Vector2D>::iterator it = matefront.begin();
         it != matefront.end() ; ++it )
    {
        if( wm.self().pos().x < (*it).x )
        {
            veryfront = false;
            break;
        }
    }
    if( veryfront )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: I am the very front."  ,__FILE__, __LINE__ );
    }
    if( (frontier && wm.self().pos().dist( goal_center ) > 15.0 ) || veryfront )
    {

        rcsc::Vector2D d_target( 52.5, 0.0 );

        if( wm.self().pos().x >= 50.0 )
        {
            if( wm.getOpponentGoalie() )
            {
                // キーパーが見えていれば、キーパに向かっていく
                d_target = wm.getOpponentGoalie()->pos();
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: Attack to goalie side."
                                    ,__FILE__, __LINE__ );
            }
            else
            {
                d_target = goal_center;
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: Attack to goal_center."
                                    ,__FILE__, __LINE__ );
            }
        }
        else
        {
            // 最近傍の敵が内側ではなく、他の敵は内側領域にいない場合はキーパの横を目指す
            if( canDribble2GoalieWhenFrontier( agent ) )
            {
                const rcsc::PlayerObject * goalie = wm.getOpponentGoalie();
                if( goalie )
                {
                    d_target = goalie->pos();
                    if( d_target.x < 50.0 )
                        d_target.x = 51.0;
                    if( std::fabs( wm.self().pos().y ) < 7.0 )
                        d_target.y = wm.self().pos().y;
                    else if( wm.self().pos().y > 0.0 )
                        d_target.y = 8.0;
                    else
                        d_target.y = -8.0;
                }
                else
                    d_target = goal_center;

                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: Attack to goalie."
                                    ,__FILE__, __LINE__ );
            }
            // それ以外はまっすぐ
            else
            {
                // ほぼゴール方向を向いていればターンしないように目標値を直進方向に設定
                rcsc::Line2D l_body( wm.self().pos(), wm.self().body() );
                if( -9.3 < l_body.getY( 52.5 ) && l_body.getY( 52.5 ) < 9.3 )
                    d_target.assign( 52.5, l_body.getY( 52.5 ) );
                else
                    d_target.assign( 52.5, wm.self().pos().y );
                rcsc::dlog.addText( rcsc::Logger::TEAM, "%s:%d: Dribble target (%f, %f)"
                                    ,__FILE__, __LINE__, d_target.x, d_target.y );

                agent->debugClient().addMessage( "FrontierDrib%d", __LINE__ );
            }
        }

        //Opuci_SelfPass( d_target.th() ).execute( agent, false );
        Body_DribbleOpuci( d_target, 1.0, sp.maxPower(), 2, false ).execute( agent );
        return;
    }
    
/*
    // 自分より前にいるゴール方向フリーな味方にパス
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin() ;
         it != wm.teammatesFromSelf().end() ; ++it )
    {
        if( (*it)->pos().dist( goal_center ) > wm.self().pos().dist( goal_center ) )
            continue;

        if( wm.offsideLineX() < (*it)->pos().x )
            continue;

        rcsc::Vector2D s2r = (*it)->pos() - wm.self().pos();
        rcsc::Sector2D sector( wm.self().pos(), 0.0, wm.self().pos().dist( (*it)->pos() ),
                               s2r.th() + rcsc::AngleDeg( -20.0 ),
                               s2r.th() + rcsc::AngleDeg( +20.0 ) );
        if( wm.existOpponentIn( sector, 100, false ) )
            continue;

        double tmp_dist = wm.ball().pos().dist( (*it)->pos() );
        double first_speed = rcsc::calc_first_term_geom_series_last( 1.2, tmp_dist,
                                                                     sp.ballDecay() );

        agent->debugClient().addMessage( "Tgt%d:%d", (*it)->unum(), __LINE__ );
        agent->debugClient().setTarget( (*it)->pos() );
        if( rcsc::Body_SmartKick( (*it)->pos(), first_speed, first_speed, 3 ).execute( agent ) )
        {
            agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(), (*it)->pos(),
                                                         agent->effector().queuedNextBallPos(),
                                                         agent->effector().queuedNextBallVel() ) );
            return;
        }
    }
*/

/*
    // パスできる味方にパス
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin() ;
         it != wm.teammatesFromSelf().end() ; ++it )
    {
        if( !(*it) || (*it)->unum() == wm.self().unum() )
            continue;

        if( wm.offsideLineX() < (*it)->pos().x )
            continue;

        rcsc::Vector2D s2r = (*it)->pos() - wm.self().pos();
        rcsc::Sector2D sector( wm.self().pos(), 0.0, wm.self().pos().dist( (*it)->pos() ),
                               s2r.th() + rcsc::AngleDeg( -20.0 ),
                               s2r.th() + rcsc::AngleDeg( +20.0 ) );
        if( wm.existOpponentIn( sector, 100, false ) )
            continue;

        double tmp_dist = wm.ball().pos().dist( (*it)->pos() );
        double first_speed = rcsc::calc_first_term_geom_series_last( 1.2, tmp_dist,
                                                                     sp.ballDecay() );

        agent->debugClient().addMessage( "Tgt%d:%d", (*it)->unum(), __LINE__ );
        agent->debugClient().setTarget( (*it)->pos() );
        if( rcsc::Body_SmartKick( (*it)->pos(), first_speed, first_speed, 3 ).execute( agent ) )
        {
            agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(), (*it)->pos(),
                                                         agent->effector().queuedNextBallPos(),
                                                         agent->effector().queuedNextBallVel() ) );
        }
        return;
    }
*/
    
    //強制パス
    if( wm.interceptTable()->opponentReachCycle() <= 2 )
    {
        double first_speed = sp.ballSpeedMax();
        for( std::vector< rcsc::PlayerObject* >::iterator it = cand_mates.begin();
             it != cand_mates.end();
             ++it )
        {
            rcsc::Vector2D pass_target = (*it)->pos();
            agent->debugClient().addMessage( "ForcePass" );
            if( rcsc::Body_KickOneStep( pass_target, first_speed, true ).execute( agent ) )
            {
                agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(), pass_target,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
            }
            return;
        }

    }



    // ドリブルで前進
    rcsc::Line2D me2center( wm.self().pos(), rcsc::Vector2D( 52.5, 0.0 ) );
    const rcsc::PlayerObject * goalie = wm.getOpponentGoalie();
    if( goalie && me2center.dist( goalie->pos() ) < 1.0 )
    {
        rcsc::Vector2D d_target;
        if( goalie->pos().x < 47.0 )
        {
            int tmp_i;
            d_target.x = 52.5;
            tmp_i = (int)(goalie->pos().y/5);
            d_target.y = tmp_i > 0 ? (tmp_i + 1) * 5.0: (tmp_i - 1) * 5.0;
        }
        else
            d_target.assign( 52.5, wm.self().pos().y );

        if( Bhv_Offense().countSectorOpp( agent, wm.self().pos(), d_target.th(), 20, 30 ) <= 1 )
        {
            if( Opuci_SelfPass( d_target.th() ).execute( agent, goalie ) )
            {
                agent->debugClient().addMessage( "GoalieSelfPass" );
                return;
            }
        }
    }

    // 最後の手段：１ドリブル
    Body_DribbleOpuci( goal_center, 1.0, sp.maxPower(), 2, true ).execute( agent );
    agent->debugClient().addMessage( "Drib%d", __LINE__ );
    return;


}

/*-------------------------------------------------------------------*/
/*!

 */
/*-------------------------------------------------------------------*/

bool
Bhv_Shootchance::doKick( rcsc::PlayerAgent * agent )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d:Bhv_Shootchance::doKick"
                        ,__FILE__, __LINE__ );

    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();
    rcsc::Vector2D next_self_pos = wm.self().pos() + wm.self().vel();

    rcsc::Vector2D tmp_target = M_target_point - next_self_pos;
    rcsc::AngleDeg angle_dif = tmp_target.th() - wm.self().body();

    bool acted = false;

    bool danger = false;
    const rcsc::PlayerPtrCont::const_iterator opp_end = wm.opponentsFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
         it != opp_end;
         ++it )
    {
        if( (*it)->isTackling() )
            continue;

        rcsc::Vector2D next_opp_pos = (*it)->pos() + (*it)->vel();
        rcsc::Circle2D turned_opp_circle( next_opp_pos, (*it)->playerTypePtr()->kickableMargin() );
        rcsc::Vector2D opp_dash;
        opp_dash.setPolar( sp.defaultDashPowerRate() * 100, (*it)->body() );
        next_opp_pos += opp_dash;
        rcsc::Circle2D dashed_opp_circle( next_opp_pos, (*it)->playerTypePtr()->kickableMargin() );
        if( turned_opp_circle.contains( wm.ball().pos() ) 
            || dashed_opp_circle.contains( wm.ball().pos() ) )
        {
            danger = true;
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: I'm in danger." ,__FILE__, __LINE__ );
            break;
        }

    }

    rcsc::Line2D me2center( wm.self().pos(), rcsc::Vector2D( 52.5, 0.0 ) );

    const rcsc::PlayerObject * goalie = wm.getOpponentGoalie();
    if( goalie && me2center.dist( goalie->pos() ) < 1.0 )
    {
        rcsc::Vector2D d_target;
        if( goalie->pos().x < 47.0 )
        {
            int tmp_i;
            d_target.x = 52.5;
            tmp_i = (int)(goalie->pos().y/5);
            d_target.y = tmp_i > 0 ? (tmp_i + 1) * 5.0: (tmp_i - 1) * 5.0;
        }
        else
            d_target.assign( 52.5, 0.0 );

        if( Bhv_Offense().countSectorOpp( agent, wm.self().pos(), d_target.th(), 20, 30 ) <= 1 )
        {
            if( Opuci_SelfPass( d_target.th() ).execute( agent, goalie ) )
            {
                agent->debugClient().addMessage( "GoalieSelfPass" );
                acted = true;
            }
            
        }
    }



    rcsc::Vector2D dribble_target = M_target_point;

    rcsc::Vector2D goal_center( 52, 0 );
    rcsc::Vector2D goal_left( 48, -10 );
    rcsc::Vector2D goal_right( 48, 10 );

    rcsc::Vector2D back_target = wm.self().pos() + rcsc::Vector2D( -5.0, 0 );

    if( wm.getOpponentGoalie() )
    {
        double tmp_d = std::min( 51.0, std::max( 48.0, wm.getOpponentGoalie()->pos().x ) );
        goal_center.x = tmp_d;
        goal_left.x = tmp_d;
        goal_right.x = tmp_d;
    }
    rcsc::Vector2D self_front = wm.self().pos() + rcsc::Vector2D( 10, 0 );
    if( self_front.y > 32 )
    {
        self_front.y = 32;
    }
    else if( self_front.y < -32 )
    {
        self_front.y = -32;
    }
    if( self_front.x > 50 )
    {
        self_front.x = 50;
    }
    
    int opp_num = 10, tmp_num = 0;
    if( wm.self().pos().y < -10 )
    {
        tmp_num = Bhv_Offense().checkReceiver( agent, wm.self().pos(), goal_left );
        if( tmp_num < opp_num )
        {
            opp_num = tmp_num;
            dribble_target = goal_left;
        }
    }
    else if( wm.self().pos().y <= 10 )
    {
        tmp_num = Bhv_Offense().checkReceiver( agent, wm.self().pos(), goal_center );
        if( tmp_num < opp_num )
        {
            opp_num = tmp_num;
            dribble_target = goal_center;
        }
    }
    else
    {
        tmp_num = Bhv_Offense().checkReceiver( agent, wm.self().pos(), goal_right );
        if( tmp_num < opp_num )
        {
            opp_num = tmp_num;
            dribble_target = goal_right;
        }
    }
    if( wm.self().pos().x < 45 )
    {
        tmp_num = Bhv_Offense().checkReceiver( agent, wm.self().pos(), self_front );
        if( tmp_num < opp_num )
        {
            opp_num = tmp_num;
            dribble_target = self_front;
        }

    }
    
    if( wm.self().pos().x > 40 )
    {
        rcsc::Vector2D top_left( wm.self().pos().x - 5, -10 );
        rcsc::Vector2D bottom_right( wm.self().pos().x, 10 );
        if( wm.self().pos().y < 0 )
        {
            top_left.y = 0;
            bottom_right.y = wm.self().pos().y;
        }
        else
        {
            top_left.y = wm.self().pos().y;
            bottom_right.y = 0;
        }
        rcsc::Rect2D rect( top_left, bottom_right );
        agent->debugClient().addLine( top_left, bottom_right );
        if( !wm.existOpponentIn( rect, 20, true ) 
            && Bhv_Offense().checkReceiver( agent, wm.self().pos(), back_target ) == 0 )
        {
            dribble_target = back_target;
            opp_num = 10;
        }
    }






    rcsc::AngleDeg vel_dif = wm.ball().vel().th() - tmp_target.th();
    if( !acted && vel_dif.abs() > 90 && wm.ball().vel().r() > 1.0 )
    {
        agent->debugClient().addMessage( "Trap" );
        rcsc::Body_HoldBall().execute( agent );
        agent->addSayMessage( new rcsc::BallPlayerMessage( agent->effector().queuedNextBallPos(), 
                                                           agent->effector().queuedNextBallVel(),
                                                           wm.self().unum(),
                                                           agent->effector().queuedNextMyPos(),
                                                           agent->effector().queuedNextMyBody() ) );
        acted = true;
    }


    rcsc::Vector2D pass_target = next_self_pos;

    double first_speed;


    rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromSelf().end();
    std::vector< rcsc::PlayerObject* > cand_mates;
//    std::vector< rcsc::PlayerObject* > sub_mates;
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
         it != end;
         ++it )
    {
        if( (*it)->pos().x >= wm.self().pos().x - 15
            && (*it)->pos().dist( wm.self().pos() ) >= 5 
            && (*it)->posCount() <= 3 
            && (*it)->pos().x < wm.offsideLineX() )
        {
            cand_mates.push_back( (*it) );
        }
        /*      
        if( (*it)->pos().x >= wm.self().pos().x - 5
            && (*it)->pos().dist( wm.self().pos() ) >= 5 
            && (*it)->posCount() <= 3 )
        {
            sub_mates.push_back( (*it) );
        }
        */
    }



    if( !acted && !cand_mates.empty() )
    {
        for( std::vector< rcsc::PlayerObject* >::iterator it1 = cand_mates.begin();
             it1 != cand_mates.end();
             ++it1 )
        {
            for( std::vector< rcsc::PlayerObject* >::iterator it2 = it1;
                 it2 != cand_mates.end();
                 ++it2 )
            {
                if( (*it2)->pos().absY() < (*it1)->pos().absY() )
                {
                    swap_ranges( it1, it1 + 1, it2 );
                }
            }
        }
        int self_opp = opp_num;
        for( int i = 0; i < (int)cand_mates.size(); i++ )
        {
            rcsc::PlayerObject * receiver = cand_mates.at( i );
            std::vector< rcsc::PlayerObject* > subs;
            rcsc::Vector2D mate_front = receiver->pos() + rcsc::Vector2D( 10, 0 );
            int mate_opp = 10;
            if( receiver->pos().y < -10 )
            {
                tmp_num = Bhv_Offense().checkReceiver( agent, receiver->pos(), goal_left );
                if( tmp_num < mate_opp )
                {
                    mate_opp = tmp_num;
                }
            }
            else if( wm.self().pos().y <= 10 )
            {
                tmp_num = Bhv_Offense().checkReceiver( agent, receiver->pos(), goal_center );
                if( tmp_num < mate_opp )
                {
                    mate_opp = tmp_num;
                }
            }
            else
            {
                tmp_num = Bhv_Offense().checkReceiver( agent, receiver->pos(), goal_right );
                if( tmp_num < mate_opp )
                {
                    mate_opp = tmp_num;
                }
            }

            if( receiver->pos().x < 50 )
            {
                tmp_num = Bhv_Offense().checkReceiver( agent, wm.self().pos(), mate_front );
                if( tmp_num < mate_opp )
                {
                    mate_opp = tmp_num;
                }
            }

            if( receiver->pos().x > wm.self().pos().x 
                && mate_opp == self_opp )
            {
                mate_opp--;
            }

            if( mate_opp < self_opp )
            {

                if( Opuci_OtherPass().search( agent, receiver, pass_target, first_speed, false ) )
                {
                    agent->debugClient().addMessage( "Target%d", receiver->unum() );
                    agent->debugClient().setTarget( pass_target );
                    if( rcsc::Body_SmartKick( pass_target, first_speed, first_speed * 0.8, 3 ).execute( agent ) )
                    {
                        agent->addSayMessage( new rcsc::PassMessage( receiver->unum(), pass_target,
                                                                     agent->effector().queuedNextBallPos(),
                                                                     agent->effector().queuedNextBallVel() ) );
                    }
                    
                    acted = true;
                    break;
                }     
                else if( Opuci_DirectPass().search( agent, receiver, pass_target, first_speed ) )
                {
                    agent->debugClient().addMessage( "Target%d", receiver->unum() );
                    agent->debugClient().setTarget( pass_target );
                    if( rcsc::Body_SmartKick( pass_target, first_speed, first_speed * 0.8, 3 ).execute( agent ) )
                    {
                        agent->addSayMessage( new rcsc::PassMessage( receiver->unum(), pass_target,
                                                                     agent->effector().queuedNextBallPos(),
                                                                     agent->effector().queuedNextBallVel() ) );
                    }

                    acted = true;
                    break;
                }
                else if( Bhv_Offense().getSubReceiver( agent, cand_mates.at( i ), subs ) )
                {
                    for( int j = 0; j < (int)subs.size(); j++ )
                    {
                        receiver = subs.at( j );
                        if( Opuci_DirectPass().search( agent, receiver, pass_target, first_speed ) )
                        {
                            agent->debugClient().addMessage( "Target%d", receiver->unum() );
                            agent->debugClient().addLine( wm.self().pos(), subs.at( j )->pos() );
                            agent->debugClient().addLine( subs.at( j )->pos(), cand_mates.at( i )->pos() );
                            agent->debugClient().setTarget( pass_target );
                            if( rcsc::Body_SmartKick( pass_target, first_speed, first_speed * 0.8, 3 ).execute( agent ) )
                            {
                                agent->addSayMessage( new rcsc::PassMessage( receiver->unum(), pass_target,
                                                                             agent->effector().queuedNextBallPos(),
                                                                             agent->effector().queuedNextBallVel() ) );
                            }
                            acted = true;
                            break;
                        }
                        else if( Opuci_OtherPass().search( agent, receiver, pass_target, first_speed, false ) )
                        {
                            agent->debugClient().addMessage( "Target%d", receiver->unum() );
                            agent->debugClient().addLine( wm.self().pos(), subs.at( j )->pos() );
                            agent->debugClient().addLine( subs.at( j )->pos(), cand_mates.at( i )->pos() );
                            agent->debugClient().setTarget( pass_target );
                            if( rcsc::Body_SmartKick( pass_target, first_speed, first_speed * 0.8, 3 ).execute( agent ) )
                            {
                                agent->addSayMessage( new rcsc::PassMessage( receiver->unum(), pass_target,
                                                                             agent->effector().queuedNextBallPos(),
                                                                             agent->effector().queuedNextBallVel() ) );
                            }
                            
                            acted = true;
                        }
                    }
                }
            }
        }
    }
    if( !acted && opp_num == 0 )
    {
        rcsc::Vector2D self_to_drib = dribble_target - wm.self().pos();
        if( dribble_target != back_target 
            && Opuci_SelfPass( self_to_drib.th() ).execute( agent, true ) )
        {
            agent->debugClient().addMessage( "BodySelfPass" );
            acted = true;
        }
        else if( Body_DribbleOpuci( dribble_target, 1.0, sp.maxPower(), 4, false ).execute( agent ) )
        {
            agent->debugClient().addMessage( "FDrib%d", __LINE__ );
            acted = true;
        }
    }


/*
    bool dodge = false;

    if( wm.self().pos().x > 30
        && wm.self().pos().absY() < 20 )
    {
        dodge = false;
    }

    if( opp_num <= 1 )
    {
        if( Body_DribbleOpuci( dribble_target, 1.0, sp.maxPower(), 3, dodge ).execute( agent ) )
        {
            agent->debugClient().addMessage( "3Dribble" );
            return true;
        }
    }
*/




    if( !acted && danger )
    {
        rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromSelf().end();
        for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
             it != end;
             ++it )
        {        
            const rcsc::PlayerObject * receiver = (*it);
            if( receiver->pos().absY() < 10 )
            {
                if( Opuci_OtherPass().search( agent, receiver, pass_target, first_speed, true ) )
                {
                    agent->debugClient().addMessage( "Target%d", receiver->unum() );
                    agent->debugClient().setTarget( pass_target );
                    if( rcsc::Body_SmartKick( pass_target, first_speed, first_speed * 0.5, 1 ).execute( agent ) )
                    {
                        agent->addSayMessage( new rcsc::PassMessage( receiver->unum(), pass_target,
                                                                     agent->effector().queuedNextBallPos(),
                                                                     agent->effector().queuedNextBallVel() ) );
                    }
                
                    acted = true;
                    break;
                }
            }
        }
    }
    
    

    if( !acted )
    {
        Body_DribbleOpuci( dribble_target, 1.0, sp.maxPower(), 1, false ).execute( agent );
        agent->debugClient().addMessage( "OneDribble" );
    }
    
    
    
    agent->addSayMessage( new rcsc::BallPlayerMessage( agent->effector().queuedNextBallPos(), 
                                                       agent->effector().queuedNextBallVel(),
                                                       wm.self().unum(),
                                                       agent->effector().queuedNextMyPos(),
                                                       agent->effector().queuedNextMyBody() ) );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */

/*-------------------------------------------------------------------*/

bool 
Bhv_Shootchance::testPass( rcsc::PlayerAgent * agent, rcsc::Vector2D & target_pos, int & mate_unum, bool allow_back )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    std::vector< rcsc::Vector2D > candidates;
    setFormationPos( agent, candidates );
    

    int max_opp = 0;
    int unum = 0;
    rcsc::Vector2D max_target = wm.self().pos();


    const rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromBall().begin();
         it != end;
         ++it )
    {
        if( (*it)->unum() < 7 )
        {
            continue;
        }
        rcsc::Vector2D tmp_target( 52, 0 );
        switch( (*it)->unum() )
        {
        case 7:
            tmp_target = candidates.at( 0 );
            break;
        case 8:
            tmp_target = candidates.at( 1 );
            break;
        case 9:
            tmp_target = candidates.at( 2 );
            break;
        case 10:
            tmp_target = candidates.at( 3 );
            break;
        case 11:
            tmp_target = candidates.at( 4 );
            break;
        }
        if( !allow_back
            && tmp_target.x < wm.self().pos().x )
        {
            continue;
        }


        int opp_cycle = 0;

        if( checkPassSafe( agent, tmp_target, *it, opp_cycle ) )
        {
            if( opp_cycle > max_opp )
            {
                max_opp = opp_cycle;
                max_target = tmp_target;
                unum = (*it)->unum();
            }
        }



        rcsc::Vector2D middle_target = ( tmp_target + (*it)->pos() ) / 2.0;
        if( allow_back
            || middle_target.x > wm.self().pos().x )
        {
	  if( checkPassSafe( agent, middle_target, *it, opp_cycle ) )
	    {
	      if( opp_cycle > max_opp )
		{
		  max_opp = opp_cycle;
		  max_target = middle_target;
		  unum = (*it)->unum();
		}
	    }
        }

        if( allow_back
            || (*it)->pos().x > wm.self().pos().x )
        {
	  if( checkPassSafe( agent, (*it)->pos(), *it, opp_cycle ) )
	    {
	      if( opp_cycle > max_opp )
		{
		  max_opp = opp_cycle;
		  max_target = middle_target;
		  unum = (*it)->unum();
		}
	    }
        }
        
    }

    if( max_opp != 0 )
    {
        target_pos = max_target;
        mate_unum = unum;
        return true;
    }
    
    return false;
}


bool 
Bhv_Shootchance::checkPassSafe( rcsc::PlayerAgent * agent, rcsc::Vector2D target_pos, rcsc::PlayerObject * receiver, int & opp_cycle )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();
    int turn_cycle = 1;
    if( receiver->pos().x > target_pos.x )
    {
        turn_cycle++;
    }
    
    opp_cycle = 0;
    double dist = ( target_pos - receiver->pos() ).r();
    dist -= receiver->playerTypePtr()->kickableMargin();
    int dash_cycle = receiver->playerTypePtr()->cyclesToReachDistance( dist );
    
    int reach_cycle = turn_cycle + dash_cycle;

    if( reach_cycle >= 10 )
    {
        return false;
    }
    //agent->debugClient().addMessage( "M%d:%d", receiver->unum(), reach_cycle );    
    double tmp_dist = wm.ball().pos().dist( target_pos );
    double first_speed = rcsc::calc_first_term_geom_series_last( 1.2, tmp_dist, sp.ballDecay() );
    int max_ball_cycle = rcsc::calc_length_geom_series( first_speed, tmp_dist, sp.ballDecay() );
    if( max_ball_cycle <= 0 )
    {
        max_ball_cycle = 1;
    }
    if( max_ball_cycle < reach_cycle )
    {
        //agent->debugClient().addMessage( "b%d", max_ball_cycle );
        return false;
    }
    rcsc::Vector2D rvec = target_pos - wm.ball().pos();
    rcsc::Vector2D max_accel( wm.self().kickRate() * 100, 0 );
    max_accel.rotate( rvec.th() );
    rcsc::Vector2D acceled_ball = wm.ball().vel() + max_accel;
    if( acceled_ball.r() < first_speed * 0.9 )
    {
        //agent->debugClient().addMessage( "BadAccel" );
        return false;
    }
    
    int opp_min = 50;
    const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
         it != end;
         ++it )
    {    
        rcsc::Segment2D ball_to_target( wm.ball().pos(), target_pos );
        rcsc::Vector2D next_opp_pos = (*it)->pos() + (*it)->vel();

        rcsc::Circle2D opp_area( next_opp_pos, (*it)->playerTypePtr()->kickableMargin() * 1.2 );
        if( opp_area.intersection( ball_to_target, NULL, NULL ) > 0 )
        {
            return false;
        }


        double opp_dist = ball_to_target.dist( next_opp_pos );
        opp_dist -= (*it)->playerTypePtr()->kickableMargin();
        int opp_dash = (*it)->playerTypePtr()->cyclesToReachDistance( opp_dist );
        int opp_turn = 1;
        int tmp_opp_cycle = opp_turn + opp_dash;
        
        rcsc::Vector2D opp_target =  ball_to_target.nearestPoint( next_opp_pos );
        double ball_dist = wm.ball().pos().dist( opp_target );
        
        int ball_cycle = rcsc::calc_length_geom_series( first_speed, ball_dist, sp.ballDecay() );
        if( ball_cycle <= 0 )
        {
            ball_cycle = 1;
        }
        if( tmp_opp_cycle <= ball_cycle )
        {
            return false;
        }

        opp_dist = target_pos.dist( next_opp_pos );
        opp_dist -= (*it)->playerTypePtr()->kickableMargin();
        opp_dash = (*it)->playerTypePtr()->cyclesToReachDistance( opp_dist );
        tmp_opp_cycle = opp_turn + opp_dash;
        

        if( max_ball_cycle >= tmp_opp_cycle )
        {
            return false;
        }
//        agent->debugClient().addMessage( "O%d,%d>%d", (*it)->unum(), tmp_opp_cycle, ball_cycle );
        if( tmp_opp_cycle < opp_min )
        {
            opp_min = tmp_opp_cycle;
        }
    }
    if( opp_min < 50 )
    {
        opp_cycle = opp_min;
        return true;
    }
    return false;
}

bool
Bhv_Shootchance::checkDribbleSafe( rcsc::PlayerAgent * agent, rcsc::Vector2D target_pos, int dash_cycle )
{
    const rcsc::WorldModel & wm = agent->world();
    int kick_cycle = 1;
    int turn_cycle = 1;
    int dribble_cycle = kick_cycle + turn_cycle + dash_cycle;
   
    rcsc::Vector2D inertial_self_pos = wm.self().inertiaPoint( dribble_cycle );

    rcsc::Vector2D rpos = target_pos - inertial_self_pos;
    
    rcsc::Vector2D move_pos = inertial_self_pos;
    double self_speed = 0;
    double self_dist = 0;
    for( int j = 0; j < dash_cycle; j++ )
    {
        self_speed += wm.self().dashRate() * 100;
        if( self_speed >= wm.self().playerType().playerSpeedMax() )
            self_speed = wm.self().playerType().playerSpeedMax();
        self_dist += self_speed;
        self_speed *= wm.self().playerTypePtr()->playerDecay();
    }    
    rcsc::Vector2D dash_vector( self_dist, 0 );
    dash_vector.rotate( rpos.th() );
    move_pos += dash_vector;
    const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
         it != end;
         ++it )
    {
        rcsc::Vector2D inertial_opp_pos = (*it)->playerTypePtr()->inertiaPoint( (*it)->pos(),
                                                                                (*it)->vel(),
                                                                                dribble_cycle );
        double dist = inertial_opp_pos.dist( move_pos );
        dist -= wm.self().kickableArea();
        dist -= (*it)->playerTypePtr()->kickableMargin();
        if( dist <= 0 )
        {
            return false;
        }
        if( (*it)->playerTypePtr()->cyclesToReachDistance( dist ) < dribble_cycle )
        {
            return false;
        }
    }
    return true;
}


bool
Bhv_Shootchance::doSelfPass( rcsc::PlayerAgent * agent, rcsc::AngleDeg angle, int dash_cycle )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    int turn_cycle = 1;
    int kick_cycle = 1;
    
    double degree = angle.degree();
    double moment = degree - wm.self().body().degree();
    if( fabs( moment ) < 10 )
    {
        turn_cycle = 0;
    }
    if( wm.self().canTurn( moment ) )
    {
#if 1
        int total_cycle = kick_cycle + turn_cycle + dash_cycle;
        rcsc::Vector2D moved_pos = wm.self().pos();
        rcsc::Vector2D inertial_move = wm.self().vel();
        
        double stamina = wm.self().stamina();
        double effort = wm.self().effort();
        double recovery = wm.self().recovery();
        double power = 100;

        for( int i = 0; i < kick_cycle; i++ )
        {
            moved_pos += inertial_move;
            inertial_move *= wm.self().playerType().playerDecay();
            stamina += recovery * wm.self().playerType().staminaIncMax();
            if( stamina > sp.staminaMax() )
                stamina = sp.staminaMax();
        }

        for( int i = 0; i < turn_cycle; i++ )
        {
            moved_pos += inertial_move;
            inertial_move *= wm.self().playerType().playerDecay();
            stamina += recovery * wm.self().playerType().staminaIncMax();
            if( stamina > sp.staminaMax() )
                stamina = sp.staminaMax();
        }

        rcsc::Vector2D dash_move( 1, 0 );
        dash_move.rotate( degree );
        for( int i = 0; i < dash_cycle; i++ )
        {
            if( i == 0 )
            {
                dash_move.setLength( effort * wm.self().playerType().dashPowerRate() * power );
            }
            else
            {
                dash_move.setLength( dash_move.r() + effort * wm.self().playerType().dashPowerRate() * power );
            }
	    if( dash_move.r() > wm.self().playerType().playerSpeedMax() )
                dash_move.setLength( wm.self().playerType().playerSpeedMax() );
            
            rcsc::Vector2D tmp_move = dash_move + inertial_move;
            if( tmp_move.r() > wm.self().playerType().playerSpeedMax() )
                tmp_move.setLength( wm.self().playerType().playerSpeedMax() );

            moved_pos += tmp_move;
            
            stamina -= power;
            if( stamina <= sp.recoverDecThr() * sp.staminaMax() )
            {
                if( recovery > sp.recoverMin() )
                    recovery -= sp.recoverDec();
                if( recovery < sp.recoverMin() )
                    recovery = sp.recoverMin();
            }
            if( stamina <= sp.effortDecThr() * sp.staminaMax() )
            {
                if( effort > wm.self().playerType().effortMin() )
                    effort -= sp.recoverDec();
                if( effort < wm.self().playerType().effortMin() )
                    effort = wm.self().playerType().effortMin();
            }
            if( stamina >= sp.effortIncThr() * sp.staminaMax() )
            {
                if( effort < wm.self().playerType().effortMax() )
                {
                    effort += sp.effortInc();
                    if( effort > wm.self().playerType().effortMax() )
                        effort = wm.self().playerType().effortMax();
                }
            }
            stamina += recovery * wm.self().playerType().staminaIncMax();
            if( stamina > sp.staminaMax() )
                stamina = sp.staminaMax();

            dash_move *= wm.self().playerType().playerDecay();
            inertial_move *= wm.self().playerType().playerDecay();
        }
#endif 
/*old
        int total_cycle = kick_cycle + turn_cycle + dash_cycle;
        rcsc::Vector2D moved_pos = wm.self().inertiaPoint( total_cycle );
        double self_speed = 0;
        double self_dist = 0;
        for( int j = 0; j < dash_cycle; j++ )
        {
            self_speed += wm.self().dashRate() * 100;
	    if( self_speed >= wm.self().playerType().playerSpeedMax() )
                self_speed = wm.self().playerType().playerSpeedMax();
            self_dist += self_speed;
            self_speed *= wm.self().playerTypePtr()->playerDecay();
        }
        rcsc::Vector2D dash_vector( self_dist, 0 );
        dash_vector.rotate( degree );
        moved_pos += dash_vector;        
*/

        rcsc::Vector2D ball_margin( wm.self().kickableArea() * 0.5, 0 );
        ball_margin.rotate( degree );

        rcsc::Vector2D ball_target = moved_pos + ball_margin;

        if( ball_target.absX() > 51
            || ball_target.absY() > 33 )
        {
            return false;
        }



        double ball_dist = wm.ball().pos().dist( ball_target );
        double ball_speed = rcsc::calc_first_term_geom_series( ball_dist, sp.ballDecay(), total_cycle );
        rcsc::Vector2D ball_vector( ball_speed, 0 );
        rcsc::Vector2D rpos = ball_target - wm.ball().pos();
        ball_vector.rotate( rpos.th() );
        rcsc::Vector2D accel = ball_vector - wm.ball().vel();
        double kick_power = accel.r() / wm.self().kickRate();
        
        if( kick_power > sp.maxPower() )
        {
            return false;
        }
        
        rcsc::Vector2D next_ball_pos = wm.ball().pos() + ball_vector;
        rcsc::Circle2D next_ball_area( next_ball_pos, sp.ballSize() );

        rcsc::Vector2D next_self_pos = wm.self().pos() + wm.self().vel();
        rcsc::Circle2D next_self_area( next_self_pos, wm.self().playerTypePtr()->playerSize() + 0.1 );

        if( next_self_area.intersection( next_ball_area, NULL, NULL ) > 0 
            || next_self_area.contains( next_ball_pos ) )
        {
            return false;
        }
        
        const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
        for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
             it != end;
             ++it )
        {
            if( (*it)->isTackling() )
                continue;
	  
            const rcsc::Vector2D selfToIt = (*it)->pos() - wm.self().pos();
            const rcsc::AngleDeg difAD( selfToIt.th().degree() - angle.degree() );
            if( fabs( difAD.tan() ) > 6 )
                continue;  // たぶん抜ける

            rcsc::Vector2D inertial_opp_pos = (*it)->playerTypePtr()->inertiaPoint( (*it)->pos(),
                                                                                    (*it)->vel(),
                                                                                    total_cycle );
            double opp_dist = inertial_opp_pos.dist( ball_target );
            opp_dist -= sp.tackleDist();//(*it)->playerTypePtr()->kickableMargin() * 1.2;
            if( opp_dist <= 0 )
            {
                return false;
            }
            if( (*it)->playerTypePtr()->cyclesToReachDistance( opp_dist ) <= total_cycle )
            {
                return false;
            }
        }

        agent->doKick( kick_power, accel.th() - wm.self().body() );
        agent->debugClient().addCircle( ball_target, 0.1 );
        agent->debugClient().addCircle( moved_pos, 1.0 );

        agent->setIntention( new Opuci_IntentionDribble( ball_target, dash_cycle + 1 ) );
        

        return true;

    }//end if canturn
    return false;
}

/*-------------------------------------------------------------------*/
/*!
  ホームポジションを決める関数
*/
rcsc::Vector2D
Bhv_Shootchance::getPosition( const rcsc::WorldModel & wm,
                          const int unum )
{
    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    int ball_reach_step = std::min( opp_min, std::min( self_min, mate_min ) );

    rcsc::Vector2D static_pos = Strategy::i().getBeforeKickOffPos( unum ); //試合開始時の自分の位置
    rcsc::Vector2D base_pos = wm.ball().inertiaPoint( ball_reach_step ); //ボールの予定位置
    rcsc::Vector2D home_pos = base_pos + static_pos;

    home_pos.x = base_pos.x + ( static_pos.x * 2.0 + 30.0 ) - 1.0 / 2.0 * base_pos.x;
    home_pos.y = static_pos.y;
    if( home_pos.x > wm.offsideLineX() - 0.5 )
    {
        home_pos.x = wm.offsideLineX() - 0.5;
    }
    return home_pos;
}
bool 
Bhv_Shootchance::existTri( rcsc::PlayerAgent * agent, rcsc::Vector2D player_pos )
{
    const rcsc::WorldModel & wm = agent->world();
    rcsc::Vector2D goal_left( 52, -7 );
    rcsc::Vector2D goal_right( 52, 7 );
    rcsc::Triangle2D tri( player_pos, goal_left, goal_right );
    bool exist_tri = false;
    const rcsc::PlayerPtrCont::const_iterator end = wm.opponentsFromSelf().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromSelf().begin();
         it != end;
         ++it )
    {
        if( !(*it)->goalie()
            && tri.contains( (*it)->pos() ) )
        {
            exist_tri = true;
            break;
        }
    }
    if( !exist_tri )
    {
        agent->debugClient().addLine( player_pos, goal_left );
        agent->debugClient().addLine( player_pos, goal_right );
        agent->debugClient().addLine( goal_left, goal_right );
    }
    return exist_tri;
}
