#include <stdlib.h>
#include <vector>
#include <math.h>
#include <iostream>
#include <fstream>

template< int INPUT,
  int HIDDEN,
  int OUTPUT>

class Neuro{
 private:
  const double M_eta;
  const double M_alpha;

  double Weight_i_to_h[HIDDEN][INPUT + 1];
  double Weight_h_to_o[OUTPUT][HIDDEN + 1];
  double Delta_weight_i_to_h[HIDDEN][INPUT + 1];
  double Delta_weight_h_to_o[OUTPUT][HIDDEN + 1];
  double Hidden_layer[HIDDEN];

 public:
  Neuro( double eta, double alpha )
    :M_eta( eta ),
    M_alpha( alpha )

    { }

  void init()
  {
    for( int i = 0; i < HIDDEN; ++i )
      {
	for( int j = 0; j < INPUT + 1; ++j )
	  {
	    Weight_i_to_h[i][j] = 0.0;
	    Delta_weight_i_to_h[i][j] = 0.0;
	  }
      }
    for( int i = 0; i < OUTPUT; ++i )
      {
	for( int j = 0; j < HIDDEN + 1; ++j )
	  {
	    Weight_h_to_o[i][j] = 0.0;
	    Delta_weight_h_to_o[i][j] = 0.0;
	  }
      }
    for( int i = 0; i < HIDDEN; ++i )
      {
	Hidden_layer[i] = 0.0;
      }
  }

  void randomize()
  {
    srand( (unsigned)time( NULL ) );
    for( int i = 0; i < HIDDEN; ++i )
      {
	for( int j = 0; j < INPUT + 1; ++j )
	  {
	    int rnd = rand();
	    double rng = (double)rnd / (double)RAND_MAX - 0.5;
	    Weight_i_to_h[i][j] = rng;
	  }
      }
    for( int i = 0; i < OUTPUT; ++i )
      {
	for( int j = 0; j < HIDDEN + 1; ++j )
	  {
	    int rnd = rand();
	    double rng = (double)rnd / (double)RAND_MAX - 0.5;
	    Weight_h_to_o[i][j] = rng;
	  }
      }
  }

  void propagate( const double input[],
		  double output[] )
  {
    for( int i = 0; i < HIDDEN; ++i )
      {
	double sum = 0;
	for ( int j = 0; j < INPUT; ++j )
	  {
	    sum += input[j] * Weight_i_to_h[i][j];
	  }
	sum += Weight_i_to_h[i][INPUT];
	Hidden_layer[i] = func_sig( sum );
      }
    for( int i = 0; i < OUTPUT; ++i )
      {
	double sum = 0;
	for ( int j = 0; j < HIDDEN; ++j )
	  {
	    sum += Hidden_layer[j] * Weight_h_to_o[i][j];
	  }
	sum += Weight_h_to_o[i][HIDDEN];
	output[i] = func_sig( sum );
      }
  }

  void train( const double input[],
	      const double teacher[] )
  {
    double output[OUTPUT];
    double output_back[OUTPUT];
    double hidden_back[HIDDEN];

    propagate( input, output );

    for( int i = 0; i < OUTPUT; ++i )
      {
	double err = teacher[i] - output[i];
	output_back[i] = err * func_back( output[i] );
      }
    for ( int i = 0; i < HIDDEN; ++i )
      {
	double sum = 0.0;
	for( int j = 0; j < OUTPUT; ++j )
	  {
	    sum += output_back[j] * Weight_h_to_o[j][i];
	  }
	hidden_back[i] = sum * func_back( Hidden_layer[i] );
      }

    for ( int i = 0; i < OUTPUT; ++i )
      {
	for( int j = 0; j < HIDDEN; ++j )
	  {
	    Delta_weight_h_to_o[i][j]
	      = M_eta * Hidden_layer[j] * output_back[i]
	      + M_alpha * Delta_weight_h_to_o[i][j];
	    Weight_h_to_o[i][j]
	      += Delta_weight_h_to_o[i][j];
	  }
	Delta_weight_h_to_o[i][HIDDEN]
	  = M_eta * output_back[i]
	  + M_alpha * Delta_weight_h_to_o[i][HIDDEN];
	Weight_h_to_o[i][HIDDEN]
	  += Delta_weight_h_to_o[i][HIDDEN];
      }

    for( int i = 0; i < HIDDEN; ++ i )
      {
	for ( int j = 0; j < INPUT; ++j )
	  {
	    Delta_weight_i_to_h[i][j]
	      = M_eta * input[j] * hidden_back[i]
	      + M_alpha * Delta_weight_i_to_h[i][j];
	    Weight_i_to_h[i][j]
	      += Delta_weight_i_to_h[i][j];
	  }
	Delta_weight_i_to_h[i][INPUT]
	  = M_eta * hidden_back[i]
	  + M_alpha * Delta_weight_i_to_h[i][INPUT];
	Weight_i_to_h[i][INPUT]
	  += Delta_weight_i_to_h[i][INPUT];
      }
  }

  std::ostream & savefile( std::ostream & os ) const
  {
    for( int i = 0; i < HIDDEN; i++ )
      {
	for( int j = 0; j < INPUT + 1; j++ )
	  {
	    os << Weight_i_to_h[i][j] << " ";
	  }
      }
    for( int i = 0; i < OUTPUT; i ++ )
      {
	for( int j = 0; j < HIDDEN + 1; j++ )
	  {
	    os << Weight_h_to_o[i][j] << " ";
	  }
      }
    return os;
  }

  bool loadfile( std::istream & is )
  {
    for( int i = 0; i < HIDDEN; i++ )
      {
	for( int j = 0; j < INPUT + 1; j++ )
	  {
	    if( ! is.good() ) return false;
	    is >> Weight_i_to_h[i][j];
	  }
      }
    for( int i = 0; i < OUTPUT; i++ )
      {
	for( int j = 0; j < HIDDEN + 1; j++ )
	  {
	    if( ! is.good() ) return false;
	    is >> Weight_h_to_o[i][j];
	  }
      }
    return true;
  }

 private:
  double func_sig( const double x )
  {
    return 1.0 / ( 1.0 + exp( -x ) );
  }

  double func_back( const double x )
  {
    return x * ( 1.0 - x );
  }
};
