#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "opuci_chase_ball.h"
#include "opuci_move.h"
#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/geom/angle_deg.h>
#include <rcsc/geom/line_2d.h>
#include <rcsc/geom/ray_2d.h>
/*-------------------------------------------------------------------*/
/*!

 */
int
Opuci_ChaseBall::selfCycle( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    rcsc::Vector2D ball_pos = wm.ball().pos();
    rcsc::Vector2D ball_vel = wm.ball().vel();

    if( wm.ball().pos().dist( wm.self().pos() ) < wm.self().playerType().kickableArea() )
    {
        return 0;
    }

    for( int i = 0; i <= 30; i++ )
    {
        int turn_cycle = turnCycle( ball_pos, agent );
        int dash_cycle = dashCycle( ball_pos, turn_cycle, agent );
        if( turn_cycle + dash_cycle <= i )
	{
            agent->debugClient().addMessage( "cycle:%d,%d", turn_cycle, dash_cycle );
            return i;
	}
        ball_pos += ball_vel;
        ball_vel *= rcsc::ServerParam::i().ballDecay();
    }
    return 30;
}
/*-------------------------------------------------------------------*/
/*!

 */
int
Opuci_ChaseBall::turnCycle( const rcsc::Vector2D target_pos,
			    rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
 
    const double player_size = wm.self().playerType().kickableArea();
    const rcsc::Line2D body_right_line( rcsc::Vector2D( wm.self().pos().x + player_size * wm.self().body().sin(),
                                                        wm.self().pos().y - player_size * wm.self().body().cos() ),
                                        wm.self().body() );
    const rcsc::Line2D body_left_line( rcsc::Vector2D( wm.self().pos().x - player_size * wm.self().body().sin(),
                                                       wm.self().pos().y + player_size * wm.self().body().cos() ),
                                       wm.self().body() );

    const double target_inside =
        ( body_right_line.getA() * target_pos.x + body_right_line.getB() * target_pos.y + body_right_line.getC() )
        * ( body_left_line.getA() * target_pos.x + body_left_line.getB() * target_pos.y + body_left_line.getC() );

    if( target_inside <= 0.0
        && fabs( ( wm.self().body() - rcsc::Vector2D( target_pos - wm.self().pos() ).th() ).degree() ) < 90.0 )
    {
        return 0;
    }

    rcsc::AngleDeg target_angle = ( wm.self().body() - ( target_pos - wm.self().pos() ).th() ).abs();
    double player_speed = wm.self().vel().r();

    int i;
    for( i = 1; i <= 5; i++ )
    {
        rcsc::AngleDeg angle = 180.0 / ( 1.0 + wm.self().playerType().inertiaMoment() * player_speed );
        target_angle -= angle;
        player_speed *= wm.self().playerType().playerDecay();

        if( target_angle.degree() <= 0.0 )
	{
            return i;
	}
    }

    return i;
}
/*-------------------------------------------------------------------*/
/*!

 */
int
Opuci_ChaseBall::dashCycle( const rcsc::Vector2D target_pos,
			    const int turn_cycle,
			    rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    double dist_to_point = wm.self().pos().dist( target_pos );
    double player_speed = wm.self().vel().r() * std::pow( wm.self().playerType().playerDecay(), turn_cycle );

    int i;
    for( i = 0; i <= 25; i++ )
    {
        if( dist_to_point < wm.self().playerType().kickableArea() )
	{
            return i;
	}
        player_speed += wm.self().effort() * wm.self().playerType().dashPowerRate() * rcsc::ServerParam::i().maxPower();
        dist_to_point -= player_speed;
        player_speed *= wm.self().playerType().playerDecay();
    }

    return i;
}


/*-------------------------------------------------------------------*/
/*!

 */
/*
bool
Opuci_ChaseBall::execute( rcsc::PlayerAgent * agent )
{
    return execute2( agent );
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    agent->debugClient().addMessage( "ChaseBall" );
    if( wm.self().isKickable() )
    {
        agent->debugClient().addMessage( "AlreadyKickable" ); 
        return false;
    }
    if( wm.ball().vel().r() < 0.1 )
    {
        if( !Opuci_Move( wm.ball().pos(), 100, 0.5 ).execute( agent ) )
        {
            //reached
            rcsc::Vector2D turn_pos = wm.ball().pos() + wm.ball().vel();
            rcsc::Vector2D dif = turn_pos - wm.self().pos() - wm.self().vel();
            agent->doTurn( dif.th() - wm.self().body() );
        }
        
        return true;
    }

    int self_cycle = calcCycle( agent );
    int self_min = wm.interceptTable()->selfReachCycle();
    agent->debugClient().addMessage( "WorldSelfMin%d,CalcedSelfMin%d", self_min, self_cycle );
    rcsc::Vector2D inertial_ball_pos = wm.ball().inertiaPoint( self_cycle );
    rcsc::Vector2D inertial_self_pos = wm.self().inertiaPoint( self_cycle );
    rcsc::Ray2D ball_ray( wm.ball().pos(), wm.ball().vel().th() );
    rcsc::Ray2D self_ray( inertial_self_pos, wm.self().body() );
    rcsc::Line2D ball_line( wm.ball().pos(), wm.ball().vel().th() );
    
    rcsc::Vector2D tmp_target = self_ray.intersection( ball_ray );
    rcsc::Vector2D next_self_pos = wm.self().pos() + wm.self().vel();
    rcsc::Circle2D twostep_self_area( wm.self().inertiaPoint( 2 ),
                                      wm.self().kickableArea() * 0.9 );
    rcsc::Circle2D next_self_area( wm.self().inertiaPoint( 1 ),
                                   wm.self().kickableArea() * 0.9 );
    rcsc::Circle2D current_self_area( wm.self().pos(),
                                      wm.self().kickableArea() * 0.9 );

    if( current_self_area.intersection( ball_ray, NULL, NULL ) == 2 )
    {
        if( next_self_area.intersection( ball_ray, NULL, NULL ) < 2 
            || twostep_self_area.intersection( ball_ray, NULL, NULL ) < 2 )
        {
            //brake
            double dash_power = wm.self().vel().r() / wm.self().dashRate();
            agent->doDash( -dash_power );
            agent->debugClient().addMessage( "Brake1" );
            return true;
        }
        //wait
        if( self_cycle > 10 )
        {
            if( Opuci_Move( inertial_ball_pos, 100, 0.5 ).execute( agent ) )
            {
                return true;
            }
        }

        rcsc::Vector2D target_pos( 51, 0 );
        rcsc::Vector2D rpos = target_pos - next_self_pos;
	if( fabs( rpos.th().degree() - wm.self().body().degree() ) < 5.0 )
        {
	    double min_dist = 100000.0;
	    double opt_dir = -1000.0;
	    rcsc::Vector2D sim_next_pos;
	    rcsc::Vector2D sim_next_pos_final;
	    double effort = wm.self().effort();
	    rcsc::Circle2D sim_nextself_area;
            double dist2bline = ball_line.dist( next_self_pos );

	    for( int i = 0 ; i < 8 ; i++ )  // clockwise. 0: back, 1:back-left,..., 7:back-right
            {
		double degree = ((double)i-4.0) * 45.0;
		rcsc::Vector2D move( 1.0, 0 );
		move.rotate( wm.self().body().degree() + degree );
		double dash_power;
		if( i == 0 )
                    dash_power = 25.0;
		else if( i == 1 || i == 7 )
                    dash_power = 37.75;
		else if( i == 2 || i == 6 )
                    dash_power = 50.0;
		else if( i == 3 || i == 5 )
                    dash_power = 75.0;
		else if( i == 4 )
                    dash_power = 100.0;

		move.setLength( effort * wm.self().playerType().dashPowerRate() * dash_power );
		move += wm.self().vel();
		if( move.length() > wm.self().playerType().playerSpeedMax() )
                    move.setLength( wm.self().playerType().playerSpeedMax() );
		sim_next_pos = wm.self().pos() + move;
		sim_nextself_area.assign( sim_next_pos, wm.self().kickableArea() * 0.8 );
		// condition: get closer after the move
		if( wm.self().pos().dist( wm.ball().pos() ) < sim_next_pos.dist( wm.ball().pos() ) )
                    continue;
		// condition: intersect with ball_ray
		if( sim_nextself_area.intersection( ball_ray, NULL, NULL ) != 2 )
                    continue;
                // condition: should not be further then current situation to the ball line
                if( ball_line.dist( sim_next_pos ) > dist2bline )
                    continue;

		if( sim_next_pos.dist( wm.ball().pos() ) < min_dist )
                {
		    sim_next_pos_final = sim_next_pos;
		    min_dist = sim_next_pos.dist( wm.ball().pos() );
		    opt_dir = degree;
                }
            }

	    if( opt_dir != -1000.0 )
            {
		const rcsc::AngleDeg dashDir( opt_dir );
		agent->doDash( 100.0, dashDir );
		agent->debugClient().addCircle( sim_next_pos_final, 0.1 );
		agent->debugClient().addMessage( "Approach%f", opt_dir );
		return true;
            }
        }


	agent->doTurn( rpos.th() - wm.self().body() );
	agent->debugClient().addMessage( "WaitTurn" );
	return true;

    }

    if( current_self_area.intersection( ball_line, NULL, NULL ) == 2 )
    {
        rcsc::AngleDeg angle_dif = wm.ball().vel().th() - wm.self().body();
        if( angle_dif.abs() > 15 )
        {
            agent->doTurn( angle_dif );
            agent->debugClient().addMessage( "TurnLine" );
            return true;
        }
        else
        {
            rcsc::Vector2D next_ball_pos = wm.ball().pos() + wm.ball().vel();
            rcsc::Vector2D dash_vector = next_ball_pos - wm.self().pos();
	    dash_vector.setLength( dash_vector.r() - wm.self().playerType().playerSize() );
            rcsc::Vector2D accel = dash_vector - wm.self().vel();
            double dash_power = accel.r() / wm.self().dashRate();
            if( dash_power > 100 )
            {
                dash_power = 100;
            }
            agent->doDash( dash_power );
            agent->debugClient().addMessage( "SDash" );
            return true;            
        }
    }
    

    if( !tmp_target.valid() )
    {
        //turn
        rcsc::Vector2D rpos = inertial_ball_pos - inertial_self_pos;
        rcsc::AngleDeg angle_dif = rpos.th() - wm.self().body();
        agent->doTurn( angle_dif );
        agent->debugClient().addMessage( "Turn1" );
        return true;
    }
    else
    {
        agent->debugClient().addLine( wm.self().pos(), tmp_target );
//        double dash_dist = inertial_self_pos.dist( tmp_target );
//        int dash_cycle = wm.self().playerTypePtr()->cyclesToReachDistance( dash_dist );
//        double ball_dist = rcsc::calc_sum_geom_series( wm.ball().vel().r(), sp.ballDecay(), dash_cycle );
#if 0
          if( wm.ball().pos().dist( tmp_target ) < ball_dist - wm.self().kickableArea() * 0.8 )
          {
          //$B4V$K9g$o$J$$>l9g(B
          rcsc::Vector2D rpos = inertial_ball_pos - inertial_self_pos;
          rcsc::AngleDeg angle_dif = rpos.th() - wm.self().body();
          agent->doTurn( angle_dif );
          agent->debugClient().addMessage( "Turn2" );
          return true;
          }
          else
endif
        {
            //dash
            rcsc::Vector2D dash_vector = tmp_target - wm.self().pos();
	    dash_vector.setLength( dash_vector.r() - wm.self().playerType().playerSize() );
            rcsc::Vector2D accel = dash_vector - wm.self().vel();
            double dash_power = accel.r() / wm.self().dashRate();
            if( dash_power > 100 )
            {
                dash_power = 100;
            }
            agent->doDash( dash_power );
            agent->debugClient().addMessage( "LDash" );
            return true;
        }
        
        
    }
    
    return true;
}
*/

/*-------------------------------------------------------------------*/
/*!

 */
int
Opuci_ChaseBall::calcCycle( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
//    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    if( wm.self().isKickable() )
    {
        return 0;
    }
    for( int i = 1 ; i <= 5 ; i++ )
    {
        if( approachWithinCycles( agent, i, true ) )
            return i;
    }
    

    int straight_cycle = checkStraight( agent );
    if( straight_cycle < 100 )
    {
        return straight_cycle;
    }
    //turn and dash
    int turn_cycle = 1;

    int cycle = 0;
    bool finish = false;
    while( !finish )
    {
        ++cycle;
        rcsc::Vector2D inertial_ball_pos = wm.ball().inertiaPoint( cycle );
        rcsc::Vector2D inertial_self_pos = wm.self().inertiaPoint( cycle );

        rcsc::Vector2D rpos = inertial_ball_pos - inertial_self_pos;
        double moment = rpos.th().degree() - wm.self().body().degree();

        if( wm.self().canTurn( moment ) )
        {
            turn_cycle = 1;
        }
        else
        {
            turn_cycle = 2;
        }
        double dist = inertial_self_pos.dist( inertial_ball_pos );
        int dash_cycle = wm.self().playerTypePtr()->cyclesToReachDistance( dist );
        
        if( turn_cycle + dash_cycle <= cycle )
        {
            agent->debugClient().addCircle( inertial_ball_pos, 0.6 );
            finish = true;
        }
        if( cycle >= 100 )
        {
            finish = true;
        }
    }
/*
    //dash only
    rcsc::Ray2D ball_ray( wm.ball().pos(), wm.ball().vel() );
    rcsc::Ray2D self_ray( wm.self().pos() + wm.self().vel(), wm.self().body() );
    rcsc::Vector2D intersection = ball_ray.intersection( self_ray );

    if( intersection.valid() )
    {
        double dist = intersection.dist( wm.self().pos() + wm.self().vel() );
        int tmp_cycle = wm.self().playerTypePtr()->cyclesToReachDistance( dist );
        if( tmp_cycle < cycle )
        {
            agent->debugClient().addCircle( intersection, 0.5 );
            return tmp_cycle;
        }
    }
*/    

    
    return cycle;
}
/*-------------------------------------------------------------------*/
/*!

 */
bool
Opuci_ChaseBall::execute( rcsc::PlayerAgent * agent )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: Opuci_ChaseBall::execute"
                        ,__FILE__, __LINE__ );

    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    agent->debugClient().addMessage( "ChaseBall" );
    if( wm.self().isKickable() )
    {
        agent->debugClient().addMessage( "AlreadyKickable" );
        return false;
    }






    int self_cycle = calcCycle( agent );

    rcsc::Ray2D ball_ray( wm.ball().pos(), wm.ball().vel().th() );
    if( ball_ray.line().dist( wm.self().pos() ) > wm.self().kickableArea() 
        && approachWithinCycles( agent, 5 ) )
    {
        agent->debugClient().addMessage( "Go2Bline" );
        return true;
    }



    rcsc::Vector2D self_to_ball = wm.ball().pos() - wm.self().pos();
    rcsc::AngleDeg dif = self_to_ball.th() - wm.self().body();
    if( dif.abs() < 90 )
    {
        if( checkStraight( agent ) < 100 )
        {
            rcsc::Vector2D next_ball_pos = wm.ball().pos() + wm.ball().vel();
            rcsc::Vector2D next_self_pos = wm.self().pos() + wm.self().vel();
            double dash_power = next_ball_pos.dist( next_self_pos ) / wm.self().dashRate();
            if( dash_power > sp.maxPower() )
            {
                dash_power = sp.maxPower();
            }
            agent->debugClient().addMessage( "Straight%.0f", dash_power );
            agent->doDash( dash_power );
            return true;
        }
    }




    agent->debugClient().addMessage( "WM%d SELF%d", wm.interceptTable()->selfReachCycle(), self_cycle );
    rcsc::Vector2D inertial_ball_pos = wm.ball().inertiaPoint( self_cycle );
    rcsc::Vector2D inertial_self_pos = wm.self().inertiaPoint( self_cycle );
    
    agent->debugClient().addCircle( inertial_ball_pos, 0.2 );

    rcsc::Line2D ball_line( wm.ball().pos(), wm.ball().vel().th() );
    rcsc::Vector2D body_dir( wm.self().kickableArea() * 0.5, 0 );
    body_dir.rotate( wm.self().body() );
    rcsc::Ray2D self_ray_center( inertial_self_pos, wm.self().body() );
    rcsc::Ray2D self_ray_left( inertial_self_pos + body_dir.rotatedVector( -90 ), 
                               wm.self().body() );
    rcsc::Ray2D self_ray_right( inertial_self_pos + body_dir.rotatedVector( 90 ),
                                wm.self().body() );
    
    rcsc::Vector2D next_self_pos = wm.self().pos() + wm.self().vel();
    rcsc::Circle2D next_self_area( next_self_pos,
                                   wm.self().kickableArea() * 0.9 );
    rcsc::Circle2D twostep_self_area( wm.self().inertiaPoint( 2 ),
                                      wm.self().kickableArea() * 0.9 );
    rcsc::Circle2D current_self_area( wm.self().pos(),
                                      wm.self().kickableArea() * 0.9 );
    
    rcsc::Vector2D center_target = self_ray_center.intersection( ball_ray );
    rcsc::Vector2D left_target = self_ray_left.intersection( ball_ray );
    rcsc::Vector2D right_target = self_ray_right.intersection( ball_ray );


    rcsc::Vector2D top_left( -52.2, -33.7 );
    rcsc::Vector2D bottom_right( 52, 33.7 );
    rcsc::Rect2D field( top_left, bottom_right );
    if( !field.contains( inertial_ball_pos ) )
    {
        rcsc::Vector2D line_pos;
        if( !field.intersection( ball_ray, &line_pos, NULL ) == 1 )
        {
            line_pos = wm.ball().pos();
        }
        rcsc::Vector2D rpos = line_pos - next_self_pos;
        rcsc::AngleDeg angle_dif = rpos.th() - wm.self().body();
        if( angle_dif.abs() > 15 )
        {
            agent->doTurn( angle_dif );
            agent->debugClient().addMessage( "TurnFieldLine" );
            return true;
        }
        else
        {
            double dash_power = 50;
            agent->doDash( dash_power );
            agent->debugClient().addMessage( "FieldLineDash" );
            return true; 
        }

    }




    if( current_self_area.intersection( ball_ray, NULL, NULL ) == 2 )
    {
        if( next_self_area.intersection( ball_ray, NULL, NULL ) < 2
            || twostep_self_area.intersection( ball_ray, NULL, NULL ) < 2 )
        {
            //brake
            double dash_power = wm.self().vel().r() / wm.self().dashRate();
            agent->doDash( -dash_power );
            agent->debugClient().addMessage( "Brake2" );
            return true;
        }
        
        double dist = wm.ball().pos().dist( wm.self().pos() );
        int wait_cycle = rcsc::calc_length_geom_series( wm.ball().vel().r(),
                                                        dist, sp.ballDecay() );
        int opp_min = wm.interceptTable()->opponentReachCycle();
        //go to ball
        if( opp_min < wait_cycle )
        {
            rcsc::Vector2D rpos = wm.ball().pos() - wm.self().pos();
            rcsc::AngleDeg angle_dif = rpos.th() - wm.self().body();
            
            if( fabs( angle_dif.degree() ) > 5 )
            {
                agent->doTurn( angle_dif );
                agent->debugClient().addMessage( "NoWaitTurn" );
                return true;        
            }
            else
            {
                rcsc::Vector2D accel = rpos - wm.self().vel();
                double dash_power = accel.r() / wm.self().dashRate();
                if( dash_power > 100 )
                {
                    dash_power = 100;
                }
                agent->doDash( dash_power );
                agent->debugClient().addMessage( "NoWaitDash" );
                return true;
            }

            
        }
        ///////////////////
        rcsc::Vector2D target_pos( 51, 0 );
        rcsc::Vector2D rpos = target_pos - next_self_pos;
	if( fabs( rpos.th().degree() - wm.self().body().degree() ) < 5.0 )
        {
	    double min_dist = 100000.0;
	    double opt_dir = -1000.0;
	    rcsc::Vector2D sim_next_pos;
	    rcsc::Vector2D sim_next_pos_final;
	    double effort = wm.self().effort();
	    rcsc::Circle2D sim_nextself_area;
            double dist2bline = ball_line.dist( next_self_pos );

	    for( int i = 0 ; i < 8 ; i++ )  // clockwise. 0: back, 1:back-left,..., 7:back-right
            {
		double degree = ((double)i-4.0) * 45.0;
		rcsc::Vector2D move( 1.0, 0 );
		move.rotate( wm.self().body().degree() + degree );
		double dash_power;
		if( i == 0 )
                    dash_power = 25.0;
		else if( i == 1 || i == 7 )
                    dash_power = 37.75;
		else if( i == 2 || i == 6 )
                    dash_power = 50.0;
		else if( i == 3 || i == 5 )
                    dash_power = 75.0;
		else if( i == 4 )
                    dash_power = 100.0;

		move.setLength( effort * wm.self().playerType().dashPowerRate() * dash_power );
		move += wm.self().vel();
		if( move.length() > wm.self().playerType().playerSpeedMax() )
                    move.setLength( wm.self().playerType().playerSpeedMax() );
		sim_next_pos = wm.self().pos() + move;
		sim_nextself_area.assign( sim_next_pos, wm.self().kickableArea() * 0.8 );
		// condition: get closer after the move
		if( next_self_pos.dist( wm.ball().pos() + wm.ball().vel() )
                    < sim_next_pos.dist( wm.ball().pos() + wm.ball().vel() ) )
                    continue;
		// condition: intersect with ball_ray
		if( sim_nextself_area.intersection( ball_ray, NULL, NULL ) != 2 )
                    continue;
                // condition: should not be further then current situation to the ball line
                if( ball_line.dist( sim_next_pos ) > dist2bline )
                    continue;

		if( sim_next_pos.dist( wm.ball().pos() + wm.ball().vel() ) < min_dist )
                {
		    sim_next_pos_final = sim_next_pos;
		    min_dist = sim_next_pos.dist( wm.ball().pos() + wm.ball().vel() );
		    opt_dir = degree;
                }
            }

	    if( opt_dir != -1000.0 )
            {
		const rcsc::AngleDeg dashDir( opt_dir );
		agent->doDash( 100.0, dashDir );
		agent->debugClient().addCircle( sim_next_pos_final, 0.1 );
		agent->debugClient().addMessage( "Approach%f", opt_dir );
		return true;
            }
        }//end if degree
        
        agent->doTurn( rpos.th() - wm.self().body() );
	agent->debugClient().addMessage( "WaitTurn" );
	return true;
    }

/////////////////////////////////////////////////////////
    int ball_stop_cycle = 0;
    double speed = wm.ball().vel().r();
    while( speed > 0.2 )
    {
        ball_stop_cycle++;
        speed *= sp.ballDecay();
    }
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: Ball will stop after %d cycles"
                        ,__FILE__, __LINE__, ball_stop_cycle );
    if( ball_stop_cycle < self_cycle )
    {
        rcsc::Vector2D target_pos = wm.ball().inertiaPoint( ball_stop_cycle );
        rcsc::Vector2D rpos = target_pos - wm.self().inertiaPoint( ball_stop_cycle );
        rcsc::AngleDeg angle_dif = rpos.th() - wm.self().body();
        agent->debugClient().setTarget( target_pos );
        if( angle_dif.abs() > 5 )
        {
            agent->doTurn( angle_dif );
            agent->debugClient().addMessage( "BallStopTurn" );
            return true;
        }
        else
        {
            double dash_dist = target_pos.dist( wm.self().pos() ) - wm.self().playerType().playerSize() ;
            double dash_power = dash_dist / wm.self().dashRate();
            if( dash_power > sp.maxPower() )
            {
                dash_power = 100;
            }
            agent->debugClient().addMessage( "BallStopDash" );
            agent->doDash( dash_power );
            return true;
        }
        
    }

/////////////////////////////////////////////////////////
    
    if( !center_target.valid() )
    {
        if( left_target.valid() )
        {
            agent->debugClient().addLine( wm.self().pos(), left_target );
            rcsc::Vector2D dash_vector = left_target - wm.self().pos();
            dash_vector.setLength( dash_vector.r() - wm.self().playerType().playerSize() );
            rcsc::Vector2D accel = dash_vector - wm.self().vel();
            double dash_power = accel.r() / wm.self().dashRate();
            if( dash_power > 100 )
            {
                dash_power = 100;
            }
            agent->doDash( dash_power );
            agent->debugClient().addMessage( "DashL" );
            return true;
            
        }
        else if( right_target.valid() )
        {
            agent->debugClient().addLine( wm.self().pos(), right_target );
            rcsc::Vector2D dash_vector = right_target - wm.self().pos();
            dash_vector.setLength( dash_vector.r() - wm.self().playerType().playerSize() );
            rcsc::Vector2D accel = dash_vector - wm.self().vel();
            double dash_power = accel.r() / wm.self().dashRate();
            if( dash_power > 100 )
            {
                dash_power = 100;
            }
            agent->doDash( dash_power );
            agent->debugClient().addMessage( "DashR" );
            return true;
        }
        else
        {
            //turn
            inertial_ball_pos = wm.ball().inertiaPoint( self_cycle + 1 );
            rcsc::Vector2D rpos = inertial_ball_pos - inertial_self_pos;
            rcsc::AngleDeg angle_dif = rpos.th() - wm.self().body();
            agent->debugClient().addMessage( "TurnRay" );
            agent->doTurn( angle_dif );
            return true;
        }
    }
    else
    {
        if( !field.contains( center_target ) )
        {
            rcsc::Vector2D line_pos;
            if( !field.intersection( ball_ray, &line_pos, NULL ) == 1 )
            {
                line_pos = wm.ball().pos();
            }
            rcsc::Vector2D rpos = line_pos - next_self_pos;
            rcsc::AngleDeg angle_dif = rpos.th() - wm.self().body();
            if( angle_dif.abs() > 5 )
            {
                agent->doTurn( angle_dif );
                agent->debugClient().addMessage( "TurnFieldLine" );
                return true;
            }
            else
            {
                rcsc::Vector2D dash_vector = center_target - wm.self().pos();
                dash_vector.setLength( dash_vector.r() - wm.self().playerType().playerSize() );
                rcsc::Vector2D accel = dash_vector - wm.self().vel();
                double dash_power = accel.r() / wm.self().dashRate();
                if( dash_power > 100 )
                {
                    dash_power = 100;
                }
                agent->doDash( dash_power );
                agent->debugClient().addMessage( "FieldLineDash" );
                return true; 
            }
            
        }
        agent->debugClient().addLine( wm.self().pos(), center_target );
        rcsc::Vector2D dash_vector = center_target - wm.self().pos();
        dash_vector.setLength( dash_vector.r() - wm.self().playerType().playerSize() );
        rcsc::Vector2D accel = dash_vector - wm.self().vel();
        double dash_power = accel.r() / wm.self().dashRate();
        if( dash_power > 100 )
        {
            dash_power = 100;
        }
        agent->doDash( dash_power );
        agent->debugClient().addMessage( "DashC" );
        return true;
    }
    
    if( current_self_area.intersection( ball_line, NULL, NULL ) == 2 )
    {
        rcsc::AngleDeg angle_dif = wm.ball().vel().th() - wm.self().body();
        if( angle_dif.abs() > 5 )
        {
            agent->doTurn( angle_dif );
            agent->debugClient().addMessage( "TurnLine" );
            return true;
        }
        else
        {
            rcsc::Vector2D next_ball_pos = wm.ball().pos() + wm.ball().vel();
            rcsc::Vector2D dash_vector = next_ball_pos - wm.self().pos();
	    dash_vector.setLength( dash_vector.r() - wm.self().playerType().playerSize() );
            rcsc::Vector2D accel = dash_vector - wm.self().vel();
            double dash_power = accel.r() / wm.self().dashRate();
            if( dash_power > 100 )
            {
                dash_power = 100;
            }
            agent->doDash( dash_power );
            agent->debugClient().addMessage( "SDash" );
            return true;            
        }
    }    

    return false;
}
/*-------------------------------------------------------------------*/
/*!

 */
bool
Opuci_ChaseBall::approachWithinCycles( rcsc::PlayerAgent *agent, int limit, bool test )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: Opuci_ChaseBall::approachWithinCycles"
                        ,__FILE__, __LINE__ );

    const rcsc::WorldModel &wm = agent->world();
    const rcsc::ServerParam &sp = rcsc::ServerParam::i();

    rcsc::Vector2D sim_spos;
    rcsc::Vector2D sim_bpos;
    rcsc::Vector2D sim_vec;
    double effort;
    double stamina;
    double recovery;
    double dist_b2s;

    double min_dist = 10000.0;
    double min_degree = 10000.0;
    int min_i = 100;
    rcsc::Vector2D min_spos( 0.0, 0.0 );

    rcsc::Ray2D ball_ray( wm.ball().pos(), wm.ball().vel().th() );
    rcsc::Vector2D accel;
    double d_rate;

    if( ball_ray.line().dist( wm.self().pos() + wm.self().vel() ) < wm.self().kickableArea() * 0.9 )
    {
        if( test == false )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: Already on Bline. quit."
                                ,__FILE__, __LINE__ );
        }
        return false;
    }

    for( int dir = 0 ; dir < 8 ; dir++ )
    {
        double degree = ((double)dir-4.0) * 45.0;
        if( dir == 0 )
            d_rate = 25.0;
        else if( dir == 1 || dir == 7 )
            d_rate = 37.75;
        else if( dir == 2 || dir == 6 )
            d_rate = 50.0;
        else if( dir == 3 || dir == 5 )
            d_rate = 75.0;
        else if( dir == 4 )
            d_rate = 100.0;

        sim_spos = wm.self().pos();
        sim_vec = wm.self().vel();
        effort = wm.self().effort();
        stamina = wm.self().stamina();
        recovery = wm.self().recovery();

        dist_b2s = wm.self().pos().dist( wm.ball().pos() );

        for( int i = 1 ; i <= limit ; i++ )
        {
            // $B2CB.EY$N7hDj(B
            accel.assign( 1.0, 0 );
            accel.rotate( wm.self().body().degree() + degree );
            accel.setLength( std::min( sp.playerAccelMax(),
                                  effort * wm.self().playerType().dashPowerRate() * d_rate ) );

            // $B$3$NJ}8~$K(B i $B%5%$%/%kB3$1$F0\F0$7$?$"$H$N0LCV$r5a$a$k(B
            sim_vec += accel;
            if( sim_vec.r() > wm.self().playerType().playerSpeedMax() )
                sim_vec.setLength( wm.self().playerType().playerSpeedMax() );
            sim_spos += sim_vec;
            sim_bpos = wm.ball().inertiaPoint( i );

            // $B%\!<%k$,DL$j$9$.$k$h$&$G$"$l$P%k!<%W=*N;(B -> $B5wN%$,Bg$-$/$J$C$?$i=*N;(B
            if( sim_spos.dist( sim_bpos ) > wm.self().kickableArea()
                && sim_spos.dist( sim_bpos ) > dist_b2s )
                break;
            dist_b2s = sim_spos.dist( sim_bpos );

            // $B$b$7%\!<%k%i%$%s$K=E$J$k$J$i$P!$$3$N9TF0$r8uJd$H$7$F5-O?(B
            if( ball_ray.line().dist( sim_spos ) < wm.self().kickableArea() * 0.9 )
            {
                if( min_dist > ball_ray.line().dist( sim_spos ) )
                {
                    min_dist = ball_ray.line().dist( sim_spos );
                    min_degree = degree;
                    min_i = i;
                    min_spos = sim_spos;
                }
                break;
            }

            // $B%9%?%_%J8:>/(B
            stamina -= 100;

            // $B%9%?%_%JA}2C(B
            if( stamina <= sp.recoverDecThr() * sp.staminaMax() )
            {
                if( recovery > sp.recoverMin() )
                    recovery -= sp.recoverDec();
                if( recovery < sp.recoverMin() )
                    recovery = sp.recoverMin();
            }
            if( stamina <= sp.effortDecThr() * sp.staminaMax() )
            {
                if( effort > wm.self().playerType().effortMin() )
                    effort -= sp.effortDec();
                if( effort < wm.self().playerType().effortMin() )
                    effort = wm.self().playerType().effortMin();
            }
            if( stamina >= sp.effortIncThr() * sp.staminaMax() )
            {
                if( effort < wm.self().playerType().effortMin() )
                {
                    effort += sp.effortInc();
                    if( effort > wm.self().playerType().effortMin() )
                        effort = wm.self().playerType().effortMin();
                }
            }
            stamina += recovery * wm.self().playerType().staminaIncMax();
            
            // $B<!%5%$%/%kB.EY(B
            sim_vec *= wm.self().playerType().playerDecay();
        }
    }

    if( min_degree != 10000.0 )
    {
        const rcsc::AngleDeg dashDir( min_degree );
        if( test == false )
        {
            agent->doDash( 100.0, dashDir );
            agent->debugClient().addCircle( sim_spos, 0.1 );
            agent->debugClient().addMessage( "App2Ray%f", min_degree );
        }
        if( test == true )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: Testing mode of approachWithinCycle finished."
                                ,__FILE__, __LINE__, min_i );
        }
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: Reach Bline after %d cycle."
                            ,__FILE__, __LINE__, min_i );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: Expected distance at that cycle: %f"
                            ,__FILE__, __LINE__, min_dist );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: Expected self-position at that cycle: (%f, %f)"
                            ,__FILE__, __LINE__, min_spos.x, min_spos.y );
        return true;
    }
    else
        return false;
}


int 
Opuci_ChaseBall::checkStraight( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();
    double player_size = wm.self().kickableArea();
    rcsc::Line2D body_right_line( rcsc::Vector2D( wm.self().pos().x + player_size * wm.self().body().sin(),
                                                  wm.self().pos().y - player_size * wm.self().body().cos() ),
                                  wm.self().body() );
    rcsc::Line2D body_left_line( rcsc::Vector2D( wm.self().pos().x - player_size * wm.self().body().sin(),
                                                 wm.self().pos().y + player_size * wm.self().body().cos() ),
                                 wm.self().body() );
    
    double target_inside =
        ( body_right_line.getA() * wm.ball().pos().x + body_right_line.getB() * wm.ball().pos().y + body_right_line.getC() )
        * ( body_left_line.getA() * wm.ball().pos().x + body_left_line.getB() * wm.ball().pos().y + body_left_line.getC() );
    
    rcsc::Circle2D self_area( wm.self().pos(), wm.self().kickableArea() );
    rcsc::Vector2D tmp_self_pos = wm.self().pos();
    rcsc::Vector2D tmp_self_vel = wm.self().vel();
    rcsc::Vector2D straight_dash;
    rcsc::Vector2D tmp_ball_pos = wm.ball().pos();
    rcsc::Vector2D tmp_ball_vel = wm.ball().vel();
    
    straight_dash.setPolar( wm.self().dashRate() * 100, wm.self().body() );
    bool front = false;

    rcsc::Vector2D self_to_ball = wm.ball().pos() - wm.self().pos();
    rcsc::AngleDeg angle_dif = self_to_ball.th() - wm.self().body();
    if( target_inside < 0 
        && angle_dif.abs() < 90 ) 
    {
        front = true;
    }
    int tmp_dash_cycle = 1;
    while( front )
    {
        tmp_self_vel += straight_dash;
        tmp_self_pos += tmp_self_vel;
        tmp_ball_pos += tmp_ball_vel;

        self_area.assign( tmp_self_pos, wm.self().kickableArea() );
        if( self_area.contains( tmp_ball_pos ) )
        {
            agent->debugClient().addCircle( tmp_self_pos, wm.self().kickableArea() );
            agent->debugClient().addCircle( tmp_ball_pos, 0.5 );
            return tmp_dash_cycle;
        }

        body_right_line.assign( rcsc::Vector2D( tmp_self_pos.x + player_size * wm.self().body().sin(),
                                                tmp_self_pos.y - player_size * wm.self().body().cos() ),
                                wm.self().body() );
        body_left_line.assign( rcsc::Vector2D( tmp_self_pos.x - player_size * wm.self().body().sin(),
                                               tmp_self_pos.y + player_size * wm.self().body().cos() ),
                               wm.self().body() );
        target_inside =
            ( body_right_line.getA() * tmp_ball_pos.x + body_right_line.getB() * tmp_ball_pos.y + body_right_line.getC() )
            * ( body_left_line.getA() * tmp_ball_pos.x + body_left_line.getB() * tmp_ball_pos.y + body_left_line.getC() );
        rcsc::Vector2D rpos = tmp_ball_pos - tmp_self_pos;
        if( target_inside >= 0
            || fabs( ( rpos.th() - wm.self().body() ).degree() >= 90 ) )
        {
            front = false;
        }

        tmp_self_vel *= wm.self().playerTypePtr()->playerDecay();
        tmp_ball_pos *= sp.ballDecay();
        tmp_dash_cycle++;
        if( tmp_dash_cycle >= 100 )
        {
            front = false;
        }
    }    
    return 100;
}
