#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "opuci_dribble.h"
#include "opuci_intention_dribble.h"

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>

#include <rcsc/geom/angle_deg.h>

/*-------------------------------------------------------------------*/
/*!

 */
bool
Opuci_Dribble::execute( rcsc::PlayerAgent * agent )
{
  rcsc::dlog.addText( rcsc::Logger::KICK,
		      "%s:%d:Opuci_Dribble"
		      ,__FILE__, __LINE__ );

  const rcsc::WorldModel & wm = agent->world();
  const rcsc::ServerParam & SP = rcsc::ServerParam::i();
  const rcsc::PlayerType & PT = wm.self().playerType();

  const rcsc::Vector2D next_ball_pos = wm.ball().pos() + wm.ball().vel();
  const rcsc::Vector2D next_self_pos = wm.self().pos() + wm.self().vel();

  const rcsc::Line2D body_right_line( rcsc::Vector2D( wm.self().pos().x + PT.kickableArea() * wm.self().body().sin(),
						      wm.self().pos().y - PT.kickableArea() * wm.self().body().cos() ),
				      wm.self().body() );
  const rcsc::Line2D body_left_line( rcsc::Vector2D( wm.self().pos().x - PT.kickableArea() * wm.self().body().sin(),
						     wm.self().pos().y + PT.kickableArea() * wm.self().body().cos() ),
				     wm.self().body() );

  const double target_inside =
    ( body_right_line.getA() * M_target_point.x + body_right_line.getB() * M_target_point.y + body_right_line.getC() )
    * ( body_left_line.getA() * M_target_point.x + body_left_line.getB() * M_target_point.y + body_left_line.getC() );

  if( fabs( ( ( wm.ball().pos() - wm.self().pos() ).th() 
	      - rcsc::Vector2D( M_target_point - wm.self().pos() ).th() ).degree() ) > 90.0 )
    {
      const rcsc::Vector2D kick_target = next_self_pos 
	+ rcsc::Vector2D( ( PT.playerSize() + SP.ballSize() * 2.0 ) * ( M_target_point - next_self_pos ).th().cos(),
			  ( PT.playerSize() + SP.ballSize() * 2.0 ) * ( M_target_point - next_self_pos ).th().sin() );
      const rcsc::Vector2D kick_accel = ( kick_target - wm.ball().pos() ) - wm.ball().vel();
      const double kick_power = std::min( kick_accel.r() / wm.self().kickRate(), SP.maxPower() );

      agent->doKick( kick_power, kick_accel.th() - wm.self().body() );
      agent->debugClient().setTarget( kick_target );
      agent->debugClient().addMessage( "opuciDribbleFirstKick" );
    }
  else if( next_self_pos.dist( next_ball_pos ) < PT.kickableArea() * 0.8
      && ( target_inside > 0.0
	   || fabs( ( wm.self().body() - rcsc::Vector2D( M_target_point - wm.self().pos() ).th() ).degree() ) > 90.0 ) )
    {
      agent->doTurn( ( M_target_point - wm.self().pos() ).th() - wm.self().body() );
      agent->debugClient().addMessage( "opuciDribbleTurn" );
    }
  else
    {
      double self_stamina = wm.self().stamina();
      const double self_stamina_inc = wm.self().playerType().staminaIncMax() * wm.self().recovery();
      double power = SP.maxPower();
      double dash_power = wm.self().effort() * PT.dashPowerRate() * power;
      rcsc::Vector2D dash_dist( dash_power * ( M_target_point - wm.self().pos() ).th().cos(),
				dash_power * ( M_target_point - wm.self().pos() ).th().sin() );
      rcsc::Vector2D self_pos = wm.self().pos();
      rcsc::Vector2D self_vel = wm.self().vel();
      rcsc::Vector2D after_dash_self_pos;

      double ball_decay = 1.0;

      for( int i = 1; i <= M_dash_count; i++ )
	{
	  after_dash_self_pos = self_pos + self_vel + dash_dist;
	  self_vel = ( after_dash_self_pos - self_pos ) * PT.playerDecay();
	  self_pos = after_dash_self_pos;

	  self_stamina += self_stamina_inc - power;
	  if( self_stamina < SP.staminaMax() * SP.effortDecThr() * 1.2
	      || self_stamina < SP.staminaMax() * SP.recoverDecThr() * 1.2 )
	    {
	      power = 0.0;
	    }
	  else
	    {
	      power = SP.maxPower();
	    }
	  dash_power = wm.self().effort() * PT.dashPowerRate() * power;
	  dash_dist.assign( dash_power * ( M_target_point - wm.self().pos() ).th().cos(),
			    dash_power * ( M_target_point - wm.self().pos() ).th().sin() );

	  ball_decay += pow( SP.ballDecay(), i );
	}

      const rcsc::Vector2D dribble_target = after_dash_self_pos 
	+ rcsc::Vector2D( ( PT.playerSize() + SP.ballSize() * 2.0 ) * wm.self().body().cos(),
			  ( PT.playerSize() + SP.ballSize() * 2.0 ) * wm.self().body().sin() );

      const rcsc::Vector2D kick_target = ( dribble_target - wm.ball().pos() ) / ball_decay;
      const rcsc::Vector2D kick_accel = kick_target - wm.ball().vel();
      const double kick_power = std::min( kick_accel.r() / wm.self().kickRate(), SP.maxPower() );

      agent->doKick( kick_power, kick_accel.th() - wm.self().body() );
      agent->setIntention( new Opuci_IntentionDribble( M_target_point, M_dash_count ) );
      agent->debugClient().addMessage( "opuciDribbleKick" );
      agent->debugClient().setTarget( dribble_target );
    }

  return true;
}
