#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "opuci_through_pass.h"
#include "opuci_one_step_kick.h"
#include "bhv_offense.h"
#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>

#include <rcsc/geom/vector_2d.h>
#include <rcsc/geom/ray_2d.h>
#include <rcsc/player/say_message_builder.h>
#include <rcsc/action/body_smart_kick.h>
/*-------------------------------------------------------------------*/
/*!

 */
/*-------------------------------------------------------------------*/
bool
Opuci_ThroughPass::execute( rcsc::PlayerAgent * agent )
{
    
    const rcsc::WorldModel & wm = agent->world();
    if( !wm.self().isKickable() )
    {
        return false;
    }
    rcsc::Vector2D pass_target;
    double first_speed;
    rcsc::PlayerPtrCont::const_iterator end = wm.teammatesFromSelf().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.teammatesFromSelf().begin();
         it != end;
         ++it )
    {
        const rcsc::PlayerObject * receiver = (*it);
        if( search( agent, receiver, pass_target, first_speed ) )
        {
            agent->debugClient().setTarget( pass_target );
            if( rcsc::Body_SmartKick( pass_target, first_speed, first_speed * 0.8, 3 ).execute( agent ) )
            {
                agent->debugClient().addMessage( "Target%d", receiver->unum() );
                agent->addSayMessage( new rcsc::PassMessage( receiver->unum(), pass_target,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
            }  
            return true;
        }
    }
    
    
    
    return false;
}
/*-------------------------------------------------------------------*/
/*!

 */
/*-------------------------------------------------------------------*/
bool 
Opuci_ThroughPass::search( rcsc::PlayerAgent * agent, const rcsc::PlayerObject * receiver, 
                           rcsc::Vector2D & target_pos, double & kick_power, rcsc::AngleDeg & kick_angle )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();



    std::vector< rcsc::PlayerObject* > opponents;
    const rcsc::PlayerPtrCont::const_iterator opp_end = wm.opponentsFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
         it != opp_end;
         ++it )
    {
        if( (*it)->isTackling() )
            continue;

        if( (*it)->pos().x > wm.self().pos().x )
        {
            opponents.push_back( (*it) );
        }
    }

    if( opponents.empty() )
    {
        return false;
    }

    for( std::vector< rcsc::PlayerObject* >::iterator it1 = opponents.begin();
         it1 != opponents.end();
         ++it1 )
    {
        for( std::vector< rcsc::PlayerObject* >::iterator it2 = it1;
             it2 != opponents.end();
             ++it2 )
        {
            if( (*it2)->pos().y < (*it1)->pos().y )
            {
                swap_ranges( it1, it1 + 1, it2 );
            }
        }
    }    
    
    std::vector< rcsc::Vector2D > middle_pos;
    if( opponents.size() == 1 )
    {
        return false;
    }
    else
    {
        for( int i = 0; i < (int)opponents.size() - 1; i++ )
        {
            rcsc::Vector2D mpos = ( opponents.at( i )->pos() + opponents.at( i + 1 )->pos() ) / 2.0;
            middle_pos.push_back( mpos );
        }
    }
    rcsc::Vector2D tmp_target;    
    int cycle_margin = 0;
    double tmp_power = 0;
    rcsc::AngleDeg tmp_angle = 0;
    for( int i = 0; i < (int)middle_pos.size(); i++ )
    {
        double opp_dist = 0;
        int opp_dash_cycle = 2;
        int opp_turn_cycle = 2;
        const rcsc::PlayerObject * nearest_opp = wm.getOpponentNearestTo( middle_pos.at( i ), 10, &opp_dist );
        if( nearest_opp )
        {
            opp_dash_cycle = nearest_opp->playerTypePtr()->cyclesToReachDistance( opp_dist );
        }
        int opp_reach_cycle = opp_dash_cycle + opp_turn_cycle;
        

        rcsc::Vector2D ball_to_middle = middle_pos.at( i ) - wm.ball().pos();
        double ball_dist = middle_pos.at( i ).dist( wm.ball().pos() );
        double ball_first_speed = rcsc::calc_first_term_geom_series( ball_dist, sp.ballDecay(), opp_reach_cycle - 1 );
        
        if( ball_first_speed > sp.ballSpeedMax() )
        {
            continue;
        }

        rcsc::AngleDeg angle = ball_to_middle.th();        
        rcsc::Vector2D ball_kick( ball_first_speed, 0 );
        ball_kick.rotate( angle );
        rcsc::Vector2D accel = ball_kick - wm.ball().vel();
        double power = accel.r() / wm.self().kickRate();

        if( power > sp.maxPower() )
        {
            continue;
        }
        for( int p = 0; p < 10; p++ )
        {
            bool finish = false;
            power += p * 10;
            if( power > sp.maxPower() )
            {
                finish = true;
                power = sp.maxPower();
            }
            int tmp_cycle = check( agent, receiver, power, angle, tmp_target );
            if( tmp_target.x < receiver->pos().x )
            {
                continue;
            }
            if( tmp_cycle > cycle_margin )
            {
                cycle_margin = tmp_cycle;
                tmp_power = power;
                tmp_angle = angle;
            }
            if( finish )
            {
                break;
            }
        }

    }//end for middle_pos
    
    if( cycle_margin > 0 )
    {
        agent->debugClient().addMessage( "CanThroughPass" );
        target_pos = tmp_target;
        kick_power = tmp_power;
        kick_angle = tmp_angle - wm.self().body();
        return true;
    }
    
    return false;
}
/*-------------------------------------------------------------------*/
/*!

 */
/*-------------------------------------------------------------------*/
int 
Opuci_ThroughPass::check( rcsc::PlayerAgent * agent, const rcsc::PlayerObject * receiver,
                          double power, rcsc::AngleDeg angle, rcsc::Vector2D & tmp_receive_pos )
{
    int cycle_margin = 0;
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    int inertial_cycle = 1;
    rcsc::Vector2D inertial_self_pos = wm.self().inertiaPoint( inertial_cycle );
    rcsc::Vector2D inertial_mate_pos = receiver->playerTypePtr()->inertiaPoint( receiver->pos(), receiver->vel(), inertial_cycle );
    rcsc::Ray2D ball_ray( wm.ball().pos(), angle );
    rcsc::Ray2D mate_ray( inertial_mate_pos, rcsc::AngleDeg( 0 ) );
    rcsc::Vector2D receive_pos = ball_ray.intersection( mate_ray );
    if( !receive_pos.valid() )
    {
        return -1;
    }



    double tmp_first_speed = power * wm.self().kickRate();
    rcsc::Vector2D kick_vector( tmp_first_speed, 0 );
    kick_vector.rotate( angle );
    int max_ball_cycle = rcsc::calc_length_geom_series( tmp_first_speed, 
                                                        receive_pos.dist( wm.ball().pos() ),
                                                        sp.ballDecay() );
        
    bool success = false;
    int opp_min_cycle = 100;
        
    tmp_receive_pos = wm.ball().pos();
    for( int j = 2; j <= max_ball_cycle + 2; j++ )
    {
        tmp_receive_pos += kick_vector;
            
        double mate_dist = tmp_receive_pos.dist( inertial_mate_pos );
        int mate_turn_cycle = 1;
        int mate_dash_cycle = receiver->playerTypePtr()->cyclesToReachDistance( mate_dist );
        if( mate_dash_cycle <= 0 )
        {
            mate_dash_cycle = 1;
        }
        int mate_reach_cycle = mate_turn_cycle + mate_dash_cycle;
            
        rcsc::Segment2D pass_seg( wm.ball().pos(), tmp_receive_pos );
            
        if( mate_reach_cycle < j )
        {
            success = true;
            const rcsc::PlayerPtrCont::const_iterator opp_end = wm.opponentsFromBall().end();
            for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
                 it != opp_end;
                 ++it )
            {
                if( (*it)->isTackling() )
                    continue;

                rcsc::Vector2D inertial_opp_pos = (*it)->playerTypePtr()->inertiaPoint( (*it)->pos(), (*it)->vel(), inertial_cycle );
                rcsc::Circle2D opp_area( inertial_opp_pos, (*it)->playerTypePtr()->kickableMargin() );
                rcsc::Circle2D opp_area2( (*it)->pos(), (*it)->playerTypePtr()->kickableMargin() );

                if( opp_area.intersection( pass_seg, NULL, NULL ) > 0 
                    || opp_area.intersection( pass_seg, NULL, NULL ) > 0 )
                {
                    success = false;
                    break;
                }
                double tmp_opp_dist = pass_seg.dist( inertial_opp_pos );
                tmp_opp_dist -= (*it)->playerTypePtr()->kickableMargin();
                if( tmp_opp_dist <= 0 )
                {
                    success = false;
                    break;
                }
                int tmp_opp_dash = (*it)->playerTypePtr()->cyclesToReachDistance( tmp_opp_dist );
                int tmp_opp_turn = 2;
                int tmp_opp_cycle = tmp_opp_dash + tmp_opp_turn;
                    
                rcsc::Vector2D opp_target = pass_seg.nearestPoint( inertial_opp_pos );
                double tmp_ball_dist = wm.ball().pos().dist( opp_target );
                int tmp_ball_cycle = rcsc::calc_length_geom_series( tmp_first_speed, tmp_ball_dist, sp.ballDecay() );

                if( tmp_ball_cycle <= 0 
                    || tmp_ball_cycle >= tmp_opp_cycle )
                {
                    success = false;
                    break;
                }
                else if( tmp_opp_cycle < opp_min_cycle )
                {
                    opp_min_cycle = tmp_opp_cycle;
                }
            }
            if( success && opp_min_cycle > cycle_margin )
            {
                cycle_margin = opp_min_cycle;
            }
        }//end if mate_reach_cycle
        kick_vector *= sp.ballDecay();
    }//end for ball cycle    
    return cycle_margin;
}

/*-------------------------------------------------------------------*/
/*!

 */
/*-------------------------------------------------------------------*/

bool 
Opuci_ThroughPass::search( rcsc::PlayerAgent * agent, const rcsc::PlayerObject * receiver, 
                           rcsc::Vector2D & target_pos, double & first_speed )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    if( receiver->pos().x > wm.offsideLineX() )
    {
        return false;
    }
/*
    std::vector< rcsc::PlayerObject* > opponents;
    const rcsc::PlayerPtrCont::const_iterator opp_end = wm.opponentsFromBall().end();
    for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
         it != opp_end;
         ++it )
    {
        if( (*it)->isTackling() )
            continue;

        if( (*it)->pos().x > wm.self().pos().x )
        {
            opponents.push_back( (*it) );
        }
    }

    if( opponents.empty() )
    {
        return false;
    }

    for( std::vector< rcsc::PlayerObject* >::iterator it1 = opponents.begin();
         it1 != opponents.end();
         ++it1 )
    {
        for( std::vector< rcsc::PlayerObject* >::iterator it2 = it1;
             it2 != opponents.end();
             ++it2 )
        {
            if( (*it2)->pos().y < (*it1)->pos().y )
            {
                swap_ranges( it1, it1 + 1, it2 );
            }
        }
    }    
    
    std::vector< rcsc::Vector2D > middle_pos;
    if( opponents.size() == 1 )
    {
        return false;
    }
    else
    {
        for( int i = 0; i < (int)opponents.size() - 1; i++ )
        {
            rcsc::Vector2D mpos = ( opponents.at( i )->pos() + opponents.at( i + 1 )->pos() ) / 2.0;
            middle_pos.push_back( mpos );
            double x = std::max( opponents.at( i )->pos().x, opponents.at( i + 1 )->pos().x );
            double y = mpos.y;
            middle_pos.push_back( rcsc::Vector2D( x, y ) );
        }
    }
*/
    std::vector< rcsc::Vector2D > oppline;
    Bhv_Offense().getOppEndLine( agent, oppline );
    if( oppline.size() < 2 )
    {
        return false;
    }
    std::vector< rcsc::Vector2D > middle_pos;
    for( int i = 1; i < (int)oppline.size(); i++ )
    {
        rcsc::Vector2D mpos = ( oppline.at( i - 1 ) + oppline.at( i ) ) / 2.0;
        middle_pos.push_back( mpos );
        double x = wm.offsideLineX();
        double y = mpos.y;
        middle_pos.push_back( rcsc::Vector2D( x, y ) );
    }
    rcsc::Vector2D last_target = receiver->pos();
    int cycle_margin = 0;
    double tmp_first_speed = 0;

    for( int i = 0; i < (int)middle_pos.size(); i++ )
    {
        double opp_dist = 0;
        int opp_dash_cycle = 2;
        int opp_turn_cycle = 2;
        const rcsc::PlayerObject * nearest_opp = wm.getOpponentNearestTo( middle_pos.at( i ), 10, &opp_dist );
        if( nearest_opp )
        {
            opp_dash_cycle = nearest_opp->playerTypePtr()->cyclesToReachDistance( opp_dist );
        }
        int opp_reach_cycle = opp_dash_cycle + opp_turn_cycle;
        

        rcsc::Vector2D ball_to_middle = middle_pos.at( i ) - wm.ball().pos();
        double ball_dist = middle_pos.at( i ).dist( wm.ball().pos() );
        double ball_first_speed = rcsc::calc_first_term_geom_series( ball_dist, sp.ballDecay(), opp_reach_cycle - 1 );
        
        if( ball_first_speed > sp.ballSpeedMax() )
        {
            continue;
        }
        rcsc::Vector2D tmp_target;
        double rest = sp.ballSpeedMax() - ball_first_speed;
        for( int p = 0; p <= 10; p++ )
        {
            ball_first_speed += rest * p / 10.0;
            if( ball_first_speed > sp.ballSpeedMax() )
            {
                ball_first_speed = sp.ballSpeedMax();
            }

            int tmp_cycle = check( agent, receiver, ball_to_middle.th(), ball_first_speed, tmp_target );
            if( tmp_target.x < wm.offsideLineX() )
            {
                continue;
            }
            if( tmp_cycle >= cycle_margin )
            {
                cycle_margin = tmp_cycle;
            }
            if( tmp_target.x > last_target.x )
            {
                tmp_first_speed = ball_first_speed;
                last_target = tmp_target;
            }
        }

    }//end for middle_pos
    
    if( cycle_margin > 0 )
    {
        target_pos = last_target;
        first_speed = tmp_first_speed;
        agent->debugClient().addMessage( "ThroughPass%.1f", first_speed );
        return true;
    }
    
    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
/*-------------------------------------------------------------------*/

int 
Opuci_ThroughPass::check( rcsc::PlayerAgent * agent, const rcsc::PlayerObject * receiver,
                          rcsc::AngleDeg angle, double tmp_first_speed, rcsc::Vector2D & last_receive_pos )
{
    int cycle_margin = 0;
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    int inertial_cycle = 1;
    rcsc::Vector2D inertial_self_pos = wm.self().inertiaPoint( inertial_cycle );
    rcsc::Vector2D inertial_mate_pos = receiver->playerTypePtr()->inertiaPoint( receiver->pos(), receiver->vel(), inertial_cycle );
    rcsc::Ray2D ball_ray( wm.ball().pos(), angle );
    rcsc::Ray2D mate_ray( inertial_mate_pos, rcsc::AngleDeg( 0 ) );
    rcsc::Vector2D receive_pos = ball_ray.intersection( mate_ray );
    if( !receive_pos.valid() )
    {
        return -1;
    }

    if( receive_pos.x > 50 
        || receive_pos.absY() > 30 )
    {
        return -1;
    }

    agent->debugClient().addLine( receive_pos, wm.ball().pos() );
    rcsc::Vector2D kick_vector( tmp_first_speed, 0 );
    kick_vector.rotate( angle );
    int max_ball_cycle = rcsc::calc_length_geom_series( tmp_first_speed, 
                                                        receive_pos.dist( wm.ball().pos() ),
                                                        sp.ballDecay() );
    

    int opp_min_cycle = 100;
    rcsc::Vector2D tmp_receive_pos = wm.ball().pos();
    for( int j = 1; j <= max_ball_cycle + 5; j++ )
    {
        bool success = false;
        tmp_receive_pos += kick_vector;
            
        double mate_dist = tmp_receive_pos.dist( inertial_mate_pos );
        mate_dist -= receiver->playerTypePtr()->kickableMargin();
        int mate_turn_cycle = 1;
        int mate_dash_cycle = receiver->playerTypePtr()->cyclesToReachDistance( mate_dist );
        if( mate_dash_cycle <= 0 )
        {
            mate_dash_cycle = 1;
        }
        int mate_reach_cycle = mate_turn_cycle + mate_dash_cycle;
            
        rcsc::Segment2D pass_seg( wm.ball().pos(), tmp_receive_pos );
            
        if( mate_reach_cycle <= j )
        {
            success = true;
            const rcsc::PlayerPtrCont::const_iterator opp_end = wm.opponentsFromBall().end();
            for( rcsc::PlayerPtrCont::const_iterator it = wm.opponentsFromBall().begin();
                 it != opp_end;
                 ++it )
            {
                if( (*it)->isTackling() )
                    continue;

                rcsc::Vector2D inertial_opp_pos = (*it)->playerTypePtr()->inertiaPoint( (*it)->pos(), (*it)->vel(), inertial_cycle );
                rcsc::Circle2D opp_area( inertial_opp_pos, (*it)->playerTypePtr()->kickableMargin() );
                rcsc::Circle2D opp_area2( (*it)->pos(), (*it)->playerTypePtr()->kickableMargin() );

                if( opp_area.intersection( pass_seg, NULL, NULL ) > 0 
                    || opp_area.intersection( pass_seg, NULL, NULL ) > 0 )
                {
                    success = false;
                    break;
                }
                double tmp_opp_dist = pass_seg.dist( inertial_opp_pos );
                tmp_opp_dist -= (*it)->playerTypePtr()->kickableMargin();
                if( tmp_opp_dist <= 0 )
                {
                    success = false;
                    break;
                }
                int tmp_opp_dash = (*it)->playerTypePtr()->cyclesToReachDistance( tmp_opp_dist );
                int tmp_opp_turn = 1;
                int tmp_opp_cycle = tmp_opp_dash + tmp_opp_turn;
                if( tmp_opp_cycle < j 
                    || tmp_opp_cycle < mate_reach_cycle )
                {
                    success = false;
                    break;
                }
/*                    
                rcsc::Vector2D opp_target = pass_seg.nearestPoint( inertial_opp_pos );
                double tmp_ball_dist = wm.ball().pos().dist( opp_target );
                int tmp_ball_cycle = rcsc::calc_length_geom_series( tmp_first_speed, tmp_ball_dist, sp.ballDecay() );

                if( tmp_ball_cycle <= 0 
                    || tmp_ball_cycle > tmp_opp_cycle )
                {
                    success = false;
                    break;
                }
*/
                if( tmp_opp_cycle < opp_min_cycle )
                {
                    opp_min_cycle = tmp_opp_cycle;
                }
            }
            if( success && tmp_receive_pos.x > last_receive_pos.x )
            {
                cycle_margin = opp_min_cycle;
                last_receive_pos = tmp_receive_pos;
            }
        }//end if mate_reach_cycle
        kick_vector *= sp.ballDecay();
    }//end for ball cycle    
/*
    if( cycle_margin == 0 )
    {
        double mate_dist = receiver->pos().dist( receive_pos );
        const rcsc::PlayerObject * nearest_opp = wm.getOpponentNearestTo( receiver, 20, NULL );
        if( nearest_opp )
        {
            double opp_dist = nearest_opp->pos().dist( receive_pos );
            int mate_cycle = receiver->playerTypePtr()->cyclesToReachDistance( mate_dist );
            int opp_cycle =  nearest_opp->playerTypePtr()->cyclesToReachDistance( opp_dist );
            return mate_cycle - opp_cycle;
        }
        
        
    }
*/
    return cycle_margin;
}

/*-------------------------------------------------------------------*/
/*!

 */
/*-------------------------------------------------------------------*/
